//=================================================================
// console.cs
//=================================================================
// PowerSDR is a C# implementation of a Software Defined Radio.
// Copyright (C) 2004, 2005  FlexRadio Systems
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// You may contact us via email at: sales@flex-radio.com.
// Paper mail may be sent to: 
//    FlexRadio Systems, 8900 Marybank Dr., Austin, TX  78750, USA.
//=================================================================

#define NEW_DTTSP
#define DBG_PRINT

using System;
using System.Collections;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Globalization;
using System.IO;
using System.Net;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Threading;
using System.Text;
using System.Windows.Forms;
using SDRSerialSupport; 

namespace PowerSDR
{
	#region Enums

	public enum BPFBand
	{
		NONE = -1,
		B160,
		B60,
		B20,
		B40,		
		B10,
		B6,
	}

	public enum RFELPFBand
	{
		NONE = -1,
		AUX,
		B6,
		B60,
		B20,
		B30,
		B40,
		B1210,
		B80,
		B1715,
		B160,
	}

	public enum PAFBand
	{
		NONE = 0,
		B1210,
		B1715,
		B3020,
		B6040,
		B80,
		B160,
	}

	public enum XVTRTRMode
	{
		POSITIVE = 0,	// DEMI144-28FRS
		NEGATIVE,		// 25W DEMI
		NONE,			// For Elecraft or similar XVTR
	}

	public enum SoundCard
	{
		FIRST = -1,
		SANTA_CRUZ,
		AUDIGY,
		AUDIGY_2,
		AUDIGY_2_ZS,
		EXTIGY,
		MP3_PLUS,
		DELTA_44,
		UNSUPPORTED_CARD,
		LAST,
	}

	public enum DisplayMode
	{
		FIRST = -1,
		SPECTRUM,
		PANADAPTER,
		SCOPE,
		PHASE,
		PHASE2,
		WATERFALL,
		HISTOGRAM,
		OFF,
		LAST,
	}

	public enum AGCMode
	{
		FIRST = -1,
		FIXD,
		LONG,
		SLOW,
		MED,
		FAST,
		LAST,
	}

	public enum MeterRXMode
	{
		FIRST = -1,
		SIGNAL_STRENGTH,
		SIGNAL_AVERAGE,
		ADC_L,
		ADC_R,
		LAST,
	}

	public enum MeterTXMode
	{
		FIRST = -1,
		ALC,
		FORWARD_POWER,
		PEAK_POWER,
		REVERSE_POWER,
		SWR,
		LAST,
	}

	public enum BandPlan
	{
		IARU1 = 1,
		IARU2 = 2,
		IARU3 = 3,
	}

	public enum PreampMode
	{
		FIRST = -1,
		OFF,
		LOW,
		MED,
		HIGH,
		LAST,
	}

	public enum StatusPin
	{
		PIN_12	= 0x08,		//S3
		Dash	= 0x10,		//S4
		Dot		= 0x20,		//S5
		PA_DATA	= 0x40,		//S6
		PIN_11	= 0x80,		//S7\
	}

	public enum ATUTuneMode
	{
		FIRST = -1,
		BYPASS,
		MEMORY,
		FULL,
		LAST,
	}

	public enum DSPMode
	{
		FIRST = -1,
		LSB,
		USB,
		DSB,
		CWL,
		CWU,
		FMN,
		AM,
		PSK,
		SPEC,
		RTTY,
		SAM,
		DRM,
		LAST,
	}

	public enum Band
	{
		FIRST = -1,
		GEN,
		B160M,
		B80M,
		B60M,
		B40M,
		B30M,
		B20M,
		B17M,
		B15M,
		B12M,
		B10M,
		B6M,
		B2M,
		WWV,
		LAST,
	}

	public enum Filter
	{
		FIRST = -1,
		F6000,
		F4000,
		F2600,
		F2100,
		F1000,
		F500,
		F250,
		F100,
		F50,
		F25,
		VAR1,
		VAR2,
		NONE,
		LAST,
	}

	#endregion

	unsafe public class Console : System.Windows.Forms.Form
	{
		#region Variable Declarations
		// ======================================================
		// Variable Declarations
		// ======================================================

		private int main_volume;
		private int wave_volume;

		private const float CLEAR_FLAG = -999.999F;				// for resetting the average buffer

		private PA19.PaStreamCallback callback1;				// audio callback for regular RX/TX
		private PA19.PaStreamCallback callback2;				// audio callback for second sound card
		private PA19.PaStreamCallback callback4port;			// audio callback for 4port cards

		// BT 04/2005 Added serial IO event
		public event SIOEventHandler NotifySIO;
		public SIOListener siolisten = null; 

		private Thread audio_process_thread;				// fields calls to DttSP functions
		private Thread draw_display_thread;					// draws the main display 
		private Thread multimeter_thread;					// draws/updates the multimeter
		private Thread poll_ptt_thread;						// polls the PTT line on the parallel port
		private Thread poll_pa_pwr_thread;					// polls the FWD and REV power if the PA is installed

		private Mutex pa_power_mutex;
		private Mutex high_swr_mutex;

		public Setup SetupForm;
		public CW CWForm;
		public Memory MemForm;
		public HW hw;										// will eventually be an array of rigs to support multiple radios
		public WaveControl WaveForm;
		public PAQualify PAQualForm;

		private bool run_setup_wizard;						// Used to run the wizard the first time the software comes up
	
		private Mutex display_image_mutex;					// used to lock the display image while editting it
		private Mutex background_image_mutex;				// used to lock the base display image

		private int band_160m_index;						// These band indexes are used to keep track of which
		private int band_80m_index;							// location in the bandstack was last saved/recalled
		private int band_60m_index;
		private int band_40m_index;
		private int band_30m_index;
		private int band_20m_index;
		private int band_17m_index;
		private int band_15m_index;
		private int band_12m_index;
		private int band_10m_index;
		private int band_6m_index;
		private int band_2m_index;
		private int band_wwv_index;
		private int band_gen_index;

		private int band_160m_register;						// These integers are the number of band stack registers
		private int band_80m_register;						// found in the database for each band
		private int band_60m_register;
		private int band_40m_register;
		private int band_30m_register;
		private int band_20m_register;
		private int band_17m_register;
		private int band_15m_register;
		private int band_12m_register;
		private int band_10m_register;
		private int band_6m_register;
		private int band_2m_register;
		private int band_wwv_register;
		private int band_gen_register;

		private double[] wheel_tune_list;					// A list of available tuning steps
		private int wheel_tune_index;						// An index into the above array

		private float display_max_x;						// x-coord of maxmimum over one display pass							
		private float display_max_y;						// y-coord of maxmimum over one display pass
		private int display_cursor_x;						// x-coord of the cursor when over the display
		private int display_cursor_y;						// y-coord of the cursor when over the display
		private Graphics display_graphics;					// graphics object used to draw the bitmap

		private float[] preamp_offset;						// offset values for each preamp mode in dB
		private float display_cal_offset;					// display calibration offset per volume setting in dB
		private float multimeter_cal_offset;				// multimeter calibration offset per volume setting in dB

		private int meter_peak_count;						// Counter for peak hold on multimeter
		private int meter_peak_value;						// Value for peak hold on multimeter
		private float[] meter_text_history;					// Array used to output the peak power over a period of time
		private int meter_text_history_index;				// index used with above variable to do peak power

		private int pa_fwd_power;							// forward power as read by the ADC on the PA
		private int pa_rev_power;							// reverse power as read by the ADC on the PA
		private bool atu_tuning;							// true while the atu is tuning
		private Band tuned_band;							// last band that the atu was tuned on
	
		private bool shift_down;							// used to modify tuning rate
		private bool calibrating;							// true if running a calibration routine
		private bool manual_mox;							// True if the MOX button was clicked on (not PTT)
		private bool long_crosshair;						// If true, the long yellow cursors are displayed
		private bool average_display;						// True if the Average button is pressed
		private float[] average_buffer;						// Averaged display data buffer
		private int[] histogram_data;						// histogram display buffer
		private int[] histogram_history;					// histogram counter

		private DSPMode vfob_dsp_mode;						// Saves control pointer for last mode used on VFO B 
		private Filter vfob_filter;							// Saves control pointer for last filter used on VFO B
		private int vfo_pixel_offset;						// Used to calibrate mousewheel tuning
		private int vfo_char_width;							// Used to calibrate mousewheel tuning
		private int vfo_decimal_width;						// Used to calibrate mousewheel tuning
		private int vfoa_hover_digit;						// Digit for hover display
		private int vfob_hover_digit;						// Digit for hover display
		private string last_band;							// Used in bandstacking algorithm

		private DSPMode quick_save_mode;					// Quick Save Mode
		private Filter quick_save_filter;					// Quick Save Filter

		private Filter[] last_filter;						// Last filter used indexed by mode
		private int[] var1_low;								// Saved variable filter 1 low cut indexed by mode
		private int[] var1_high;							// Saved variable filter 1 high cut indexed by mode
		private int[] var2_low;								// Saved variable filter 2 low cut indexed by mode
		private int[] var2_high;							// Saved variable filter 2 high cut indexed by mode

		private int waterfall_counter;
		private Bitmap display_bitmap;						// Bitmap for use when drawing
		private Bitmap waterfall_bmp;						// saved waterfall picture for display
		private Bitmap background_bmp;						// saved background picture for display
		private float[] display_data;						// Buffer used to draw the display

		private DisplayMode current_display_mode;			// selected display mode
		private string separator;							// contains the locations specific decimal separator

		private int last_filter_shift;						// VK6APH
		private int last_var1_shift;						// VK6APH 
		private int last_var2_shift;						// VK6APH
		private int pre_shift_filter_low;
		private int pre_shift_filter_high;

		private int[] sc2_cal;								// array for calibrating second sound card output
		public string[] CmdLineArgs;
		private CWKeyer2 Keyer;

		#region Windows Form Generated Code

		private System.Windows.Forms.MainMenu mainMenu1;
		private System.Windows.Forms.MenuItem menuItem1;
		private System.Windows.Forms.ButtonTS btnHidden;
		private System.Windows.Forms.TextBoxTS txtVFOAFreq;
		private System.Windows.Forms.TextBoxTS txtVFOABand;
		private System.Windows.Forms.TextBoxTS txtVFOBFreq;
		private System.Windows.Forms.PictureBox picDisplay;
		private System.Windows.Forms.GroupBoxTS grpVFOA;
		private System.Windows.Forms.GroupBoxTS grpVFOB;
		private System.Windows.Forms.TextBoxTS txtVFOBBand;
		private System.Windows.Forms.GroupBoxTS grpMode;
		private System.Windows.Forms.GroupBoxTS grpBand;
		private System.Windows.Forms.GroupBoxTS grpFilter;
		private System.Windows.Forms.GroupBoxTS grpDisplay;
		private System.Windows.Forms.ButtonTS btnBandGEN;
		private System.Windows.Forms.ButtonTS btnBandWWV;
		private System.Windows.Forms.ButtonTS btnBand2;
		private System.Windows.Forms.ButtonTS btnBand6;
		private System.Windows.Forms.ButtonTS btnBand10;
		private System.Windows.Forms.ButtonTS btnBand12;
		private System.Windows.Forms.ButtonTS btnBand15;
		private System.Windows.Forms.ButtonTS btnBand17;
		private System.Windows.Forms.ButtonTS btnBand20;
		private System.Windows.Forms.ButtonTS btnBand40;
		private System.Windows.Forms.ButtonTS btnBand60;
		private System.Windows.Forms.ButtonTS btnBand80;
		private System.Windows.Forms.ButtonTS btnBand160;
		private System.Windows.Forms.CheckBoxTS chkPower;
		private System.Windows.Forms.TextBoxTS txtDisplayCurPos;
		private System.Windows.Forms.ButtonTS btnBand13;
		private System.Windows.Forms.ButtonTS btnBand30;
		private System.Windows.Forms.LabelTS lblCPUMeter;
		private System.Windows.Forms.ComboBoxTS comboDisplayMode;
		private System.Windows.Forms.NumericUpDownTS udFilterLow;
		private System.Windows.Forms.NumericUpDownTS udFilterHigh;
		private System.Windows.Forms.RadioButtonTS radFilter6000;
		private System.Windows.Forms.RadioButtonTS radFilter4000;
		private System.Windows.Forms.RadioButtonTS radFilter2600;
		private System.Windows.Forms.RadioButtonTS radFilter2100;
		private System.Windows.Forms.RadioButtonTS radFilter1000;
		private System.Windows.Forms.RadioButtonTS radFilter500;
		private System.Windows.Forms.RadioButtonTS radFilter250;
		private System.Windows.Forms.RadioButtonTS radFilter100;
		private System.Windows.Forms.RadioButtonTS radFilter50;
		private System.Windows.Forms.RadioButtonTS radFilter25;
		private System.Windows.Forms.RadioButtonTS radFilterVar1;
		private System.Windows.Forms.RadioButtonTS radFilterVar2;
		private System.Windows.Forms.RadioButtonTS radModeSPEC;
		private System.Windows.Forms.RadioButtonTS radModeLSB;
		private System.Windows.Forms.RadioButtonTS radModeRTTY;
		private System.Windows.Forms.RadioButtonTS radModePSK;
		private System.Windows.Forms.RadioButtonTS radModeCWU;
		private System.Windows.Forms.RadioButtonTS radModeDSB;
		private System.Windows.Forms.RadioButtonTS radModeSAM;
		private System.Windows.Forms.RadioButtonTS radModeAM;
		private System.Windows.Forms.RadioButtonTS radModeCWL;
		private System.Windows.Forms.RadioButtonTS radModeUSB;
		private System.Windows.Forms.RadioButtonTS radModeFMN;
		private System.Windows.Forms.RadioButtonTS radModeDRM;
		private System.Windows.Forms.GroupBoxTS grpDSP;
		private System.Windows.Forms.LabelTS lblAGC;
		private System.Windows.Forms.ComboBoxTS comboAGC;
		private System.Windows.Forms.CheckBoxTS chkNB;
		private System.Windows.Forms.CheckBoxTS chkANF;
		private System.Windows.Forms.CheckBoxTS chkNR;
		private System.Windows.Forms.CheckBoxTS chkMON;
		private System.Windows.Forms.CheckBoxTS chkTUN;
		private System.Windows.Forms.CheckBoxTS chkMOX;
		private System.Windows.Forms.NumericUpDownTS udXIT;
		private System.Windows.Forms.NumericUpDownTS udRIT;
		private System.Windows.Forms.CheckBoxTS chkMUT;
		private System.Windows.Forms.CheckBoxTS chkXIT;
		private System.Windows.Forms.CheckBoxTS chkRIT;
		private System.Windows.Forms.LabelTS lblPWR;
		private System.Windows.Forms.NumericUpDownTS udPWR;
		private System.Windows.Forms.LabelTS lblAF;
		private System.Windows.Forms.NumericUpDownTS udAF;
		private System.Windows.Forms.LabelTS lblMIC;
		private System.Windows.Forms.NumericUpDownTS udMIC;
		private System.Windows.Forms.TextBoxTS txtWheelTune;
		private System.Windows.Forms.CheckBoxTS chkBIN;
		private System.Windows.Forms.CheckBoxTS chkPipe;
		private System.Windows.Forms.MenuItem menuItem3;
		private System.Windows.Forms.GroupBoxTS grpMultimeter;
		private System.Windows.Forms.ButtonTS button6;
		private System.Windows.Forms.ButtonTS btnVFOSwap;
		private System.Windows.Forms.ButtonTS btnVFOBtoA;
		private System.Windows.Forms.ButtonTS btnVFOAtoB;
		private System.Windows.Forms.GroupBoxTS grpVFO;
		private System.Windows.Forms.CheckBoxTS chkVFOSplit;
		private System.Windows.Forms.GroupBoxTS grpDisplay2;
		private System.Windows.Forms.CheckBoxTS chkDisplayAVG;
		private System.Windows.Forms.CheckBoxTS checkBox2;
		private System.Windows.Forms.GroupBoxTS grpMemory;
		private System.Windows.Forms.CheckBoxTS checkBox1;
		private System.Windows.Forms.TextBoxTS txtMultiText;
		private System.Windows.Forms.Timer timer_cpu_meter;
		private System.Windows.Forms.ButtonTS btnMemoryRecall;
		private System.Windows.Forms.ButtonTS btnSave;
		private System.Windows.Forms.LabelTS lblFilterHigh;
		private System.Windows.Forms.LabelTS lblFilterLow;
		private System.Windows.Forms.CheckBoxTS chkMemoryScanner;
		private System.Windows.Forms.NumericUpDownTS udScannerLow;
		private System.Windows.Forms.NumericUpDownTS udScannerHigh;
		private System.Windows.Forms.NumericUpDownTS udScannerStep;
		private System.Windows.Forms.Timer timer_scanner;
		private System.Windows.Forms.LabelTS labelScannerDelay;
		private System.Windows.Forms.NumericUpDownTS udScannerDelay;
		private System.Windows.Forms.RadioButtonTS radScannerFreq;
		private System.Windows.Forms.RadioButtonTS radScannerChannel;
		private System.Windows.Forms.LabelTS lblMultiSMeter;
		private System.Windows.Forms.PictureBox picMultimeterAnalog;
		private System.Windows.Forms.PictureBox picMultiMeterDigital;
		private System.Windows.Forms.NumericUpDownTS udSquelch;
		private System.Windows.Forms.CheckBoxTS chkSquelch;
		private System.Windows.Forms.Timer timer_peak_text;
		private System.Windows.Forms.TextBoxTS txtMemoryQuick;
		private System.Windows.Forms.ButtonTS btnMemoryQuickSave;
		private System.Windows.Forms.ButtonTS btnMemoryQuickRestore;
		private System.Windows.Forms.LabelTS lblMemoryStep;
		private System.Windows.Forms.LabelTS lblMemoryHigh;
		private System.Windows.Forms.LabelTS lblMemoryLow;
		private System.Windows.Forms.ToolTip toolTip1;
		private System.Windows.Forms.MenuItem menuItem2;
		private System.Windows.Forms.TrackBarTS tbFilterShift;
		private System.Windows.Forms.LabelTS lblFilterShift;
		private System.Windows.Forms.ButtonTS btnFilterShiftReset;
		private System.Windows.Forms.Timer timer_clock;
		private System.Windows.Forms.TextBoxTS txtDate;
		private System.Windows.Forms.TextBoxTS txtLocalTime;
		private System.Windows.Forms.TextBoxTS txtUTCTime;
		private System.Windows.Forms.Panel panelVFOAHover;
		private System.Windows.Forms.Panel panelVFOBHover;
		private System.Windows.Forms.ComboBoxTS comboMeterRXMode;
		private System.Windows.Forms.ComboBoxTS comboMeterTXMode;
		private System.Windows.Forms.GroupBoxTS grpSoundControls;
		private System.Windows.Forms.GroupBoxTS grpOptions;
		private System.Windows.Forms.GroupBoxTS grpDateTime;
		private System.Windows.Forms.ButtonTS btnXITReset;
		private System.Windows.Forms.ButtonTS btnRITReset;
		private System.Windows.Forms.ComboBoxTS comboTuneMode;
		private System.Windows.Forms.ComboBoxTS comboPreamp;
		private System.Windows.Forms.LabelTS lblPreamp;
		private System.Windows.Forms.CheckBoxTS chkDSPComp;
		private System.Windows.Forms.CheckBoxTS chkDSPNB2;
		private System.Windows.Forms.TextBoxTS txtDisplayPeak;
		private System.Windows.Forms.TrackBarTS tbWidth;
		private TypematicButton button_IFShiftDown;
		private TypematicButton button_IFShiftUp;
		private System.Windows.Forms.ButtonTS button_IFShiftZero;
		private TypematicButton button_IFShiftDownSmall;
		private TypematicButton button_IFShiftUpSmall;
	
		// private System.Windows.Forms.ButtonTS button_WidthDown;
		private TypematicButton button_WidthDown;
		private TypematicButton button_WidthUp;
		private System.ComponentModel.IContainer components;

		#endregion

		#endregion

		#region Constructor and Destructor
		// ======================================================
		// Constructor and Destructor
		// ======================================================

		public Console(string[] args)
		{
			CmdLineArgs = args; 
			CheckForOpenProcesses();

			callback1 = new PA19.PaStreamCallback(Audio.Callback1);	// Init callbacks to prevent GC
			callback2 = new PA19.PaStreamCallback(Audio.Callback2);
			callback4port = new PA19.PaStreamCallback(Audio.Callback4Port);

			Splash.ShowSplashScreen();							// Start splash screen
			
			Splash.SetStatus("Initializing Components");		// Set progress point
			InitializeComponent();								// Windows Forms Generated Code

			Splash.SetStatus("Saving Mixer Values");			// Set progress point
			main_volume = Mixer.GetMainVolume(0);
			wave_volume = Mixer.GetWaveOutVolume(0);
			
			Splash.SetStatus("Initializing Database");			// Set progress point
			DB.Init();											// Initialize the database

			Splash.SetStatus("Initializing DSP");				// Set progress point
			DttSP.Init();										// Initialize the DSP processor

			Splash.SetStatus("Initializing Parallel Port");		// Set progress point
			Parallel.InitPortTalk();							// Initialize parallel port

			Splash.SetStatus("Initializing PortAudio");			// Set progress point
			PA19.PA_Initialize();								// Initialize the audio interface

			Splash.SetStatus("Loading Main Form");				// Set progress point
			Splash.SplashForm.Owner = this;						// So that main form will show when splash disappears
			InitConsole();										// Initialize all forms and main variables
			
			Splash.SetStatus("Finished");						// Set progress point
			// Activates double buffering
			SetStyle(ControlStyles.DoubleBuffer, true);

			Splash.CloseForm();									// End splash screen

			if(File.Exists("extended.edf"))						// Check for extended capabilities file
			{
				ArrayList a = DB.GetVars("State");				
				a.Sort();
				ArrayList names = new ArrayList();
				ArrayList vals = new ArrayList();

				foreach(string s in a)
				{
					string[] str = s.Split('/');
					if(str.Length > 2)
					{
						for(int j=2; j<str.Length; j++)
							str[1] += "/"+str[j];
					}
					names.Add(str[0]);
					vals.Add(str[1]);
				}

				int i = names.BinarySearch("extended");
				if(i < 0) // If not found, prompt for logon info
				{
					LogOn LogOnForm = new LogOn(this);
					LogOnForm.ShowDialog();
				}
				else  // If found, check for existing logon info
				{
					string text = (string)vals[i];
					
					StreamReader sr = File.OpenText("extended.edf");
					string data = sr.ReadLine();
					sr.Close();

					if(text == data)
						extended = true;
					else	// Logon information found, but doesn't match
					{
						MessageBox.Show("Error reading logon information.", "Logon Error",
							MessageBoxButtons.OK, MessageBoxIcon.Error);
						LogOn LogOnForm = new LogOn(this);
						LogOnForm.ShowDialog();
					}
				}
			}

			if(run_setup_wizard)
			{
				SetupWizard w = new SetupWizard(this, (int)SoundCard.DELTA_44);
				w.ShowDialog();
				w.Focus();
			}
			
			if(multimeter_cal_offset == 0.0f)
			{
				switch((SoundCard)SetupForm.SoundCardIndex)
				{
					case SoundCard.SANTA_CRUZ:
						multimeter_cal_offset = -26.39952f;
						break;
					case SoundCard.AUDIGY_2_ZS:
						multimeter_cal_offset = 1.024933f;
						break;
					case SoundCard.MP3_PLUS:
						multimeter_cal_offset = -33.40224f;
						break;
					case SoundCard.UNSUPPORTED_CARD:
						multimeter_cal_offset = -52.43533f;
						break;
				}
			}

			if(display_cal_offset == 0.0f)
			{
				switch((SoundCard)SetupForm.SoundCardIndex)
				{
					case SoundCard.SANTA_CRUZ:
						display_cal_offset = -56.56675f;
						break;
					case SoundCard.AUDIGY_2_ZS:
						display_cal_offset = -29.20928f;
						break;
					case SoundCard.MP3_PLUS:
						display_cal_offset = -62.84578f;
						break;
					case SoundCard.UNSUPPORTED_CARD:
						display_cal_offset = -82.62103f;
						break;
				}
			}

			if(notify_on_beta || notify_on_release)
			{
				Thread t = new Thread(new ThreadStart(CheckForUpdates));
				t.IsBackground = true;
				t.Priority = ThreadPriority.Lowest;
				t.Name = "Update Check Thread";
				t.Start();
			}

			// BT 04/2005
			// instantiates the listener and raises event to initialize commander
			// if it has been opened before the console.

			siolisten = new SIOListener(this);
			//                      EventArgs sioe = new EventArgs();
			//                      OnNotifySIO(sioe);

			
		}

		protected override void Dispose( bool disposing )
		{
			USB.Exit();
			if( disposing )
			{
				if (components != null) 
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
			ExitConsole();
		}

		#endregion

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.components = new System.ComponentModel.Container();
			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(Console));
			this.txtVFOAFreq = new System.Windows.Forms.TextBoxTS();
			this.grpVFOA = new System.Windows.Forms.GroupBoxTS();
			this.panelVFOAHover = new System.Windows.Forms.Panel();
			this.txtWheelTune = new System.Windows.Forms.TextBoxTS();
			this.txtVFOABand = new System.Windows.Forms.TextBoxTS();
			this.btnHidden = new System.Windows.Forms.ButtonTS();
			this.grpVFOB = new System.Windows.Forms.GroupBoxTS();
			this.panelVFOBHover = new System.Windows.Forms.Panel();
			this.txtVFOBBand = new System.Windows.Forms.TextBoxTS();
			this.txtVFOBFreq = new System.Windows.Forms.TextBoxTS();
			this.grpDisplay = new System.Windows.Forms.GroupBoxTS();
			this.txtDisplayCurPos = new System.Windows.Forms.TextBoxTS();
			this.txtDisplayPeak = new System.Windows.Forms.TextBoxTS();
			this.picDisplay = new System.Windows.Forms.PictureBox();
			this.mainMenu1 = new System.Windows.Forms.MainMenu();
			this.menuItem1 = new System.Windows.Forms.MenuItem();
			this.menuItem3 = new System.Windows.Forms.MenuItem();
			this.menuItem2 = new System.Windows.Forms.MenuItem();
			this.grpMode = new System.Windows.Forms.GroupBoxTS();
			this.radModeAM = new System.Windows.Forms.RadioButtonTS();
			this.radModeSAM = new System.Windows.Forms.RadioButtonTS();
			this.radModeDSB = new System.Windows.Forms.RadioButtonTS();
			this.radModeCWU = new System.Windows.Forms.RadioButtonTS();
			this.radModePSK = new System.Windows.Forms.RadioButtonTS();
			this.radModeRTTY = new System.Windows.Forms.RadioButtonTS();
			this.radModeLSB = new System.Windows.Forms.RadioButtonTS();
			this.radModeSPEC = new System.Windows.Forms.RadioButtonTS();
			this.radModeDRM = new System.Windows.Forms.RadioButtonTS();
			this.radModeFMN = new System.Windows.Forms.RadioButtonTS();
			this.radModeUSB = new System.Windows.Forms.RadioButtonTS();
			this.radModeCWL = new System.Windows.Forms.RadioButtonTS();
			this.grpBand = new System.Windows.Forms.GroupBoxTS();
			this.btnBandGEN = new System.Windows.Forms.ButtonTS();
			this.btnBandWWV = new System.Windows.Forms.ButtonTS();
			this.btnBand13 = new System.Windows.Forms.ButtonTS();
			this.btnBand2 = new System.Windows.Forms.ButtonTS();
			this.btnBand6 = new System.Windows.Forms.ButtonTS();
			this.btnBand10 = new System.Windows.Forms.ButtonTS();
			this.btnBand12 = new System.Windows.Forms.ButtonTS();
			this.btnBand15 = new System.Windows.Forms.ButtonTS();
			this.btnBand17 = new System.Windows.Forms.ButtonTS();
			this.btnBand20 = new System.Windows.Forms.ButtonTS();
			this.btnBand30 = new System.Windows.Forms.ButtonTS();
			this.btnBand40 = new System.Windows.Forms.ButtonTS();
			this.btnBand60 = new System.Windows.Forms.ButtonTS();
			this.btnBand80 = new System.Windows.Forms.ButtonTS();
			this.btnBand160 = new System.Windows.Forms.ButtonTS();
			this.grpFilter = new System.Windows.Forms.GroupBoxTS();
			this.radFilter6000 = new System.Windows.Forms.RadioButtonTS();
			this.radFilter4000 = new System.Windows.Forms.RadioButtonTS();
			this.radFilter2600 = new System.Windows.Forms.RadioButtonTS();
			this.radFilter2100 = new System.Windows.Forms.RadioButtonTS();
			this.radFilter1000 = new System.Windows.Forms.RadioButtonTS();
			this.radFilter500 = new System.Windows.Forms.RadioButtonTS();
			this.radFilter250 = new System.Windows.Forms.RadioButtonTS();
			this.radFilter100 = new System.Windows.Forms.RadioButtonTS();
			this.radFilter50 = new System.Windows.Forms.RadioButtonTS();
			this.radFilter25 = new System.Windows.Forms.RadioButtonTS();
			this.radFilterVar1 = new System.Windows.Forms.RadioButtonTS();
			this.radFilterVar2 = new System.Windows.Forms.RadioButtonTS();
			this.btnFilterShiftReset = new System.Windows.Forms.ButtonTS();
			this.tbFilterShift = new System.Windows.Forms.TrackBarTS();
			this.lblFilterShift = new System.Windows.Forms.LabelTS();
			this.lblFilterHigh = new System.Windows.Forms.LabelTS();
			this.lblFilterLow = new System.Windows.Forms.LabelTS();
			this.udFilterHigh = new System.Windows.Forms.NumericUpDownTS();
			this.udFilterLow = new System.Windows.Forms.NumericUpDownTS();
			this.chkPower = new System.Windows.Forms.CheckBoxTS();
			this.comboDisplayMode = new System.Windows.Forms.ComboBoxTS();
			this.lblCPUMeter = new System.Windows.Forms.LabelTS();
			this.timer_cpu_meter = new System.Windows.Forms.Timer(this.components);
			this.grpDSP = new System.Windows.Forms.GroupBoxTS();
			this.chkDSPComp = new System.Windows.Forms.CheckBoxTS();
			this.chkANF = new System.Windows.Forms.CheckBoxTS();
			this.chkNR = new System.Windows.Forms.CheckBoxTS();
			this.checkBox2 = new System.Windows.Forms.CheckBoxTS();
			this.chkDSPNB2 = new System.Windows.Forms.CheckBoxTS();

			this.button_IFShiftUp = new TypematicButton();
			this.button_IFShiftDown = new TypematicButton();

			this.chkNB = new System.Windows.Forms.CheckBoxTS();
			this.button_IFShiftZero = new System.Windows.Forms.ButtonTS();
			this.tbWidth = new System.Windows.Forms.TrackBarTS();
			this.lblAGC = new System.Windows.Forms.LabelTS();
			this.comboAGC = new System.Windows.Forms.ComboBoxTS();
			this.chkBIN = new System.Windows.Forms.CheckBoxTS();
			this.comboPreamp = new System.Windows.Forms.ComboBoxTS();
			this.lblPreamp = new System.Windows.Forms.LabelTS();
			this.udXIT = new System.Windows.Forms.NumericUpDownTS();
			this.chkMON = new System.Windows.Forms.CheckBoxTS();
			this.chkTUN = new System.Windows.Forms.CheckBoxTS();
			this.chkMOX = new System.Windows.Forms.CheckBoxTS();
			this.udRIT = new System.Windows.Forms.NumericUpDownTS();
			this.chkMUT = new System.Windows.Forms.CheckBoxTS();
			this.chkXIT = new System.Windows.Forms.CheckBoxTS();
			this.chkRIT = new System.Windows.Forms.CheckBoxTS();
			this.lblPWR = new System.Windows.Forms.LabelTS();
			this.udPWR = new System.Windows.Forms.NumericUpDownTS();
			this.lblAF = new System.Windows.Forms.LabelTS();
			this.udAF = new System.Windows.Forms.NumericUpDownTS();
			this.lblMIC = new System.Windows.Forms.LabelTS();
			this.udMIC = new System.Windows.Forms.NumericUpDownTS();
			this.chkPipe = new System.Windows.Forms.CheckBoxTS();
			this.grpMultimeter = new System.Windows.Forms.GroupBoxTS();
			this.comboMeterTXMode = new System.Windows.Forms.ComboBoxTS();
			this.picMultiMeterDigital = new System.Windows.Forms.PictureBox();
			this.lblMultiSMeter = new System.Windows.Forms.LabelTS();
			this.comboMeterRXMode = new System.Windows.Forms.ComboBoxTS();
			this.txtMultiText = new System.Windows.Forms.TextBoxTS();
			this.picMultimeterAnalog = new System.Windows.Forms.PictureBox();
			this.grpVFO = new System.Windows.Forms.GroupBoxTS();
			this.btnRITReset = new System.Windows.Forms.ButtonTS();
			this.btnXITReset = new System.Windows.Forms.ButtonTS();
			this.button6 = new System.Windows.Forms.ButtonTS();
			this.btnVFOSwap = new System.Windows.Forms.ButtonTS();
			this.btnVFOBtoA = new System.Windows.Forms.ButtonTS();
			this.btnVFOAtoB = new System.Windows.Forms.ButtonTS();
			this.chkVFOSplit = new System.Windows.Forms.CheckBoxTS();
			this.checkBox1 = new System.Windows.Forms.CheckBoxTS();
			this.grpSoundControls = new System.Windows.Forms.GroupBoxTS();
			this.udSquelch = new System.Windows.Forms.NumericUpDownTS();
			this.chkSquelch = new System.Windows.Forms.CheckBoxTS();
			this.grpOptions = new System.Windows.Forms.GroupBoxTS();
			this.comboTuneMode = new System.Windows.Forms.ComboBoxTS();
			this.grpDisplay2 = new System.Windows.Forms.GroupBoxTS();
			this.chkDisplayAVG = new System.Windows.Forms.CheckBoxTS();
			this.grpMemory = new System.Windows.Forms.GroupBoxTS();
			this.btnMemoryQuickRestore = new System.Windows.Forms.ButtonTS();
			this.btnMemoryQuickSave = new System.Windows.Forms.ButtonTS();
			this.txtMemoryQuick = new System.Windows.Forms.TextBoxTS();
			this.labelScannerDelay = new System.Windows.Forms.LabelTS();
			this.udScannerDelay = new System.Windows.Forms.NumericUpDownTS();
			this.lblMemoryStep = new System.Windows.Forms.LabelTS();
			this.lblMemoryHigh = new System.Windows.Forms.LabelTS();
			this.lblMemoryLow = new System.Windows.Forms.LabelTS();
			this.udScannerStep = new System.Windows.Forms.NumericUpDownTS();
			this.udScannerHigh = new System.Windows.Forms.NumericUpDownTS();
			this.udScannerLow = new System.Windows.Forms.NumericUpDownTS();
			this.chkMemoryScanner = new System.Windows.Forms.CheckBoxTS();
			this.radScannerFreq = new System.Windows.Forms.RadioButtonTS();
			this.radScannerChannel = new System.Windows.Forms.RadioButtonTS();
			this.btnSave = new System.Windows.Forms.ButtonTS();
			this.btnMemoryRecall = new System.Windows.Forms.ButtonTS();
			this.timer_scanner = new System.Windows.Forms.Timer(this.components);
			this.timer_peak_text = new System.Windows.Forms.Timer(this.components);
			this.toolTip1 = new System.Windows.Forms.ToolTip(this.components);
			this.grpDateTime = new System.Windows.Forms.GroupBoxTS();
			this.txtUTCTime = new System.Windows.Forms.TextBoxTS();
			this.txtLocalTime = new System.Windows.Forms.TextBoxTS();
			this.txtDate = new System.Windows.Forms.TextBoxTS();
			this.button_IFShiftDownSmall = new TypematicButton();
			this.button_IFShiftUpSmall = new TypematicButton();
			this.button_WidthDown = new PowerSDR.TypematicButton();
			this.button_WidthUp = new PowerSDR.TypematicButton();
			this.timer_clock = new System.Windows.Forms.Timer(this.components);
			this.grpVFOA.SuspendLayout();
			this.grpVFOB.SuspendLayout();
			this.grpDisplay.SuspendLayout();
			this.grpMode.SuspendLayout();
			this.grpBand.SuspendLayout();
			this.grpFilter.SuspendLayout();
			((System.ComponentModel.ISupportInitialize)(this.tbFilterShift)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.udFilterHigh)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.udFilterLow)).BeginInit();
			this.grpDSP.SuspendLayout();
			((System.ComponentModel.ISupportInitialize)(this.tbWidth)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.udXIT)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.udRIT)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.udPWR)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.udAF)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.udMIC)).BeginInit();
			this.grpMultimeter.SuspendLayout();
			this.grpVFO.SuspendLayout();
			this.grpSoundControls.SuspendLayout();
			((System.ComponentModel.ISupportInitialize)(this.udSquelch)).BeginInit();
			this.grpOptions.SuspendLayout();
			this.grpDisplay2.SuspendLayout();
			this.grpMemory.SuspendLayout();
			((System.ComponentModel.ISupportInitialize)(this.udScannerDelay)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.udScannerStep)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.udScannerHigh)).BeginInit();
			((System.ComponentModel.ISupportInitialize)(this.udScannerLow)).BeginInit();
			this.grpDateTime.SuspendLayout();
			this.SuspendLayout();
			// 
			// txtVFOAFreq
			// 
			this.txtVFOAFreq.AccessibleDescription = resources.GetString("txtVFOAFreq.AccessibleDescription");
			this.txtVFOAFreq.AccessibleName = resources.GetString("txtVFOAFreq.AccessibleName");
			this.txtVFOAFreq.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtVFOAFreq.Anchor")));
			this.txtVFOAFreq.AutoSize = ((bool)(resources.GetObject("txtVFOAFreq.AutoSize")));
			this.txtVFOAFreq.BackColor = System.Drawing.Color.Black;
			this.txtVFOAFreq.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtVFOAFreq.BackgroundImage")));
			this.txtVFOAFreq.Cursor = System.Windows.Forms.Cursors.Default;
			this.txtVFOAFreq.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtVFOAFreq.Dock")));
			this.txtVFOAFreq.Enabled = ((bool)(resources.GetObject("txtVFOAFreq.Enabled")));
			this.txtVFOAFreq.Font = ((System.Drawing.Font)(resources.GetObject("txtVFOAFreq.Font")));
			this.txtVFOAFreq.ForeColor = System.Drawing.Color.Olive;
			this.txtVFOAFreq.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtVFOAFreq.ImeMode")));
			this.txtVFOAFreq.Location = ((System.Drawing.Point)(resources.GetObject("txtVFOAFreq.Location")));
			this.txtVFOAFreq.MaxLength = ((int)(resources.GetObject("txtVFOAFreq.MaxLength")));
			this.txtVFOAFreq.Multiline = ((bool)(resources.GetObject("txtVFOAFreq.Multiline")));
			this.txtVFOAFreq.Name = "txtVFOAFreq";
			this.txtVFOAFreq.PasswordChar = ((char)(resources.GetObject("txtVFOAFreq.PasswordChar")));
			this.txtVFOAFreq.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtVFOAFreq.RightToLeft")));
			this.txtVFOAFreq.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtVFOAFreq.ScrollBars")));
			this.txtVFOAFreq.Size = ((System.Drawing.Size)(resources.GetObject("txtVFOAFreq.Size")));
			this.txtVFOAFreq.TabIndex = ((int)(resources.GetObject("txtVFOAFreq.TabIndex")));
			this.txtVFOAFreq.Text = resources.GetString("txtVFOAFreq.Text");
			this.txtVFOAFreq.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtVFOAFreq.TextAlign")));
			this.toolTip1.SetToolTip(this.txtVFOAFreq, resources.GetString("txtVFOAFreq.ToolTip"));
			this.txtVFOAFreq.Visible = ((bool)(resources.GetObject("txtVFOAFreq.Visible")));
			this.txtVFOAFreq.WordWrap = ((bool)(resources.GetObject("txtVFOAFreq.WordWrap")));
			this.txtVFOAFreq.MouseDown += new System.Windows.Forms.MouseEventHandler(this.WheelTune_MouseDown);
			this.txtVFOAFreq.LostFocus += new System.EventHandler(this.txtVFOAFreq_LostFocus);
			this.txtVFOAFreq.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.txtVFOAFreq_KeyPress);
			this.txtVFOAFreq.MouseMove += new System.Windows.Forms.MouseEventHandler(this.txtVFOAFreq_MouseMove);
			this.txtVFOAFreq.MouseLeave += new System.EventHandler(this.txtVFOAFreq_MouseLeave);
			// 
			// grpVFOA
			// 
			this.grpVFOA.AccessibleDescription = resources.GetString("grpVFOA.AccessibleDescription");
			this.grpVFOA.AccessibleName = resources.GetString("grpVFOA.AccessibleName");
			this.grpVFOA.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpVFOA.Anchor")));
			this.grpVFOA.BackColor = System.Drawing.SystemColors.Control;
			this.grpVFOA.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpVFOA.BackgroundImage")));
			this.grpVFOA.Controls.Add(this.panelVFOAHover);
			this.grpVFOA.Controls.Add(this.txtWheelTune);
			this.grpVFOA.Controls.Add(this.txtVFOABand);
			this.grpVFOA.Controls.Add(this.txtVFOAFreq);
			this.grpVFOA.Controls.Add(this.btnHidden);
			this.grpVFOA.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpVFOA.Dock")));
			this.grpVFOA.Enabled = ((bool)(resources.GetObject("grpVFOA.Enabled")));
			this.grpVFOA.Font = ((System.Drawing.Font)(resources.GetObject("grpVFOA.Font")));
			this.grpVFOA.ForeColor = System.Drawing.Color.Black;
			this.grpVFOA.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpVFOA.ImeMode")));
			this.grpVFOA.Location = ((System.Drawing.Point)(resources.GetObject("grpVFOA.Location")));
			this.grpVFOA.Name = "grpVFOA";
			this.grpVFOA.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpVFOA.RightToLeft")));
			this.grpVFOA.Size = ((System.Drawing.Size)(resources.GetObject("grpVFOA.Size")));
			this.grpVFOA.TabIndex = ((int)(resources.GetObject("grpVFOA.TabIndex")));
			this.grpVFOA.TabStop = false;
			this.grpVFOA.Text = resources.GetString("grpVFOA.Text");
			this.toolTip1.SetToolTip(this.grpVFOA, resources.GetString("grpVFOA.ToolTip"));
			this.grpVFOA.Visible = ((bool)(resources.GetObject("grpVFOA.Visible")));
			// 
			// panelVFOAHover
			// 
			this.panelVFOAHover.AccessibleDescription = resources.GetString("panelVFOAHover.AccessibleDescription");
			this.panelVFOAHover.AccessibleName = resources.GetString("panelVFOAHover.AccessibleName");
			this.panelVFOAHover.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("panelVFOAHover.Anchor")));
			this.panelVFOAHover.AutoScroll = ((bool)(resources.GetObject("panelVFOAHover.AutoScroll")));
			this.panelVFOAHover.AutoScrollMargin = ((System.Drawing.Size)(resources.GetObject("panelVFOAHover.AutoScrollMargin")));
			this.panelVFOAHover.AutoScrollMinSize = ((System.Drawing.Size)(resources.GetObject("panelVFOAHover.AutoScrollMinSize")));
			this.panelVFOAHover.BackColor = System.Drawing.Color.Black;
			this.panelVFOAHover.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("panelVFOAHover.BackgroundImage")));
			this.panelVFOAHover.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("panelVFOAHover.Dock")));
			this.panelVFOAHover.Enabled = ((bool)(resources.GetObject("panelVFOAHover.Enabled")));
			this.panelVFOAHover.Font = ((System.Drawing.Font)(resources.GetObject("panelVFOAHover.Font")));
			this.panelVFOAHover.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("panelVFOAHover.ImeMode")));
			this.panelVFOAHover.Location = ((System.Drawing.Point)(resources.GetObject("panelVFOAHover.Location")));
			this.panelVFOAHover.Name = "panelVFOAHover";
			this.panelVFOAHover.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("panelVFOAHover.RightToLeft")));
			this.panelVFOAHover.Size = ((System.Drawing.Size)(resources.GetObject("panelVFOAHover.Size")));
			this.panelVFOAHover.TabIndex = ((int)(resources.GetObject("panelVFOAHover.TabIndex")));
			this.panelVFOAHover.Text = resources.GetString("panelVFOAHover.Text");
			this.toolTip1.SetToolTip(this.panelVFOAHover, resources.GetString("panelVFOAHover.ToolTip"));
			this.panelVFOAHover.Visible = ((bool)(resources.GetObject("panelVFOAHover.Visible")));
			this.panelVFOAHover.Paint += new System.Windows.Forms.PaintEventHandler(this.panelVFOAHover_Paint);
			this.panelVFOAHover.MouseMove += new System.Windows.Forms.MouseEventHandler(this.panelVFOAHover_MouseMove);
			// 
			// txtWheelTune
			// 
			this.txtWheelTune.AccessibleDescription = resources.GetString("txtWheelTune.AccessibleDescription");
			this.txtWheelTune.AccessibleName = resources.GetString("txtWheelTune.AccessibleName");
			this.txtWheelTune.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtWheelTune.Anchor")));
			this.txtWheelTune.AutoSize = ((bool)(resources.GetObject("txtWheelTune.AutoSize")));
			this.txtWheelTune.BackColor = System.Drawing.Color.Black;
			this.txtWheelTune.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtWheelTune.BackgroundImage")));
			this.txtWheelTune.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtWheelTune.Dock")));
			this.txtWheelTune.Enabled = ((bool)(resources.GetObject("txtWheelTune.Enabled")));
			this.txtWheelTune.Font = ((System.Drawing.Font)(resources.GetObject("txtWheelTune.Font")));
			this.txtWheelTune.ForeColor = System.Drawing.Color.White;
			this.txtWheelTune.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtWheelTune.ImeMode")));
			this.txtWheelTune.Location = ((System.Drawing.Point)(resources.GetObject("txtWheelTune.Location")));
			this.txtWheelTune.MaxLength = ((int)(resources.GetObject("txtWheelTune.MaxLength")));
			this.txtWheelTune.Multiline = ((bool)(resources.GetObject("txtWheelTune.Multiline")));
			this.txtWheelTune.Name = "txtWheelTune";
			this.txtWheelTune.PasswordChar = ((char)(resources.GetObject("txtWheelTune.PasswordChar")));
			this.txtWheelTune.ReadOnly = true;
			this.txtWheelTune.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtWheelTune.RightToLeft")));
			this.txtWheelTune.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtWheelTune.ScrollBars")));
			this.txtWheelTune.Size = ((System.Drawing.Size)(resources.GetObject("txtWheelTune.Size")));
			this.txtWheelTune.TabIndex = ((int)(resources.GetObject("txtWheelTune.TabIndex")));
			this.txtWheelTune.Text = resources.GetString("txtWheelTune.Text");
			this.txtWheelTune.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtWheelTune.TextAlign")));
			this.toolTip1.SetToolTip(this.txtWheelTune, resources.GetString("txtWheelTune.ToolTip"));
			this.txtWheelTune.Visible = ((bool)(resources.GetObject("txtWheelTune.Visible")));
			this.txtWheelTune.WordWrap = ((bool)(resources.GetObject("txtWheelTune.WordWrap")));
			this.txtWheelTune.MouseDown += new System.Windows.Forms.MouseEventHandler(this.WheelTune_MouseDown);
			this.txtWheelTune.GotFocus += new System.EventHandler(this.HideFocus);
			// 
			// txtVFOABand
			// 
			this.txtVFOABand.AccessibleDescription = resources.GetString("txtVFOABand.AccessibleDescription");
			this.txtVFOABand.AccessibleName = resources.GetString("txtVFOABand.AccessibleName");
			this.txtVFOABand.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtVFOABand.Anchor")));
			this.txtVFOABand.AutoSize = ((bool)(resources.GetObject("txtVFOABand.AutoSize")));
			this.txtVFOABand.BackColor = System.Drawing.Color.Black;
			this.txtVFOABand.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtVFOABand.BackgroundImage")));
			this.txtVFOABand.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtVFOABand.Dock")));
			this.txtVFOABand.Enabled = ((bool)(resources.GetObject("txtVFOABand.Enabled")));
			this.txtVFOABand.Font = ((System.Drawing.Font)(resources.GetObject("txtVFOABand.Font")));
			this.txtVFOABand.ForeColor = System.Drawing.Color.Green;
			this.txtVFOABand.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtVFOABand.ImeMode")));
			this.txtVFOABand.Location = ((System.Drawing.Point)(resources.GetObject("txtVFOABand.Location")));
			this.txtVFOABand.MaxLength = ((int)(resources.GetObject("txtVFOABand.MaxLength")));
			this.txtVFOABand.Multiline = ((bool)(resources.GetObject("txtVFOABand.Multiline")));
			this.txtVFOABand.Name = "txtVFOABand";
			this.txtVFOABand.PasswordChar = ((char)(resources.GetObject("txtVFOABand.PasswordChar")));
			this.txtVFOABand.ReadOnly = true;
			this.txtVFOABand.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtVFOABand.RightToLeft")));
			this.txtVFOABand.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtVFOABand.ScrollBars")));
			this.txtVFOABand.Size = ((System.Drawing.Size)(resources.GetObject("txtVFOABand.Size")));
			this.txtVFOABand.TabIndex = ((int)(resources.GetObject("txtVFOABand.TabIndex")));
			this.txtVFOABand.Text = resources.GetString("txtVFOABand.Text");
			this.txtVFOABand.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtVFOABand.TextAlign")));
			this.toolTip1.SetToolTip(this.txtVFOABand, resources.GetString("txtVFOABand.ToolTip"));
			this.txtVFOABand.Visible = ((bool)(resources.GetObject("txtVFOABand.Visible")));
			this.txtVFOABand.WordWrap = ((bool)(resources.GetObject("txtVFOABand.WordWrap")));
			this.txtVFOABand.GotFocus += new System.EventHandler(this.HideFocus);
			// 
			// btnHidden
			// 
			this.btnHidden.AccessibleDescription = resources.GetString("btnHidden.AccessibleDescription");
			this.btnHidden.AccessibleName = resources.GetString("btnHidden.AccessibleName");
			this.btnHidden.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnHidden.Anchor")));
			this.btnHidden.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnHidden.BackgroundImage")));
			this.btnHidden.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnHidden.Dock")));
			this.btnHidden.Enabled = ((bool)(resources.GetObject("btnHidden.Enabled")));
			this.btnHidden.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnHidden.FlatStyle")));
			this.btnHidden.Font = ((System.Drawing.Font)(resources.GetObject("btnHidden.Font")));
			this.btnHidden.Image = ((System.Drawing.Image)(resources.GetObject("btnHidden.Image")));
			this.btnHidden.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnHidden.ImageAlign")));
			this.btnHidden.ImageIndex = ((int)(resources.GetObject("btnHidden.ImageIndex")));
			this.btnHidden.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnHidden.ImeMode")));
			this.btnHidden.Location = ((System.Drawing.Point)(resources.GetObject("btnHidden.Location")));
			this.btnHidden.Name = "btnHidden";
			this.btnHidden.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnHidden.RightToLeft")));
			this.btnHidden.Size = ((System.Drawing.Size)(resources.GetObject("btnHidden.Size")));
			this.btnHidden.TabIndex = ((int)(resources.GetObject("btnHidden.TabIndex")));
			this.btnHidden.Text = resources.GetString("btnHidden.Text");
			this.btnHidden.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnHidden.TextAlign")));
			this.toolTip1.SetToolTip(this.btnHidden, resources.GetString("btnHidden.ToolTip"));
			this.btnHidden.Visible = ((bool)(resources.GetObject("btnHidden.Visible")));
			// 
			// grpVFOB
			// 
			this.grpVFOB.AccessibleDescription = resources.GetString("grpVFOB.AccessibleDescription");
			this.grpVFOB.AccessibleName = resources.GetString("grpVFOB.AccessibleName");
			this.grpVFOB.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpVFOB.Anchor")));
			this.grpVFOB.BackColor = System.Drawing.SystemColors.Control;
			this.grpVFOB.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpVFOB.BackgroundImage")));
			this.grpVFOB.Controls.Add(this.panelVFOBHover);
			this.grpVFOB.Controls.Add(this.txtVFOBBand);
			this.grpVFOB.Controls.Add(this.txtVFOBFreq);
			this.grpVFOB.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpVFOB.Dock")));
			this.grpVFOB.Enabled = ((bool)(resources.GetObject("grpVFOB.Enabled")));
			this.grpVFOB.Font = ((System.Drawing.Font)(resources.GetObject("grpVFOB.Font")));
			this.grpVFOB.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpVFOB.ImeMode")));
			this.grpVFOB.Location = ((System.Drawing.Point)(resources.GetObject("grpVFOB.Location")));
			this.grpVFOB.Name = "grpVFOB";
			this.grpVFOB.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpVFOB.RightToLeft")));
			this.grpVFOB.Size = ((System.Drawing.Size)(resources.GetObject("grpVFOB.Size")));
			this.grpVFOB.TabIndex = ((int)(resources.GetObject("grpVFOB.TabIndex")));
			this.grpVFOB.TabStop = false;
			this.grpVFOB.Text = resources.GetString("grpVFOB.Text");
			this.toolTip1.SetToolTip(this.grpVFOB, resources.GetString("grpVFOB.ToolTip"));
			this.grpVFOB.Visible = ((bool)(resources.GetObject("grpVFOB.Visible")));
			// 
			// panelVFOBHover
			// 
			this.panelVFOBHover.AccessibleDescription = resources.GetString("panelVFOBHover.AccessibleDescription");
			this.panelVFOBHover.AccessibleName = resources.GetString("panelVFOBHover.AccessibleName");
			this.panelVFOBHover.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("panelVFOBHover.Anchor")));
			this.panelVFOBHover.AutoScroll = ((bool)(resources.GetObject("panelVFOBHover.AutoScroll")));
			this.panelVFOBHover.AutoScrollMargin = ((System.Drawing.Size)(resources.GetObject("panelVFOBHover.AutoScrollMargin")));
			this.panelVFOBHover.AutoScrollMinSize = ((System.Drawing.Size)(resources.GetObject("panelVFOBHover.AutoScrollMinSize")));
			this.panelVFOBHover.BackColor = System.Drawing.Color.Black;
			this.panelVFOBHover.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("panelVFOBHover.BackgroundImage")));
			this.panelVFOBHover.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("panelVFOBHover.Dock")));
			this.panelVFOBHover.Enabled = ((bool)(resources.GetObject("panelVFOBHover.Enabled")));
			this.panelVFOBHover.Font = ((System.Drawing.Font)(resources.GetObject("panelVFOBHover.Font")));
			this.panelVFOBHover.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("panelVFOBHover.ImeMode")));
			this.panelVFOBHover.Location = ((System.Drawing.Point)(resources.GetObject("panelVFOBHover.Location")));
			this.panelVFOBHover.Name = "panelVFOBHover";
			this.panelVFOBHover.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("panelVFOBHover.RightToLeft")));
			this.panelVFOBHover.Size = ((System.Drawing.Size)(resources.GetObject("panelVFOBHover.Size")));
			this.panelVFOBHover.TabIndex = ((int)(resources.GetObject("panelVFOBHover.TabIndex")));
			this.panelVFOBHover.Text = resources.GetString("panelVFOBHover.Text");
			this.toolTip1.SetToolTip(this.panelVFOBHover, resources.GetString("panelVFOBHover.ToolTip"));
			this.panelVFOBHover.Visible = ((bool)(resources.GetObject("panelVFOBHover.Visible")));
			this.panelVFOBHover.Paint += new System.Windows.Forms.PaintEventHandler(this.panelVFOBHover_Paint);
			this.panelVFOBHover.MouseMove += new System.Windows.Forms.MouseEventHandler(this.panelVFOBHover_MouseMove);
			// 
			// txtVFOBBand
			// 
			this.txtVFOBBand.AccessibleDescription = resources.GetString("txtVFOBBand.AccessibleDescription");
			this.txtVFOBBand.AccessibleName = resources.GetString("txtVFOBBand.AccessibleName");
			this.txtVFOBBand.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtVFOBBand.Anchor")));
			this.txtVFOBBand.AutoSize = ((bool)(resources.GetObject("txtVFOBBand.AutoSize")));
			this.txtVFOBBand.BackColor = System.Drawing.Color.Black;
			this.txtVFOBBand.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtVFOBBand.BackgroundImage")));
			this.txtVFOBBand.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtVFOBBand.Dock")));
			this.txtVFOBBand.Enabled = ((bool)(resources.GetObject("txtVFOBBand.Enabled")));
			this.txtVFOBBand.Font = ((System.Drawing.Font)(resources.GetObject("txtVFOBBand.Font")));
			this.txtVFOBBand.ForeColor = System.Drawing.Color.Green;
			this.txtVFOBBand.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtVFOBBand.ImeMode")));
			this.txtVFOBBand.Location = ((System.Drawing.Point)(resources.GetObject("txtVFOBBand.Location")));
			this.txtVFOBBand.MaxLength = ((int)(resources.GetObject("txtVFOBBand.MaxLength")));
			this.txtVFOBBand.Multiline = ((bool)(resources.GetObject("txtVFOBBand.Multiline")));
			this.txtVFOBBand.Name = "txtVFOBBand";
			this.txtVFOBBand.PasswordChar = ((char)(resources.GetObject("txtVFOBBand.PasswordChar")));
			this.txtVFOBBand.ReadOnly = true;
			this.txtVFOBBand.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtVFOBBand.RightToLeft")));
			this.txtVFOBBand.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtVFOBBand.ScrollBars")));
			this.txtVFOBBand.Size = ((System.Drawing.Size)(resources.GetObject("txtVFOBBand.Size")));
			this.txtVFOBBand.TabIndex = ((int)(resources.GetObject("txtVFOBBand.TabIndex")));
			this.txtVFOBBand.Text = resources.GetString("txtVFOBBand.Text");
			this.txtVFOBBand.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtVFOBBand.TextAlign")));
			this.toolTip1.SetToolTip(this.txtVFOBBand, resources.GetString("txtVFOBBand.ToolTip"));
			this.txtVFOBBand.Visible = ((bool)(resources.GetObject("txtVFOBBand.Visible")));
			this.txtVFOBBand.WordWrap = ((bool)(resources.GetObject("txtVFOBBand.WordWrap")));
			this.txtVFOBBand.GotFocus += new System.EventHandler(this.HideFocus);
			// 
			// txtVFOBFreq
			// 
			this.txtVFOBFreq.AccessibleDescription = resources.GetString("txtVFOBFreq.AccessibleDescription");
			this.txtVFOBFreq.AccessibleName = resources.GetString("txtVFOBFreq.AccessibleName");
			this.txtVFOBFreq.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtVFOBFreq.Anchor")));
			this.txtVFOBFreq.AutoSize = ((bool)(resources.GetObject("txtVFOBFreq.AutoSize")));
			this.txtVFOBFreq.BackColor = System.Drawing.Color.Black;
			this.txtVFOBFreq.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtVFOBFreq.BackgroundImage")));
			this.txtVFOBFreq.Cursor = System.Windows.Forms.Cursors.Default;
			this.txtVFOBFreq.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtVFOBFreq.Dock")));
			this.txtVFOBFreq.Enabled = ((bool)(resources.GetObject("txtVFOBFreq.Enabled")));
			this.txtVFOBFreq.Font = ((System.Drawing.Font)(resources.GetObject("txtVFOBFreq.Font")));
			this.txtVFOBFreq.ForeColor = System.Drawing.Color.Olive;
			this.txtVFOBFreq.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtVFOBFreq.ImeMode")));
			this.txtVFOBFreq.Location = ((System.Drawing.Point)(resources.GetObject("txtVFOBFreq.Location")));
			this.txtVFOBFreq.MaxLength = ((int)(resources.GetObject("txtVFOBFreq.MaxLength")));
			this.txtVFOBFreq.Multiline = ((bool)(resources.GetObject("txtVFOBFreq.Multiline")));
			this.txtVFOBFreq.Name = "txtVFOBFreq";
			this.txtVFOBFreq.PasswordChar = ((char)(resources.GetObject("txtVFOBFreq.PasswordChar")));
			this.txtVFOBFreq.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtVFOBFreq.RightToLeft")));
			this.txtVFOBFreq.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtVFOBFreq.ScrollBars")));
			this.txtVFOBFreq.Size = ((System.Drawing.Size)(resources.GetObject("txtVFOBFreq.Size")));
			this.txtVFOBFreq.TabIndex = ((int)(resources.GetObject("txtVFOBFreq.TabIndex")));
			this.txtVFOBFreq.Text = resources.GetString("txtVFOBFreq.Text");
			this.txtVFOBFreq.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtVFOBFreq.TextAlign")));
			this.toolTip1.SetToolTip(this.txtVFOBFreq, resources.GetString("txtVFOBFreq.ToolTip"));
			this.txtVFOBFreq.Visible = ((bool)(resources.GetObject("txtVFOBFreq.Visible")));
			this.txtVFOBFreq.WordWrap = ((bool)(resources.GetObject("txtVFOBFreq.WordWrap")));
			this.txtVFOBFreq.MouseDown += new System.Windows.Forms.MouseEventHandler(this.WheelTune_MouseDown);
			this.txtVFOBFreq.LostFocus += new System.EventHandler(this.txtVFOBFreq_LostFocus);
			this.txtVFOBFreq.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.txtVFOBFreq_KeyPress);
			this.txtVFOBFreq.MouseMove += new System.Windows.Forms.MouseEventHandler(this.txtVFOBFreq_MouseMove);
			this.txtVFOBFreq.MouseLeave += new System.EventHandler(this.txtVFOBFreq_MouseLeave);
			// 
			// grpDisplay
			// 
			this.grpDisplay.AccessibleDescription = resources.GetString("grpDisplay.AccessibleDescription");
			this.grpDisplay.AccessibleName = resources.GetString("grpDisplay.AccessibleName");
			this.grpDisplay.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpDisplay.Anchor")));
			this.grpDisplay.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpDisplay.BackgroundImage")));
			this.grpDisplay.Controls.Add(this.txtDisplayCurPos);
			this.grpDisplay.Controls.Add(this.txtDisplayPeak);
			this.grpDisplay.Controls.Add(this.picDisplay);
			this.grpDisplay.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpDisplay.Dock")));
			this.grpDisplay.Enabled = ((bool)(resources.GetObject("grpDisplay.Enabled")));
			this.grpDisplay.Font = ((System.Drawing.Font)(resources.GetObject("grpDisplay.Font")));
			this.grpDisplay.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpDisplay.ImeMode")));
			this.grpDisplay.Location = ((System.Drawing.Point)(resources.GetObject("grpDisplay.Location")));
			this.grpDisplay.Name = "grpDisplay";
			this.grpDisplay.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpDisplay.RightToLeft")));
			this.grpDisplay.Size = ((System.Drawing.Size)(resources.GetObject("grpDisplay.Size")));
			this.grpDisplay.TabIndex = ((int)(resources.GetObject("grpDisplay.TabIndex")));
			this.grpDisplay.TabStop = false;
			this.grpDisplay.Text = resources.GetString("grpDisplay.Text");
			this.toolTip1.SetToolTip(this.grpDisplay, resources.GetString("grpDisplay.ToolTip"));
			this.grpDisplay.Visible = ((bool)(resources.GetObject("grpDisplay.Visible")));
			// 
			// txtDisplayCurPos
			// 
			this.txtDisplayCurPos.AccessibleDescription = resources.GetString("txtDisplayCurPos.AccessibleDescription");
			this.txtDisplayCurPos.AccessibleName = resources.GetString("txtDisplayCurPos.AccessibleName");
			this.txtDisplayCurPos.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtDisplayCurPos.Anchor")));
			this.txtDisplayCurPos.AutoSize = ((bool)(resources.GetObject("txtDisplayCurPos.AutoSize")));
			this.txtDisplayCurPos.BackColor = System.Drawing.Color.Black;
			this.txtDisplayCurPos.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtDisplayCurPos.BackgroundImage")));
			this.txtDisplayCurPos.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
			this.txtDisplayCurPos.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtDisplayCurPos.Dock")));
			this.txtDisplayCurPos.Enabled = ((bool)(resources.GetObject("txtDisplayCurPos.Enabled")));
			this.txtDisplayCurPos.Font = ((System.Drawing.Font)(resources.GetObject("txtDisplayCurPos.Font")));
			this.txtDisplayCurPos.ForeColor = System.Drawing.Color.DodgerBlue;
			this.txtDisplayCurPos.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtDisplayCurPos.ImeMode")));
			this.txtDisplayCurPos.Location = ((System.Drawing.Point)(resources.GetObject("txtDisplayCurPos.Location")));
			this.txtDisplayCurPos.MaxLength = ((int)(resources.GetObject("txtDisplayCurPos.MaxLength")));
			this.txtDisplayCurPos.Multiline = ((bool)(resources.GetObject("txtDisplayCurPos.Multiline")));
			this.txtDisplayCurPos.Name = "txtDisplayCurPos";
			this.txtDisplayCurPos.PasswordChar = ((char)(resources.GetObject("txtDisplayCurPos.PasswordChar")));
			this.txtDisplayCurPos.ReadOnly = true;
			this.txtDisplayCurPos.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtDisplayCurPos.RightToLeft")));
			this.txtDisplayCurPos.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtDisplayCurPos.ScrollBars")));
			this.txtDisplayCurPos.Size = ((System.Drawing.Size)(resources.GetObject("txtDisplayCurPos.Size")));
			this.txtDisplayCurPos.TabIndex = ((int)(resources.GetObject("txtDisplayCurPos.TabIndex")));
			this.txtDisplayCurPos.Text = resources.GetString("txtDisplayCurPos.Text");
			this.txtDisplayCurPos.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtDisplayCurPos.TextAlign")));
			this.toolTip1.SetToolTip(this.txtDisplayCurPos, resources.GetString("txtDisplayCurPos.ToolTip"));
			this.txtDisplayCurPos.Visible = ((bool)(resources.GetObject("txtDisplayCurPos.Visible")));
			this.txtDisplayCurPos.WordWrap = ((bool)(resources.GetObject("txtDisplayCurPos.WordWrap")));
			this.txtDisplayCurPos.GotFocus += new System.EventHandler(this.HideFocus);
			// 
			// txtDisplayPeak
			// 
			this.txtDisplayPeak.AccessibleDescription = resources.GetString("txtDisplayPeak.AccessibleDescription");
			this.txtDisplayPeak.AccessibleName = resources.GetString("txtDisplayPeak.AccessibleName");
			this.txtDisplayPeak.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtDisplayPeak.Anchor")));
			this.txtDisplayPeak.AutoSize = ((bool)(resources.GetObject("txtDisplayPeak.AutoSize")));
			this.txtDisplayPeak.BackColor = System.Drawing.Color.Black;
			this.txtDisplayPeak.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtDisplayPeak.BackgroundImage")));
			this.txtDisplayPeak.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
			this.txtDisplayPeak.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtDisplayPeak.Dock")));
			this.txtDisplayPeak.Enabled = ((bool)(resources.GetObject("txtDisplayPeak.Enabled")));
			this.txtDisplayPeak.Font = ((System.Drawing.Font)(resources.GetObject("txtDisplayPeak.Font")));
			this.txtDisplayPeak.ForeColor = System.Drawing.Color.DodgerBlue;
			this.txtDisplayPeak.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtDisplayPeak.ImeMode")));
			this.txtDisplayPeak.Location = ((System.Drawing.Point)(resources.GetObject("txtDisplayPeak.Location")));
			this.txtDisplayPeak.MaxLength = ((int)(resources.GetObject("txtDisplayPeak.MaxLength")));
			this.txtDisplayPeak.Multiline = ((bool)(resources.GetObject("txtDisplayPeak.Multiline")));
			this.txtDisplayPeak.Name = "txtDisplayPeak";
			this.txtDisplayPeak.PasswordChar = ((char)(resources.GetObject("txtDisplayPeak.PasswordChar")));
			this.txtDisplayPeak.ReadOnly = true;
			this.txtDisplayPeak.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtDisplayPeak.RightToLeft")));
			this.txtDisplayPeak.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtDisplayPeak.ScrollBars")));
			this.txtDisplayPeak.Size = ((System.Drawing.Size)(resources.GetObject("txtDisplayPeak.Size")));
			this.txtDisplayPeak.TabIndex = ((int)(resources.GetObject("txtDisplayPeak.TabIndex")));
			this.txtDisplayPeak.Text = resources.GetString("txtDisplayPeak.Text");
			this.txtDisplayPeak.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtDisplayPeak.TextAlign")));
			this.toolTip1.SetToolTip(this.txtDisplayPeak, resources.GetString("txtDisplayPeak.ToolTip"));
			this.txtDisplayPeak.Visible = ((bool)(resources.GetObject("txtDisplayPeak.Visible")));
			this.txtDisplayPeak.WordWrap = ((bool)(resources.GetObject("txtDisplayPeak.WordWrap")));
			this.txtDisplayPeak.GotFocus += new System.EventHandler(this.HideFocus);
			// 
			// picDisplay
			// 
			this.picDisplay.AccessibleDescription = resources.GetString("picDisplay.AccessibleDescription");
			this.picDisplay.AccessibleName = resources.GetString("picDisplay.AccessibleName");
			this.picDisplay.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("picDisplay.Anchor")));
			this.picDisplay.BackColor = System.Drawing.Color.Black;
			this.picDisplay.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("picDisplay.BackgroundImage")));
			this.picDisplay.Cursor = System.Windows.Forms.Cursors.Cross;
			this.picDisplay.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("picDisplay.Dock")));
			this.picDisplay.Enabled = ((bool)(resources.GetObject("picDisplay.Enabled")));
			this.picDisplay.Font = ((System.Drawing.Font)(resources.GetObject("picDisplay.Font")));
			this.picDisplay.Image = ((System.Drawing.Image)(resources.GetObject("picDisplay.Image")));
			this.picDisplay.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("picDisplay.ImeMode")));
			this.picDisplay.Location = ((System.Drawing.Point)(resources.GetObject("picDisplay.Location")));
			this.picDisplay.Name = "picDisplay";
			this.picDisplay.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("picDisplay.RightToLeft")));
			this.picDisplay.Size = ((System.Drawing.Size)(resources.GetObject("picDisplay.Size")));
			this.picDisplay.SizeMode = ((System.Windows.Forms.PictureBoxSizeMode)(resources.GetObject("picDisplay.SizeMode")));
			this.picDisplay.TabIndex = ((int)(resources.GetObject("picDisplay.TabIndex")));
			this.picDisplay.TabStop = false;
			this.picDisplay.Text = resources.GetString("picDisplay.Text");
			this.toolTip1.SetToolTip(this.picDisplay, resources.GetString("picDisplay.ToolTip"));
			this.picDisplay.Visible = ((bool)(resources.GetObject("picDisplay.Visible")));
			this.picDisplay.Paint += new System.Windows.Forms.PaintEventHandler(this.picDisplay_Paint);
			this.picDisplay.MouseMove += new System.Windows.Forms.MouseEventHandler(this.picDisplay_MouseMove);
			this.picDisplay.MouseLeave += new System.EventHandler(this.picDisplay_MouseLeave);
			this.picDisplay.MouseDown += new System.Windows.Forms.MouseEventHandler(this.picDisplay_MouseDown);
			// 
			// mainMenu1
			// 
			this.mainMenu1.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					  this.menuItem1,
																					  this.menuItem3,
																					  this.menuItem2});
			this.mainMenu1.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("mainMenu1.RightToLeft")));
			// 
			// menuItem1
			// 
			this.menuItem1.Enabled = ((bool)(resources.GetObject("menuItem1.Enabled")));
			this.menuItem1.Index = 0;
			this.menuItem1.Shortcut = ((System.Windows.Forms.Shortcut)(resources.GetObject("menuItem1.Shortcut")));
			this.menuItem1.ShowShortcut = ((bool)(resources.GetObject("menuItem1.ShowShortcut")));
			this.menuItem1.Text = resources.GetString("menuItem1.Text");
			this.menuItem1.Visible = ((bool)(resources.GetObject("menuItem1.Visible")));
			this.menuItem1.Click += new System.EventHandler(this.menuItem1_Click);
			// 
			// menuItem3
			// 
			this.menuItem3.Enabled = ((bool)(resources.GetObject("menuItem3.Enabled")));
			this.menuItem3.Index = 1;
			this.menuItem3.Shortcut = ((System.Windows.Forms.Shortcut)(resources.GetObject("menuItem3.Shortcut")));
			this.menuItem3.ShowShortcut = ((bool)(resources.GetObject("menuItem3.ShowShortcut")));
			this.menuItem3.Text = resources.GetString("menuItem3.Text");
			this.menuItem3.Visible = ((bool)(resources.GetObject("menuItem3.Visible")));
			this.menuItem3.Click += new System.EventHandler(this.menuItem3_Click);
			// 
			// menuItem2
			// 
			this.menuItem2.Enabled = ((bool)(resources.GetObject("menuItem2.Enabled")));
			this.menuItem2.Index = 2;
			this.menuItem2.Shortcut = ((System.Windows.Forms.Shortcut)(resources.GetObject("menuItem2.Shortcut")));
			this.menuItem2.ShowShortcut = ((bool)(resources.GetObject("menuItem2.ShowShortcut")));
			this.menuItem2.Text = resources.GetString("menuItem2.Text");
			this.menuItem2.Visible = ((bool)(resources.GetObject("menuItem2.Visible")));
			this.menuItem2.Click += new System.EventHandler(this.menuItem2_Click);
			// 
			// grpMode
			// 
			this.grpMode.AccessibleDescription = resources.GetString("grpMode.AccessibleDescription");
			this.grpMode.AccessibleName = resources.GetString("grpMode.AccessibleName");
			this.grpMode.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpMode.Anchor")));
			this.grpMode.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpMode.BackgroundImage")));
			this.grpMode.Controls.Add(this.radModeAM);
			this.grpMode.Controls.Add(this.radModeSAM);
			this.grpMode.Controls.Add(this.radModeDSB);
			this.grpMode.Controls.Add(this.radModeCWU);
			this.grpMode.Controls.Add(this.radModePSK);
			this.grpMode.Controls.Add(this.radModeRTTY);
			this.grpMode.Controls.Add(this.radModeLSB);
			this.grpMode.Controls.Add(this.radModeSPEC);
			this.grpMode.Controls.Add(this.radModeDRM);
			this.grpMode.Controls.Add(this.radModeFMN);
			this.grpMode.Controls.Add(this.radModeUSB);
			this.grpMode.Controls.Add(this.radModeCWL);
			this.grpMode.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpMode.Dock")));
			this.grpMode.Enabled = ((bool)(resources.GetObject("grpMode.Enabled")));
			this.grpMode.Font = ((System.Drawing.Font)(resources.GetObject("grpMode.Font")));
			this.grpMode.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpMode.ImeMode")));
			this.grpMode.Location = ((System.Drawing.Point)(resources.GetObject("grpMode.Location")));
			this.grpMode.Name = "grpMode";
			this.grpMode.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpMode.RightToLeft")));
			this.grpMode.Size = ((System.Drawing.Size)(resources.GetObject("grpMode.Size")));
			this.grpMode.TabIndex = ((int)(resources.GetObject("grpMode.TabIndex")));
			this.grpMode.TabStop = false;
			this.grpMode.Text = resources.GetString("grpMode.Text");
			this.toolTip1.SetToolTip(this.grpMode, resources.GetString("grpMode.ToolTip"));
			this.grpMode.Visible = ((bool)(resources.GetObject("grpMode.Visible")));
			// 
			// radModeAM
			// 
			this.radModeAM.AccessibleDescription = resources.GetString("radModeAM.AccessibleDescription");
			this.radModeAM.AccessibleName = resources.GetString("radModeAM.AccessibleName");
			this.radModeAM.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModeAM.Anchor")));
			this.radModeAM.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModeAM.Appearance")));
			this.radModeAM.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModeAM.BackgroundImage")));
			this.radModeAM.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeAM.CheckAlign")));
			this.radModeAM.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModeAM.Dock")));
			this.radModeAM.Enabled = ((bool)(resources.GetObject("radModeAM.Enabled")));
			this.radModeAM.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModeAM.FlatStyle")));
			this.radModeAM.Font = ((System.Drawing.Font)(resources.GetObject("radModeAM.Font")));
			this.radModeAM.Image = ((System.Drawing.Image)(resources.GetObject("radModeAM.Image")));
			this.radModeAM.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeAM.ImageAlign")));
			this.radModeAM.ImageIndex = ((int)(resources.GetObject("radModeAM.ImageIndex")));
			this.radModeAM.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModeAM.ImeMode")));
			this.radModeAM.Location = ((System.Drawing.Point)(resources.GetObject("radModeAM.Location")));
			this.radModeAM.Name = "radModeAM";
			this.radModeAM.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModeAM.RightToLeft")));
			this.radModeAM.Size = ((System.Drawing.Size)(resources.GetObject("radModeAM.Size")));
			this.radModeAM.TabIndex = ((int)(resources.GetObject("radModeAM.TabIndex")));
			this.radModeAM.Text = resources.GetString("radModeAM.Text");
			this.radModeAM.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeAM.TextAlign")));
			this.toolTip1.SetToolTip(this.radModeAM, resources.GetString("radModeAM.ToolTip"));
			this.radModeAM.Visible = ((bool)(resources.GetObject("radModeAM.Visible")));
			this.radModeAM.CheckedChanged += new System.EventHandler(this.radModeAM_CheckedChanged);
			// 
			// radModeSAM
			// 
			this.radModeSAM.AccessibleDescription = resources.GetString("radModeSAM.AccessibleDescription");
			this.radModeSAM.AccessibleName = resources.GetString("radModeSAM.AccessibleName");
			this.radModeSAM.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModeSAM.Anchor")));
			this.radModeSAM.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModeSAM.Appearance")));
			this.radModeSAM.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModeSAM.BackgroundImage")));
			this.radModeSAM.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeSAM.CheckAlign")));
			this.radModeSAM.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModeSAM.Dock")));
			this.radModeSAM.Enabled = ((bool)(resources.GetObject("radModeSAM.Enabled")));
			this.radModeSAM.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModeSAM.FlatStyle")));
			this.radModeSAM.Font = ((System.Drawing.Font)(resources.GetObject("radModeSAM.Font")));
			this.radModeSAM.Image = ((System.Drawing.Image)(resources.GetObject("radModeSAM.Image")));
			this.radModeSAM.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeSAM.ImageAlign")));
			this.radModeSAM.ImageIndex = ((int)(resources.GetObject("radModeSAM.ImageIndex")));
			this.radModeSAM.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModeSAM.ImeMode")));
			this.radModeSAM.Location = ((System.Drawing.Point)(resources.GetObject("radModeSAM.Location")));
			this.radModeSAM.Name = "radModeSAM";
			this.radModeSAM.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModeSAM.RightToLeft")));
			this.radModeSAM.Size = ((System.Drawing.Size)(resources.GetObject("radModeSAM.Size")));
			this.radModeSAM.TabIndex = ((int)(resources.GetObject("radModeSAM.TabIndex")));
			this.radModeSAM.Text = resources.GetString("radModeSAM.Text");
			this.radModeSAM.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeSAM.TextAlign")));
			this.toolTip1.SetToolTip(this.radModeSAM, resources.GetString("radModeSAM.ToolTip"));
			this.radModeSAM.Visible = ((bool)(resources.GetObject("radModeSAM.Visible")));
			this.radModeSAM.CheckedChanged += new System.EventHandler(this.radModeSAM_CheckedChanged);
			// 
			// radModeDSB
			// 
			this.radModeDSB.AccessibleDescription = resources.GetString("radModeDSB.AccessibleDescription");
			this.radModeDSB.AccessibleName = resources.GetString("radModeDSB.AccessibleName");
			this.radModeDSB.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModeDSB.Anchor")));
			this.radModeDSB.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModeDSB.Appearance")));
			this.radModeDSB.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModeDSB.BackgroundImage")));
			this.radModeDSB.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeDSB.CheckAlign")));
			this.radModeDSB.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModeDSB.Dock")));
			this.radModeDSB.Enabled = ((bool)(resources.GetObject("radModeDSB.Enabled")));
			this.radModeDSB.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModeDSB.FlatStyle")));
			this.radModeDSB.Font = ((System.Drawing.Font)(resources.GetObject("radModeDSB.Font")));
			this.radModeDSB.Image = ((System.Drawing.Image)(resources.GetObject("radModeDSB.Image")));
			this.radModeDSB.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeDSB.ImageAlign")));
			this.radModeDSB.ImageIndex = ((int)(resources.GetObject("radModeDSB.ImageIndex")));
			this.radModeDSB.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModeDSB.ImeMode")));
			this.radModeDSB.Location = ((System.Drawing.Point)(resources.GetObject("radModeDSB.Location")));
			this.radModeDSB.Name = "radModeDSB";
			this.radModeDSB.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModeDSB.RightToLeft")));
			this.radModeDSB.Size = ((System.Drawing.Size)(resources.GetObject("radModeDSB.Size")));
			this.radModeDSB.TabIndex = ((int)(resources.GetObject("radModeDSB.TabIndex")));
			this.radModeDSB.Text = resources.GetString("radModeDSB.Text");
			this.radModeDSB.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeDSB.TextAlign")));
			this.toolTip1.SetToolTip(this.radModeDSB, resources.GetString("radModeDSB.ToolTip"));
			this.radModeDSB.Visible = ((bool)(resources.GetObject("radModeDSB.Visible")));
			this.radModeDSB.CheckedChanged += new System.EventHandler(this.radModeDSB_CheckedChanged);
			// 
			// radModeCWU
			// 
			this.radModeCWU.AccessibleDescription = resources.GetString("radModeCWU.AccessibleDescription");
			this.radModeCWU.AccessibleName = resources.GetString("radModeCWU.AccessibleName");
			this.radModeCWU.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModeCWU.Anchor")));
			this.radModeCWU.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModeCWU.Appearance")));
			this.radModeCWU.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModeCWU.BackgroundImage")));
			this.radModeCWU.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeCWU.CheckAlign")));
			this.radModeCWU.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModeCWU.Dock")));
			this.radModeCWU.Enabled = ((bool)(resources.GetObject("radModeCWU.Enabled")));
			this.radModeCWU.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModeCWU.FlatStyle")));
			this.radModeCWU.Font = ((System.Drawing.Font)(resources.GetObject("radModeCWU.Font")));
			this.radModeCWU.Image = ((System.Drawing.Image)(resources.GetObject("radModeCWU.Image")));
			this.radModeCWU.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeCWU.ImageAlign")));
			this.radModeCWU.ImageIndex = ((int)(resources.GetObject("radModeCWU.ImageIndex")));
			this.radModeCWU.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModeCWU.ImeMode")));
			this.radModeCWU.Location = ((System.Drawing.Point)(resources.GetObject("radModeCWU.Location")));
			this.radModeCWU.Name = "radModeCWU";
			this.radModeCWU.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModeCWU.RightToLeft")));
			this.radModeCWU.Size = ((System.Drawing.Size)(resources.GetObject("radModeCWU.Size")));
			this.radModeCWU.TabIndex = ((int)(resources.GetObject("radModeCWU.TabIndex")));
			this.radModeCWU.Text = resources.GetString("radModeCWU.Text");
			this.radModeCWU.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeCWU.TextAlign")));
			this.toolTip1.SetToolTip(this.radModeCWU, resources.GetString("radModeCWU.ToolTip"));
			this.radModeCWU.Visible = ((bool)(resources.GetObject("radModeCWU.Visible")));
			this.radModeCWU.CheckedChanged += new System.EventHandler(this.radModeCWU_CheckedChanged);
			// 
			// radModePSK
			// 
			this.radModePSK.AccessibleDescription = resources.GetString("radModePSK.AccessibleDescription");
			this.radModePSK.AccessibleName = resources.GetString("radModePSK.AccessibleName");
			this.radModePSK.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModePSK.Anchor")));
			this.radModePSK.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModePSK.Appearance")));
			this.radModePSK.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModePSK.BackgroundImage")));
			this.radModePSK.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModePSK.CheckAlign")));
			this.radModePSK.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModePSK.Dock")));
			this.radModePSK.Enabled = ((bool)(resources.GetObject("radModePSK.Enabled")));
			this.radModePSK.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModePSK.FlatStyle")));
			this.radModePSK.Font = ((System.Drawing.Font)(resources.GetObject("radModePSK.Font")));
			this.radModePSK.Image = ((System.Drawing.Image)(resources.GetObject("radModePSK.Image")));
			this.radModePSK.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModePSK.ImageAlign")));
			this.radModePSK.ImageIndex = ((int)(resources.GetObject("radModePSK.ImageIndex")));
			this.radModePSK.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModePSK.ImeMode")));
			this.radModePSK.Location = ((System.Drawing.Point)(resources.GetObject("radModePSK.Location")));
			this.radModePSK.Name = "radModePSK";
			this.radModePSK.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModePSK.RightToLeft")));
			this.radModePSK.Size = ((System.Drawing.Size)(resources.GetObject("radModePSK.Size")));
			this.radModePSK.TabIndex = ((int)(resources.GetObject("radModePSK.TabIndex")));
			this.radModePSK.Text = resources.GetString("radModePSK.Text");
			this.radModePSK.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModePSK.TextAlign")));
			this.toolTip1.SetToolTip(this.radModePSK, resources.GetString("radModePSK.ToolTip"));
			this.radModePSK.Visible = ((bool)(resources.GetObject("radModePSK.Visible")));
			this.radModePSK.CheckedChanged += new System.EventHandler(this.radModePSK_CheckedChanged);
			// 
			// radModeRTTY
			// 
			this.radModeRTTY.AccessibleDescription = resources.GetString("radModeRTTY.AccessibleDescription");
			this.radModeRTTY.AccessibleName = resources.GetString("radModeRTTY.AccessibleName");
			this.radModeRTTY.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModeRTTY.Anchor")));
			this.radModeRTTY.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModeRTTY.Appearance")));
			this.radModeRTTY.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModeRTTY.BackgroundImage")));
			this.radModeRTTY.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeRTTY.CheckAlign")));
			this.radModeRTTY.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModeRTTY.Dock")));
			this.radModeRTTY.Enabled = ((bool)(resources.GetObject("radModeRTTY.Enabled")));
			this.radModeRTTY.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModeRTTY.FlatStyle")));
			this.radModeRTTY.Font = ((System.Drawing.Font)(resources.GetObject("radModeRTTY.Font")));
			this.radModeRTTY.Image = ((System.Drawing.Image)(resources.GetObject("radModeRTTY.Image")));
			this.radModeRTTY.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeRTTY.ImageAlign")));
			this.radModeRTTY.ImageIndex = ((int)(resources.GetObject("radModeRTTY.ImageIndex")));
			this.radModeRTTY.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModeRTTY.ImeMode")));
			this.radModeRTTY.Location = ((System.Drawing.Point)(resources.GetObject("radModeRTTY.Location")));
			this.radModeRTTY.Name = "radModeRTTY";
			this.radModeRTTY.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModeRTTY.RightToLeft")));
			this.radModeRTTY.Size = ((System.Drawing.Size)(resources.GetObject("radModeRTTY.Size")));
			this.radModeRTTY.TabIndex = ((int)(resources.GetObject("radModeRTTY.TabIndex")));
			this.radModeRTTY.Text = resources.GetString("radModeRTTY.Text");
			this.radModeRTTY.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeRTTY.TextAlign")));
			this.toolTip1.SetToolTip(this.radModeRTTY, resources.GetString("radModeRTTY.ToolTip"));
			this.radModeRTTY.Visible = ((bool)(resources.GetObject("radModeRTTY.Visible")));
			this.radModeRTTY.CheckedChanged += new System.EventHandler(this.radModeRTTY_CheckedChanged);
			// 
			// radModeLSB
			// 
			this.radModeLSB.AccessibleDescription = resources.GetString("radModeLSB.AccessibleDescription");
			this.radModeLSB.AccessibleName = resources.GetString("radModeLSB.AccessibleName");
			this.radModeLSB.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModeLSB.Anchor")));
			this.radModeLSB.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModeLSB.Appearance")));
			this.radModeLSB.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModeLSB.BackgroundImage")));
			this.radModeLSB.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeLSB.CheckAlign")));
			this.radModeLSB.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModeLSB.Dock")));
			this.radModeLSB.Enabled = ((bool)(resources.GetObject("radModeLSB.Enabled")));
			this.radModeLSB.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModeLSB.FlatStyle")));
			this.radModeLSB.Font = ((System.Drawing.Font)(resources.GetObject("radModeLSB.Font")));
			this.radModeLSB.Image = ((System.Drawing.Image)(resources.GetObject("radModeLSB.Image")));
			this.radModeLSB.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeLSB.ImageAlign")));
			this.radModeLSB.ImageIndex = ((int)(resources.GetObject("radModeLSB.ImageIndex")));
			this.radModeLSB.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModeLSB.ImeMode")));
			this.radModeLSB.Location = ((System.Drawing.Point)(resources.GetObject("radModeLSB.Location")));
			this.radModeLSB.Name = "radModeLSB";
			this.radModeLSB.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModeLSB.RightToLeft")));
			this.radModeLSB.Size = ((System.Drawing.Size)(resources.GetObject("radModeLSB.Size")));
			this.radModeLSB.TabIndex = ((int)(resources.GetObject("radModeLSB.TabIndex")));
			this.radModeLSB.Text = resources.GetString("radModeLSB.Text");
			this.radModeLSB.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeLSB.TextAlign")));
			this.toolTip1.SetToolTip(this.radModeLSB, resources.GetString("radModeLSB.ToolTip"));
			this.radModeLSB.Visible = ((bool)(resources.GetObject("radModeLSB.Visible")));
			this.radModeLSB.CheckedChanged += new System.EventHandler(this.radModeLSB_CheckedChanged);
			// 
			// radModeSPEC
			// 
			this.radModeSPEC.AccessibleDescription = resources.GetString("radModeSPEC.AccessibleDescription");
			this.radModeSPEC.AccessibleName = resources.GetString("radModeSPEC.AccessibleName");
			this.radModeSPEC.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModeSPEC.Anchor")));
			this.radModeSPEC.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModeSPEC.Appearance")));
			this.radModeSPEC.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModeSPEC.BackgroundImage")));
			this.radModeSPEC.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeSPEC.CheckAlign")));
			this.radModeSPEC.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModeSPEC.Dock")));
			this.radModeSPEC.Enabled = ((bool)(resources.GetObject("radModeSPEC.Enabled")));
			this.radModeSPEC.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModeSPEC.FlatStyle")));
			this.radModeSPEC.Font = ((System.Drawing.Font)(resources.GetObject("radModeSPEC.Font")));
			this.radModeSPEC.Image = ((System.Drawing.Image)(resources.GetObject("radModeSPEC.Image")));
			this.radModeSPEC.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeSPEC.ImageAlign")));
			this.radModeSPEC.ImageIndex = ((int)(resources.GetObject("radModeSPEC.ImageIndex")));
			this.radModeSPEC.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModeSPEC.ImeMode")));
			this.radModeSPEC.Location = ((System.Drawing.Point)(resources.GetObject("radModeSPEC.Location")));
			this.radModeSPEC.Name = "radModeSPEC";
			this.radModeSPEC.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModeSPEC.RightToLeft")));
			this.radModeSPEC.Size = ((System.Drawing.Size)(resources.GetObject("radModeSPEC.Size")));
			this.radModeSPEC.TabIndex = ((int)(resources.GetObject("radModeSPEC.TabIndex")));
			this.radModeSPEC.Text = resources.GetString("radModeSPEC.Text");
			this.radModeSPEC.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeSPEC.TextAlign")));
			this.toolTip1.SetToolTip(this.radModeSPEC, resources.GetString("radModeSPEC.ToolTip"));
			this.radModeSPEC.Visible = ((bool)(resources.GetObject("radModeSPEC.Visible")));
			this.radModeSPEC.CheckedChanged += new System.EventHandler(this.radModeSPEC_CheckedChanged);
			// 
			// radModeDRM
			// 
			this.radModeDRM.AccessibleDescription = resources.GetString("radModeDRM.AccessibleDescription");
			this.radModeDRM.AccessibleName = resources.GetString("radModeDRM.AccessibleName");
			this.radModeDRM.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModeDRM.Anchor")));
			this.radModeDRM.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModeDRM.Appearance")));
			this.radModeDRM.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModeDRM.BackgroundImage")));
			this.radModeDRM.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeDRM.CheckAlign")));
			this.radModeDRM.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModeDRM.Dock")));
			this.radModeDRM.Enabled = ((bool)(resources.GetObject("radModeDRM.Enabled")));
			this.radModeDRM.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModeDRM.FlatStyle")));
			this.radModeDRM.Font = ((System.Drawing.Font)(resources.GetObject("radModeDRM.Font")));
			this.radModeDRM.Image = ((System.Drawing.Image)(resources.GetObject("radModeDRM.Image")));
			this.radModeDRM.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeDRM.ImageAlign")));
			this.radModeDRM.ImageIndex = ((int)(resources.GetObject("radModeDRM.ImageIndex")));
			this.radModeDRM.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModeDRM.ImeMode")));
			this.radModeDRM.Location = ((System.Drawing.Point)(resources.GetObject("radModeDRM.Location")));
			this.radModeDRM.Name = "radModeDRM";
			this.radModeDRM.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModeDRM.RightToLeft")));
			this.radModeDRM.Size = ((System.Drawing.Size)(resources.GetObject("radModeDRM.Size")));
			this.radModeDRM.TabIndex = ((int)(resources.GetObject("radModeDRM.TabIndex")));
			this.radModeDRM.Text = resources.GetString("radModeDRM.Text");
			this.radModeDRM.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeDRM.TextAlign")));
			this.toolTip1.SetToolTip(this.radModeDRM, resources.GetString("radModeDRM.ToolTip"));
			this.radModeDRM.Visible = ((bool)(resources.GetObject("radModeDRM.Visible")));
			this.radModeDRM.CheckedChanged += new System.EventHandler(this.radModeDRM_CheckedChanged);
			// 
			// radModeFMN
			// 
			this.radModeFMN.AccessibleDescription = resources.GetString("radModeFMN.AccessibleDescription");
			this.radModeFMN.AccessibleName = resources.GetString("radModeFMN.AccessibleName");
			this.radModeFMN.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModeFMN.Anchor")));
			this.radModeFMN.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModeFMN.Appearance")));
			this.radModeFMN.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModeFMN.BackgroundImage")));
			this.radModeFMN.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeFMN.CheckAlign")));
			this.radModeFMN.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModeFMN.Dock")));
			this.radModeFMN.Enabled = ((bool)(resources.GetObject("radModeFMN.Enabled")));
			this.radModeFMN.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModeFMN.FlatStyle")));
			this.radModeFMN.Font = ((System.Drawing.Font)(resources.GetObject("radModeFMN.Font")));
			this.radModeFMN.Image = ((System.Drawing.Image)(resources.GetObject("radModeFMN.Image")));
			this.radModeFMN.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeFMN.ImageAlign")));
			this.radModeFMN.ImageIndex = ((int)(resources.GetObject("radModeFMN.ImageIndex")));
			this.radModeFMN.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModeFMN.ImeMode")));
			this.radModeFMN.Location = ((System.Drawing.Point)(resources.GetObject("radModeFMN.Location")));
			this.radModeFMN.Name = "radModeFMN";
			this.radModeFMN.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModeFMN.RightToLeft")));
			this.radModeFMN.Size = ((System.Drawing.Size)(resources.GetObject("radModeFMN.Size")));
			this.radModeFMN.TabIndex = ((int)(resources.GetObject("radModeFMN.TabIndex")));
			this.radModeFMN.Text = resources.GetString("radModeFMN.Text");
			this.radModeFMN.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeFMN.TextAlign")));
			this.toolTip1.SetToolTip(this.radModeFMN, resources.GetString("radModeFMN.ToolTip"));
			this.radModeFMN.Visible = ((bool)(resources.GetObject("radModeFMN.Visible")));
			this.radModeFMN.CheckedChanged += new System.EventHandler(this.radModeFMN_CheckedChanged);
			// 
			// radModeUSB
			// 
			this.radModeUSB.AccessibleDescription = resources.GetString("radModeUSB.AccessibleDescription");
			this.radModeUSB.AccessibleName = resources.GetString("radModeUSB.AccessibleName");
			this.radModeUSB.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModeUSB.Anchor")));
			this.radModeUSB.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModeUSB.Appearance")));
			this.radModeUSB.BackColor = System.Drawing.SystemColors.Control;
			this.radModeUSB.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModeUSB.BackgroundImage")));
			this.radModeUSB.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeUSB.CheckAlign")));
			this.radModeUSB.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModeUSB.Dock")));
			this.radModeUSB.Enabled = ((bool)(resources.GetObject("radModeUSB.Enabled")));
			this.radModeUSB.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModeUSB.FlatStyle")));
			this.radModeUSB.Font = ((System.Drawing.Font)(resources.GetObject("radModeUSB.Font")));
			this.radModeUSB.Image = ((System.Drawing.Image)(resources.GetObject("radModeUSB.Image")));
			this.radModeUSB.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeUSB.ImageAlign")));
			this.radModeUSB.ImageIndex = ((int)(resources.GetObject("radModeUSB.ImageIndex")));
			this.radModeUSB.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModeUSB.ImeMode")));
			this.radModeUSB.Location = ((System.Drawing.Point)(resources.GetObject("radModeUSB.Location")));
			this.radModeUSB.Name = "radModeUSB";
			this.radModeUSB.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModeUSB.RightToLeft")));
			this.radModeUSB.Size = ((System.Drawing.Size)(resources.GetObject("radModeUSB.Size")));
			this.radModeUSB.TabIndex = ((int)(resources.GetObject("radModeUSB.TabIndex")));
			this.radModeUSB.Text = resources.GetString("radModeUSB.Text");
			this.radModeUSB.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeUSB.TextAlign")));
			this.toolTip1.SetToolTip(this.radModeUSB, resources.GetString("radModeUSB.ToolTip"));
			this.radModeUSB.Visible = ((bool)(resources.GetObject("radModeUSB.Visible")));
			this.radModeUSB.CheckedChanged += new System.EventHandler(this.radModeUSB_CheckedChanged);
			// 
			// radModeCWL
			// 
			this.radModeCWL.AccessibleDescription = resources.GetString("radModeCWL.AccessibleDescription");
			this.radModeCWL.AccessibleName = resources.GetString("radModeCWL.AccessibleName");
			this.radModeCWL.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radModeCWL.Anchor")));
			this.radModeCWL.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radModeCWL.Appearance")));
			this.radModeCWL.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radModeCWL.BackgroundImage")));
			this.radModeCWL.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeCWL.CheckAlign")));
			this.radModeCWL.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radModeCWL.Dock")));
			this.radModeCWL.Enabled = ((bool)(resources.GetObject("radModeCWL.Enabled")));
			this.radModeCWL.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radModeCWL.FlatStyle")));
			this.radModeCWL.Font = ((System.Drawing.Font)(resources.GetObject("radModeCWL.Font")));
			this.radModeCWL.Image = ((System.Drawing.Image)(resources.GetObject("radModeCWL.Image")));
			this.radModeCWL.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeCWL.ImageAlign")));
			this.radModeCWL.ImageIndex = ((int)(resources.GetObject("radModeCWL.ImageIndex")));
			this.radModeCWL.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radModeCWL.ImeMode")));
			this.radModeCWL.Location = ((System.Drawing.Point)(resources.GetObject("radModeCWL.Location")));
			this.radModeCWL.Name = "radModeCWL";
			this.radModeCWL.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radModeCWL.RightToLeft")));
			this.radModeCWL.Size = ((System.Drawing.Size)(resources.GetObject("radModeCWL.Size")));
			this.radModeCWL.TabIndex = ((int)(resources.GetObject("radModeCWL.TabIndex")));
			this.radModeCWL.Text = resources.GetString("radModeCWL.Text");
			this.radModeCWL.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radModeCWL.TextAlign")));
			this.toolTip1.SetToolTip(this.radModeCWL, resources.GetString("radModeCWL.ToolTip"));
			this.radModeCWL.Visible = ((bool)(resources.GetObject("radModeCWL.Visible")));
			this.radModeCWL.CheckedChanged += new System.EventHandler(this.radModeCWL_CheckedChanged);
			// 
			// grpBand
			// 
			this.grpBand.AccessibleDescription = resources.GetString("grpBand.AccessibleDescription");
			this.grpBand.AccessibleName = resources.GetString("grpBand.AccessibleName");
			this.grpBand.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpBand.Anchor")));
			this.grpBand.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpBand.BackgroundImage")));
			this.grpBand.Controls.Add(this.btnBandGEN);
			this.grpBand.Controls.Add(this.btnBandWWV);
			this.grpBand.Controls.Add(this.btnBand13);
			this.grpBand.Controls.Add(this.btnBand2);
			this.grpBand.Controls.Add(this.btnBand6);
			this.grpBand.Controls.Add(this.btnBand10);
			this.grpBand.Controls.Add(this.btnBand12);
			this.grpBand.Controls.Add(this.btnBand15);
			this.grpBand.Controls.Add(this.btnBand17);
			this.grpBand.Controls.Add(this.btnBand20);
			this.grpBand.Controls.Add(this.btnBand30);
			this.grpBand.Controls.Add(this.btnBand40);
			this.grpBand.Controls.Add(this.btnBand60);
			this.grpBand.Controls.Add(this.btnBand80);
			this.grpBand.Controls.Add(this.btnBand160);
			this.grpBand.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpBand.Dock")));
			this.grpBand.Enabled = ((bool)(resources.GetObject("grpBand.Enabled")));
			this.grpBand.Font = ((System.Drawing.Font)(resources.GetObject("grpBand.Font")));
			this.grpBand.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpBand.ImeMode")));
			this.grpBand.Location = ((System.Drawing.Point)(resources.GetObject("grpBand.Location")));
			this.grpBand.Name = "grpBand";
			this.grpBand.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpBand.RightToLeft")));
			this.grpBand.Size = ((System.Drawing.Size)(resources.GetObject("grpBand.Size")));
			this.grpBand.TabIndex = ((int)(resources.GetObject("grpBand.TabIndex")));
			this.grpBand.TabStop = false;
			this.grpBand.Text = resources.GetString("grpBand.Text");
			this.toolTip1.SetToolTip(this.grpBand, resources.GetString("grpBand.ToolTip"));
			this.grpBand.Visible = ((bool)(resources.GetObject("grpBand.Visible")));
			this.grpBand.Enter += new System.EventHandler(this.grpBand_Enter);
			// 
			// btnBandGEN
			// 
			this.btnBandGEN.AccessibleDescription = resources.GetString("btnBandGEN.AccessibleDescription");
			this.btnBandGEN.AccessibleName = resources.GetString("btnBandGEN.AccessibleName");
			this.btnBandGEN.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBandGEN.Anchor")));
			this.btnBandGEN.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBandGEN.BackgroundImage")));
			this.btnBandGEN.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBandGEN.Dock")));
			this.btnBandGEN.Enabled = ((bool)(resources.GetObject("btnBandGEN.Enabled")));
			this.btnBandGEN.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBandGEN.FlatStyle")));
			this.btnBandGEN.Font = ((System.Drawing.Font)(resources.GetObject("btnBandGEN.Font")));
			this.btnBandGEN.Image = ((System.Drawing.Image)(resources.GetObject("btnBandGEN.Image")));
			this.btnBandGEN.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBandGEN.ImageAlign")));
			this.btnBandGEN.ImageIndex = ((int)(resources.GetObject("btnBandGEN.ImageIndex")));
			this.btnBandGEN.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBandGEN.ImeMode")));
			this.btnBandGEN.Location = ((System.Drawing.Point)(resources.GetObject("btnBandGEN.Location")));
			this.btnBandGEN.Name = "btnBandGEN";
			this.btnBandGEN.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBandGEN.RightToLeft")));
			this.btnBandGEN.Size = ((System.Drawing.Size)(resources.GetObject("btnBandGEN.Size")));
			this.btnBandGEN.TabIndex = ((int)(resources.GetObject("btnBandGEN.TabIndex")));
			this.btnBandGEN.Text = resources.GetString("btnBandGEN.Text");
			this.btnBandGEN.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBandGEN.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBandGEN, resources.GetString("btnBandGEN.ToolTip"));
			this.btnBandGEN.Visible = ((bool)(resources.GetObject("btnBandGEN.Visible")));
			this.btnBandGEN.Click += new System.EventHandler(this.btnBandGEN_Click);
			// 
			// btnBandWWV
			// 
			this.btnBandWWV.AccessibleDescription = resources.GetString("btnBandWWV.AccessibleDescription");
			this.btnBandWWV.AccessibleName = resources.GetString("btnBandWWV.AccessibleName");
			this.btnBandWWV.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBandWWV.Anchor")));
			this.btnBandWWV.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBandWWV.BackgroundImage")));
			this.btnBandWWV.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBandWWV.Dock")));
			this.btnBandWWV.Enabled = ((bool)(resources.GetObject("btnBandWWV.Enabled")));
			this.btnBandWWV.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBandWWV.FlatStyle")));
			this.btnBandWWV.Font = ((System.Drawing.Font)(resources.GetObject("btnBandWWV.Font")));
			this.btnBandWWV.Image = ((System.Drawing.Image)(resources.GetObject("btnBandWWV.Image")));
			this.btnBandWWV.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBandWWV.ImageAlign")));
			this.btnBandWWV.ImageIndex = ((int)(resources.GetObject("btnBandWWV.ImageIndex")));
			this.btnBandWWV.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBandWWV.ImeMode")));
			this.btnBandWWV.Location = ((System.Drawing.Point)(resources.GetObject("btnBandWWV.Location")));
			this.btnBandWWV.Name = "btnBandWWV";
			this.btnBandWWV.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBandWWV.RightToLeft")));
			this.btnBandWWV.Size = ((System.Drawing.Size)(resources.GetObject("btnBandWWV.Size")));
			this.btnBandWWV.TabIndex = ((int)(resources.GetObject("btnBandWWV.TabIndex")));
			this.btnBandWWV.Text = resources.GetString("btnBandWWV.Text");
			this.btnBandWWV.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBandWWV.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBandWWV, resources.GetString("btnBandWWV.ToolTip"));
			this.btnBandWWV.Visible = ((bool)(resources.GetObject("btnBandWWV.Visible")));
			this.btnBandWWV.Click += new System.EventHandler(this.btnBandWWV_Click);
			// 
			// btnBand13
			// 
			this.btnBand13.AccessibleDescription = resources.GetString("btnBand13.AccessibleDescription");
			this.btnBand13.AccessibleName = resources.GetString("btnBand13.AccessibleName");
			this.btnBand13.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand13.Anchor")));
			this.btnBand13.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand13.BackgroundImage")));
			this.btnBand13.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand13.Dock")));
			this.btnBand13.Enabled = ((bool)(resources.GetObject("btnBand13.Enabled")));
			this.btnBand13.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand13.FlatStyle")));
			this.btnBand13.Font = ((System.Drawing.Font)(resources.GetObject("btnBand13.Font")));
			this.btnBand13.Image = ((System.Drawing.Image)(resources.GetObject("btnBand13.Image")));
			this.btnBand13.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand13.ImageAlign")));
			this.btnBand13.ImageIndex = ((int)(resources.GetObject("btnBand13.ImageIndex")));
			this.btnBand13.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand13.ImeMode")));
			this.btnBand13.Location = ((System.Drawing.Point)(resources.GetObject("btnBand13.Location")));
			this.btnBand13.Name = "btnBand13";
			this.btnBand13.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand13.RightToLeft")));
			this.btnBand13.Size = ((System.Drawing.Size)(resources.GetObject("btnBand13.Size")));
			this.btnBand13.TabIndex = ((int)(resources.GetObject("btnBand13.TabIndex")));
			this.btnBand13.Text = resources.GetString("btnBand13.Text");
			this.btnBand13.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand13.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand13, resources.GetString("btnBand13.ToolTip"));
			this.btnBand13.Visible = ((bool)(resources.GetObject("btnBand13.Visible")));
			// 
			// btnBand2
			// 
			this.btnBand2.AccessibleDescription = resources.GetString("btnBand2.AccessibleDescription");
			this.btnBand2.AccessibleName = resources.GetString("btnBand2.AccessibleName");
			this.btnBand2.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand2.Anchor")));
			this.btnBand2.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand2.BackgroundImage")));
			this.btnBand2.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand2.Dock")));
			this.btnBand2.Enabled = ((bool)(resources.GetObject("btnBand2.Enabled")));
			this.btnBand2.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand2.FlatStyle")));
			this.btnBand2.Font = ((System.Drawing.Font)(resources.GetObject("btnBand2.Font")));
			this.btnBand2.Image = ((System.Drawing.Image)(resources.GetObject("btnBand2.Image")));
			this.btnBand2.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand2.ImageAlign")));
			this.btnBand2.ImageIndex = ((int)(resources.GetObject("btnBand2.ImageIndex")));
			this.btnBand2.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand2.ImeMode")));
			this.btnBand2.Location = ((System.Drawing.Point)(resources.GetObject("btnBand2.Location")));
			this.btnBand2.Name = "btnBand2";
			this.btnBand2.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand2.RightToLeft")));
			this.btnBand2.Size = ((System.Drawing.Size)(resources.GetObject("btnBand2.Size")));
			this.btnBand2.TabIndex = ((int)(resources.GetObject("btnBand2.TabIndex")));
			this.btnBand2.Text = resources.GetString("btnBand2.Text");
			this.btnBand2.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand2.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand2, resources.GetString("btnBand2.ToolTip"));
			this.btnBand2.Visible = ((bool)(resources.GetObject("btnBand2.Visible")));
			this.btnBand2.Click += new System.EventHandler(this.btnBand2_Click);
			// 
			// btnBand6
			// 
			this.btnBand6.AccessibleDescription = resources.GetString("btnBand6.AccessibleDescription");
			this.btnBand6.AccessibleName = resources.GetString("btnBand6.AccessibleName");
			this.btnBand6.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand6.Anchor")));
			this.btnBand6.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand6.BackgroundImage")));
			this.btnBand6.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand6.Dock")));
			this.btnBand6.Enabled = ((bool)(resources.GetObject("btnBand6.Enabled")));
			this.btnBand6.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand6.FlatStyle")));
			this.btnBand6.Font = ((System.Drawing.Font)(resources.GetObject("btnBand6.Font")));
			this.btnBand6.Image = ((System.Drawing.Image)(resources.GetObject("btnBand6.Image")));
			this.btnBand6.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand6.ImageAlign")));
			this.btnBand6.ImageIndex = ((int)(resources.GetObject("btnBand6.ImageIndex")));
			this.btnBand6.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand6.ImeMode")));
			this.btnBand6.Location = ((System.Drawing.Point)(resources.GetObject("btnBand6.Location")));
			this.btnBand6.Name = "btnBand6";
			this.btnBand6.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand6.RightToLeft")));
			this.btnBand6.Size = ((System.Drawing.Size)(resources.GetObject("btnBand6.Size")));
			this.btnBand6.TabIndex = ((int)(resources.GetObject("btnBand6.TabIndex")));
			this.btnBand6.Text = resources.GetString("btnBand6.Text");
			this.btnBand6.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand6.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand6, resources.GetString("btnBand6.ToolTip"));
			this.btnBand6.Visible = ((bool)(resources.GetObject("btnBand6.Visible")));
			this.btnBand6.Click += new System.EventHandler(this.btnBand6_Click);
			// 
			// btnBand10
			// 
			this.btnBand10.AccessibleDescription = resources.GetString("btnBand10.AccessibleDescription");
			this.btnBand10.AccessibleName = resources.GetString("btnBand10.AccessibleName");
			this.btnBand10.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand10.Anchor")));
			this.btnBand10.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand10.BackgroundImage")));
			this.btnBand10.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand10.Dock")));
			this.btnBand10.Enabled = ((bool)(resources.GetObject("btnBand10.Enabled")));
			this.btnBand10.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand10.FlatStyle")));
			this.btnBand10.Font = ((System.Drawing.Font)(resources.GetObject("btnBand10.Font")));
			this.btnBand10.Image = ((System.Drawing.Image)(resources.GetObject("btnBand10.Image")));
			this.btnBand10.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand10.ImageAlign")));
			this.btnBand10.ImageIndex = ((int)(resources.GetObject("btnBand10.ImageIndex")));
			this.btnBand10.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand10.ImeMode")));
			this.btnBand10.Location = ((System.Drawing.Point)(resources.GetObject("btnBand10.Location")));
			this.btnBand10.Name = "btnBand10";
			this.btnBand10.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand10.RightToLeft")));
			this.btnBand10.Size = ((System.Drawing.Size)(resources.GetObject("btnBand10.Size")));
			this.btnBand10.TabIndex = ((int)(resources.GetObject("btnBand10.TabIndex")));
			this.btnBand10.Text = resources.GetString("btnBand10.Text");
			this.btnBand10.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand10.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand10, resources.GetString("btnBand10.ToolTip"));
			this.btnBand10.Visible = ((bool)(resources.GetObject("btnBand10.Visible")));
			this.btnBand10.Click += new System.EventHandler(this.btnBand10_Click);
			// 
			// btnBand12
			// 
			this.btnBand12.AccessibleDescription = resources.GetString("btnBand12.AccessibleDescription");
			this.btnBand12.AccessibleName = resources.GetString("btnBand12.AccessibleName");
			this.btnBand12.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand12.Anchor")));
			this.btnBand12.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand12.BackgroundImage")));
			this.btnBand12.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand12.Dock")));
			this.btnBand12.Enabled = ((bool)(resources.GetObject("btnBand12.Enabled")));
			this.btnBand12.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand12.FlatStyle")));
			this.btnBand12.Font = ((System.Drawing.Font)(resources.GetObject("btnBand12.Font")));
			this.btnBand12.Image = ((System.Drawing.Image)(resources.GetObject("btnBand12.Image")));
			this.btnBand12.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand12.ImageAlign")));
			this.btnBand12.ImageIndex = ((int)(resources.GetObject("btnBand12.ImageIndex")));
			this.btnBand12.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand12.ImeMode")));
			this.btnBand12.Location = ((System.Drawing.Point)(resources.GetObject("btnBand12.Location")));
			this.btnBand12.Name = "btnBand12";
			this.btnBand12.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand12.RightToLeft")));
			this.btnBand12.Size = ((System.Drawing.Size)(resources.GetObject("btnBand12.Size")));
			this.btnBand12.TabIndex = ((int)(resources.GetObject("btnBand12.TabIndex")));
			this.btnBand12.Text = resources.GetString("btnBand12.Text");
			this.btnBand12.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand12.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand12, resources.GetString("btnBand12.ToolTip"));
			this.btnBand12.Visible = ((bool)(resources.GetObject("btnBand12.Visible")));
			this.btnBand12.Click += new System.EventHandler(this.btnBand12_Click);
			// 
			// btnBand15
			// 
			this.btnBand15.AccessibleDescription = resources.GetString("btnBand15.AccessibleDescription");
			this.btnBand15.AccessibleName = resources.GetString("btnBand15.AccessibleName");
			this.btnBand15.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand15.Anchor")));
			this.btnBand15.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand15.BackgroundImage")));
			this.btnBand15.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand15.Dock")));
			this.btnBand15.Enabled = ((bool)(resources.GetObject("btnBand15.Enabled")));
			this.btnBand15.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand15.FlatStyle")));
			this.btnBand15.Font = ((System.Drawing.Font)(resources.GetObject("btnBand15.Font")));
			this.btnBand15.Image = ((System.Drawing.Image)(resources.GetObject("btnBand15.Image")));
			this.btnBand15.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand15.ImageAlign")));
			this.btnBand15.ImageIndex = ((int)(resources.GetObject("btnBand15.ImageIndex")));
			this.btnBand15.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand15.ImeMode")));
			this.btnBand15.Location = ((System.Drawing.Point)(resources.GetObject("btnBand15.Location")));
			this.btnBand15.Name = "btnBand15";
			this.btnBand15.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand15.RightToLeft")));
			this.btnBand15.Size = ((System.Drawing.Size)(resources.GetObject("btnBand15.Size")));
			this.btnBand15.TabIndex = ((int)(resources.GetObject("btnBand15.TabIndex")));
			this.btnBand15.Text = resources.GetString("btnBand15.Text");
			this.btnBand15.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand15.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand15, resources.GetString("btnBand15.ToolTip"));
			this.btnBand15.Visible = ((bool)(resources.GetObject("btnBand15.Visible")));
			this.btnBand15.Click += new System.EventHandler(this.btnBand15_Click);
			// 
			// btnBand17
			// 
			this.btnBand17.AccessibleDescription = resources.GetString("btnBand17.AccessibleDescription");
			this.btnBand17.AccessibleName = resources.GetString("btnBand17.AccessibleName");
			this.btnBand17.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand17.Anchor")));
			this.btnBand17.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand17.BackgroundImage")));
			this.btnBand17.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand17.Dock")));
			this.btnBand17.Enabled = ((bool)(resources.GetObject("btnBand17.Enabled")));
			this.btnBand17.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand17.FlatStyle")));
			this.btnBand17.Font = ((System.Drawing.Font)(resources.GetObject("btnBand17.Font")));
			this.btnBand17.Image = ((System.Drawing.Image)(resources.GetObject("btnBand17.Image")));
			this.btnBand17.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand17.ImageAlign")));
			this.btnBand17.ImageIndex = ((int)(resources.GetObject("btnBand17.ImageIndex")));
			this.btnBand17.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand17.ImeMode")));
			this.btnBand17.Location = ((System.Drawing.Point)(resources.GetObject("btnBand17.Location")));
			this.btnBand17.Name = "btnBand17";
			this.btnBand17.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand17.RightToLeft")));
			this.btnBand17.Size = ((System.Drawing.Size)(resources.GetObject("btnBand17.Size")));
			this.btnBand17.TabIndex = ((int)(resources.GetObject("btnBand17.TabIndex")));
			this.btnBand17.Text = resources.GetString("btnBand17.Text");
			this.btnBand17.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand17.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand17, resources.GetString("btnBand17.ToolTip"));
			this.btnBand17.Visible = ((bool)(resources.GetObject("btnBand17.Visible")));
			this.btnBand17.Click += new System.EventHandler(this.btnBand17_Click);
			// 
			// btnBand20
			// 
			this.btnBand20.AccessibleDescription = resources.GetString("btnBand20.AccessibleDescription");
			this.btnBand20.AccessibleName = resources.GetString("btnBand20.AccessibleName");
			this.btnBand20.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand20.Anchor")));
			this.btnBand20.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand20.BackgroundImage")));
			this.btnBand20.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand20.Dock")));
			this.btnBand20.Enabled = ((bool)(resources.GetObject("btnBand20.Enabled")));
			this.btnBand20.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand20.FlatStyle")));
			this.btnBand20.Font = ((System.Drawing.Font)(resources.GetObject("btnBand20.Font")));
			this.btnBand20.Image = ((System.Drawing.Image)(resources.GetObject("btnBand20.Image")));
			this.btnBand20.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand20.ImageAlign")));
			this.btnBand20.ImageIndex = ((int)(resources.GetObject("btnBand20.ImageIndex")));
			this.btnBand20.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand20.ImeMode")));
			this.btnBand20.Location = ((System.Drawing.Point)(resources.GetObject("btnBand20.Location")));
			this.btnBand20.Name = "btnBand20";
			this.btnBand20.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand20.RightToLeft")));
			this.btnBand20.Size = ((System.Drawing.Size)(resources.GetObject("btnBand20.Size")));
			this.btnBand20.TabIndex = ((int)(resources.GetObject("btnBand20.TabIndex")));
			this.btnBand20.Text = resources.GetString("btnBand20.Text");
			this.btnBand20.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand20.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand20, resources.GetString("btnBand20.ToolTip"));
			this.btnBand20.Visible = ((bool)(resources.GetObject("btnBand20.Visible")));
			this.btnBand20.Click += new System.EventHandler(this.btnBand20_Click);
			// 
			// btnBand30
			// 
			this.btnBand30.AccessibleDescription = resources.GetString("btnBand30.AccessibleDescription");
			this.btnBand30.AccessibleName = resources.GetString("btnBand30.AccessibleName");
			this.btnBand30.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand30.Anchor")));
			this.btnBand30.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand30.BackgroundImage")));
			this.btnBand30.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand30.Dock")));
			this.btnBand30.Enabled = ((bool)(resources.GetObject("btnBand30.Enabled")));
			this.btnBand30.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand30.FlatStyle")));
			this.btnBand30.Font = ((System.Drawing.Font)(resources.GetObject("btnBand30.Font")));
			this.btnBand30.Image = ((System.Drawing.Image)(resources.GetObject("btnBand30.Image")));
			this.btnBand30.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand30.ImageAlign")));
			this.btnBand30.ImageIndex = ((int)(resources.GetObject("btnBand30.ImageIndex")));
			this.btnBand30.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand30.ImeMode")));
			this.btnBand30.Location = ((System.Drawing.Point)(resources.GetObject("btnBand30.Location")));
			this.btnBand30.Name = "btnBand30";
			this.btnBand30.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand30.RightToLeft")));
			this.btnBand30.Size = ((System.Drawing.Size)(resources.GetObject("btnBand30.Size")));
			this.btnBand30.TabIndex = ((int)(resources.GetObject("btnBand30.TabIndex")));
			this.btnBand30.Text = resources.GetString("btnBand30.Text");
			this.btnBand30.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand30.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand30, resources.GetString("btnBand30.ToolTip"));
			this.btnBand30.Visible = ((bool)(resources.GetObject("btnBand30.Visible")));
			this.btnBand30.Click += new System.EventHandler(this.btnBand30_Click);
			// 
			// btnBand40
			// 
			this.btnBand40.AccessibleDescription = resources.GetString("btnBand40.AccessibleDescription");
			this.btnBand40.AccessibleName = resources.GetString("btnBand40.AccessibleName");
			this.btnBand40.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand40.Anchor")));
			this.btnBand40.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand40.BackgroundImage")));
			this.btnBand40.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand40.Dock")));
			this.btnBand40.Enabled = ((bool)(resources.GetObject("btnBand40.Enabled")));
			this.btnBand40.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand40.FlatStyle")));
			this.btnBand40.Font = ((System.Drawing.Font)(resources.GetObject("btnBand40.Font")));
			this.btnBand40.Image = ((System.Drawing.Image)(resources.GetObject("btnBand40.Image")));
			this.btnBand40.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand40.ImageAlign")));
			this.btnBand40.ImageIndex = ((int)(resources.GetObject("btnBand40.ImageIndex")));
			this.btnBand40.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand40.ImeMode")));
			this.btnBand40.Location = ((System.Drawing.Point)(resources.GetObject("btnBand40.Location")));
			this.btnBand40.Name = "btnBand40";
			this.btnBand40.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand40.RightToLeft")));
			this.btnBand40.Size = ((System.Drawing.Size)(resources.GetObject("btnBand40.Size")));
			this.btnBand40.TabIndex = ((int)(resources.GetObject("btnBand40.TabIndex")));
			this.btnBand40.Text = resources.GetString("btnBand40.Text");
			this.btnBand40.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand40.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand40, resources.GetString("btnBand40.ToolTip"));
			this.btnBand40.Visible = ((bool)(resources.GetObject("btnBand40.Visible")));
			this.btnBand40.Click += new System.EventHandler(this.btnBand40_Click);
			// 
			// btnBand60
			// 
			this.btnBand60.AccessibleDescription = resources.GetString("btnBand60.AccessibleDescription");
			this.btnBand60.AccessibleName = resources.GetString("btnBand60.AccessibleName");
			this.btnBand60.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand60.Anchor")));
			this.btnBand60.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand60.BackgroundImage")));
			this.btnBand60.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand60.Dock")));
			this.btnBand60.Enabled = ((bool)(resources.GetObject("btnBand60.Enabled")));
			this.btnBand60.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand60.FlatStyle")));
			this.btnBand60.Font = ((System.Drawing.Font)(resources.GetObject("btnBand60.Font")));
			this.btnBand60.Image = ((System.Drawing.Image)(resources.GetObject("btnBand60.Image")));
			this.btnBand60.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand60.ImageAlign")));
			this.btnBand60.ImageIndex = ((int)(resources.GetObject("btnBand60.ImageIndex")));
			this.btnBand60.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand60.ImeMode")));
			this.btnBand60.Location = ((System.Drawing.Point)(resources.GetObject("btnBand60.Location")));
			this.btnBand60.Name = "btnBand60";
			this.btnBand60.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand60.RightToLeft")));
			this.btnBand60.Size = ((System.Drawing.Size)(resources.GetObject("btnBand60.Size")));
			this.btnBand60.TabIndex = ((int)(resources.GetObject("btnBand60.TabIndex")));
			this.btnBand60.Text = resources.GetString("btnBand60.Text");
			this.btnBand60.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand60.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand60, resources.GetString("btnBand60.ToolTip"));
			this.btnBand60.Visible = ((bool)(resources.GetObject("btnBand60.Visible")));
			this.btnBand60.Click += new System.EventHandler(this.btnBand60_Click);
			// 
			// btnBand80
			// 
			this.btnBand80.AccessibleDescription = resources.GetString("btnBand80.AccessibleDescription");
			this.btnBand80.AccessibleName = resources.GetString("btnBand80.AccessibleName");
			this.btnBand80.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand80.Anchor")));
			this.btnBand80.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand80.BackgroundImage")));
			this.btnBand80.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand80.Dock")));
			this.btnBand80.Enabled = ((bool)(resources.GetObject("btnBand80.Enabled")));
			this.btnBand80.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand80.FlatStyle")));
			this.btnBand80.Font = ((System.Drawing.Font)(resources.GetObject("btnBand80.Font")));
			this.btnBand80.Image = ((System.Drawing.Image)(resources.GetObject("btnBand80.Image")));
			this.btnBand80.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand80.ImageAlign")));
			this.btnBand80.ImageIndex = ((int)(resources.GetObject("btnBand80.ImageIndex")));
			this.btnBand80.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand80.ImeMode")));
			this.btnBand80.Location = ((System.Drawing.Point)(resources.GetObject("btnBand80.Location")));
			this.btnBand80.Name = "btnBand80";
			this.btnBand80.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand80.RightToLeft")));
			this.btnBand80.Size = ((System.Drawing.Size)(resources.GetObject("btnBand80.Size")));
			this.btnBand80.TabIndex = ((int)(resources.GetObject("btnBand80.TabIndex")));
			this.btnBand80.Text = resources.GetString("btnBand80.Text");
			this.btnBand80.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand80.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand80, resources.GetString("btnBand80.ToolTip"));
			this.btnBand80.Visible = ((bool)(resources.GetObject("btnBand80.Visible")));
			this.btnBand80.Click += new System.EventHandler(this.btnBand80_Click);
			// 
			// btnBand160
			// 
			this.btnBand160.AccessibleDescription = resources.GetString("btnBand160.AccessibleDescription");
			this.btnBand160.AccessibleName = resources.GetString("btnBand160.AccessibleName");
			this.btnBand160.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnBand160.Anchor")));
			this.btnBand160.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnBand160.BackgroundImage")));
			this.btnBand160.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnBand160.Dock")));
			this.btnBand160.Enabled = ((bool)(resources.GetObject("btnBand160.Enabled")));
			this.btnBand160.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnBand160.FlatStyle")));
			this.btnBand160.Font = ((System.Drawing.Font)(resources.GetObject("btnBand160.Font")));
			this.btnBand160.Image = ((System.Drawing.Image)(resources.GetObject("btnBand160.Image")));
			this.btnBand160.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand160.ImageAlign")));
			this.btnBand160.ImageIndex = ((int)(resources.GetObject("btnBand160.ImageIndex")));
			this.btnBand160.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnBand160.ImeMode")));
			this.btnBand160.Location = ((System.Drawing.Point)(resources.GetObject("btnBand160.Location")));
			this.btnBand160.Name = "btnBand160";
			this.btnBand160.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnBand160.RightToLeft")));
			this.btnBand160.Size = ((System.Drawing.Size)(resources.GetObject("btnBand160.Size")));
			this.btnBand160.TabIndex = ((int)(resources.GetObject("btnBand160.TabIndex")));
			this.btnBand160.Text = resources.GetString("btnBand160.Text");
			this.btnBand160.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnBand160.TextAlign")));
			this.toolTip1.SetToolTip(this.btnBand160, resources.GetString("btnBand160.ToolTip"));
			this.btnBand160.Visible = ((bool)(resources.GetObject("btnBand160.Visible")));
			this.btnBand160.Click += new System.EventHandler(this.btnBand160_Click);
			// 
			// grpFilter
			// 
			this.grpFilter.AccessibleDescription = resources.GetString("grpFilter.AccessibleDescription");
			this.grpFilter.AccessibleName = resources.GetString("grpFilter.AccessibleName");
			this.grpFilter.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpFilter.Anchor")));
			this.grpFilter.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpFilter.BackgroundImage")));
			this.grpFilter.Controls.Add(this.radFilter6000);
			this.grpFilter.Controls.Add(this.radFilter4000);
			this.grpFilter.Controls.Add(this.radFilter2600);
			this.grpFilter.Controls.Add(this.radFilter2100);
			this.grpFilter.Controls.Add(this.radFilter1000);
			this.grpFilter.Controls.Add(this.radFilter500);
			this.grpFilter.Controls.Add(this.radFilter250);
			this.grpFilter.Controls.Add(this.radFilter100);
			this.grpFilter.Controls.Add(this.radFilter50);
			this.grpFilter.Controls.Add(this.radFilter25);
			this.grpFilter.Controls.Add(this.radFilterVar1);
			this.grpFilter.Controls.Add(this.radFilterVar2);
			this.grpFilter.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpFilter.Dock")));
			this.grpFilter.Enabled = ((bool)(resources.GetObject("grpFilter.Enabled")));
			this.grpFilter.Font = ((System.Drawing.Font)(resources.GetObject("grpFilter.Font")));
			this.grpFilter.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpFilter.ImeMode")));
			this.grpFilter.Location = ((System.Drawing.Point)(resources.GetObject("grpFilter.Location")));
			this.grpFilter.Name = "grpFilter";
			this.grpFilter.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpFilter.RightToLeft")));
			this.grpFilter.Size = ((System.Drawing.Size)(resources.GetObject("grpFilter.Size")));
			this.grpFilter.TabIndex = ((int)(resources.GetObject("grpFilter.TabIndex")));
			this.grpFilter.TabStop = false;
			this.grpFilter.Text = resources.GetString("grpFilter.Text");
			this.toolTip1.SetToolTip(this.grpFilter, resources.GetString("grpFilter.ToolTip"));
			this.grpFilter.Visible = ((bool)(resources.GetObject("grpFilter.Visible")));
			// 
			// radFilter6000
			// 
			this.radFilter6000.AccessibleDescription = resources.GetString("radFilter6000.AccessibleDescription");
			this.radFilter6000.AccessibleName = resources.GetString("radFilter6000.AccessibleName");
			this.radFilter6000.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilter6000.Anchor")));
			this.radFilter6000.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilter6000.Appearance")));
			this.radFilter6000.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilter6000.BackgroundImage")));
			this.radFilter6000.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter6000.CheckAlign")));
			this.radFilter6000.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilter6000.Dock")));
			this.radFilter6000.Enabled = ((bool)(resources.GetObject("radFilter6000.Enabled")));
			this.radFilter6000.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilter6000.FlatStyle")));
			this.radFilter6000.Font = ((System.Drawing.Font)(resources.GetObject("radFilter6000.Font")));
			this.radFilter6000.Image = ((System.Drawing.Image)(resources.GetObject("radFilter6000.Image")));
			this.radFilter6000.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter6000.ImageAlign")));
			this.radFilter6000.ImageIndex = ((int)(resources.GetObject("radFilter6000.ImageIndex")));
			this.radFilter6000.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilter6000.ImeMode")));
			this.radFilter6000.Location = ((System.Drawing.Point)(resources.GetObject("radFilter6000.Location")));
			this.radFilter6000.Name = "radFilter6000";
			this.radFilter6000.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilter6000.RightToLeft")));
			this.radFilter6000.Size = ((System.Drawing.Size)(resources.GetObject("radFilter6000.Size")));
			this.radFilter6000.TabIndex = ((int)(resources.GetObject("radFilter6000.TabIndex")));
			this.radFilter6000.Text = resources.GetString("radFilter6000.Text");
			this.radFilter6000.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter6000.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilter6000, resources.GetString("radFilter6000.ToolTip"));
			this.radFilter6000.Visible = ((bool)(resources.GetObject("radFilter6000.Visible")));
			this.radFilter6000.CheckedChanged += new System.EventHandler(this.radFilter6000_CheckedChanged);
			// 
			// radFilter4000
			// 
			this.radFilter4000.AccessibleDescription = resources.GetString("radFilter4000.AccessibleDescription");
			this.radFilter4000.AccessibleName = resources.GetString("radFilter4000.AccessibleName");
			this.radFilter4000.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilter4000.Anchor")));
			this.radFilter4000.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilter4000.Appearance")));
			this.radFilter4000.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilter4000.BackgroundImage")));
			this.radFilter4000.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter4000.CheckAlign")));
			this.radFilter4000.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilter4000.Dock")));
			this.radFilter4000.Enabled = ((bool)(resources.GetObject("radFilter4000.Enabled")));
			this.radFilter4000.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilter4000.FlatStyle")));
			this.radFilter4000.Font = ((System.Drawing.Font)(resources.GetObject("radFilter4000.Font")));
			this.radFilter4000.Image = ((System.Drawing.Image)(resources.GetObject("radFilter4000.Image")));
			this.radFilter4000.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter4000.ImageAlign")));
			this.radFilter4000.ImageIndex = ((int)(resources.GetObject("radFilter4000.ImageIndex")));
			this.radFilter4000.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilter4000.ImeMode")));
			this.radFilter4000.Location = ((System.Drawing.Point)(resources.GetObject("radFilter4000.Location")));
			this.radFilter4000.Name = "radFilter4000";
			this.radFilter4000.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilter4000.RightToLeft")));
			this.radFilter4000.Size = ((System.Drawing.Size)(resources.GetObject("radFilter4000.Size")));
			this.radFilter4000.TabIndex = ((int)(resources.GetObject("radFilter4000.TabIndex")));
			this.radFilter4000.Text = resources.GetString("radFilter4000.Text");
			this.radFilter4000.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter4000.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilter4000, resources.GetString("radFilter4000.ToolTip"));
			this.radFilter4000.Visible = ((bool)(resources.GetObject("radFilter4000.Visible")));
			this.radFilter4000.CheckedChanged += new System.EventHandler(this.radFilter4000_CheckedChanged);
			// 
			// radFilter2600
			// 
			this.radFilter2600.AccessibleDescription = resources.GetString("radFilter2600.AccessibleDescription");
			this.radFilter2600.AccessibleName = resources.GetString("radFilter2600.AccessibleName");
			this.radFilter2600.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilter2600.Anchor")));
			this.radFilter2600.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilter2600.Appearance")));
			this.radFilter2600.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilter2600.BackgroundImage")));
			this.radFilter2600.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter2600.CheckAlign")));
			this.radFilter2600.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilter2600.Dock")));
			this.radFilter2600.Enabled = ((bool)(resources.GetObject("radFilter2600.Enabled")));
			this.radFilter2600.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilter2600.FlatStyle")));
			this.radFilter2600.Font = ((System.Drawing.Font)(resources.GetObject("radFilter2600.Font")));
			this.radFilter2600.Image = ((System.Drawing.Image)(resources.GetObject("radFilter2600.Image")));
			this.radFilter2600.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter2600.ImageAlign")));
			this.radFilter2600.ImageIndex = ((int)(resources.GetObject("radFilter2600.ImageIndex")));
			this.radFilter2600.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilter2600.ImeMode")));
			this.radFilter2600.Location = ((System.Drawing.Point)(resources.GetObject("radFilter2600.Location")));
			this.radFilter2600.Name = "radFilter2600";
			this.radFilter2600.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilter2600.RightToLeft")));
			this.radFilter2600.Size = ((System.Drawing.Size)(resources.GetObject("radFilter2600.Size")));
			this.radFilter2600.TabIndex = ((int)(resources.GetObject("radFilter2600.TabIndex")));
			this.radFilter2600.Text = resources.GetString("radFilter2600.Text");
			this.radFilter2600.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter2600.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilter2600, resources.GetString("radFilter2600.ToolTip"));
			this.radFilter2600.Visible = ((bool)(resources.GetObject("radFilter2600.Visible")));
			this.radFilter2600.CheckedChanged += new System.EventHandler(this.radFilter2600_CheckedChanged);
			// 
			// radFilter2100
			// 
			this.radFilter2100.AccessibleDescription = resources.GetString("radFilter2100.AccessibleDescription");
			this.radFilter2100.AccessibleName = resources.GetString("radFilter2100.AccessibleName");
			this.radFilter2100.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilter2100.Anchor")));
			this.radFilter2100.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilter2100.Appearance")));
			this.radFilter2100.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilter2100.BackgroundImage")));
			this.radFilter2100.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter2100.CheckAlign")));
			this.radFilter2100.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilter2100.Dock")));
			this.radFilter2100.Enabled = ((bool)(resources.GetObject("radFilter2100.Enabled")));
			this.radFilter2100.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilter2100.FlatStyle")));
			this.radFilter2100.Font = ((System.Drawing.Font)(resources.GetObject("radFilter2100.Font")));
			this.radFilter2100.Image = ((System.Drawing.Image)(resources.GetObject("radFilter2100.Image")));
			this.radFilter2100.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter2100.ImageAlign")));
			this.radFilter2100.ImageIndex = ((int)(resources.GetObject("radFilter2100.ImageIndex")));
			this.radFilter2100.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilter2100.ImeMode")));
			this.radFilter2100.Location = ((System.Drawing.Point)(resources.GetObject("radFilter2100.Location")));
			this.radFilter2100.Name = "radFilter2100";
			this.radFilter2100.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilter2100.RightToLeft")));
			this.radFilter2100.Size = ((System.Drawing.Size)(resources.GetObject("radFilter2100.Size")));
			this.radFilter2100.TabIndex = ((int)(resources.GetObject("radFilter2100.TabIndex")));
			this.radFilter2100.Text = resources.GetString("radFilter2100.Text");
			this.radFilter2100.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter2100.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilter2100, resources.GetString("radFilter2100.ToolTip"));
			this.radFilter2100.Visible = ((bool)(resources.GetObject("radFilter2100.Visible")));
			this.radFilter2100.CheckedChanged += new System.EventHandler(this.radFilter2100_CheckedChanged);
			// 
			// radFilter1000
			// 
			this.radFilter1000.AccessibleDescription = resources.GetString("radFilter1000.AccessibleDescription");
			this.radFilter1000.AccessibleName = resources.GetString("radFilter1000.AccessibleName");
			this.radFilter1000.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilter1000.Anchor")));
			this.radFilter1000.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilter1000.Appearance")));
			this.radFilter1000.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilter1000.BackgroundImage")));
			this.radFilter1000.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter1000.CheckAlign")));
			this.radFilter1000.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilter1000.Dock")));
			this.radFilter1000.Enabled = ((bool)(resources.GetObject("radFilter1000.Enabled")));
			this.radFilter1000.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilter1000.FlatStyle")));
			this.radFilter1000.Font = ((System.Drawing.Font)(resources.GetObject("radFilter1000.Font")));
			this.radFilter1000.Image = ((System.Drawing.Image)(resources.GetObject("radFilter1000.Image")));
			this.radFilter1000.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter1000.ImageAlign")));
			this.radFilter1000.ImageIndex = ((int)(resources.GetObject("radFilter1000.ImageIndex")));
			this.radFilter1000.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilter1000.ImeMode")));
			this.radFilter1000.Location = ((System.Drawing.Point)(resources.GetObject("radFilter1000.Location")));
			this.radFilter1000.Name = "radFilter1000";
			this.radFilter1000.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilter1000.RightToLeft")));
			this.radFilter1000.Size = ((System.Drawing.Size)(resources.GetObject("radFilter1000.Size")));
			this.radFilter1000.TabIndex = ((int)(resources.GetObject("radFilter1000.TabIndex")));
			this.radFilter1000.Text = resources.GetString("radFilter1000.Text");
			this.radFilter1000.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter1000.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilter1000, resources.GetString("radFilter1000.ToolTip"));
			this.radFilter1000.Visible = ((bool)(resources.GetObject("radFilter1000.Visible")));
			this.radFilter1000.CheckedChanged += new System.EventHandler(this.radFilter1000_CheckedChanged);
			// 
			// radFilter500
			// 
			this.radFilter500.AccessibleDescription = resources.GetString("radFilter500.AccessibleDescription");
			this.radFilter500.AccessibleName = resources.GetString("radFilter500.AccessibleName");
			this.radFilter500.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilter500.Anchor")));
			this.radFilter500.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilter500.Appearance")));
			this.radFilter500.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilter500.BackgroundImage")));
			this.radFilter500.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter500.CheckAlign")));
			this.radFilter500.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilter500.Dock")));
			this.radFilter500.Enabled = ((bool)(resources.GetObject("radFilter500.Enabled")));
			this.radFilter500.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilter500.FlatStyle")));
			this.radFilter500.Font = ((System.Drawing.Font)(resources.GetObject("radFilter500.Font")));
			this.radFilter500.Image = ((System.Drawing.Image)(resources.GetObject("radFilter500.Image")));
			this.radFilter500.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter500.ImageAlign")));
			this.radFilter500.ImageIndex = ((int)(resources.GetObject("radFilter500.ImageIndex")));
			this.radFilter500.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilter500.ImeMode")));
			this.radFilter500.Location = ((System.Drawing.Point)(resources.GetObject("radFilter500.Location")));
			this.radFilter500.Name = "radFilter500";
			this.radFilter500.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilter500.RightToLeft")));
			this.radFilter500.Size = ((System.Drawing.Size)(resources.GetObject("radFilter500.Size")));
			this.radFilter500.TabIndex = ((int)(resources.GetObject("radFilter500.TabIndex")));
			this.radFilter500.Text = resources.GetString("radFilter500.Text");
			this.radFilter500.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter500.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilter500, resources.GetString("radFilter500.ToolTip"));
			this.radFilter500.Visible = ((bool)(resources.GetObject("radFilter500.Visible")));
			this.radFilter500.CheckedChanged += new System.EventHandler(this.radFilter500_CheckedChanged);
			// 
			// radFilter250
			// 
			this.radFilter250.AccessibleDescription = resources.GetString("radFilter250.AccessibleDescription");
			this.radFilter250.AccessibleName = resources.GetString("radFilter250.AccessibleName");
			this.radFilter250.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilter250.Anchor")));
			this.radFilter250.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilter250.Appearance")));
			this.radFilter250.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilter250.BackgroundImage")));
			this.radFilter250.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter250.CheckAlign")));
			this.radFilter250.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilter250.Dock")));
			this.radFilter250.Enabled = ((bool)(resources.GetObject("radFilter250.Enabled")));
			this.radFilter250.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilter250.FlatStyle")));
			this.radFilter250.Font = ((System.Drawing.Font)(resources.GetObject("radFilter250.Font")));
			this.radFilter250.Image = ((System.Drawing.Image)(resources.GetObject("radFilter250.Image")));
			this.radFilter250.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter250.ImageAlign")));
			this.radFilter250.ImageIndex = ((int)(resources.GetObject("radFilter250.ImageIndex")));
			this.radFilter250.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilter250.ImeMode")));
			this.radFilter250.Location = ((System.Drawing.Point)(resources.GetObject("radFilter250.Location")));
			this.radFilter250.Name = "radFilter250";
			this.radFilter250.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilter250.RightToLeft")));
			this.radFilter250.Size = ((System.Drawing.Size)(resources.GetObject("radFilter250.Size")));
			this.radFilter250.TabIndex = ((int)(resources.GetObject("radFilter250.TabIndex")));
			this.radFilter250.Text = resources.GetString("radFilter250.Text");
			this.radFilter250.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter250.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilter250, resources.GetString("radFilter250.ToolTip"));
			this.radFilter250.Visible = ((bool)(resources.GetObject("radFilter250.Visible")));
			this.radFilter250.CheckedChanged += new System.EventHandler(this.radFilter250_CheckedChanged);
			// 
			// radFilter100
			// 
			this.radFilter100.AccessibleDescription = resources.GetString("radFilter100.AccessibleDescription");
			this.radFilter100.AccessibleName = resources.GetString("radFilter100.AccessibleName");
			this.radFilter100.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilter100.Anchor")));
			this.radFilter100.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilter100.Appearance")));
			this.radFilter100.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilter100.BackgroundImage")));
			this.radFilter100.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter100.CheckAlign")));
			this.radFilter100.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilter100.Dock")));
			this.radFilter100.Enabled = ((bool)(resources.GetObject("radFilter100.Enabled")));
			this.radFilter100.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilter100.FlatStyle")));
			this.radFilter100.Font = ((System.Drawing.Font)(resources.GetObject("radFilter100.Font")));
			this.radFilter100.Image = ((System.Drawing.Image)(resources.GetObject("radFilter100.Image")));
			this.radFilter100.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter100.ImageAlign")));
			this.radFilter100.ImageIndex = ((int)(resources.GetObject("radFilter100.ImageIndex")));
			this.radFilter100.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilter100.ImeMode")));
			this.radFilter100.Location = ((System.Drawing.Point)(resources.GetObject("radFilter100.Location")));
			this.radFilter100.Name = "radFilter100";
			this.radFilter100.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilter100.RightToLeft")));
			this.radFilter100.Size = ((System.Drawing.Size)(resources.GetObject("radFilter100.Size")));
			this.radFilter100.TabIndex = ((int)(resources.GetObject("radFilter100.TabIndex")));
			this.radFilter100.Text = resources.GetString("radFilter100.Text");
			this.radFilter100.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter100.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilter100, resources.GetString("radFilter100.ToolTip"));
			this.radFilter100.Visible = ((bool)(resources.GetObject("radFilter100.Visible")));
			this.radFilter100.CheckedChanged += new System.EventHandler(this.radFilter100_CheckedChanged);
			// 
			// radFilter50
			// 
			this.radFilter50.AccessibleDescription = resources.GetString("radFilter50.AccessibleDescription");
			this.radFilter50.AccessibleName = resources.GetString("radFilter50.AccessibleName");
			this.radFilter50.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilter50.Anchor")));
			this.radFilter50.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilter50.Appearance")));
			this.radFilter50.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilter50.BackgroundImage")));
			this.radFilter50.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter50.CheckAlign")));
			this.radFilter50.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilter50.Dock")));
			this.radFilter50.Enabled = ((bool)(resources.GetObject("radFilter50.Enabled")));
			this.radFilter50.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilter50.FlatStyle")));
			this.radFilter50.Font = ((System.Drawing.Font)(resources.GetObject("radFilter50.Font")));
			this.radFilter50.Image = ((System.Drawing.Image)(resources.GetObject("radFilter50.Image")));
			this.radFilter50.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter50.ImageAlign")));
			this.radFilter50.ImageIndex = ((int)(resources.GetObject("radFilter50.ImageIndex")));
			this.radFilter50.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilter50.ImeMode")));
			this.radFilter50.Location = ((System.Drawing.Point)(resources.GetObject("radFilter50.Location")));
			this.radFilter50.Name = "radFilter50";
			this.radFilter50.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilter50.RightToLeft")));
			this.radFilter50.Size = ((System.Drawing.Size)(resources.GetObject("radFilter50.Size")));
			this.radFilter50.TabIndex = ((int)(resources.GetObject("radFilter50.TabIndex")));
			this.radFilter50.Text = resources.GetString("radFilter50.Text");
			this.radFilter50.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter50.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilter50, resources.GetString("radFilter50.ToolTip"));
			this.radFilter50.Visible = ((bool)(resources.GetObject("radFilter50.Visible")));
			this.radFilter50.CheckedChanged += new System.EventHandler(this.radFilter50_CheckedChanged);
			// 
			// radFilter25
			// 
			this.radFilter25.AccessibleDescription = resources.GetString("radFilter25.AccessibleDescription");
			this.radFilter25.AccessibleName = resources.GetString("radFilter25.AccessibleName");
			this.radFilter25.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilter25.Anchor")));
			this.radFilter25.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilter25.Appearance")));
			this.radFilter25.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilter25.BackgroundImage")));
			this.radFilter25.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter25.CheckAlign")));
			this.radFilter25.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilter25.Dock")));
			this.radFilter25.Enabled = ((bool)(resources.GetObject("radFilter25.Enabled")));
			this.radFilter25.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilter25.FlatStyle")));
			this.radFilter25.Font = ((System.Drawing.Font)(resources.GetObject("radFilter25.Font")));
			this.radFilter25.Image = ((System.Drawing.Image)(resources.GetObject("radFilter25.Image")));
			this.radFilter25.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter25.ImageAlign")));
			this.radFilter25.ImageIndex = ((int)(resources.GetObject("radFilter25.ImageIndex")));
			this.radFilter25.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilter25.ImeMode")));
			this.radFilter25.Location = ((System.Drawing.Point)(resources.GetObject("radFilter25.Location")));
			this.radFilter25.Name = "radFilter25";
			this.radFilter25.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilter25.RightToLeft")));
			this.radFilter25.Size = ((System.Drawing.Size)(resources.GetObject("radFilter25.Size")));
			this.radFilter25.TabIndex = ((int)(resources.GetObject("radFilter25.TabIndex")));
			this.radFilter25.Text = resources.GetString("radFilter25.Text");
			this.radFilter25.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilter25.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilter25, resources.GetString("radFilter25.ToolTip"));
			this.radFilter25.Visible = ((bool)(resources.GetObject("radFilter25.Visible")));
			this.radFilter25.CheckedChanged += new System.EventHandler(this.radFilter25_CheckedChanged);
			// 
			// radFilterVar1
			// 
			this.radFilterVar1.AccessibleDescription = resources.GetString("radFilterVar1.AccessibleDescription");
			this.radFilterVar1.AccessibleName = resources.GetString("radFilterVar1.AccessibleName");
			this.radFilterVar1.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilterVar1.Anchor")));
			this.radFilterVar1.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilterVar1.Appearance")));
			this.radFilterVar1.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilterVar1.BackgroundImage")));
			this.radFilterVar1.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilterVar1.CheckAlign")));
			this.radFilterVar1.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilterVar1.Dock")));
			this.radFilterVar1.Enabled = ((bool)(resources.GetObject("radFilterVar1.Enabled")));
			this.radFilterVar1.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilterVar1.FlatStyle")));
			this.radFilterVar1.Font = ((System.Drawing.Font)(resources.GetObject("radFilterVar1.Font")));
			this.radFilterVar1.Image = ((System.Drawing.Image)(resources.GetObject("radFilterVar1.Image")));
			this.radFilterVar1.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilterVar1.ImageAlign")));
			this.radFilterVar1.ImageIndex = ((int)(resources.GetObject("radFilterVar1.ImageIndex")));
			this.radFilterVar1.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilterVar1.ImeMode")));
			this.radFilterVar1.Location = ((System.Drawing.Point)(resources.GetObject("radFilterVar1.Location")));
			this.radFilterVar1.Name = "radFilterVar1";
			this.radFilterVar1.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilterVar1.RightToLeft")));
			this.radFilterVar1.Size = ((System.Drawing.Size)(resources.GetObject("radFilterVar1.Size")));
			this.radFilterVar1.TabIndex = ((int)(resources.GetObject("radFilterVar1.TabIndex")));
			this.radFilterVar1.Text = resources.GetString("radFilterVar1.Text");
			this.radFilterVar1.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilterVar1.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilterVar1, resources.GetString("radFilterVar1.ToolTip"));
			this.radFilterVar1.Visible = ((bool)(resources.GetObject("radFilterVar1.Visible")));
			this.radFilterVar1.CheckedChanged += new System.EventHandler(this.radFilterVar1_CheckedChanged);
			// 
			// radFilterVar2
			// 
			this.radFilterVar2.AccessibleDescription = resources.GetString("radFilterVar2.AccessibleDescription");
			this.radFilterVar2.AccessibleName = resources.GetString("radFilterVar2.AccessibleName");
			this.radFilterVar2.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radFilterVar2.Anchor")));
			this.radFilterVar2.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radFilterVar2.Appearance")));
			this.radFilterVar2.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radFilterVar2.BackgroundImage")));
			this.radFilterVar2.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilterVar2.CheckAlign")));
			this.radFilterVar2.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radFilterVar2.Dock")));
			this.radFilterVar2.Enabled = ((bool)(resources.GetObject("radFilterVar2.Enabled")));
			this.radFilterVar2.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radFilterVar2.FlatStyle")));
			this.radFilterVar2.Font = ((System.Drawing.Font)(resources.GetObject("radFilterVar2.Font")));
			this.radFilterVar2.Image = ((System.Drawing.Image)(resources.GetObject("radFilterVar2.Image")));
			this.radFilterVar2.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilterVar2.ImageAlign")));
			this.radFilterVar2.ImageIndex = ((int)(resources.GetObject("radFilterVar2.ImageIndex")));
			this.radFilterVar2.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radFilterVar2.ImeMode")));
			this.radFilterVar2.Location = ((System.Drawing.Point)(resources.GetObject("radFilterVar2.Location")));
			this.radFilterVar2.Name = "radFilterVar2";
			this.radFilterVar2.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radFilterVar2.RightToLeft")));
			this.radFilterVar2.Size = ((System.Drawing.Size)(resources.GetObject("radFilterVar2.Size")));
			this.radFilterVar2.TabIndex = ((int)(resources.GetObject("radFilterVar2.TabIndex")));
			this.radFilterVar2.Text = resources.GetString("radFilterVar2.Text");
			this.radFilterVar2.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radFilterVar2.TextAlign")));
			this.toolTip1.SetToolTip(this.radFilterVar2, resources.GetString("radFilterVar2.ToolTip"));
			this.radFilterVar2.Visible = ((bool)(resources.GetObject("radFilterVar2.Visible")));
			this.radFilterVar2.CheckedChanged += new System.EventHandler(this.radFilterVar2_CheckedChanged);
			// 
			// btnFilterShiftReset
			// 
			this.btnFilterShiftReset.AccessibleDescription = resources.GetString("btnFilterShiftReset.AccessibleDescription");
			this.btnFilterShiftReset.AccessibleName = resources.GetString("btnFilterShiftReset.AccessibleName");
			this.btnFilterShiftReset.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnFilterShiftReset.Anchor")));
			this.btnFilterShiftReset.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnFilterShiftReset.BackgroundImage")));
			this.btnFilterShiftReset.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnFilterShiftReset.Dock")));
			this.btnFilterShiftReset.Enabled = ((bool)(resources.GetObject("btnFilterShiftReset.Enabled")));
			this.btnFilterShiftReset.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnFilterShiftReset.FlatStyle")));
			this.btnFilterShiftReset.Font = ((System.Drawing.Font)(resources.GetObject("btnFilterShiftReset.Font")));
			this.btnFilterShiftReset.Image = ((System.Drawing.Image)(resources.GetObject("btnFilterShiftReset.Image")));
			this.btnFilterShiftReset.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnFilterShiftReset.ImageAlign")));
			this.btnFilterShiftReset.ImageIndex = ((int)(resources.GetObject("btnFilterShiftReset.ImageIndex")));
			this.btnFilterShiftReset.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnFilterShiftReset.ImeMode")));
			this.btnFilterShiftReset.Location = ((System.Drawing.Point)(resources.GetObject("btnFilterShiftReset.Location")));
			this.btnFilterShiftReset.Name = "btnFilterShiftReset";
			this.btnFilterShiftReset.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnFilterShiftReset.RightToLeft")));
			this.btnFilterShiftReset.Size = ((System.Drawing.Size)(resources.GetObject("btnFilterShiftReset.Size")));
			this.btnFilterShiftReset.TabIndex = ((int)(resources.GetObject("btnFilterShiftReset.TabIndex")));
			this.btnFilterShiftReset.Tag = "Reset Filter Shift";
			this.btnFilterShiftReset.Text = resources.GetString("btnFilterShiftReset.Text");
			this.btnFilterShiftReset.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnFilterShiftReset.TextAlign")));
			this.toolTip1.SetToolTip(this.btnFilterShiftReset, resources.GetString("btnFilterShiftReset.ToolTip"));
			this.btnFilterShiftReset.Visible = ((bool)(resources.GetObject("btnFilterShiftReset.Visible")));
			this.btnFilterShiftReset.Click += new System.EventHandler(this.btnFilterShiftReset_Click);
			// 
			// tbFilterShift
			// 
			this.tbFilterShift.AccessibleDescription = resources.GetString("tbFilterShift.AccessibleDescription");
			this.tbFilterShift.AccessibleName = resources.GetString("tbFilterShift.AccessibleName");
			this.tbFilterShift.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("tbFilterShift.Anchor")));
			this.tbFilterShift.AutoSize = false;
			this.tbFilterShift.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("tbFilterShift.BackgroundImage")));
			this.tbFilterShift.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("tbFilterShift.Dock")));
			this.tbFilterShift.Enabled = ((bool)(resources.GetObject("tbFilterShift.Enabled")));
			this.tbFilterShift.Font = ((System.Drawing.Font)(resources.GetObject("tbFilterShift.Font")));
			this.tbFilterShift.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("tbFilterShift.ImeMode")));
			this.tbFilterShift.Location = ((System.Drawing.Point)(resources.GetObject("tbFilterShift.Location")));
			this.tbFilterShift.Maximum = 1000;
			this.tbFilterShift.Minimum = -1000;
			this.tbFilterShift.Name = "tbFilterShift";
			this.tbFilterShift.Orientation = ((System.Windows.Forms.Orientation)(resources.GetObject("tbFilterShift.Orientation")));
			this.tbFilterShift.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("tbFilterShift.RightToLeft")));
			this.tbFilterShift.Size = ((System.Drawing.Size)(resources.GetObject("tbFilterShift.Size")));
			this.tbFilterShift.TabIndex = ((int)(resources.GetObject("tbFilterShift.TabIndex")));
			this.tbFilterShift.Text = resources.GetString("tbFilterShift.Text");
			this.tbFilterShift.TickStyle = System.Windows.Forms.TickStyle.None;
			this.toolTip1.SetToolTip(this.tbFilterShift, resources.GetString("tbFilterShift.ToolTip"));
			this.tbFilterShift.Visible = ((bool)(resources.GetObject("tbFilterShift.Visible")));
			this.tbFilterShift.Scroll += new System.EventHandler(this.tbFilterShift_Scroll);
			// 
			// lblFilterShift
			// 
			this.lblFilterShift.AccessibleDescription = resources.GetString("lblFilterShift.AccessibleDescription");
			this.lblFilterShift.AccessibleName = resources.GetString("lblFilterShift.AccessibleName");
			this.lblFilterShift.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblFilterShift.Anchor")));
			this.lblFilterShift.AutoSize = ((bool)(resources.GetObject("lblFilterShift.AutoSize")));
			this.lblFilterShift.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblFilterShift.Dock")));
			this.lblFilterShift.Enabled = ((bool)(resources.GetObject("lblFilterShift.Enabled")));
			this.lblFilterShift.Font = ((System.Drawing.Font)(resources.GetObject("lblFilterShift.Font")));
			this.lblFilterShift.Image = ((System.Drawing.Image)(resources.GetObject("lblFilterShift.Image")));
			this.lblFilterShift.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblFilterShift.ImageAlign")));
			this.lblFilterShift.ImageIndex = ((int)(resources.GetObject("lblFilterShift.ImageIndex")));
			this.lblFilterShift.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblFilterShift.ImeMode")));
			this.lblFilterShift.Location = ((System.Drawing.Point)(resources.GetObject("lblFilterShift.Location")));
			this.lblFilterShift.Name = "lblFilterShift";
			this.lblFilterShift.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblFilterShift.RightToLeft")));
			this.lblFilterShift.Size = ((System.Drawing.Size)(resources.GetObject("lblFilterShift.Size")));
			this.lblFilterShift.TabIndex = ((int)(resources.GetObject("lblFilterShift.TabIndex")));
			this.lblFilterShift.Text = resources.GetString("lblFilterShift.Text");
			this.lblFilterShift.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblFilterShift.TextAlign")));
			this.toolTip1.SetToolTip(this.lblFilterShift, resources.GetString("lblFilterShift.ToolTip"));
			this.lblFilterShift.Visible = ((bool)(resources.GetObject("lblFilterShift.Visible")));
			// 
			// lblFilterHigh
			// 
			this.lblFilterHigh.AccessibleDescription = resources.GetString("lblFilterHigh.AccessibleDescription");
			this.lblFilterHigh.AccessibleName = resources.GetString("lblFilterHigh.AccessibleName");
			this.lblFilterHigh.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblFilterHigh.Anchor")));
			this.lblFilterHigh.AutoSize = ((bool)(resources.GetObject("lblFilterHigh.AutoSize")));
			this.lblFilterHigh.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblFilterHigh.Dock")));
			this.lblFilterHigh.Enabled = ((bool)(resources.GetObject("lblFilterHigh.Enabled")));
			this.lblFilterHigh.Font = ((System.Drawing.Font)(resources.GetObject("lblFilterHigh.Font")));
			this.lblFilterHigh.Image = ((System.Drawing.Image)(resources.GetObject("lblFilterHigh.Image")));
			this.lblFilterHigh.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblFilterHigh.ImageAlign")));
			this.lblFilterHigh.ImageIndex = ((int)(resources.GetObject("lblFilterHigh.ImageIndex")));
			this.lblFilterHigh.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblFilterHigh.ImeMode")));
			this.lblFilterHigh.Location = ((System.Drawing.Point)(resources.GetObject("lblFilterHigh.Location")));
			this.lblFilterHigh.Name = "lblFilterHigh";
			this.lblFilterHigh.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblFilterHigh.RightToLeft")));
			this.lblFilterHigh.Size = ((System.Drawing.Size)(resources.GetObject("lblFilterHigh.Size")));
			this.lblFilterHigh.TabIndex = ((int)(resources.GetObject("lblFilterHigh.TabIndex")));
			this.lblFilterHigh.Text = resources.GetString("lblFilterHigh.Text");
			this.lblFilterHigh.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblFilterHigh.TextAlign")));
			this.toolTip1.SetToolTip(this.lblFilterHigh, resources.GetString("lblFilterHigh.ToolTip"));
			this.lblFilterHigh.Visible = ((bool)(resources.GetObject("lblFilterHigh.Visible")));
			// 
			// lblFilterLow
			// 
			this.lblFilterLow.AccessibleDescription = resources.GetString("lblFilterLow.AccessibleDescription");
			this.lblFilterLow.AccessibleName = resources.GetString("lblFilterLow.AccessibleName");
			this.lblFilterLow.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblFilterLow.Anchor")));
			this.lblFilterLow.AutoSize = ((bool)(resources.GetObject("lblFilterLow.AutoSize")));
			this.lblFilterLow.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblFilterLow.Dock")));
			this.lblFilterLow.Enabled = ((bool)(resources.GetObject("lblFilterLow.Enabled")));
			this.lblFilterLow.Font = ((System.Drawing.Font)(resources.GetObject("lblFilterLow.Font")));
			this.lblFilterLow.Image = ((System.Drawing.Image)(resources.GetObject("lblFilterLow.Image")));
			this.lblFilterLow.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblFilterLow.ImageAlign")));
			this.lblFilterLow.ImageIndex = ((int)(resources.GetObject("lblFilterLow.ImageIndex")));
			this.lblFilterLow.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblFilterLow.ImeMode")));
			this.lblFilterLow.Location = ((System.Drawing.Point)(resources.GetObject("lblFilterLow.Location")));
			this.lblFilterLow.Name = "lblFilterLow";
			this.lblFilterLow.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblFilterLow.RightToLeft")));
			this.lblFilterLow.Size = ((System.Drawing.Size)(resources.GetObject("lblFilterLow.Size")));
			this.lblFilterLow.TabIndex = ((int)(resources.GetObject("lblFilterLow.TabIndex")));
			this.lblFilterLow.Text = resources.GetString("lblFilterLow.Text");
			this.lblFilterLow.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblFilterLow.TextAlign")));
			this.toolTip1.SetToolTip(this.lblFilterLow, resources.GetString("lblFilterLow.ToolTip"));
			this.lblFilterLow.Visible = ((bool)(resources.GetObject("lblFilterLow.Visible")));
			// 
			// udFilterHigh
			// 
			this.udFilterHigh.AccessibleDescription = resources.GetString("udFilterHigh.AccessibleDescription");
			this.udFilterHigh.AccessibleName = resources.GetString("udFilterHigh.AccessibleName");
			this.udFilterHigh.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udFilterHigh.Anchor")));
			this.udFilterHigh.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udFilterHigh.Dock")));
			this.udFilterHigh.Enabled = ((bool)(resources.GetObject("udFilterHigh.Enabled")));
			this.udFilterHigh.Font = ((System.Drawing.Font)(resources.GetObject("udFilterHigh.Font")));
			this.udFilterHigh.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udFilterHigh.ImeMode")));
			this.udFilterHigh.Increment = new System.Decimal(new int[] {
																		   10,
																		   0,
																		   0,
																		   0});
			this.udFilterHigh.Location = ((System.Drawing.Point)(resources.GetObject("udFilterHigh.Location")));
			this.udFilterHigh.Maximum = new System.Decimal(new int[] {
																		 9999,
																		 0,
																		 0,
																		 0});
			this.udFilterHigh.Minimum = new System.Decimal(new int[] {
																		 9999,
																		 0,
																		 0,
																		 -2147483648});
			this.udFilterHigh.Name = "udFilterHigh";
			this.udFilterHigh.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udFilterHigh.RightToLeft")));
			this.udFilterHigh.Size = ((System.Drawing.Size)(resources.GetObject("udFilterHigh.Size")));
			this.udFilterHigh.TabIndex = ((int)(resources.GetObject("udFilterHigh.TabIndex")));
			this.udFilterHigh.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udFilterHigh.TextAlign")));
			this.udFilterHigh.ThousandsSeparator = ((bool)(resources.GetObject("udFilterHigh.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udFilterHigh, resources.GetString("udFilterHigh.ToolTip"));
			this.udFilterHigh.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udFilterHigh.UpDownAlign")));
			this.udFilterHigh.Value = new System.Decimal(new int[] {
																	   6070,
																	   0,
																	   0,
																	   0});
			this.udFilterHigh.Visible = ((bool)(resources.GetObject("udFilterHigh.Visible")));
			this.udFilterHigh.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udFilterHigh.ValueChanged += new System.EventHandler(this.udFilterHigh_ValueChanged);
			// 
			// udFilterLow
			// 
			this.udFilterLow.AccessibleDescription = resources.GetString("udFilterLow.AccessibleDescription");
			this.udFilterLow.AccessibleName = resources.GetString("udFilterLow.AccessibleName");
			this.udFilterLow.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udFilterLow.Anchor")));
			this.udFilterLow.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udFilterLow.Dock")));
			this.udFilterLow.Enabled = ((bool)(resources.GetObject("udFilterLow.Enabled")));
			this.udFilterLow.Font = ((System.Drawing.Font)(resources.GetObject("udFilterLow.Font")));
			this.udFilterLow.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udFilterLow.ImeMode")));
			this.udFilterLow.Increment = new System.Decimal(new int[] {
																		  10,
																		  0,
																		  0,
																		  0});
			this.udFilterLow.Location = ((System.Drawing.Point)(resources.GetObject("udFilterLow.Location")));
			this.udFilterLow.Maximum = new System.Decimal(new int[] {
																		9999,
																		0,
																		0,
																		0});
			this.udFilterLow.Minimum = new System.Decimal(new int[] {
																		9999,
																		0,
																		0,
																		-2147483648});
			this.udFilterLow.Name = "udFilterLow";
			this.udFilterLow.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udFilterLow.RightToLeft")));
			this.udFilterLow.Size = ((System.Drawing.Size)(resources.GetObject("udFilterLow.Size")));
			this.udFilterLow.TabIndex = ((int)(resources.GetObject("udFilterLow.TabIndex")));
			this.udFilterLow.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udFilterLow.TextAlign")));
			this.udFilterLow.ThousandsSeparator = ((bool)(resources.GetObject("udFilterLow.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udFilterLow, resources.GetString("udFilterLow.ToolTip"));
			this.udFilterLow.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udFilterLow.UpDownAlign")));
			this.udFilterLow.Value = new System.Decimal(new int[] {
																	  540,
																	  0,
																	  0,
																	  0});
			this.udFilterLow.Visible = ((bool)(resources.GetObject("udFilterLow.Visible")));
			this.udFilterLow.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udFilterLow.ValueChanged += new System.EventHandler(this.udFilterLow_ValueChanged);
			// 
			// chkPower
			// 
			this.chkPower.AccessibleDescription = resources.GetString("chkPower.AccessibleDescription");
			this.chkPower.AccessibleName = resources.GetString("chkPower.AccessibleName");
			this.chkPower.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkPower.Anchor")));
			this.chkPower.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkPower.Appearance")));
			this.chkPower.BackColor = System.Drawing.SystemColors.Control;
			this.chkPower.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkPower.BackgroundImage")));
			this.chkPower.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkPower.CheckAlign")));
			this.chkPower.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkPower.Dock")));
			this.chkPower.Enabled = ((bool)(resources.GetObject("chkPower.Enabled")));
			this.chkPower.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkPower.FlatStyle")));
			this.chkPower.Font = ((System.Drawing.Font)(resources.GetObject("chkPower.Font")));
			this.chkPower.Image = ((System.Drawing.Image)(resources.GetObject("chkPower.Image")));
			this.chkPower.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkPower.ImageAlign")));
			this.chkPower.ImageIndex = ((int)(resources.GetObject("chkPower.ImageIndex")));
			this.chkPower.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkPower.ImeMode")));
			this.chkPower.Location = ((System.Drawing.Point)(resources.GetObject("chkPower.Location")));
			this.chkPower.Name = "chkPower";
			this.chkPower.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkPower.RightToLeft")));
			this.chkPower.Size = ((System.Drawing.Size)(resources.GetObject("chkPower.Size")));
			this.chkPower.TabIndex = ((int)(resources.GetObject("chkPower.TabIndex")));
			this.chkPower.Text = resources.GetString("chkPower.Text");
			this.chkPower.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkPower.TextAlign")));
			this.toolTip1.SetToolTip(this.chkPower, resources.GetString("chkPower.ToolTip"));
			this.chkPower.Visible = ((bool)(resources.GetObject("chkPower.Visible")));
			this.chkPower.CheckedChanged += new System.EventHandler(this.chkPower_CheckedChanged);
			// 
			// comboDisplayMode
			// 
			this.comboDisplayMode.AccessibleDescription = resources.GetString("comboDisplayMode.AccessibleDescription");
			this.comboDisplayMode.AccessibleName = resources.GetString("comboDisplayMode.AccessibleName");
			this.comboDisplayMode.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("comboDisplayMode.Anchor")));
			this.comboDisplayMode.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("comboDisplayMode.BackgroundImage")));
			this.comboDisplayMode.DisplayMember = "0";
			this.comboDisplayMode.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("comboDisplayMode.Dock")));
			this.comboDisplayMode.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.comboDisplayMode.DropDownWidth = 88;
			this.comboDisplayMode.Enabled = ((bool)(resources.GetObject("comboDisplayMode.Enabled")));
			this.comboDisplayMode.Font = ((System.Drawing.Font)(resources.GetObject("comboDisplayMode.Font")));
			this.comboDisplayMode.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("comboDisplayMode.ImeMode")));
			this.comboDisplayMode.IntegralHeight = ((bool)(resources.GetObject("comboDisplayMode.IntegralHeight")));
			this.comboDisplayMode.ItemHeight = ((int)(resources.GetObject("comboDisplayMode.ItemHeight")));
			this.comboDisplayMode.Location = ((System.Drawing.Point)(resources.GetObject("comboDisplayMode.Location")));
			this.comboDisplayMode.MaxDropDownItems = ((int)(resources.GetObject("comboDisplayMode.MaxDropDownItems")));
			this.comboDisplayMode.MaxLength = ((int)(resources.GetObject("comboDisplayMode.MaxLength")));
			this.comboDisplayMode.Name = "comboDisplayMode";
			this.comboDisplayMode.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("comboDisplayMode.RightToLeft")));
			this.comboDisplayMode.Size = ((System.Drawing.Size)(resources.GetObject("comboDisplayMode.Size")));
			this.comboDisplayMode.TabIndex = ((int)(resources.GetObject("comboDisplayMode.TabIndex")));
			this.comboDisplayMode.Text = resources.GetString("comboDisplayMode.Text");
			this.toolTip1.SetToolTip(this.comboDisplayMode, resources.GetString("comboDisplayMode.ToolTip"));
			this.comboDisplayMode.Visible = ((bool)(resources.GetObject("comboDisplayMode.Visible")));
			this.comboDisplayMode.SelectedIndexChanged += new System.EventHandler(this.comboDisplayMode_SelectedIndexChanged);
			// 
			// lblCPUMeter
			// 
			this.lblCPUMeter.AccessibleDescription = resources.GetString("lblCPUMeter.AccessibleDescription");
			this.lblCPUMeter.AccessibleName = resources.GetString("lblCPUMeter.AccessibleName");
			this.lblCPUMeter.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblCPUMeter.Anchor")));
			this.lblCPUMeter.AutoSize = ((bool)(resources.GetObject("lblCPUMeter.AutoSize")));
			this.lblCPUMeter.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblCPUMeter.Dock")));
			this.lblCPUMeter.Enabled = ((bool)(resources.GetObject("lblCPUMeter.Enabled")));
			this.lblCPUMeter.Font = ((System.Drawing.Font)(resources.GetObject("lblCPUMeter.Font")));
			this.lblCPUMeter.Image = ((System.Drawing.Image)(resources.GetObject("lblCPUMeter.Image")));
			this.lblCPUMeter.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblCPUMeter.ImageAlign")));
			this.lblCPUMeter.ImageIndex = ((int)(resources.GetObject("lblCPUMeter.ImageIndex")));
			this.lblCPUMeter.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblCPUMeter.ImeMode")));
			this.lblCPUMeter.Location = ((System.Drawing.Point)(resources.GetObject("lblCPUMeter.Location")));
			this.lblCPUMeter.Name = "lblCPUMeter";
			this.lblCPUMeter.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblCPUMeter.RightToLeft")));
			this.lblCPUMeter.Size = ((System.Drawing.Size)(resources.GetObject("lblCPUMeter.Size")));
			this.lblCPUMeter.TabIndex = ((int)(resources.GetObject("lblCPUMeter.TabIndex")));
			this.lblCPUMeter.Text = resources.GetString("lblCPUMeter.Text");
			this.lblCPUMeter.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblCPUMeter.TextAlign")));
			this.toolTip1.SetToolTip(this.lblCPUMeter, resources.GetString("lblCPUMeter.ToolTip"));
			this.lblCPUMeter.Visible = ((bool)(resources.GetObject("lblCPUMeter.Visible")));
			// 
			// timer_cpu_meter
			// 
			this.timer_cpu_meter.Enabled = true;
			this.timer_cpu_meter.Interval = 1000;
			this.timer_cpu_meter.Tick += new System.EventHandler(this.timer_cpu_meter_Tick);
			// 
			// grpDSP
			// 
			this.grpDSP.AccessibleDescription = resources.GetString("grpDSP.AccessibleDescription");
			this.grpDSP.AccessibleName = resources.GetString("grpDSP.AccessibleName");
			this.grpDSP.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpDSP.Anchor")));
			this.grpDSP.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpDSP.BackgroundImage")));
			this.grpDSP.Controls.Add(this.chkDSPComp);
			this.grpDSP.Controls.Add(this.chkANF);
			this.grpDSP.Controls.Add(this.chkNR);
			this.grpDSP.Controls.Add(this.checkBox2);
			this.grpDSP.Controls.Add(this.chkDSPNB2);
			this.grpDSP.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpDSP.Dock")));
			this.grpDSP.Enabled = ((bool)(resources.GetObject("grpDSP.Enabled")));
			this.grpDSP.Font = ((System.Drawing.Font)(resources.GetObject("grpDSP.Font")));
			this.grpDSP.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpDSP.ImeMode")));
			this.grpDSP.Location = ((System.Drawing.Point)(resources.GetObject("grpDSP.Location")));
			this.grpDSP.Name = "grpDSP";
			this.grpDSP.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpDSP.RightToLeft")));
			this.grpDSP.Size = ((System.Drawing.Size)(resources.GetObject("grpDSP.Size")));
			this.grpDSP.TabIndex = ((int)(resources.GetObject("grpDSP.TabIndex")));
			this.grpDSP.TabStop = false;
			this.grpDSP.Text = resources.GetString("grpDSP.Text");
			this.toolTip1.SetToolTip(this.grpDSP, resources.GetString("grpDSP.ToolTip"));
			this.grpDSP.Visible = ((bool)(resources.GetObject("grpDSP.Visible")));
			// 
			// chkDSPComp
			// 
			this.chkDSPComp.AccessibleDescription = resources.GetString("chkDSPComp.AccessibleDescription");
			this.chkDSPComp.AccessibleName = resources.GetString("chkDSPComp.AccessibleName");
			this.chkDSPComp.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkDSPComp.Anchor")));
			this.chkDSPComp.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkDSPComp.Appearance")));
			this.chkDSPComp.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkDSPComp.BackgroundImage")));
			this.chkDSPComp.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkDSPComp.CheckAlign")));
			this.chkDSPComp.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkDSPComp.Dock")));
			this.chkDSPComp.Enabled = ((bool)(resources.GetObject("chkDSPComp.Enabled")));
			this.chkDSPComp.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkDSPComp.FlatStyle")));
			this.chkDSPComp.Font = ((System.Drawing.Font)(resources.GetObject("chkDSPComp.Font")));
			this.chkDSPComp.Image = ((System.Drawing.Image)(resources.GetObject("chkDSPComp.Image")));
			this.chkDSPComp.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkDSPComp.ImageAlign")));
			this.chkDSPComp.ImageIndex = ((int)(resources.GetObject("chkDSPComp.ImageIndex")));
			this.chkDSPComp.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkDSPComp.ImeMode")));
			this.chkDSPComp.Location = ((System.Drawing.Point)(resources.GetObject("chkDSPComp.Location")));
			this.chkDSPComp.Name = "chkDSPComp";
			this.chkDSPComp.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkDSPComp.RightToLeft")));
			this.chkDSPComp.Size = ((System.Drawing.Size)(resources.GetObject("chkDSPComp.Size")));
			this.chkDSPComp.TabIndex = ((int)(resources.GetObject("chkDSPComp.TabIndex")));
			this.chkDSPComp.Text = resources.GetString("chkDSPComp.Text");
			this.chkDSPComp.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkDSPComp.TextAlign")));
			this.toolTip1.SetToolTip(this.chkDSPComp, resources.GetString("chkDSPComp.ToolTip"));
			this.chkDSPComp.Visible = ((bool)(resources.GetObject("chkDSPComp.Visible")));
			this.chkDSPComp.CheckedChanged += new System.EventHandler(this.chkDSPComp_CheckedChanged);
			// 
			// chkANF
			// 
			this.chkANF.AccessibleDescription = resources.GetString("chkANF.AccessibleDescription");
			this.chkANF.AccessibleName = resources.GetString("chkANF.AccessibleName");
			this.chkANF.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkANF.Anchor")));
			this.chkANF.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkANF.Appearance")));
			this.chkANF.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkANF.BackgroundImage")));
			this.chkANF.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkANF.CheckAlign")));
			this.chkANF.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkANF.Dock")));
			this.chkANF.Enabled = ((bool)(resources.GetObject("chkANF.Enabled")));
			this.chkANF.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkANF.FlatStyle")));
			this.chkANF.Font = ((System.Drawing.Font)(resources.GetObject("chkANF.Font")));
			this.chkANF.Image = ((System.Drawing.Image)(resources.GetObject("chkANF.Image")));
			this.chkANF.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkANF.ImageAlign")));
			this.chkANF.ImageIndex = ((int)(resources.GetObject("chkANF.ImageIndex")));
			this.chkANF.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkANF.ImeMode")));
			this.chkANF.Location = ((System.Drawing.Point)(resources.GetObject("chkANF.Location")));
			this.chkANF.Name = "chkANF";
			this.chkANF.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkANF.RightToLeft")));
			this.chkANF.Size = ((System.Drawing.Size)(resources.GetObject("chkANF.Size")));
			this.chkANF.TabIndex = ((int)(resources.GetObject("chkANF.TabIndex")));
			this.chkANF.Text = resources.GetString("chkANF.Text");
			this.chkANF.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkANF.TextAlign")));
			this.toolTip1.SetToolTip(this.chkANF, resources.GetString("chkANF.ToolTip"));
			this.chkANF.Visible = ((bool)(resources.GetObject("chkANF.Visible")));
			this.chkANF.CheckedChanged += new System.EventHandler(this.chkANF_CheckedChanged);
			// 
			// chkNR
			// 
			this.chkNR.AccessibleDescription = resources.GetString("chkNR.AccessibleDescription");
			this.chkNR.AccessibleName = resources.GetString("chkNR.AccessibleName");
			this.chkNR.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkNR.Anchor")));
			this.chkNR.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkNR.Appearance")));
			this.chkNR.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkNR.BackgroundImage")));
			this.chkNR.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkNR.CheckAlign")));
			this.chkNR.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkNR.Dock")));
			this.chkNR.Enabled = ((bool)(resources.GetObject("chkNR.Enabled")));
			this.chkNR.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkNR.FlatStyle")));
			this.chkNR.Font = ((System.Drawing.Font)(resources.GetObject("chkNR.Font")));
			this.chkNR.Image = ((System.Drawing.Image)(resources.GetObject("chkNR.Image")));
			this.chkNR.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkNR.ImageAlign")));
			this.chkNR.ImageIndex = ((int)(resources.GetObject("chkNR.ImageIndex")));
			this.chkNR.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkNR.ImeMode")));
			this.chkNR.Location = ((System.Drawing.Point)(resources.GetObject("chkNR.Location")));
			this.chkNR.Name = "chkNR";
			this.chkNR.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkNR.RightToLeft")));
			this.chkNR.Size = ((System.Drawing.Size)(resources.GetObject("chkNR.Size")));
			this.chkNR.TabIndex = ((int)(resources.GetObject("chkNR.TabIndex")));
			this.chkNR.Text = resources.GetString("chkNR.Text");
			this.chkNR.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkNR.TextAlign")));
			this.toolTip1.SetToolTip(this.chkNR, resources.GetString("chkNR.ToolTip"));
			this.chkNR.Visible = ((bool)(resources.GetObject("chkNR.Visible")));
			this.chkNR.CheckedChanged += new System.EventHandler(this.chkNR_CheckedChanged);
			// 
			// checkBox2
			// 
			this.checkBox2.AccessibleDescription = resources.GetString("checkBox2.AccessibleDescription");
			this.checkBox2.AccessibleName = resources.GetString("checkBox2.AccessibleName");
			this.checkBox2.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("checkBox2.Anchor")));
			this.checkBox2.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("checkBox2.Appearance")));
			this.checkBox2.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("checkBox2.BackgroundImage")));
			this.checkBox2.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("checkBox2.CheckAlign")));
			this.checkBox2.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("checkBox2.Dock")));
			this.checkBox2.Enabled = ((bool)(resources.GetObject("checkBox2.Enabled")));
			this.checkBox2.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("checkBox2.FlatStyle")));
			this.checkBox2.Font = ((System.Drawing.Font)(resources.GetObject("checkBox2.Font")));
			this.checkBox2.Image = ((System.Drawing.Image)(resources.GetObject("checkBox2.Image")));
			this.checkBox2.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("checkBox2.ImageAlign")));
			this.checkBox2.ImageIndex = ((int)(resources.GetObject("checkBox2.ImageIndex")));
			this.checkBox2.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("checkBox2.ImeMode")));
			this.checkBox2.Location = ((System.Drawing.Point)(resources.GetObject("checkBox2.Location")));
			this.checkBox2.Name = "checkBox2";
			this.checkBox2.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("checkBox2.RightToLeft")));
			this.checkBox2.Size = ((System.Drawing.Size)(resources.GetObject("checkBox2.Size")));
			this.checkBox2.TabIndex = ((int)(resources.GetObject("checkBox2.TabIndex")));
			this.checkBox2.Text = resources.GetString("checkBox2.Text");
			this.checkBox2.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("checkBox2.TextAlign")));
			this.toolTip1.SetToolTip(this.checkBox2, resources.GetString("checkBox2.ToolTip"));
			this.checkBox2.Visible = ((bool)(resources.GetObject("checkBox2.Visible")));
			// 
			// chkDSPNB2
			// 
			this.chkDSPNB2.AccessibleDescription = resources.GetString("chkDSPNB2.AccessibleDescription");
			this.chkDSPNB2.AccessibleName = resources.GetString("chkDSPNB2.AccessibleName");
			this.chkDSPNB2.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkDSPNB2.Anchor")));
			this.chkDSPNB2.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkDSPNB2.Appearance")));
			this.chkDSPNB2.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkDSPNB2.BackgroundImage")));
			this.chkDSPNB2.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkDSPNB2.CheckAlign")));
			this.chkDSPNB2.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkDSPNB2.Dock")));
			this.chkDSPNB2.Enabled = ((bool)(resources.GetObject("chkDSPNB2.Enabled")));
			this.chkDSPNB2.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkDSPNB2.FlatStyle")));
			this.chkDSPNB2.Font = ((System.Drawing.Font)(resources.GetObject("chkDSPNB2.Font")));
			this.chkDSPNB2.Image = ((System.Drawing.Image)(resources.GetObject("chkDSPNB2.Image")));
			this.chkDSPNB2.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkDSPNB2.ImageAlign")));
			this.chkDSPNB2.ImageIndex = ((int)(resources.GetObject("chkDSPNB2.ImageIndex")));
			this.chkDSPNB2.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkDSPNB2.ImeMode")));
			this.chkDSPNB2.Location = ((System.Drawing.Point)(resources.GetObject("chkDSPNB2.Location")));
			this.chkDSPNB2.Name = "chkDSPNB2";
			this.chkDSPNB2.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkDSPNB2.RightToLeft")));
			this.chkDSPNB2.Size = ((System.Drawing.Size)(resources.GetObject("chkDSPNB2.Size")));
			this.chkDSPNB2.TabIndex = ((int)(resources.GetObject("chkDSPNB2.TabIndex")));
			this.chkDSPNB2.Text = resources.GetString("chkDSPNB2.Text");
			this.chkDSPNB2.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkDSPNB2.TextAlign")));
			this.toolTip1.SetToolTip(this.chkDSPNB2, resources.GetString("chkDSPNB2.ToolTip"));
			this.chkDSPNB2.Visible = ((bool)(resources.GetObject("chkDSPNB2.Visible")));
			this.chkDSPNB2.CheckedChanged += new System.EventHandler(this.chkDSPNB2_CheckedChanged);
			// 
			// button_IFShiftUp
			// 
			this.button_IFShiftUp.AccessibleDescription = resources.GetString("button_IFShiftUp.AccessibleDescription");
			this.button_IFShiftUp.AccessibleName = resources.GetString("button_IFShiftUp.AccessibleName");
			this.button_IFShiftUp.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("button_IFShiftUp.Anchor")));
			this.button_IFShiftUp.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("button_IFShiftUp.BackgroundImage")));
			this.button_IFShiftUp.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("button_IFShiftUp.Dock")));
			this.button_IFShiftUp.Enabled = ((bool)(resources.GetObject("button_IFShiftUp.Enabled")));
			this.button_IFShiftUp.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("button_IFShiftUp.FlatStyle")));
			this.button_IFShiftUp.Font = ((System.Drawing.Font)(resources.GetObject("button_IFShiftUp.Font")));
			this.button_IFShiftUp.Image = ((System.Drawing.Image)(resources.GetObject("button_IFShiftUp.Image")));
			this.button_IFShiftUp.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_IFShiftUp.ImageAlign")));
			this.button_IFShiftUp.ImageIndex = ((int)(resources.GetObject("button_IFShiftUp.ImageIndex")));
			this.button_IFShiftUp.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("button_IFShiftUp.ImeMode")));
			this.button_IFShiftUp.Location = ((System.Drawing.Point)(resources.GetObject("button_IFShiftUp.Location")));
			this.button_IFShiftUp.Name = "button_IFShiftUp";
			this.button_IFShiftUp.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("button_IFShiftUp.RightToLeft")));
			this.button_IFShiftUp.Size = ((System.Drawing.Size)(resources.GetObject("button_IFShiftUp.Size")));
			this.button_IFShiftUp.TabIndex = ((int)(resources.GetObject("button_IFShiftUp.TabIndex")));
			this.button_IFShiftUp.Text = resources.GetString("button_IFShiftUp.Text");
			this.button_IFShiftUp.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_IFShiftUp.TextAlign")));
			this.toolTip1.SetToolTip(this.button_IFShiftUp, resources.GetString("button_IFShiftUp.ToolTip"));
			this.button_IFShiftUp.Visible = ((bool)(resources.GetObject("button_IFShiftUp.Visible")));
			this.button_IFShiftUp.Click += new System.EventHandler(this.button_IFShiftUp_Click);
			// 
			// button_IFShiftDown
			// 
			this.button_IFShiftDown.AccessibleDescription = resources.GetString("button_IFShiftDown.AccessibleDescription");
			this.button_IFShiftDown.AccessibleName = resources.GetString("button_IFShiftDown.AccessibleName");
			this.button_IFShiftDown.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("button_IFShiftDown.Anchor")));
			this.button_IFShiftDown.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("button_IFShiftDown.BackgroundImage")));
			this.button_IFShiftDown.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("button_IFShiftDown.Dock")));
			this.button_IFShiftDown.Enabled = ((bool)(resources.GetObject("button_IFShiftDown.Enabled")));
			this.button_IFShiftDown.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("button_IFShiftDown.FlatStyle")));
			this.button_IFShiftDown.Font = ((System.Drawing.Font)(resources.GetObject("button_IFShiftDown.Font")));
			this.button_IFShiftDown.Image = ((System.Drawing.Image)(resources.GetObject("button_IFShiftDown.Image")));
			this.button_IFShiftDown.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_IFShiftDown.ImageAlign")));
			this.button_IFShiftDown.ImageIndex = ((int)(resources.GetObject("button_IFShiftDown.ImageIndex")));
			this.button_IFShiftDown.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("button_IFShiftDown.ImeMode")));
			this.button_IFShiftDown.Location = ((System.Drawing.Point)(resources.GetObject("button_IFShiftDown.Location")));
			this.button_IFShiftDown.Name = "button_IFShiftDown";
			this.button_IFShiftDown.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("button_IFShiftDown.RightToLeft")));
			this.button_IFShiftDown.Size = ((System.Drawing.Size)(resources.GetObject("button_IFShiftDown.Size")));
			this.button_IFShiftDown.TabIndex = ((int)(resources.GetObject("button_IFShiftDown.TabIndex")));
			this.button_IFShiftDown.Text = resources.GetString("button_IFShiftDown.Text");
			this.button_IFShiftDown.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_IFShiftDown.TextAlign")));
			this.toolTip1.SetToolTip(this.button_IFShiftDown, resources.GetString("button_IFShiftDown.ToolTip"));
			this.button_IFShiftDown.Visible = ((bool)(resources.GetObject("button_IFShiftDown.Visible")));
			this.button_IFShiftDown.Click += new System.EventHandler(this.button_IFShiftDown_Click);
			// 
			// chkNB
			// 
			this.chkNB.AccessibleDescription = resources.GetString("chkNB.AccessibleDescription");
			this.chkNB.AccessibleName = resources.GetString("chkNB.AccessibleName");
			this.chkNB.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkNB.Anchor")));
			this.chkNB.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkNB.Appearance")));
			this.chkNB.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkNB.BackgroundImage")));
			this.chkNB.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkNB.CheckAlign")));
			this.chkNB.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkNB.Dock")));
			this.chkNB.Enabled = ((bool)(resources.GetObject("chkNB.Enabled")));
			this.chkNB.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkNB.FlatStyle")));
			this.chkNB.Font = ((System.Drawing.Font)(resources.GetObject("chkNB.Font")));
			this.chkNB.Image = ((System.Drawing.Image)(resources.GetObject("chkNB.Image")));
			this.chkNB.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkNB.ImageAlign")));
			this.chkNB.ImageIndex = ((int)(resources.GetObject("chkNB.ImageIndex")));
			this.chkNB.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkNB.ImeMode")));
			this.chkNB.Location = ((System.Drawing.Point)(resources.GetObject("chkNB.Location")));
			this.chkNB.Name = "chkNB";
			this.chkNB.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkNB.RightToLeft")));
			this.chkNB.Size = ((System.Drawing.Size)(resources.GetObject("chkNB.Size")));
			this.chkNB.TabIndex = ((int)(resources.GetObject("chkNB.TabIndex")));
			this.chkNB.Text = resources.GetString("chkNB.Text");
			this.chkNB.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkNB.TextAlign")));
			this.toolTip1.SetToolTip(this.chkNB, resources.GetString("chkNB.ToolTip"));
			this.chkNB.Visible = ((bool)(resources.GetObject("chkNB.Visible")));
			this.chkNB.CheckedChanged += new System.EventHandler(this.chkNB_CheckedChanged);
			// 
			// button_IFShiftZero
			// 
			this.button_IFShiftZero.AccessibleDescription = resources.GetString("button_IFShiftZero.AccessibleDescription");
			this.button_IFShiftZero.AccessibleName = resources.GetString("button_IFShiftZero.AccessibleName");
			this.button_IFShiftZero.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("button_IFShiftZero.Anchor")));
			this.button_IFShiftZero.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("button_IFShiftZero.BackgroundImage")));
			this.button_IFShiftZero.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("button_IFShiftZero.Dock")));
			this.button_IFShiftZero.Enabled = ((bool)(resources.GetObject("button_IFShiftZero.Enabled")));
			this.button_IFShiftZero.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("button_IFShiftZero.FlatStyle")));
			this.button_IFShiftZero.Font = ((System.Drawing.Font)(resources.GetObject("button_IFShiftZero.Font")));
			this.button_IFShiftZero.Image = ((System.Drawing.Image)(resources.GetObject("button_IFShiftZero.Image")));
			this.button_IFShiftZero.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_IFShiftZero.ImageAlign")));
			this.button_IFShiftZero.ImageIndex = ((int)(resources.GetObject("button_IFShiftZero.ImageIndex")));
			this.button_IFShiftZero.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("button_IFShiftZero.ImeMode")));
			this.button_IFShiftZero.Location = ((System.Drawing.Point)(resources.GetObject("button_IFShiftZero.Location")));
			this.button_IFShiftZero.Name = "button_IFShiftZero";
			this.button_IFShiftZero.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("button_IFShiftZero.RightToLeft")));
			this.button_IFShiftZero.Size = ((System.Drawing.Size)(resources.GetObject("button_IFShiftZero.Size")));
			this.button_IFShiftZero.TabIndex = ((int)(resources.GetObject("button_IFShiftZero.TabIndex")));
			this.button_IFShiftZero.Text = resources.GetString("button_IFShiftZero.Text");
			this.button_IFShiftZero.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_IFShiftZero.TextAlign")));
			this.toolTip1.SetToolTip(this.button_IFShiftZero, resources.GetString("button_IFShiftZero.ToolTip"));
			this.button_IFShiftZero.Visible = ((bool)(resources.GetObject("button_IFShiftZero.Visible")));
			this.button_IFShiftZero.Click += new System.EventHandler(this.button_IFShiftZero_Click);
			// 
			// tbWidth
			// 
			this.tbWidth.AccessibleDescription = resources.GetString("tbWidth.AccessibleDescription");
			this.tbWidth.AccessibleName = resources.GetString("tbWidth.AccessibleName");
			this.tbWidth.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("tbWidth.Anchor")));
			this.tbWidth.AutoSize = false;
			this.tbWidth.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("tbWidth.BackgroundImage")));
			this.tbWidth.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("tbWidth.Dock")));
			this.tbWidth.Enabled = ((bool)(resources.GetObject("tbWidth.Enabled")));
			this.tbWidth.Font = ((System.Drawing.Font)(resources.GetObject("tbWidth.Font")));
			this.tbWidth.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("tbWidth.ImeMode")));
			this.tbWidth.Location = ((System.Drawing.Point)(resources.GetObject("tbWidth.Location")));
			this.tbWidth.Maximum = 10000;
			this.tbWidth.Minimum = 1;
			this.tbWidth.Name = "tbWidth";
			this.tbWidth.Orientation = ((System.Windows.Forms.Orientation)(resources.GetObject("tbWidth.Orientation")));
			this.tbWidth.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("tbWidth.RightToLeft")));
			this.tbWidth.Size = ((System.Drawing.Size)(resources.GetObject("tbWidth.Size")));
			this.tbWidth.TabIndex = ((int)(resources.GetObject("tbWidth.TabIndex")));
			this.tbWidth.Text = resources.GetString("tbWidth.Text");
			this.tbWidth.TickStyle = System.Windows.Forms.TickStyle.None;
			this.toolTip1.SetToolTip(this.tbWidth, resources.GetString("tbWidth.ToolTip"));
			this.tbWidth.Value = 100;
			this.tbWidth.Visible = ((bool)(resources.GetObject("tbWidth.Visible")));
			this.tbWidth.Scroll += new System.EventHandler(this.tbWidth_Scroll);
			// 
			// lblAGC
			// 
			this.lblAGC.AccessibleDescription = resources.GetString("lblAGC.AccessibleDescription");
			this.lblAGC.AccessibleName = resources.GetString("lblAGC.AccessibleName");
			this.lblAGC.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblAGC.Anchor")));
			this.lblAGC.AutoSize = ((bool)(resources.GetObject("lblAGC.AutoSize")));
			this.lblAGC.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblAGC.Dock")));
			this.lblAGC.Enabled = ((bool)(resources.GetObject("lblAGC.Enabled")));
			this.lblAGC.Font = ((System.Drawing.Font)(resources.GetObject("lblAGC.Font")));
			this.lblAGC.Image = ((System.Drawing.Image)(resources.GetObject("lblAGC.Image")));
			this.lblAGC.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblAGC.ImageAlign")));
			this.lblAGC.ImageIndex = ((int)(resources.GetObject("lblAGC.ImageIndex")));
			this.lblAGC.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblAGC.ImeMode")));
			this.lblAGC.Location = ((System.Drawing.Point)(resources.GetObject("lblAGC.Location")));
			this.lblAGC.Name = "lblAGC";
			this.lblAGC.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblAGC.RightToLeft")));
			this.lblAGC.Size = ((System.Drawing.Size)(resources.GetObject("lblAGC.Size")));
			this.lblAGC.TabIndex = ((int)(resources.GetObject("lblAGC.TabIndex")));
			this.lblAGC.Text = resources.GetString("lblAGC.Text");
			this.lblAGC.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblAGC.TextAlign")));
			this.toolTip1.SetToolTip(this.lblAGC, resources.GetString("lblAGC.ToolTip"));
			this.lblAGC.Visible = ((bool)(resources.GetObject("lblAGC.Visible")));
			// 
			// comboAGC
			// 
			this.comboAGC.AccessibleDescription = resources.GetString("comboAGC.AccessibleDescription");
			this.comboAGC.AccessibleName = resources.GetString("comboAGC.AccessibleName");
			this.comboAGC.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("comboAGC.Anchor")));
			this.comboAGC.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("comboAGC.BackgroundImage")));
			this.comboAGC.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("comboAGC.Dock")));
			this.comboAGC.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.comboAGC.DropDownWidth = 48;
			this.comboAGC.Enabled = ((bool)(resources.GetObject("comboAGC.Enabled")));
			this.comboAGC.Font = ((System.Drawing.Font)(resources.GetObject("comboAGC.Font")));
			this.comboAGC.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("comboAGC.ImeMode")));
			this.comboAGC.IntegralHeight = ((bool)(resources.GetObject("comboAGC.IntegralHeight")));
			this.comboAGC.ItemHeight = ((int)(resources.GetObject("comboAGC.ItemHeight")));
			this.comboAGC.Location = ((System.Drawing.Point)(resources.GetObject("comboAGC.Location")));
			this.comboAGC.MaxDropDownItems = ((int)(resources.GetObject("comboAGC.MaxDropDownItems")));
			this.comboAGC.MaxLength = ((int)(resources.GetObject("comboAGC.MaxLength")));
			this.comboAGC.Name = "comboAGC";
			this.comboAGC.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("comboAGC.RightToLeft")));
			this.comboAGC.Size = ((System.Drawing.Size)(resources.GetObject("comboAGC.Size")));
			this.comboAGC.TabIndex = ((int)(resources.GetObject("comboAGC.TabIndex")));
			this.comboAGC.Text = resources.GetString("comboAGC.Text");
			this.toolTip1.SetToolTip(this.comboAGC, resources.GetString("comboAGC.ToolTip"));
			this.comboAGC.Visible = ((bool)(resources.GetObject("comboAGC.Visible")));
			this.comboAGC.SelectedIndexChanged += new System.EventHandler(this.comboAGC_SelectedIndexChanged);
			// 
			// chkBIN
			// 
			this.chkBIN.AccessibleDescription = resources.GetString("chkBIN.AccessibleDescription");
			this.chkBIN.AccessibleName = resources.GetString("chkBIN.AccessibleName");
			this.chkBIN.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkBIN.Anchor")));
			this.chkBIN.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkBIN.Appearance")));
			this.chkBIN.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkBIN.BackgroundImage")));
			this.chkBIN.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkBIN.CheckAlign")));
			this.chkBIN.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkBIN.Dock")));
			this.chkBIN.Enabled = ((bool)(resources.GetObject("chkBIN.Enabled")));
			this.chkBIN.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkBIN.FlatStyle")));
			this.chkBIN.Font = ((System.Drawing.Font)(resources.GetObject("chkBIN.Font")));
			this.chkBIN.Image = ((System.Drawing.Image)(resources.GetObject("chkBIN.Image")));
			this.chkBIN.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkBIN.ImageAlign")));
			this.chkBIN.ImageIndex = ((int)(resources.GetObject("chkBIN.ImageIndex")));
			this.chkBIN.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkBIN.ImeMode")));
			this.chkBIN.Location = ((System.Drawing.Point)(resources.GetObject("chkBIN.Location")));
			this.chkBIN.Name = "chkBIN";
			this.chkBIN.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkBIN.RightToLeft")));
			this.chkBIN.Size = ((System.Drawing.Size)(resources.GetObject("chkBIN.Size")));
			this.chkBIN.TabIndex = ((int)(resources.GetObject("chkBIN.TabIndex")));
			this.chkBIN.Text = resources.GetString("chkBIN.Text");
			this.chkBIN.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkBIN.TextAlign")));
			this.toolTip1.SetToolTip(this.chkBIN, resources.GetString("chkBIN.ToolTip"));
			this.chkBIN.Visible = ((bool)(resources.GetObject("chkBIN.Visible")));
			this.chkBIN.CheckedChanged += new System.EventHandler(this.chkBIN_CheckedChanged);
			// 
			// comboPreamp
			// 
			this.comboPreamp.AccessibleDescription = resources.GetString("comboPreamp.AccessibleDescription");
			this.comboPreamp.AccessibleName = resources.GetString("comboPreamp.AccessibleName");
			this.comboPreamp.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("comboPreamp.Anchor")));
			this.comboPreamp.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("comboPreamp.BackgroundImage")));
			this.comboPreamp.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("comboPreamp.Dock")));
			this.comboPreamp.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.comboPreamp.DropDownWidth = 48;
			this.comboPreamp.Enabled = ((bool)(resources.GetObject("comboPreamp.Enabled")));
			this.comboPreamp.Font = ((System.Drawing.Font)(resources.GetObject("comboPreamp.Font")));
			this.comboPreamp.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("comboPreamp.ImeMode")));
			this.comboPreamp.IntegralHeight = ((bool)(resources.GetObject("comboPreamp.IntegralHeight")));
			this.comboPreamp.ItemHeight = ((int)(resources.GetObject("comboPreamp.ItemHeight")));
			this.comboPreamp.Items.AddRange(new object[] {
															 resources.GetString("comboPreamp.Items"),
															 resources.GetString("comboPreamp.Items1")});
			this.comboPreamp.Location = ((System.Drawing.Point)(resources.GetObject("comboPreamp.Location")));
			this.comboPreamp.MaxDropDownItems = ((int)(resources.GetObject("comboPreamp.MaxDropDownItems")));
			this.comboPreamp.MaxLength = ((int)(resources.GetObject("comboPreamp.MaxLength")));
			this.comboPreamp.Name = "comboPreamp";
			this.comboPreamp.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("comboPreamp.RightToLeft")));
			this.comboPreamp.Size = ((System.Drawing.Size)(resources.GetObject("comboPreamp.Size")));
			this.comboPreamp.TabIndex = ((int)(resources.GetObject("comboPreamp.TabIndex")));
			this.comboPreamp.Text = resources.GetString("comboPreamp.Text");
			this.toolTip1.SetToolTip(this.comboPreamp, resources.GetString("comboPreamp.ToolTip"));
			this.comboPreamp.Visible = ((bool)(resources.GetObject("comboPreamp.Visible")));
			this.comboPreamp.SelectedIndexChanged += new System.EventHandler(this.comboPreamp_SelectedIndexChanged);
			// 
			// lblPreamp
			// 
			this.lblPreamp.AccessibleDescription = resources.GetString("lblPreamp.AccessibleDescription");
			this.lblPreamp.AccessibleName = resources.GetString("lblPreamp.AccessibleName");
			this.lblPreamp.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblPreamp.Anchor")));
			this.lblPreamp.AutoSize = ((bool)(resources.GetObject("lblPreamp.AutoSize")));
			this.lblPreamp.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblPreamp.Dock")));
			this.lblPreamp.Enabled = ((bool)(resources.GetObject("lblPreamp.Enabled")));
			this.lblPreamp.Font = ((System.Drawing.Font)(resources.GetObject("lblPreamp.Font")));
			this.lblPreamp.Image = ((System.Drawing.Image)(resources.GetObject("lblPreamp.Image")));
			this.lblPreamp.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblPreamp.ImageAlign")));
			this.lblPreamp.ImageIndex = ((int)(resources.GetObject("lblPreamp.ImageIndex")));
			this.lblPreamp.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblPreamp.ImeMode")));
			this.lblPreamp.Location = ((System.Drawing.Point)(resources.GetObject("lblPreamp.Location")));
			this.lblPreamp.Name = "lblPreamp";
			this.lblPreamp.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblPreamp.RightToLeft")));
			this.lblPreamp.Size = ((System.Drawing.Size)(resources.GetObject("lblPreamp.Size")));
			this.lblPreamp.TabIndex = ((int)(resources.GetObject("lblPreamp.TabIndex")));
			this.lblPreamp.Text = resources.GetString("lblPreamp.Text");
			this.lblPreamp.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblPreamp.TextAlign")));
			this.toolTip1.SetToolTip(this.lblPreamp, resources.GetString("lblPreamp.ToolTip"));
			this.lblPreamp.Visible = ((bool)(resources.GetObject("lblPreamp.Visible")));
			// 
			// udXIT
			// 
			this.udXIT.AccessibleDescription = resources.GetString("udXIT.AccessibleDescription");
			this.udXIT.AccessibleName = resources.GetString("udXIT.AccessibleName");
			this.udXIT.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udXIT.Anchor")));
			this.udXIT.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udXIT.Dock")));
			this.udXIT.Enabled = ((bool)(resources.GetObject("udXIT.Enabled")));
			this.udXIT.Font = ((System.Drawing.Font)(resources.GetObject("udXIT.Font")));
			this.udXIT.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udXIT.ImeMode")));
			this.udXIT.Increment = new System.Decimal(new int[] {
																	1,
																	0,
																	0,
																	0});
			this.udXIT.Location = ((System.Drawing.Point)(resources.GetObject("udXIT.Location")));
			this.udXIT.Maximum = new System.Decimal(new int[] {
																  9999,
																  0,
																  0,
																  0});
			this.udXIT.Minimum = new System.Decimal(new int[] {
																  9999,
																  0,
																  0,
																  -2147483648});
			this.udXIT.Name = "udXIT";
			this.udXIT.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udXIT.RightToLeft")));
			this.udXIT.Size = ((System.Drawing.Size)(resources.GetObject("udXIT.Size")));
			this.udXIT.TabIndex = ((int)(resources.GetObject("udXIT.TabIndex")));
			this.udXIT.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udXIT.TextAlign")));
			this.udXIT.ThousandsSeparator = ((bool)(resources.GetObject("udXIT.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udXIT, resources.GetString("udXIT.ToolTip"));
			this.udXIT.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udXIT.UpDownAlign")));
			this.udXIT.Value = new System.Decimal(new int[] {
																0,
																0,
																0,
																0});
			this.udXIT.Visible = ((bool)(resources.GetObject("udXIT.Visible")));
			this.udXIT.Click += new System.EventHandler(this.udXIT_ValueChanged);
			this.udXIT.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udXIT.ValueChanged += new System.EventHandler(this.udXIT_ValueChanged);
			// 
			// chkMON
			// 
			this.chkMON.AccessibleDescription = resources.GetString("chkMON.AccessibleDescription");
			this.chkMON.AccessibleName = resources.GetString("chkMON.AccessibleName");
			this.chkMON.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkMON.Anchor")));
			this.chkMON.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkMON.Appearance")));
			this.chkMON.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkMON.BackgroundImage")));
			this.chkMON.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMON.CheckAlign")));
			this.chkMON.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkMON.Dock")));
			this.chkMON.Enabled = ((bool)(resources.GetObject("chkMON.Enabled")));
			this.chkMON.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkMON.FlatStyle")));
			this.chkMON.Font = ((System.Drawing.Font)(resources.GetObject("chkMON.Font")));
			this.chkMON.Image = ((System.Drawing.Image)(resources.GetObject("chkMON.Image")));
			this.chkMON.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMON.ImageAlign")));
			this.chkMON.ImageIndex = ((int)(resources.GetObject("chkMON.ImageIndex")));
			this.chkMON.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkMON.ImeMode")));
			this.chkMON.Location = ((System.Drawing.Point)(resources.GetObject("chkMON.Location")));
			this.chkMON.Name = "chkMON";
			this.chkMON.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkMON.RightToLeft")));
			this.chkMON.Size = ((System.Drawing.Size)(resources.GetObject("chkMON.Size")));
			this.chkMON.TabIndex = ((int)(resources.GetObject("chkMON.TabIndex")));
			this.chkMON.Text = resources.GetString("chkMON.Text");
			this.chkMON.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMON.TextAlign")));
			this.toolTip1.SetToolTip(this.chkMON, resources.GetString("chkMON.ToolTip"));
			this.chkMON.Visible = ((bool)(resources.GetObject("chkMON.Visible")));
			this.chkMON.CheckedChanged += new System.EventHandler(this.chkMON_CheckedChanged);
			// 
			// chkTUN
			// 
			this.chkTUN.AccessibleDescription = resources.GetString("chkTUN.AccessibleDescription");
			this.chkTUN.AccessibleName = resources.GetString("chkTUN.AccessibleName");
			this.chkTUN.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkTUN.Anchor")));
			this.chkTUN.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkTUN.Appearance")));
			this.chkTUN.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkTUN.BackgroundImage")));
			this.chkTUN.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkTUN.CheckAlign")));
			this.chkTUN.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkTUN.Dock")));
			this.chkTUN.Enabled = ((bool)(resources.GetObject("chkTUN.Enabled")));
			this.chkTUN.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkTUN.FlatStyle")));
			this.chkTUN.Font = ((System.Drawing.Font)(resources.GetObject("chkTUN.Font")));
			this.chkTUN.Image = ((System.Drawing.Image)(resources.GetObject("chkTUN.Image")));
			this.chkTUN.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkTUN.ImageAlign")));
			this.chkTUN.ImageIndex = ((int)(resources.GetObject("chkTUN.ImageIndex")));
			this.chkTUN.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkTUN.ImeMode")));
			this.chkTUN.Location = ((System.Drawing.Point)(resources.GetObject("chkTUN.Location")));
			this.chkTUN.Name = "chkTUN";
			this.chkTUN.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkTUN.RightToLeft")));
			this.chkTUN.Size = ((System.Drawing.Size)(resources.GetObject("chkTUN.Size")));
			this.chkTUN.TabIndex = ((int)(resources.GetObject("chkTUN.TabIndex")));
			this.chkTUN.Text = resources.GetString("chkTUN.Text");
			this.chkTUN.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkTUN.TextAlign")));
			this.toolTip1.SetToolTip(this.chkTUN, resources.GetString("chkTUN.ToolTip"));
			this.chkTUN.Visible = ((bool)(resources.GetObject("chkTUN.Visible")));
			this.chkTUN.CheckedChanged += new System.EventHandler(this.chkTUN_CheckedChanged);
			// 
			// chkMOX
			// 
			this.chkMOX.AccessibleDescription = resources.GetString("chkMOX.AccessibleDescription");
			this.chkMOX.AccessibleName = resources.GetString("chkMOX.AccessibleName");
			this.chkMOX.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkMOX.Anchor")));
			this.chkMOX.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkMOX.Appearance")));
			this.chkMOX.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkMOX.BackgroundImage")));
			this.chkMOX.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMOX.CheckAlign")));
			this.chkMOX.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkMOX.Dock")));
			this.chkMOX.Enabled = ((bool)(resources.GetObject("chkMOX.Enabled")));
			this.chkMOX.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkMOX.FlatStyle")));
			this.chkMOX.Font = ((System.Drawing.Font)(resources.GetObject("chkMOX.Font")));
			this.chkMOX.Image = ((System.Drawing.Image)(resources.GetObject("chkMOX.Image")));
			this.chkMOX.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMOX.ImageAlign")));
			this.chkMOX.ImageIndex = ((int)(resources.GetObject("chkMOX.ImageIndex")));
			this.chkMOX.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkMOX.ImeMode")));
			this.chkMOX.Location = ((System.Drawing.Point)(resources.GetObject("chkMOX.Location")));
			this.chkMOX.Name = "chkMOX";
			this.chkMOX.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkMOX.RightToLeft")));
			this.chkMOX.Size = ((System.Drawing.Size)(resources.GetObject("chkMOX.Size")));
			this.chkMOX.TabIndex = ((int)(resources.GetObject("chkMOX.TabIndex")));
			this.chkMOX.Text = resources.GetString("chkMOX.Text");
			this.chkMOX.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMOX.TextAlign")));
			this.toolTip1.SetToolTip(this.chkMOX, resources.GetString("chkMOX.ToolTip"));
			this.chkMOX.Visible = ((bool)(resources.GetObject("chkMOX.Visible")));
			this.chkMOX.Click += new System.EventHandler(this.chkMOX_Click);
			this.chkMOX.CheckedChanged += new System.EventHandler(this.chkMOX_CheckedChanged);
			// 
			// udRIT
			// 
			this.udRIT.AccessibleDescription = resources.GetString("udRIT.AccessibleDescription");
			this.udRIT.AccessibleName = resources.GetString("udRIT.AccessibleName");
			this.udRIT.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udRIT.Anchor")));
			this.udRIT.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udRIT.Dock")));
			this.udRIT.Enabled = ((bool)(resources.GetObject("udRIT.Enabled")));
			this.udRIT.Font = ((System.Drawing.Font)(resources.GetObject("udRIT.Font")));
			this.udRIT.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udRIT.ImeMode")));
			this.udRIT.Increment = new System.Decimal(new int[] {
																	1,
																	0,
																	0,
																	0});
			this.udRIT.Location = ((System.Drawing.Point)(resources.GetObject("udRIT.Location")));
			this.udRIT.Maximum = new System.Decimal(new int[] {
																  9999,
																  0,
																  0,
																  0});
			this.udRIT.Minimum = new System.Decimal(new int[] {
																  9999,
																  0,
																  0,
																  -2147483648});
			this.udRIT.Name = "udRIT";
			this.udRIT.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udRIT.RightToLeft")));
			this.udRIT.Size = ((System.Drawing.Size)(resources.GetObject("udRIT.Size")));
			this.udRIT.TabIndex = ((int)(resources.GetObject("udRIT.TabIndex")));
			this.udRIT.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udRIT.TextAlign")));
			this.udRIT.ThousandsSeparator = ((bool)(resources.GetObject("udRIT.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udRIT, resources.GetString("udRIT.ToolTip"));
			this.udRIT.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udRIT.UpDownAlign")));
			this.udRIT.Value = new System.Decimal(new int[] {
																0,
																0,
																0,
																0});
			this.udRIT.Visible = ((bool)(resources.GetObject("udRIT.Visible")));
			this.udRIT.Click += new System.EventHandler(this.udRIT_ValueChanged);
			this.udRIT.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udRIT.ValueChanged += new System.EventHandler(this.udRIT_ValueChanged);
			// 
			// chkMUT
			// 
			this.chkMUT.AccessibleDescription = resources.GetString("chkMUT.AccessibleDescription");
			this.chkMUT.AccessibleName = resources.GetString("chkMUT.AccessibleName");
			this.chkMUT.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkMUT.Anchor")));
			this.chkMUT.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkMUT.Appearance")));
			this.chkMUT.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkMUT.BackgroundImage")));
			this.chkMUT.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMUT.CheckAlign")));
			this.chkMUT.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkMUT.Dock")));
			this.chkMUT.Enabled = ((bool)(resources.GetObject("chkMUT.Enabled")));
			this.chkMUT.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkMUT.FlatStyle")));
			this.chkMUT.Font = ((System.Drawing.Font)(resources.GetObject("chkMUT.Font")));
			this.chkMUT.Image = ((System.Drawing.Image)(resources.GetObject("chkMUT.Image")));
			this.chkMUT.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMUT.ImageAlign")));
			this.chkMUT.ImageIndex = ((int)(resources.GetObject("chkMUT.ImageIndex")));
			this.chkMUT.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkMUT.ImeMode")));
			this.chkMUT.Location = ((System.Drawing.Point)(resources.GetObject("chkMUT.Location")));
			this.chkMUT.Name = "chkMUT";
			this.chkMUT.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkMUT.RightToLeft")));
			this.chkMUT.Size = ((System.Drawing.Size)(resources.GetObject("chkMUT.Size")));
			this.chkMUT.TabIndex = ((int)(resources.GetObject("chkMUT.TabIndex")));
			this.chkMUT.Text = resources.GetString("chkMUT.Text");
			this.chkMUT.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMUT.TextAlign")));
			this.toolTip1.SetToolTip(this.chkMUT, resources.GetString("chkMUT.ToolTip"));
			this.chkMUT.Visible = ((bool)(resources.GetObject("chkMUT.Visible")));
			this.chkMUT.CheckedChanged += new System.EventHandler(this.chkMUT_CheckedChanged);
			// 
			// chkXIT
			// 
			this.chkXIT.AccessibleDescription = resources.GetString("chkXIT.AccessibleDescription");
			this.chkXIT.AccessibleName = resources.GetString("chkXIT.AccessibleName");
			this.chkXIT.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkXIT.Anchor")));
			this.chkXIT.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkXIT.Appearance")));
			this.chkXIT.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkXIT.BackgroundImage")));
			this.chkXIT.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkXIT.CheckAlign")));
			this.chkXIT.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkXIT.Dock")));
			this.chkXIT.Enabled = ((bool)(resources.GetObject("chkXIT.Enabled")));
			this.chkXIT.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkXIT.FlatStyle")));
			this.chkXIT.Font = ((System.Drawing.Font)(resources.GetObject("chkXIT.Font")));
			this.chkXIT.Image = ((System.Drawing.Image)(resources.GetObject("chkXIT.Image")));
			this.chkXIT.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkXIT.ImageAlign")));
			this.chkXIT.ImageIndex = ((int)(resources.GetObject("chkXIT.ImageIndex")));
			this.chkXIT.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkXIT.ImeMode")));
			this.chkXIT.Location = ((System.Drawing.Point)(resources.GetObject("chkXIT.Location")));
			this.chkXIT.Name = "chkXIT";
			this.chkXIT.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkXIT.RightToLeft")));
			this.chkXIT.Size = ((System.Drawing.Size)(resources.GetObject("chkXIT.Size")));
			this.chkXIT.TabIndex = ((int)(resources.GetObject("chkXIT.TabIndex")));
			this.chkXIT.Text = resources.GetString("chkXIT.Text");
			this.chkXIT.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkXIT.TextAlign")));
			this.toolTip1.SetToolTip(this.chkXIT, resources.GetString("chkXIT.ToolTip"));
			this.chkXIT.Visible = ((bool)(resources.GetObject("chkXIT.Visible")));
			this.chkXIT.CheckedChanged += new System.EventHandler(this.chkXIT_CheckedChanged);
			// 
			// chkRIT
			// 
			this.chkRIT.AccessibleDescription = resources.GetString("chkRIT.AccessibleDescription");
			this.chkRIT.AccessibleName = resources.GetString("chkRIT.AccessibleName");
			this.chkRIT.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkRIT.Anchor")));
			this.chkRIT.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkRIT.Appearance")));
			this.chkRIT.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkRIT.BackgroundImage")));
			this.chkRIT.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkRIT.CheckAlign")));
			this.chkRIT.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkRIT.Dock")));
			this.chkRIT.Enabled = ((bool)(resources.GetObject("chkRIT.Enabled")));
			this.chkRIT.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkRIT.FlatStyle")));
			this.chkRIT.Font = ((System.Drawing.Font)(resources.GetObject("chkRIT.Font")));
			this.chkRIT.Image = ((System.Drawing.Image)(resources.GetObject("chkRIT.Image")));
			this.chkRIT.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkRIT.ImageAlign")));
			this.chkRIT.ImageIndex = ((int)(resources.GetObject("chkRIT.ImageIndex")));
			this.chkRIT.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkRIT.ImeMode")));
			this.chkRIT.Location = ((System.Drawing.Point)(resources.GetObject("chkRIT.Location")));
			this.chkRIT.Name = "chkRIT";
			this.chkRIT.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkRIT.RightToLeft")));
			this.chkRIT.Size = ((System.Drawing.Size)(resources.GetObject("chkRIT.Size")));
			this.chkRIT.TabIndex = ((int)(resources.GetObject("chkRIT.TabIndex")));
			this.chkRIT.Text = resources.GetString("chkRIT.Text");
			this.chkRIT.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkRIT.TextAlign")));
			this.toolTip1.SetToolTip(this.chkRIT, resources.GetString("chkRIT.ToolTip"));
			this.chkRIT.Visible = ((bool)(resources.GetObject("chkRIT.Visible")));
			this.chkRIT.CheckedChanged += new System.EventHandler(this.chkRIT_CheckedChanged);
			// 
			// lblPWR
			// 
			this.lblPWR.AccessibleDescription = resources.GetString("lblPWR.AccessibleDescription");
			this.lblPWR.AccessibleName = resources.GetString("lblPWR.AccessibleName");
			this.lblPWR.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblPWR.Anchor")));
			this.lblPWR.AutoSize = ((bool)(resources.GetObject("lblPWR.AutoSize")));
			this.lblPWR.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblPWR.Dock")));
			this.lblPWR.Enabled = ((bool)(resources.GetObject("lblPWR.Enabled")));
			this.lblPWR.Font = ((System.Drawing.Font)(resources.GetObject("lblPWR.Font")));
			this.lblPWR.Image = ((System.Drawing.Image)(resources.GetObject("lblPWR.Image")));
			this.lblPWR.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblPWR.ImageAlign")));
			this.lblPWR.ImageIndex = ((int)(resources.GetObject("lblPWR.ImageIndex")));
			this.lblPWR.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblPWR.ImeMode")));
			this.lblPWR.Location = ((System.Drawing.Point)(resources.GetObject("lblPWR.Location")));
			this.lblPWR.Name = "lblPWR";
			this.lblPWR.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblPWR.RightToLeft")));
			this.lblPWR.Size = ((System.Drawing.Size)(resources.GetObject("lblPWR.Size")));
			this.lblPWR.TabIndex = ((int)(resources.GetObject("lblPWR.TabIndex")));
			this.lblPWR.Text = resources.GetString("lblPWR.Text");
			this.lblPWR.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblPWR.TextAlign")));
			this.toolTip1.SetToolTip(this.lblPWR, resources.GetString("lblPWR.ToolTip"));
			this.lblPWR.Visible = ((bool)(resources.GetObject("lblPWR.Visible")));
			// 
			// udPWR
			// 
			this.udPWR.AccessibleDescription = resources.GetString("udPWR.AccessibleDescription");
			this.udPWR.AccessibleName = resources.GetString("udPWR.AccessibleName");
			this.udPWR.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udPWR.Anchor")));
			this.udPWR.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udPWR.Dock")));
			this.udPWR.Enabled = ((bool)(resources.GetObject("udPWR.Enabled")));
			this.udPWR.Font = ((System.Drawing.Font)(resources.GetObject("udPWR.Font")));
			this.udPWR.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udPWR.ImeMode")));
			this.udPWR.Increment = new System.Decimal(new int[] {
																	1,
																	0,
																	0,
																	0});
			this.udPWR.Location = ((System.Drawing.Point)(resources.GetObject("udPWR.Location")));
			this.udPWR.Maximum = new System.Decimal(new int[] {
																  100,
																  0,
																  0,
																  0});
			this.udPWR.Minimum = new System.Decimal(new int[] {
																  0,
																  0,
																  0,
																  0});
			this.udPWR.Name = "udPWR";
			this.udPWR.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udPWR.RightToLeft")));
			this.udPWR.Size = ((System.Drawing.Size)(resources.GetObject("udPWR.Size")));
			this.udPWR.TabIndex = ((int)(resources.GetObject("udPWR.TabIndex")));
			this.udPWR.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udPWR.TextAlign")));
			this.udPWR.ThousandsSeparator = ((bool)(resources.GetObject("udPWR.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udPWR, resources.GetString("udPWR.ToolTip"));
			this.udPWR.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udPWR.UpDownAlign")));
			this.udPWR.Value = new System.Decimal(new int[] {
																50,
																0,
																0,
																0});
			this.udPWR.Visible = ((bool)(resources.GetObject("udPWR.Visible")));
			this.udPWR.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udPWR.ValueChanged += new System.EventHandler(this.udPWR_ValueChanged);
			// 
			// lblAF
			// 
			this.lblAF.AccessibleDescription = resources.GetString("lblAF.AccessibleDescription");
			this.lblAF.AccessibleName = resources.GetString("lblAF.AccessibleName");
			this.lblAF.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblAF.Anchor")));
			this.lblAF.AutoSize = ((bool)(resources.GetObject("lblAF.AutoSize")));
			this.lblAF.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblAF.Dock")));
			this.lblAF.Enabled = ((bool)(resources.GetObject("lblAF.Enabled")));
			this.lblAF.Font = ((System.Drawing.Font)(resources.GetObject("lblAF.Font")));
			this.lblAF.Image = ((System.Drawing.Image)(resources.GetObject("lblAF.Image")));
			this.lblAF.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblAF.ImageAlign")));
			this.lblAF.ImageIndex = ((int)(resources.GetObject("lblAF.ImageIndex")));
			this.lblAF.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblAF.ImeMode")));
			this.lblAF.Location = ((System.Drawing.Point)(resources.GetObject("lblAF.Location")));
			this.lblAF.Name = "lblAF";
			this.lblAF.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblAF.RightToLeft")));
			this.lblAF.Size = ((System.Drawing.Size)(resources.GetObject("lblAF.Size")));
			this.lblAF.TabIndex = ((int)(resources.GetObject("lblAF.TabIndex")));
			this.lblAF.Text = resources.GetString("lblAF.Text");
			this.lblAF.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblAF.TextAlign")));
			this.toolTip1.SetToolTip(this.lblAF, resources.GetString("lblAF.ToolTip"));
			this.lblAF.Visible = ((bool)(resources.GetObject("lblAF.Visible")));
			// 
			// udAF
			// 
			this.udAF.AccessibleDescription = resources.GetString("udAF.AccessibleDescription");
			this.udAF.AccessibleName = resources.GetString("udAF.AccessibleName");
			this.udAF.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udAF.Anchor")));
			this.udAF.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udAF.Dock")));
			this.udAF.Enabled = ((bool)(resources.GetObject("udAF.Enabled")));
			this.udAF.Font = ((System.Drawing.Font)(resources.GetObject("udAF.Font")));
			this.udAF.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udAF.ImeMode")));
			this.udAF.Increment = new System.Decimal(new int[] {
																   1,
																   0,
																   0,
																   0});
			this.udAF.Location = ((System.Drawing.Point)(resources.GetObject("udAF.Location")));
			this.udAF.Maximum = new System.Decimal(new int[] {
																 100,
																 0,
																 0,
																 0});
			this.udAF.Minimum = new System.Decimal(new int[] {
																 0,
																 0,
																 0,
																 0});
			this.udAF.Name = "udAF";
			this.udAF.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udAF.RightToLeft")));
			this.udAF.Size = ((System.Drawing.Size)(resources.GetObject("udAF.Size")));
			this.udAF.TabIndex = ((int)(resources.GetObject("udAF.TabIndex")));
			this.udAF.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udAF.TextAlign")));
			this.udAF.ThousandsSeparator = ((bool)(resources.GetObject("udAF.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udAF, resources.GetString("udAF.ToolTip"));
			this.udAF.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udAF.UpDownAlign")));
			this.udAF.Value = new System.Decimal(new int[] {
															   50,
															   0,
															   0,
															   0});
			this.udAF.Visible = ((bool)(resources.GetObject("udAF.Visible")));
			this.udAF.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udAF.ValueChanged += new System.EventHandler(this.udAF_ValueChanged);
			// 
			// lblMIC
			// 
			this.lblMIC.AccessibleDescription = resources.GetString("lblMIC.AccessibleDescription");
			this.lblMIC.AccessibleName = resources.GetString("lblMIC.AccessibleName");
			this.lblMIC.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblMIC.Anchor")));
			this.lblMIC.AutoSize = ((bool)(resources.GetObject("lblMIC.AutoSize")));
			this.lblMIC.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblMIC.Dock")));
			this.lblMIC.Enabled = ((bool)(resources.GetObject("lblMIC.Enabled")));
			this.lblMIC.Font = ((System.Drawing.Font)(resources.GetObject("lblMIC.Font")));
			this.lblMIC.Image = ((System.Drawing.Image)(resources.GetObject("lblMIC.Image")));
			this.lblMIC.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblMIC.ImageAlign")));
			this.lblMIC.ImageIndex = ((int)(resources.GetObject("lblMIC.ImageIndex")));
			this.lblMIC.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblMIC.ImeMode")));
			this.lblMIC.Location = ((System.Drawing.Point)(resources.GetObject("lblMIC.Location")));
			this.lblMIC.Name = "lblMIC";
			this.lblMIC.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblMIC.RightToLeft")));
			this.lblMIC.Size = ((System.Drawing.Size)(resources.GetObject("lblMIC.Size")));
			this.lblMIC.TabIndex = ((int)(resources.GetObject("lblMIC.TabIndex")));
			this.lblMIC.Text = resources.GetString("lblMIC.Text");
			this.lblMIC.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblMIC.TextAlign")));
			this.toolTip1.SetToolTip(this.lblMIC, resources.GetString("lblMIC.ToolTip"));
			this.lblMIC.Visible = ((bool)(resources.GetObject("lblMIC.Visible")));
			// 
			// udMIC
			// 
			this.udMIC.AccessibleDescription = resources.GetString("udMIC.AccessibleDescription");
			this.udMIC.AccessibleName = resources.GetString("udMIC.AccessibleName");
			this.udMIC.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udMIC.Anchor")));
			this.udMIC.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udMIC.Dock")));
			this.udMIC.Enabled = ((bool)(resources.GetObject("udMIC.Enabled")));
			this.udMIC.Font = ((System.Drawing.Font)(resources.GetObject("udMIC.Font")));
			this.udMIC.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udMIC.ImeMode")));
			this.udMIC.Increment = new System.Decimal(new int[] {
																	1,
																	0,
																	0,
																	0});
			this.udMIC.Location = ((System.Drawing.Point)(resources.GetObject("udMIC.Location")));
			this.udMIC.Maximum = new System.Decimal(new int[] {
																  100,
																  0,
																  0,
																  0});
			this.udMIC.Minimum = new System.Decimal(new int[] {
																  0,
																  0,
																  0,
																  0});
			this.udMIC.Name = "udMIC";
			this.udMIC.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udMIC.RightToLeft")));
			this.udMIC.Size = ((System.Drawing.Size)(resources.GetObject("udMIC.Size")));
			this.udMIC.TabIndex = ((int)(resources.GetObject("udMIC.TabIndex")));
			this.udMIC.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udMIC.TextAlign")));
			this.udMIC.ThousandsSeparator = ((bool)(resources.GetObject("udMIC.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udMIC, resources.GetString("udMIC.ToolTip"));
			this.udMIC.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udMIC.UpDownAlign")));
			this.udMIC.Value = new System.Decimal(new int[] {
																50,
																0,
																0,
																0});
			this.udMIC.Visible = ((bool)(resources.GetObject("udMIC.Visible")));
			this.udMIC.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udMIC.ValueChanged += new System.EventHandler(this.udMIC_ValueChanged);
			// 
			// chkPipe
			// 
			this.chkPipe.AccessibleDescription = resources.GetString("chkPipe.AccessibleDescription");
			this.chkPipe.AccessibleName = resources.GetString("chkPipe.AccessibleName");
			this.chkPipe.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkPipe.Anchor")));
			this.chkPipe.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkPipe.Appearance")));
			this.chkPipe.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkPipe.BackgroundImage")));
			this.chkPipe.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkPipe.CheckAlign")));
			this.chkPipe.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkPipe.Dock")));
			this.chkPipe.Enabled = ((bool)(resources.GetObject("chkPipe.Enabled")));
			this.chkPipe.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkPipe.FlatStyle")));
			this.chkPipe.Font = ((System.Drawing.Font)(resources.GetObject("chkPipe.Font")));
			this.chkPipe.Image = ((System.Drawing.Image)(resources.GetObject("chkPipe.Image")));
			this.chkPipe.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkPipe.ImageAlign")));
			this.chkPipe.ImageIndex = ((int)(resources.GetObject("chkPipe.ImageIndex")));
			this.chkPipe.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkPipe.ImeMode")));
			this.chkPipe.Location = ((System.Drawing.Point)(resources.GetObject("chkPipe.Location")));
			this.chkPipe.Name = "chkPipe";
			this.chkPipe.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkPipe.RightToLeft")));
			this.chkPipe.Size = ((System.Drawing.Size)(resources.GetObject("chkPipe.Size")));
			this.chkPipe.TabIndex = ((int)(resources.GetObject("chkPipe.TabIndex")));
			this.chkPipe.Text = resources.GetString("chkPipe.Text");
			this.chkPipe.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkPipe.TextAlign")));
			this.toolTip1.SetToolTip(this.chkPipe, resources.GetString("chkPipe.ToolTip"));
			this.chkPipe.Visible = ((bool)(resources.GetObject("chkPipe.Visible")));
			this.chkPipe.CheckedChanged += new System.EventHandler(this.chkPipe_CheckedChanged);
			// 
			// grpMultimeter
			// 
			this.grpMultimeter.AccessibleDescription = resources.GetString("grpMultimeter.AccessibleDescription");
			this.grpMultimeter.AccessibleName = resources.GetString("grpMultimeter.AccessibleName");
			this.grpMultimeter.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpMultimeter.Anchor")));
			this.grpMultimeter.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpMultimeter.BackgroundImage")));
			this.grpMultimeter.Controls.Add(this.comboMeterTXMode);
			this.grpMultimeter.Controls.Add(this.picMultiMeterDigital);
			this.grpMultimeter.Controls.Add(this.lblMultiSMeter);
			this.grpMultimeter.Controls.Add(this.comboMeterRXMode);
			this.grpMultimeter.Controls.Add(this.txtMultiText);
			this.grpMultimeter.Controls.Add(this.picMultimeterAnalog);
			this.grpMultimeter.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpMultimeter.Dock")));
			this.grpMultimeter.Enabled = ((bool)(resources.GetObject("grpMultimeter.Enabled")));
			this.grpMultimeter.Font = ((System.Drawing.Font)(resources.GetObject("grpMultimeter.Font")));
			this.grpMultimeter.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpMultimeter.ImeMode")));
			this.grpMultimeter.Location = ((System.Drawing.Point)(resources.GetObject("grpMultimeter.Location")));
			this.grpMultimeter.Name = "grpMultimeter";
			this.grpMultimeter.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpMultimeter.RightToLeft")));
			this.grpMultimeter.Size = ((System.Drawing.Size)(resources.GetObject("grpMultimeter.Size")));
			this.grpMultimeter.TabIndex = ((int)(resources.GetObject("grpMultimeter.TabIndex")));
			this.grpMultimeter.TabStop = false;
			this.grpMultimeter.Text = resources.GetString("grpMultimeter.Text");
			this.toolTip1.SetToolTip(this.grpMultimeter, resources.GetString("grpMultimeter.ToolTip"));
			this.grpMultimeter.Visible = ((bool)(resources.GetObject("grpMultimeter.Visible")));
			// 
			// comboMeterTXMode
			// 
			this.comboMeterTXMode.AccessibleDescription = resources.GetString("comboMeterTXMode.AccessibleDescription");
			this.comboMeterTXMode.AccessibleName = resources.GetString("comboMeterTXMode.AccessibleName");
			this.comboMeterTXMode.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("comboMeterTXMode.Anchor")));
			this.comboMeterTXMode.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("comboMeterTXMode.BackgroundImage")));
			this.comboMeterTXMode.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("comboMeterTXMode.Dock")));
			this.comboMeterTXMode.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.comboMeterTXMode.DropDownWidth = 72;
			this.comboMeterTXMode.Enabled = ((bool)(resources.GetObject("comboMeterTXMode.Enabled")));
			this.comboMeterTXMode.Font = ((System.Drawing.Font)(resources.GetObject("comboMeterTXMode.Font")));
			this.comboMeterTXMode.ForeColor = System.Drawing.Color.Gray;
			this.comboMeterTXMode.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("comboMeterTXMode.ImeMode")));
			this.comboMeterTXMode.IntegralHeight = ((bool)(resources.GetObject("comboMeterTXMode.IntegralHeight")));
			this.comboMeterTXMode.ItemHeight = ((int)(resources.GetObject("comboMeterTXMode.ItemHeight")));
			this.comboMeterTXMode.Location = ((System.Drawing.Point)(resources.GetObject("comboMeterTXMode.Location")));
			this.comboMeterTXMode.MaxDropDownItems = ((int)(resources.GetObject("comboMeterTXMode.MaxDropDownItems")));
			this.comboMeterTXMode.MaxLength = ((int)(resources.GetObject("comboMeterTXMode.MaxLength")));
			this.comboMeterTXMode.Name = "comboMeterTXMode";
			this.comboMeterTXMode.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("comboMeterTXMode.RightToLeft")));
			this.comboMeterTXMode.Size = ((System.Drawing.Size)(resources.GetObject("comboMeterTXMode.Size")));
			this.comboMeterTXMode.TabIndex = ((int)(resources.GetObject("comboMeterTXMode.TabIndex")));
			this.comboMeterTXMode.Text = resources.GetString("comboMeterTXMode.Text");
			this.toolTip1.SetToolTip(this.comboMeterTXMode, resources.GetString("comboMeterTXMode.ToolTip"));
			this.comboMeterTXMode.Visible = ((bool)(resources.GetObject("comboMeterTXMode.Visible")));
			this.comboMeterTXMode.SelectedIndexChanged += new System.EventHandler(this.comboMeterTXMode_SelectedIndexChanged);
			// 
			// picMultiMeterDigital
			// 
			this.picMultiMeterDigital.AccessibleDescription = resources.GetString("picMultiMeterDigital.AccessibleDescription");
			this.picMultiMeterDigital.AccessibleName = resources.GetString("picMultiMeterDigital.AccessibleName");
			this.picMultiMeterDigital.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("picMultiMeterDigital.Anchor")));
			this.picMultiMeterDigital.BackColor = System.Drawing.Color.Black;
			this.picMultiMeterDigital.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("picMultiMeterDigital.BackgroundImage")));
			this.picMultiMeterDigital.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
			this.picMultiMeterDigital.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("picMultiMeterDigital.Dock")));
			this.picMultiMeterDigital.Enabled = ((bool)(resources.GetObject("picMultiMeterDigital.Enabled")));
			this.picMultiMeterDigital.Font = ((System.Drawing.Font)(resources.GetObject("picMultiMeterDigital.Font")));
			this.picMultiMeterDigital.Image = ((System.Drawing.Image)(resources.GetObject("picMultiMeterDigital.Image")));
			this.picMultiMeterDigital.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("picMultiMeterDigital.ImeMode")));
			this.picMultiMeterDigital.Location = ((System.Drawing.Point)(resources.GetObject("picMultiMeterDigital.Location")));
			this.picMultiMeterDigital.Name = "picMultiMeterDigital";
			this.picMultiMeterDigital.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("picMultiMeterDigital.RightToLeft")));
			this.picMultiMeterDigital.Size = ((System.Drawing.Size)(resources.GetObject("picMultiMeterDigital.Size")));
			this.picMultiMeterDigital.SizeMode = ((System.Windows.Forms.PictureBoxSizeMode)(resources.GetObject("picMultiMeterDigital.SizeMode")));
			this.picMultiMeterDigital.TabIndex = ((int)(resources.GetObject("picMultiMeterDigital.TabIndex")));
			this.picMultiMeterDigital.TabStop = false;
			this.picMultiMeterDigital.Text = resources.GetString("picMultiMeterDigital.Text");
			this.toolTip1.SetToolTip(this.picMultiMeterDigital, resources.GetString("picMultiMeterDigital.ToolTip"));
			this.picMultiMeterDigital.Visible = ((bool)(resources.GetObject("picMultiMeterDigital.Visible")));
			this.picMultiMeterDigital.Paint += new System.Windows.Forms.PaintEventHandler(this.picMultiMeterDigital_Paint);
			// 
			// lblMultiSMeter
			// 
			this.lblMultiSMeter.AccessibleDescription = resources.GetString("lblMultiSMeter.AccessibleDescription");
			this.lblMultiSMeter.AccessibleName = resources.GetString("lblMultiSMeter.AccessibleName");
			this.lblMultiSMeter.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblMultiSMeter.Anchor")));
			this.lblMultiSMeter.AutoSize = ((bool)(resources.GetObject("lblMultiSMeter.AutoSize")));
			this.lblMultiSMeter.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblMultiSMeter.Dock")));
			this.lblMultiSMeter.Enabled = ((bool)(resources.GetObject("lblMultiSMeter.Enabled")));
			this.lblMultiSMeter.Font = ((System.Drawing.Font)(resources.GetObject("lblMultiSMeter.Font")));
			this.lblMultiSMeter.Image = ((System.Drawing.Image)(resources.GetObject("lblMultiSMeter.Image")));
			this.lblMultiSMeter.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblMultiSMeter.ImageAlign")));
			this.lblMultiSMeter.ImageIndex = ((int)(resources.GetObject("lblMultiSMeter.ImageIndex")));
			this.lblMultiSMeter.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblMultiSMeter.ImeMode")));
			this.lblMultiSMeter.Location = ((System.Drawing.Point)(resources.GetObject("lblMultiSMeter.Location")));
			this.lblMultiSMeter.Name = "lblMultiSMeter";
			this.lblMultiSMeter.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblMultiSMeter.RightToLeft")));
			this.lblMultiSMeter.Size = ((System.Drawing.Size)(resources.GetObject("lblMultiSMeter.Size")));
			this.lblMultiSMeter.TabIndex = ((int)(resources.GetObject("lblMultiSMeter.TabIndex")));
			this.lblMultiSMeter.Text = resources.GetString("lblMultiSMeter.Text");
			this.lblMultiSMeter.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblMultiSMeter.TextAlign")));
			this.toolTip1.SetToolTip(this.lblMultiSMeter, resources.GetString("lblMultiSMeter.ToolTip"));
			this.lblMultiSMeter.Visible = ((bool)(resources.GetObject("lblMultiSMeter.Visible")));
			// 
			// comboMeterRXMode
			// 
			this.comboMeterRXMode.AccessibleDescription = resources.GetString("comboMeterRXMode.AccessibleDescription");
			this.comboMeterRXMode.AccessibleName = resources.GetString("comboMeterRXMode.AccessibleName");
			this.comboMeterRXMode.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("comboMeterRXMode.Anchor")));
			this.comboMeterRXMode.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("comboMeterRXMode.BackgroundImage")));
			this.comboMeterRXMode.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("comboMeterRXMode.Dock")));
			this.comboMeterRXMode.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.comboMeterRXMode.DropDownWidth = 72;
			this.comboMeterRXMode.Enabled = ((bool)(resources.GetObject("comboMeterRXMode.Enabled")));
			this.comboMeterRXMode.Font = ((System.Drawing.Font)(resources.GetObject("comboMeterRXMode.Font")));
			this.comboMeterRXMode.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("comboMeterRXMode.ImeMode")));
			this.comboMeterRXMode.IntegralHeight = ((bool)(resources.GetObject("comboMeterRXMode.IntegralHeight")));
			this.comboMeterRXMode.ItemHeight = ((int)(resources.GetObject("comboMeterRXMode.ItemHeight")));
			this.comboMeterRXMode.Location = ((System.Drawing.Point)(resources.GetObject("comboMeterRXMode.Location")));
			this.comboMeterRXMode.MaxDropDownItems = ((int)(resources.GetObject("comboMeterRXMode.MaxDropDownItems")));
			this.comboMeterRXMode.MaxLength = ((int)(resources.GetObject("comboMeterRXMode.MaxLength")));
			this.comboMeterRXMode.Name = "comboMeterRXMode";
			this.comboMeterRXMode.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("comboMeterRXMode.RightToLeft")));
			this.comboMeterRXMode.Size = ((System.Drawing.Size)(resources.GetObject("comboMeterRXMode.Size")));
			this.comboMeterRXMode.TabIndex = ((int)(resources.GetObject("comboMeterRXMode.TabIndex")));
			this.comboMeterRXMode.Text = resources.GetString("comboMeterRXMode.Text");
			this.toolTip1.SetToolTip(this.comboMeterRXMode, resources.GetString("comboMeterRXMode.ToolTip"));
			this.comboMeterRXMode.Visible = ((bool)(resources.GetObject("comboMeterRXMode.Visible")));
			this.comboMeterRXMode.SelectedIndexChanged += new System.EventHandler(this.comboMeterRXMode_SelectedIndexChanged);
			// 
			// txtMultiText
			// 
			this.txtMultiText.AccessibleDescription = resources.GetString("txtMultiText.AccessibleDescription");
			this.txtMultiText.AccessibleName = resources.GetString("txtMultiText.AccessibleName");
			this.txtMultiText.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtMultiText.Anchor")));
			this.txtMultiText.AutoSize = ((bool)(resources.GetObject("txtMultiText.AutoSize")));
			this.txtMultiText.BackColor = System.Drawing.Color.Black;
			this.txtMultiText.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtMultiText.BackgroundImage")));
			this.txtMultiText.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtMultiText.Dock")));
			this.txtMultiText.Enabled = ((bool)(resources.GetObject("txtMultiText.Enabled")));
			this.txtMultiText.Font = ((System.Drawing.Font)(resources.GetObject("txtMultiText.Font")));
			this.txtMultiText.ForeColor = System.Drawing.Color.Yellow;
			this.txtMultiText.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtMultiText.ImeMode")));
			this.txtMultiText.Location = ((System.Drawing.Point)(resources.GetObject("txtMultiText.Location")));
			this.txtMultiText.MaxLength = ((int)(resources.GetObject("txtMultiText.MaxLength")));
			this.txtMultiText.Multiline = ((bool)(resources.GetObject("txtMultiText.Multiline")));
			this.txtMultiText.Name = "txtMultiText";
			this.txtMultiText.PasswordChar = ((char)(resources.GetObject("txtMultiText.PasswordChar")));
			this.txtMultiText.ReadOnly = true;
			this.txtMultiText.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtMultiText.RightToLeft")));
			this.txtMultiText.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtMultiText.ScrollBars")));
			this.txtMultiText.Size = ((System.Drawing.Size)(resources.GetObject("txtMultiText.Size")));
			this.txtMultiText.TabIndex = ((int)(resources.GetObject("txtMultiText.TabIndex")));
			this.txtMultiText.Text = resources.GetString("txtMultiText.Text");
			this.txtMultiText.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtMultiText.TextAlign")));
			this.toolTip1.SetToolTip(this.txtMultiText, resources.GetString("txtMultiText.ToolTip"));
			this.txtMultiText.Visible = ((bool)(resources.GetObject("txtMultiText.Visible")));
			this.txtMultiText.WordWrap = ((bool)(resources.GetObject("txtMultiText.WordWrap")));
			this.txtMultiText.GotFocus += new System.EventHandler(this.HideFocus);
			// 
			// picMultimeterAnalog
			// 
			this.picMultimeterAnalog.AccessibleDescription = resources.GetString("picMultimeterAnalog.AccessibleDescription");
			this.picMultimeterAnalog.AccessibleName = resources.GetString("picMultimeterAnalog.AccessibleName");
			this.picMultimeterAnalog.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("picMultimeterAnalog.Anchor")));
			this.picMultimeterAnalog.BackColor = System.Drawing.Color.Black;
			this.picMultimeterAnalog.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("picMultimeterAnalog.BackgroundImage")));
			this.picMultimeterAnalog.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("picMultimeterAnalog.Dock")));
			this.picMultimeterAnalog.Enabled = ((bool)(resources.GetObject("picMultimeterAnalog.Enabled")));
			this.picMultimeterAnalog.Font = ((System.Drawing.Font)(resources.GetObject("picMultimeterAnalog.Font")));
			this.picMultimeterAnalog.Image = ((System.Drawing.Image)(resources.GetObject("picMultimeterAnalog.Image")));
			this.picMultimeterAnalog.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("picMultimeterAnalog.ImeMode")));
			this.picMultimeterAnalog.Location = ((System.Drawing.Point)(resources.GetObject("picMultimeterAnalog.Location")));
			this.picMultimeterAnalog.Name = "picMultimeterAnalog";
			this.picMultimeterAnalog.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("picMultimeterAnalog.RightToLeft")));
			this.picMultimeterAnalog.Size = ((System.Drawing.Size)(resources.GetObject("picMultimeterAnalog.Size")));
			this.picMultimeterAnalog.SizeMode = ((System.Windows.Forms.PictureBoxSizeMode)(resources.GetObject("picMultimeterAnalog.SizeMode")));
			this.picMultimeterAnalog.TabIndex = ((int)(resources.GetObject("picMultimeterAnalog.TabIndex")));
			this.picMultimeterAnalog.TabStop = false;
			this.picMultimeterAnalog.Text = resources.GetString("picMultimeterAnalog.Text");
			this.toolTip1.SetToolTip(this.picMultimeterAnalog, resources.GetString("picMultimeterAnalog.ToolTip"));
			this.picMultimeterAnalog.Visible = ((bool)(resources.GetObject("picMultimeterAnalog.Visible")));
			this.picMultimeterAnalog.Paint += new System.Windows.Forms.PaintEventHandler(this.picMultimeter_Paint);
			// 
			// grpVFO
			// 
			this.grpVFO.AccessibleDescription = resources.GetString("grpVFO.AccessibleDescription");
			this.grpVFO.AccessibleName = resources.GetString("grpVFO.AccessibleName");
			this.grpVFO.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpVFO.Anchor")));
			this.grpVFO.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpVFO.BackgroundImage")));
			this.grpVFO.Controls.Add(this.btnRITReset);
			this.grpVFO.Controls.Add(this.btnXITReset);
			this.grpVFO.Controls.Add(this.button6);
			this.grpVFO.Controls.Add(this.btnVFOSwap);
			this.grpVFO.Controls.Add(this.btnVFOBtoA);
			this.grpVFO.Controls.Add(this.btnVFOAtoB);
			this.grpVFO.Controls.Add(this.udXIT);
			this.grpVFO.Controls.Add(this.chkXIT);
			this.grpVFO.Controls.Add(this.chkRIT);
			this.grpVFO.Controls.Add(this.udRIT);
			this.grpVFO.Controls.Add(this.chkVFOSplit);
			this.grpVFO.Controls.Add(this.checkBox1);
			this.grpVFO.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpVFO.Dock")));
			this.grpVFO.Enabled = ((bool)(resources.GetObject("grpVFO.Enabled")));
			this.grpVFO.Font = ((System.Drawing.Font)(resources.GetObject("grpVFO.Font")));
			this.grpVFO.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpVFO.ImeMode")));
			this.grpVFO.Location = ((System.Drawing.Point)(resources.GetObject("grpVFO.Location")));
			this.grpVFO.Name = "grpVFO";
			this.grpVFO.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpVFO.RightToLeft")));
			this.grpVFO.Size = ((System.Drawing.Size)(resources.GetObject("grpVFO.Size")));
			this.grpVFO.TabIndex = ((int)(resources.GetObject("grpVFO.TabIndex")));
			this.grpVFO.TabStop = false;
			this.grpVFO.Text = resources.GetString("grpVFO.Text");
			this.toolTip1.SetToolTip(this.grpVFO, resources.GetString("grpVFO.ToolTip"));
			this.grpVFO.Visible = ((bool)(resources.GetObject("grpVFO.Visible")));
			// 
			// btnRITReset
			// 
			this.btnRITReset.AccessibleDescription = resources.GetString("btnRITReset.AccessibleDescription");
			this.btnRITReset.AccessibleName = resources.GetString("btnRITReset.AccessibleName");
			this.btnRITReset.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnRITReset.Anchor")));
			this.btnRITReset.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnRITReset.BackgroundImage")));
			this.btnRITReset.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnRITReset.Dock")));
			this.btnRITReset.Enabled = ((bool)(resources.GetObject("btnRITReset.Enabled")));
			this.btnRITReset.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnRITReset.FlatStyle")));
			this.btnRITReset.Font = ((System.Drawing.Font)(resources.GetObject("btnRITReset.Font")));
			this.btnRITReset.Image = ((System.Drawing.Image)(resources.GetObject("btnRITReset.Image")));
			this.btnRITReset.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnRITReset.ImageAlign")));
			this.btnRITReset.ImageIndex = ((int)(resources.GetObject("btnRITReset.ImageIndex")));
			this.btnRITReset.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnRITReset.ImeMode")));
			this.btnRITReset.Location = ((System.Drawing.Point)(resources.GetObject("btnRITReset.Location")));
			this.btnRITReset.Name = "btnRITReset";
			this.btnRITReset.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnRITReset.RightToLeft")));
			this.btnRITReset.Size = ((System.Drawing.Size)(resources.GetObject("btnRITReset.Size")));
			this.btnRITReset.TabIndex = ((int)(resources.GetObject("btnRITReset.TabIndex")));
			this.btnRITReset.Text = resources.GetString("btnRITReset.Text");
			this.btnRITReset.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnRITReset.TextAlign")));
			this.toolTip1.SetToolTip(this.btnRITReset, resources.GetString("btnRITReset.ToolTip"));
			this.btnRITReset.Visible = ((bool)(resources.GetObject("btnRITReset.Visible")));
			this.btnRITReset.Click += new System.EventHandler(this.btnRITReset_Click);
			// 
			// btnXITReset
			// 
			this.btnXITReset.AccessibleDescription = resources.GetString("btnXITReset.AccessibleDescription");
			this.btnXITReset.AccessibleName = resources.GetString("btnXITReset.AccessibleName");
			this.btnXITReset.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnXITReset.Anchor")));
			this.btnXITReset.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnXITReset.BackgroundImage")));
			this.btnXITReset.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnXITReset.Dock")));
			this.btnXITReset.Enabled = ((bool)(resources.GetObject("btnXITReset.Enabled")));
			this.btnXITReset.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnXITReset.FlatStyle")));
			this.btnXITReset.Font = ((System.Drawing.Font)(resources.GetObject("btnXITReset.Font")));
			this.btnXITReset.Image = ((System.Drawing.Image)(resources.GetObject("btnXITReset.Image")));
			this.btnXITReset.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnXITReset.ImageAlign")));
			this.btnXITReset.ImageIndex = ((int)(resources.GetObject("btnXITReset.ImageIndex")));
			this.btnXITReset.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnXITReset.ImeMode")));
			this.btnXITReset.Location = ((System.Drawing.Point)(resources.GetObject("btnXITReset.Location")));
			this.btnXITReset.Name = "btnXITReset";
			this.btnXITReset.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnXITReset.RightToLeft")));
			this.btnXITReset.Size = ((System.Drawing.Size)(resources.GetObject("btnXITReset.Size")));
			this.btnXITReset.TabIndex = ((int)(resources.GetObject("btnXITReset.TabIndex")));
			this.btnXITReset.Text = resources.GetString("btnXITReset.Text");
			this.btnXITReset.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnXITReset.TextAlign")));
			this.toolTip1.SetToolTip(this.btnXITReset, resources.GetString("btnXITReset.ToolTip"));
			this.btnXITReset.Visible = ((bool)(resources.GetObject("btnXITReset.Visible")));
			this.btnXITReset.Click += new System.EventHandler(this.btnXITReset_Click);
			// 
			// button6
			// 
			this.button6.AccessibleDescription = resources.GetString("button6.AccessibleDescription");
			this.button6.AccessibleName = resources.GetString("button6.AccessibleName");
			this.button6.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("button6.Anchor")));
			this.button6.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("button6.BackgroundImage")));
			this.button6.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("button6.Dock")));
			this.button6.Enabled = ((bool)(resources.GetObject("button6.Enabled")));
			this.button6.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("button6.FlatStyle")));
			this.button6.Font = ((System.Drawing.Font)(resources.GetObject("button6.Font")));
			this.button6.Image = ((System.Drawing.Image)(resources.GetObject("button6.Image")));
			this.button6.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button6.ImageAlign")));
			this.button6.ImageIndex = ((int)(resources.GetObject("button6.ImageIndex")));
			this.button6.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("button6.ImeMode")));
			this.button6.Location = ((System.Drawing.Point)(resources.GetObject("button6.Location")));
			this.button6.Name = "button6";
			this.button6.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("button6.RightToLeft")));
			this.button6.Size = ((System.Drawing.Size)(resources.GetObject("button6.Size")));
			this.button6.TabIndex = ((int)(resources.GetObject("button6.TabIndex")));
			this.button6.Text = resources.GetString("button6.Text");
			this.button6.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button6.TextAlign")));
			this.toolTip1.SetToolTip(this.button6, resources.GetString("button6.ToolTip"));
			this.button6.Visible = ((bool)(resources.GetObject("button6.Visible")));
			// 
			// btnVFOSwap
			// 
			this.btnVFOSwap.AccessibleDescription = resources.GetString("btnVFOSwap.AccessibleDescription");
			this.btnVFOSwap.AccessibleName = resources.GetString("btnVFOSwap.AccessibleName");
			this.btnVFOSwap.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnVFOSwap.Anchor")));
			this.btnVFOSwap.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnVFOSwap.BackgroundImage")));
			this.btnVFOSwap.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnVFOSwap.Dock")));
			this.btnVFOSwap.Enabled = ((bool)(resources.GetObject("btnVFOSwap.Enabled")));
			this.btnVFOSwap.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnVFOSwap.FlatStyle")));
			this.btnVFOSwap.Font = ((System.Drawing.Font)(resources.GetObject("btnVFOSwap.Font")));
			this.btnVFOSwap.Image = ((System.Drawing.Image)(resources.GetObject("btnVFOSwap.Image")));
			this.btnVFOSwap.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnVFOSwap.ImageAlign")));
			this.btnVFOSwap.ImageIndex = ((int)(resources.GetObject("btnVFOSwap.ImageIndex")));
			this.btnVFOSwap.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnVFOSwap.ImeMode")));
			this.btnVFOSwap.Location = ((System.Drawing.Point)(resources.GetObject("btnVFOSwap.Location")));
			this.btnVFOSwap.Name = "btnVFOSwap";
			this.btnVFOSwap.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnVFOSwap.RightToLeft")));
			this.btnVFOSwap.Size = ((System.Drawing.Size)(resources.GetObject("btnVFOSwap.Size")));
			this.btnVFOSwap.TabIndex = ((int)(resources.GetObject("btnVFOSwap.TabIndex")));
			this.btnVFOSwap.Text = resources.GetString("btnVFOSwap.Text");
			this.btnVFOSwap.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnVFOSwap.TextAlign")));
			this.toolTip1.SetToolTip(this.btnVFOSwap, resources.GetString("btnVFOSwap.ToolTip"));
			this.btnVFOSwap.Visible = ((bool)(resources.GetObject("btnVFOSwap.Visible")));
			this.btnVFOSwap.Click += new System.EventHandler(this.btnVFOSwap_Click);
			// 
			// btnVFOBtoA
			// 
			this.btnVFOBtoA.AccessibleDescription = resources.GetString("btnVFOBtoA.AccessibleDescription");
			this.btnVFOBtoA.AccessibleName = resources.GetString("btnVFOBtoA.AccessibleName");
			this.btnVFOBtoA.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnVFOBtoA.Anchor")));
			this.btnVFOBtoA.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnVFOBtoA.BackgroundImage")));
			this.btnVFOBtoA.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnVFOBtoA.Dock")));
			this.btnVFOBtoA.Enabled = ((bool)(resources.GetObject("btnVFOBtoA.Enabled")));
			this.btnVFOBtoA.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnVFOBtoA.FlatStyle")));
			this.btnVFOBtoA.Font = ((System.Drawing.Font)(resources.GetObject("btnVFOBtoA.Font")));
			this.btnVFOBtoA.Image = ((System.Drawing.Image)(resources.GetObject("btnVFOBtoA.Image")));
			this.btnVFOBtoA.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnVFOBtoA.ImageAlign")));
			this.btnVFOBtoA.ImageIndex = ((int)(resources.GetObject("btnVFOBtoA.ImageIndex")));
			this.btnVFOBtoA.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnVFOBtoA.ImeMode")));
			this.btnVFOBtoA.Location = ((System.Drawing.Point)(resources.GetObject("btnVFOBtoA.Location")));
			this.btnVFOBtoA.Name = "btnVFOBtoA";
			this.btnVFOBtoA.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnVFOBtoA.RightToLeft")));
			this.btnVFOBtoA.Size = ((System.Drawing.Size)(resources.GetObject("btnVFOBtoA.Size")));
			this.btnVFOBtoA.TabIndex = ((int)(resources.GetObject("btnVFOBtoA.TabIndex")));
			this.btnVFOBtoA.Text = resources.GetString("btnVFOBtoA.Text");
			this.btnVFOBtoA.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnVFOBtoA.TextAlign")));
			this.toolTip1.SetToolTip(this.btnVFOBtoA, resources.GetString("btnVFOBtoA.ToolTip"));
			this.btnVFOBtoA.Visible = ((bool)(resources.GetObject("btnVFOBtoA.Visible")));
			this.btnVFOBtoA.Click += new System.EventHandler(this.btnVFOBtoA_Click);
			// 
			// btnVFOAtoB
			// 
			this.btnVFOAtoB.AccessibleDescription = resources.GetString("btnVFOAtoB.AccessibleDescription");
			this.btnVFOAtoB.AccessibleName = resources.GetString("btnVFOAtoB.AccessibleName");
			this.btnVFOAtoB.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnVFOAtoB.Anchor")));
			this.btnVFOAtoB.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnVFOAtoB.BackgroundImage")));
			this.btnVFOAtoB.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnVFOAtoB.Dock")));
			this.btnVFOAtoB.Enabled = ((bool)(resources.GetObject("btnVFOAtoB.Enabled")));
			this.btnVFOAtoB.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnVFOAtoB.FlatStyle")));
			this.btnVFOAtoB.Font = ((System.Drawing.Font)(resources.GetObject("btnVFOAtoB.Font")));
			this.btnVFOAtoB.Image = ((System.Drawing.Image)(resources.GetObject("btnVFOAtoB.Image")));
			this.btnVFOAtoB.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnVFOAtoB.ImageAlign")));
			this.btnVFOAtoB.ImageIndex = ((int)(resources.GetObject("btnVFOAtoB.ImageIndex")));
			this.btnVFOAtoB.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnVFOAtoB.ImeMode")));
			this.btnVFOAtoB.Location = ((System.Drawing.Point)(resources.GetObject("btnVFOAtoB.Location")));
			this.btnVFOAtoB.Name = "btnVFOAtoB";
			this.btnVFOAtoB.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnVFOAtoB.RightToLeft")));
			this.btnVFOAtoB.Size = ((System.Drawing.Size)(resources.GetObject("btnVFOAtoB.Size")));
			this.btnVFOAtoB.TabIndex = ((int)(resources.GetObject("btnVFOAtoB.TabIndex")));
			this.btnVFOAtoB.Text = resources.GetString("btnVFOAtoB.Text");
			this.btnVFOAtoB.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnVFOAtoB.TextAlign")));
			this.toolTip1.SetToolTip(this.btnVFOAtoB, resources.GetString("btnVFOAtoB.ToolTip"));
			this.btnVFOAtoB.Visible = ((bool)(resources.GetObject("btnVFOAtoB.Visible")));
			this.btnVFOAtoB.Click += new System.EventHandler(this.btnVFOAtoB_Click);
			// 
			// chkVFOSplit
			// 
			this.chkVFOSplit.AccessibleDescription = resources.GetString("chkVFOSplit.AccessibleDescription");
			this.chkVFOSplit.AccessibleName = resources.GetString("chkVFOSplit.AccessibleName");
			this.chkVFOSplit.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkVFOSplit.Anchor")));
			this.chkVFOSplit.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkVFOSplit.Appearance")));
			this.chkVFOSplit.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkVFOSplit.BackgroundImage")));
			this.chkVFOSplit.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkVFOSplit.CheckAlign")));
			this.chkVFOSplit.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkVFOSplit.Dock")));
			this.chkVFOSplit.Enabled = ((bool)(resources.GetObject("chkVFOSplit.Enabled")));
			this.chkVFOSplit.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkVFOSplit.FlatStyle")));
			this.chkVFOSplit.Font = ((System.Drawing.Font)(resources.GetObject("chkVFOSplit.Font")));
			this.chkVFOSplit.Image = ((System.Drawing.Image)(resources.GetObject("chkVFOSplit.Image")));
			this.chkVFOSplit.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkVFOSplit.ImageAlign")));
			this.chkVFOSplit.ImageIndex = ((int)(resources.GetObject("chkVFOSplit.ImageIndex")));
			this.chkVFOSplit.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkVFOSplit.ImeMode")));
			this.chkVFOSplit.Location = ((System.Drawing.Point)(resources.GetObject("chkVFOSplit.Location")));
			this.chkVFOSplit.Name = "chkVFOSplit";
			this.chkVFOSplit.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkVFOSplit.RightToLeft")));
			this.chkVFOSplit.Size = ((System.Drawing.Size)(resources.GetObject("chkVFOSplit.Size")));
			this.chkVFOSplit.TabIndex = ((int)(resources.GetObject("chkVFOSplit.TabIndex")));
			this.chkVFOSplit.Text = resources.GetString("chkVFOSplit.Text");
			this.chkVFOSplit.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkVFOSplit.TextAlign")));
			this.toolTip1.SetToolTip(this.chkVFOSplit, resources.GetString("chkVFOSplit.ToolTip"));
			this.chkVFOSplit.Visible = ((bool)(resources.GetObject("chkVFOSplit.Visible")));
			this.chkVFOSplit.CheckedChanged += new System.EventHandler(this.chkVFOSplit_CheckedChanged);
			// 
			// checkBox1
			// 
			this.checkBox1.AccessibleDescription = resources.GetString("checkBox1.AccessibleDescription");
			this.checkBox1.AccessibleName = resources.GetString("checkBox1.AccessibleName");
			this.checkBox1.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("checkBox1.Anchor")));
			this.checkBox1.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("checkBox1.Appearance")));
			this.checkBox1.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("checkBox1.BackgroundImage")));
			this.checkBox1.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("checkBox1.CheckAlign")));
			this.checkBox1.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("checkBox1.Dock")));
			this.checkBox1.Enabled = ((bool)(resources.GetObject("checkBox1.Enabled")));
			this.checkBox1.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("checkBox1.FlatStyle")));
			this.checkBox1.Font = ((System.Drawing.Font)(resources.GetObject("checkBox1.Font")));
			this.checkBox1.Image = ((System.Drawing.Image)(resources.GetObject("checkBox1.Image")));
			this.checkBox1.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("checkBox1.ImageAlign")));
			this.checkBox1.ImageIndex = ((int)(resources.GetObject("checkBox1.ImageIndex")));
			this.checkBox1.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("checkBox1.ImeMode")));
			this.checkBox1.Location = ((System.Drawing.Point)(resources.GetObject("checkBox1.Location")));
			this.checkBox1.Name = "checkBox1";
			this.checkBox1.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("checkBox1.RightToLeft")));
			this.checkBox1.Size = ((System.Drawing.Size)(resources.GetObject("checkBox1.Size")));
			this.checkBox1.TabIndex = ((int)(resources.GetObject("checkBox1.TabIndex")));
			this.checkBox1.Text = resources.GetString("checkBox1.Text");
			this.checkBox1.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("checkBox1.TextAlign")));
			this.toolTip1.SetToolTip(this.checkBox1, resources.GetString("checkBox1.ToolTip"));
			this.checkBox1.Visible = ((bool)(resources.GetObject("checkBox1.Visible")));
			// 
			// grpSoundControls
			// 
			this.grpSoundControls.AccessibleDescription = resources.GetString("grpSoundControls.AccessibleDescription");
			this.grpSoundControls.AccessibleName = resources.GetString("grpSoundControls.AccessibleName");
			this.grpSoundControls.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpSoundControls.Anchor")));
			this.grpSoundControls.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpSoundControls.BackgroundImage")));
			this.grpSoundControls.Controls.Add(this.udSquelch);
			this.grpSoundControls.Controls.Add(this.lblPWR);
			this.grpSoundControls.Controls.Add(this.udPWR);
			this.grpSoundControls.Controls.Add(this.lblAF);
			this.grpSoundControls.Controls.Add(this.udAF);
			this.grpSoundControls.Controls.Add(this.lblMIC);
			this.grpSoundControls.Controls.Add(this.udMIC);
			this.grpSoundControls.Controls.Add(this.comboPreamp);
			this.grpSoundControls.Controls.Add(this.lblPreamp);
			this.grpSoundControls.Controls.Add(this.chkSquelch);
			this.grpSoundControls.Controls.Add(this.lblAGC);
			this.grpSoundControls.Controls.Add(this.comboAGC);
			this.grpSoundControls.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpSoundControls.Dock")));
			this.grpSoundControls.Enabled = ((bool)(resources.GetObject("grpSoundControls.Enabled")));
			this.grpSoundControls.Font = ((System.Drawing.Font)(resources.GetObject("grpSoundControls.Font")));
			this.grpSoundControls.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpSoundControls.ImeMode")));
			this.grpSoundControls.Location = ((System.Drawing.Point)(resources.GetObject("grpSoundControls.Location")));
			this.grpSoundControls.Name = "grpSoundControls";
			this.grpSoundControls.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpSoundControls.RightToLeft")));
			this.grpSoundControls.Size = ((System.Drawing.Size)(resources.GetObject("grpSoundControls.Size")));
			this.grpSoundControls.TabIndex = ((int)(resources.GetObject("grpSoundControls.TabIndex")));
			this.grpSoundControls.TabStop = false;
			this.grpSoundControls.Text = resources.GetString("grpSoundControls.Text");
			this.toolTip1.SetToolTip(this.grpSoundControls, resources.GetString("grpSoundControls.ToolTip"));
			this.grpSoundControls.Visible = ((bool)(resources.GetObject("grpSoundControls.Visible")));
			// 
			// udSquelch
			// 
			this.udSquelch.AccessibleDescription = resources.GetString("udSquelch.AccessibleDescription");
			this.udSquelch.AccessibleName = resources.GetString("udSquelch.AccessibleName");
			this.udSquelch.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udSquelch.Anchor")));
			this.udSquelch.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udSquelch.Dock")));
			this.udSquelch.Enabled = ((bool)(resources.GetObject("udSquelch.Enabled")));
			this.udSquelch.Font = ((System.Drawing.Font)(resources.GetObject("udSquelch.Font")));
			this.udSquelch.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udSquelch.ImeMode")));
			this.udSquelch.Increment = new System.Decimal(new int[] {
																		1,
																		0,
																		0,
																		0});
			this.udSquelch.Location = ((System.Drawing.Point)(resources.GetObject("udSquelch.Location")));
			this.udSquelch.Maximum = new System.Decimal(new int[] {
																	  160,
																	  0,
																	  0,
																	  0});
			this.udSquelch.Minimum = new System.Decimal(new int[] {
																	  0,
																	  0,
																	  0,
																	  0});
			this.udSquelch.Name = "udSquelch";
			this.udSquelch.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udSquelch.RightToLeft")));
			this.udSquelch.Size = ((System.Drawing.Size)(resources.GetObject("udSquelch.Size")));
			this.udSquelch.TabIndex = ((int)(resources.GetObject("udSquelch.TabIndex")));
			this.udSquelch.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udSquelch.TextAlign")));
			this.udSquelch.ThousandsSeparator = ((bool)(resources.GetObject("udSquelch.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udSquelch, resources.GetString("udSquelch.ToolTip"));
			this.udSquelch.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udSquelch.UpDownAlign")));
			this.udSquelch.Value = new System.Decimal(new int[] {
																	150,
																	0,
																	0,
																	0});
			this.udSquelch.Visible = ((bool)(resources.GetObject("udSquelch.Visible")));
			this.udSquelch.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udSquelch.ValueChanged += new System.EventHandler(this.udSquelch_ValueChanged);
			// 
			// chkSquelch
			// 
			this.chkSquelch.AccessibleDescription = resources.GetString("chkSquelch.AccessibleDescription");
			this.chkSquelch.AccessibleName = resources.GetString("chkSquelch.AccessibleName");
			this.chkSquelch.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkSquelch.Anchor")));
			this.chkSquelch.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkSquelch.Appearance")));
			this.chkSquelch.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkSquelch.BackgroundImage")));
			this.chkSquelch.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkSquelch.CheckAlign")));
			this.chkSquelch.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkSquelch.Dock")));
			this.chkSquelch.Enabled = ((bool)(resources.GetObject("chkSquelch.Enabled")));
			this.chkSquelch.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkSquelch.FlatStyle")));
			this.chkSquelch.Font = ((System.Drawing.Font)(resources.GetObject("chkSquelch.Font")));
			this.chkSquelch.Image = ((System.Drawing.Image)(resources.GetObject("chkSquelch.Image")));
			this.chkSquelch.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkSquelch.ImageAlign")));
			this.chkSquelch.ImageIndex = ((int)(resources.GetObject("chkSquelch.ImageIndex")));
			this.chkSquelch.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkSquelch.ImeMode")));
			this.chkSquelch.Location = ((System.Drawing.Point)(resources.GetObject("chkSquelch.Location")));
			this.chkSquelch.Name = "chkSquelch";
			this.chkSquelch.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkSquelch.RightToLeft")));
			this.chkSquelch.Size = ((System.Drawing.Size)(resources.GetObject("chkSquelch.Size")));
			this.chkSquelch.TabIndex = ((int)(resources.GetObject("chkSquelch.TabIndex")));
			this.chkSquelch.Text = resources.GetString("chkSquelch.Text");
			this.chkSquelch.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkSquelch.TextAlign")));
			this.toolTip1.SetToolTip(this.chkSquelch, resources.GetString("chkSquelch.ToolTip"));
			this.chkSquelch.Visible = ((bool)(resources.GetObject("chkSquelch.Visible")));
			this.chkSquelch.CheckedChanged += new System.EventHandler(this.chkSquelch_CheckedChanged);
			// 
			// grpOptions
			// 
			this.grpOptions.AccessibleDescription = resources.GetString("grpOptions.AccessibleDescription");
			this.grpOptions.AccessibleName = resources.GetString("grpOptions.AccessibleName");
			this.grpOptions.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpOptions.Anchor")));
			this.grpOptions.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpOptions.BackgroundImage")));
			this.grpOptions.Controls.Add(this.comboTuneMode);
			this.grpOptions.Controls.Add(this.chkMUT);
			this.grpOptions.Controls.Add(this.chkMON);
			this.grpOptions.Controls.Add(this.chkTUN);
			this.grpOptions.Controls.Add(this.chkMOX);
			this.grpOptions.Controls.Add(this.chkBIN);
			this.grpOptions.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpOptions.Dock")));
			this.grpOptions.Enabled = ((bool)(resources.GetObject("grpOptions.Enabled")));
			this.grpOptions.Font = ((System.Drawing.Font)(resources.GetObject("grpOptions.Font")));
			this.grpOptions.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpOptions.ImeMode")));
			this.grpOptions.Location = ((System.Drawing.Point)(resources.GetObject("grpOptions.Location")));
			this.grpOptions.Name = "grpOptions";
			this.grpOptions.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpOptions.RightToLeft")));
			this.grpOptions.Size = ((System.Drawing.Size)(resources.GetObject("grpOptions.Size")));
			this.grpOptions.TabIndex = ((int)(resources.GetObject("grpOptions.TabIndex")));
			this.grpOptions.TabStop = false;
			this.grpOptions.Text = resources.GetString("grpOptions.Text");
			this.toolTip1.SetToolTip(this.grpOptions, resources.GetString("grpOptions.ToolTip"));
			this.grpOptions.Visible = ((bool)(resources.GetObject("grpOptions.Visible")));
			// 
			// comboTuneMode
			// 
			this.comboTuneMode.AccessibleDescription = resources.GetString("comboTuneMode.AccessibleDescription");
			this.comboTuneMode.AccessibleName = resources.GetString("comboTuneMode.AccessibleName");
			this.comboTuneMode.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("comboTuneMode.Anchor")));
			this.comboTuneMode.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("comboTuneMode.BackgroundImage")));
			this.comboTuneMode.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("comboTuneMode.Dock")));
			this.comboTuneMode.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.comboTuneMode.DropDownWidth = 42;
			this.comboTuneMode.Enabled = ((bool)(resources.GetObject("comboTuneMode.Enabled")));
			this.comboTuneMode.Font = ((System.Drawing.Font)(resources.GetObject("comboTuneMode.Font")));
			this.comboTuneMode.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("comboTuneMode.ImeMode")));
			this.comboTuneMode.IntegralHeight = ((bool)(resources.GetObject("comboTuneMode.IntegralHeight")));
			this.comboTuneMode.ItemHeight = ((int)(resources.GetObject("comboTuneMode.ItemHeight")));
			this.comboTuneMode.Items.AddRange(new object[] {
															   resources.GetString("comboTuneMode.Items"),
															   resources.GetString("comboTuneMode.Items1"),
															   resources.GetString("comboTuneMode.Items2")});
			this.comboTuneMode.Location = ((System.Drawing.Point)(resources.GetObject("comboTuneMode.Location")));
			this.comboTuneMode.MaxDropDownItems = ((int)(resources.GetObject("comboTuneMode.MaxDropDownItems")));
			this.comboTuneMode.MaxLength = ((int)(resources.GetObject("comboTuneMode.MaxLength")));
			this.comboTuneMode.Name = "comboTuneMode";
			this.comboTuneMode.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("comboTuneMode.RightToLeft")));
			this.comboTuneMode.Size = ((System.Drawing.Size)(resources.GetObject("comboTuneMode.Size")));
			this.comboTuneMode.TabIndex = ((int)(resources.GetObject("comboTuneMode.TabIndex")));
			this.comboTuneMode.Text = resources.GetString("comboTuneMode.Text");
			this.toolTip1.SetToolTip(this.comboTuneMode, resources.GetString("comboTuneMode.ToolTip"));
			this.comboTuneMode.Visible = ((bool)(resources.GetObject("comboTuneMode.Visible")));
			this.comboTuneMode.SelectedIndexChanged += new System.EventHandler(this.comboTuneMode_SelectedIndexChanged);
			// 
			// grpDisplay2
			// 
			this.grpDisplay2.AccessibleDescription = resources.GetString("grpDisplay2.AccessibleDescription");
			this.grpDisplay2.AccessibleName = resources.GetString("grpDisplay2.AccessibleName");
			this.grpDisplay2.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpDisplay2.Anchor")));
			this.grpDisplay2.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpDisplay2.BackgroundImage")));
			this.grpDisplay2.Controls.Add(this.comboDisplayMode);
			this.grpDisplay2.Controls.Add(this.chkDisplayAVG);
			this.grpDisplay2.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpDisplay2.Dock")));
			this.grpDisplay2.Enabled = ((bool)(resources.GetObject("grpDisplay2.Enabled")));
			this.grpDisplay2.Font = ((System.Drawing.Font)(resources.GetObject("grpDisplay2.Font")));
			this.grpDisplay2.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpDisplay2.ImeMode")));
			this.grpDisplay2.Location = ((System.Drawing.Point)(resources.GetObject("grpDisplay2.Location")));
			this.grpDisplay2.Name = "grpDisplay2";
			this.grpDisplay2.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpDisplay2.RightToLeft")));
			this.grpDisplay2.Size = ((System.Drawing.Size)(resources.GetObject("grpDisplay2.Size")));
			this.grpDisplay2.TabIndex = ((int)(resources.GetObject("grpDisplay2.TabIndex")));
			this.grpDisplay2.TabStop = false;
			this.grpDisplay2.Text = resources.GetString("grpDisplay2.Text");
			this.toolTip1.SetToolTip(this.grpDisplay2, resources.GetString("grpDisplay2.ToolTip"));
			this.grpDisplay2.Visible = ((bool)(resources.GetObject("grpDisplay2.Visible")));
			// 
			// chkDisplayAVG
			// 
			this.chkDisplayAVG.AccessibleDescription = resources.GetString("chkDisplayAVG.AccessibleDescription");
			this.chkDisplayAVG.AccessibleName = resources.GetString("chkDisplayAVG.AccessibleName");
			this.chkDisplayAVG.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkDisplayAVG.Anchor")));
			this.chkDisplayAVG.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkDisplayAVG.Appearance")));
			this.chkDisplayAVG.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkDisplayAVG.BackgroundImage")));
			this.chkDisplayAVG.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkDisplayAVG.CheckAlign")));
			this.chkDisplayAVG.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkDisplayAVG.Dock")));
			this.chkDisplayAVG.Enabled = ((bool)(resources.GetObject("chkDisplayAVG.Enabled")));
			this.chkDisplayAVG.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkDisplayAVG.FlatStyle")));
			this.chkDisplayAVG.Font = ((System.Drawing.Font)(resources.GetObject("chkDisplayAVG.Font")));
			this.chkDisplayAVG.Image = ((System.Drawing.Image)(resources.GetObject("chkDisplayAVG.Image")));
			this.chkDisplayAVG.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkDisplayAVG.ImageAlign")));
			this.chkDisplayAVG.ImageIndex = ((int)(resources.GetObject("chkDisplayAVG.ImageIndex")));
			this.chkDisplayAVG.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkDisplayAVG.ImeMode")));
			this.chkDisplayAVG.Location = ((System.Drawing.Point)(resources.GetObject("chkDisplayAVG.Location")));
			this.chkDisplayAVG.Name = "chkDisplayAVG";
			this.chkDisplayAVG.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkDisplayAVG.RightToLeft")));
			this.chkDisplayAVG.Size = ((System.Drawing.Size)(resources.GetObject("chkDisplayAVG.Size")));
			this.chkDisplayAVG.TabIndex = ((int)(resources.GetObject("chkDisplayAVG.TabIndex")));
			this.chkDisplayAVG.Text = resources.GetString("chkDisplayAVG.Text");
			this.chkDisplayAVG.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkDisplayAVG.TextAlign")));
			this.toolTip1.SetToolTip(this.chkDisplayAVG, resources.GetString("chkDisplayAVG.ToolTip"));
			this.chkDisplayAVG.Visible = ((bool)(resources.GetObject("chkDisplayAVG.Visible")));
			this.chkDisplayAVG.CheckedChanged += new System.EventHandler(this.chkDisplayAVG_CheckedChanged);
			// 
			// grpMemory
			// 
			this.grpMemory.AccessibleDescription = resources.GetString("grpMemory.AccessibleDescription");
			this.grpMemory.AccessibleName = resources.GetString("grpMemory.AccessibleName");
			this.grpMemory.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpMemory.Anchor")));
			this.grpMemory.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpMemory.BackgroundImage")));
			this.grpMemory.Controls.Add(this.btnMemoryQuickRestore);
			this.grpMemory.Controls.Add(this.btnMemoryQuickSave);
			this.grpMemory.Controls.Add(this.txtMemoryQuick);
			this.grpMemory.Controls.Add(this.labelScannerDelay);
			this.grpMemory.Controls.Add(this.udScannerDelay);
			this.grpMemory.Controls.Add(this.lblMemoryStep);
			this.grpMemory.Controls.Add(this.lblMemoryHigh);
			this.grpMemory.Controls.Add(this.lblMemoryLow);
			this.grpMemory.Controls.Add(this.udScannerStep);
			this.grpMemory.Controls.Add(this.udScannerHigh);
			this.grpMemory.Controls.Add(this.udScannerLow);
			this.grpMemory.Controls.Add(this.chkMemoryScanner);
			this.grpMemory.Controls.Add(this.radScannerFreq);
			this.grpMemory.Controls.Add(this.radScannerChannel);
			this.grpMemory.Controls.Add(this.btnSave);
			this.grpMemory.Controls.Add(this.btnMemoryRecall);
			this.grpMemory.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpMemory.Dock")));
			this.grpMemory.Enabled = ((bool)(resources.GetObject("grpMemory.Enabled")));
			this.grpMemory.Font = ((System.Drawing.Font)(resources.GetObject("grpMemory.Font")));
			this.grpMemory.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpMemory.ImeMode")));
			this.grpMemory.Location = ((System.Drawing.Point)(resources.GetObject("grpMemory.Location")));
			this.grpMemory.Name = "grpMemory";
			this.grpMemory.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpMemory.RightToLeft")));
			this.grpMemory.Size = ((System.Drawing.Size)(resources.GetObject("grpMemory.Size")));
			this.grpMemory.TabIndex = ((int)(resources.GetObject("grpMemory.TabIndex")));
			this.grpMemory.TabStop = false;
			this.grpMemory.Text = resources.GetString("grpMemory.Text");
			this.toolTip1.SetToolTip(this.grpMemory, resources.GetString("grpMemory.ToolTip"));
			this.grpMemory.Visible = ((bool)(resources.GetObject("grpMemory.Visible")));
			// 
			// btnMemoryQuickRestore
			// 
			this.btnMemoryQuickRestore.AccessibleDescription = resources.GetString("btnMemoryQuickRestore.AccessibleDescription");
			this.btnMemoryQuickRestore.AccessibleName = resources.GetString("btnMemoryQuickRestore.AccessibleName");
			this.btnMemoryQuickRestore.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnMemoryQuickRestore.Anchor")));
			this.btnMemoryQuickRestore.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnMemoryQuickRestore.BackgroundImage")));
			this.btnMemoryQuickRestore.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnMemoryQuickRestore.Dock")));
			this.btnMemoryQuickRestore.Enabled = ((bool)(resources.GetObject("btnMemoryQuickRestore.Enabled")));
			this.btnMemoryQuickRestore.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnMemoryQuickRestore.FlatStyle")));
			this.btnMemoryQuickRestore.Font = ((System.Drawing.Font)(resources.GetObject("btnMemoryQuickRestore.Font")));
			this.btnMemoryQuickRestore.Image = ((System.Drawing.Image)(resources.GetObject("btnMemoryQuickRestore.Image")));
			this.btnMemoryQuickRestore.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnMemoryQuickRestore.ImageAlign")));
			this.btnMemoryQuickRestore.ImageIndex = ((int)(resources.GetObject("btnMemoryQuickRestore.ImageIndex")));
			this.btnMemoryQuickRestore.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnMemoryQuickRestore.ImeMode")));
			this.btnMemoryQuickRestore.Location = ((System.Drawing.Point)(resources.GetObject("btnMemoryQuickRestore.Location")));
			this.btnMemoryQuickRestore.Name = "btnMemoryQuickRestore";
			this.btnMemoryQuickRestore.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnMemoryQuickRestore.RightToLeft")));
			this.btnMemoryQuickRestore.Size = ((System.Drawing.Size)(resources.GetObject("btnMemoryQuickRestore.Size")));
			this.btnMemoryQuickRestore.TabIndex = ((int)(resources.GetObject("btnMemoryQuickRestore.TabIndex")));
			this.btnMemoryQuickRestore.Text = resources.GetString("btnMemoryQuickRestore.Text");
			this.btnMemoryQuickRestore.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnMemoryQuickRestore.TextAlign")));
			this.toolTip1.SetToolTip(this.btnMemoryQuickRestore, resources.GetString("btnMemoryQuickRestore.ToolTip"));
			this.btnMemoryQuickRestore.Visible = ((bool)(resources.GetObject("btnMemoryQuickRestore.Visible")));
			this.btnMemoryQuickRestore.Click += new System.EventHandler(this.btnMemoryQuickRestore_Click);
			// 
			// btnMemoryQuickSave
			// 
			this.btnMemoryQuickSave.AccessibleDescription = resources.GetString("btnMemoryQuickSave.AccessibleDescription");
			this.btnMemoryQuickSave.AccessibleName = resources.GetString("btnMemoryQuickSave.AccessibleName");
			this.btnMemoryQuickSave.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnMemoryQuickSave.Anchor")));
			this.btnMemoryQuickSave.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnMemoryQuickSave.BackgroundImage")));
			this.btnMemoryQuickSave.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnMemoryQuickSave.Dock")));
			this.btnMemoryQuickSave.Enabled = ((bool)(resources.GetObject("btnMemoryQuickSave.Enabled")));
			this.btnMemoryQuickSave.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnMemoryQuickSave.FlatStyle")));
			this.btnMemoryQuickSave.Font = ((System.Drawing.Font)(resources.GetObject("btnMemoryQuickSave.Font")));
			this.btnMemoryQuickSave.Image = ((System.Drawing.Image)(resources.GetObject("btnMemoryQuickSave.Image")));
			this.btnMemoryQuickSave.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnMemoryQuickSave.ImageAlign")));
			this.btnMemoryQuickSave.ImageIndex = ((int)(resources.GetObject("btnMemoryQuickSave.ImageIndex")));
			this.btnMemoryQuickSave.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnMemoryQuickSave.ImeMode")));
			this.btnMemoryQuickSave.Location = ((System.Drawing.Point)(resources.GetObject("btnMemoryQuickSave.Location")));
			this.btnMemoryQuickSave.Name = "btnMemoryQuickSave";
			this.btnMemoryQuickSave.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnMemoryQuickSave.RightToLeft")));
			this.btnMemoryQuickSave.Size = ((System.Drawing.Size)(resources.GetObject("btnMemoryQuickSave.Size")));
			this.btnMemoryQuickSave.TabIndex = ((int)(resources.GetObject("btnMemoryQuickSave.TabIndex")));
			this.btnMemoryQuickSave.Text = resources.GetString("btnMemoryQuickSave.Text");
			this.btnMemoryQuickSave.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnMemoryQuickSave.TextAlign")));
			this.toolTip1.SetToolTip(this.btnMemoryQuickSave, resources.GetString("btnMemoryQuickSave.ToolTip"));
			this.btnMemoryQuickSave.Visible = ((bool)(resources.GetObject("btnMemoryQuickSave.Visible")));
			this.btnMemoryQuickSave.Click += new System.EventHandler(this.btnMemoryQuickSave_Click);
			// 
			// txtMemoryQuick
			// 
			this.txtMemoryQuick.AccessibleDescription = resources.GetString("txtMemoryQuick.AccessibleDescription");
			this.txtMemoryQuick.AccessibleName = resources.GetString("txtMemoryQuick.AccessibleName");
			this.txtMemoryQuick.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtMemoryQuick.Anchor")));
			this.txtMemoryQuick.AutoSize = ((bool)(resources.GetObject("txtMemoryQuick.AutoSize")));
			this.txtMemoryQuick.BackColor = System.Drawing.Color.White;
			this.txtMemoryQuick.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtMemoryQuick.BackgroundImage")));
			this.txtMemoryQuick.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtMemoryQuick.Dock")));
			this.txtMemoryQuick.Enabled = ((bool)(resources.GetObject("txtMemoryQuick.Enabled")));
			this.txtMemoryQuick.Font = ((System.Drawing.Font)(resources.GetObject("txtMemoryQuick.Font")));
			this.txtMemoryQuick.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtMemoryQuick.ImeMode")));
			this.txtMemoryQuick.Location = ((System.Drawing.Point)(resources.GetObject("txtMemoryQuick.Location")));
			this.txtMemoryQuick.MaxLength = ((int)(resources.GetObject("txtMemoryQuick.MaxLength")));
			this.txtMemoryQuick.Multiline = ((bool)(resources.GetObject("txtMemoryQuick.Multiline")));
			this.txtMemoryQuick.Name = "txtMemoryQuick";
			this.txtMemoryQuick.PasswordChar = ((char)(resources.GetObject("txtMemoryQuick.PasswordChar")));
			this.txtMemoryQuick.ReadOnly = true;
			this.txtMemoryQuick.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtMemoryQuick.RightToLeft")));
			this.txtMemoryQuick.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtMemoryQuick.ScrollBars")));
			this.txtMemoryQuick.Size = ((System.Drawing.Size)(resources.GetObject("txtMemoryQuick.Size")));
			this.txtMemoryQuick.TabIndex = ((int)(resources.GetObject("txtMemoryQuick.TabIndex")));
			this.txtMemoryQuick.Text = resources.GetString("txtMemoryQuick.Text");
			this.txtMemoryQuick.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtMemoryQuick.TextAlign")));
			this.toolTip1.SetToolTip(this.txtMemoryQuick, resources.GetString("txtMemoryQuick.ToolTip"));
			this.txtMemoryQuick.Visible = ((bool)(resources.GetObject("txtMemoryQuick.Visible")));
			this.txtMemoryQuick.WordWrap = ((bool)(resources.GetObject("txtMemoryQuick.WordWrap")));
			// 
			// labelScannerDelay
			// 
			this.labelScannerDelay.AccessibleDescription = resources.GetString("labelScannerDelay.AccessibleDescription");
			this.labelScannerDelay.AccessibleName = resources.GetString("labelScannerDelay.AccessibleName");
			this.labelScannerDelay.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("labelScannerDelay.Anchor")));
			this.labelScannerDelay.AutoSize = ((bool)(resources.GetObject("labelScannerDelay.AutoSize")));
			this.labelScannerDelay.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("labelScannerDelay.Dock")));
			this.labelScannerDelay.Enabled = ((bool)(resources.GetObject("labelScannerDelay.Enabled")));
			this.labelScannerDelay.Font = ((System.Drawing.Font)(resources.GetObject("labelScannerDelay.Font")));
			this.labelScannerDelay.Image = ((System.Drawing.Image)(resources.GetObject("labelScannerDelay.Image")));
			this.labelScannerDelay.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("labelScannerDelay.ImageAlign")));
			this.labelScannerDelay.ImageIndex = ((int)(resources.GetObject("labelScannerDelay.ImageIndex")));
			this.labelScannerDelay.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("labelScannerDelay.ImeMode")));
			this.labelScannerDelay.Location = ((System.Drawing.Point)(resources.GetObject("labelScannerDelay.Location")));
			this.labelScannerDelay.Name = "labelScannerDelay";
			this.labelScannerDelay.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("labelScannerDelay.RightToLeft")));
			this.labelScannerDelay.Size = ((System.Drawing.Size)(resources.GetObject("labelScannerDelay.Size")));
			this.labelScannerDelay.TabIndex = ((int)(resources.GetObject("labelScannerDelay.TabIndex")));
			this.labelScannerDelay.Text = resources.GetString("labelScannerDelay.Text");
			this.labelScannerDelay.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("labelScannerDelay.TextAlign")));
			this.toolTip1.SetToolTip(this.labelScannerDelay, resources.GetString("labelScannerDelay.ToolTip"));
			this.labelScannerDelay.Visible = ((bool)(resources.GetObject("labelScannerDelay.Visible")));
			// 
			// udScannerDelay
			// 
			this.udScannerDelay.AccessibleDescription = resources.GetString("udScannerDelay.AccessibleDescription");
			this.udScannerDelay.AccessibleName = resources.GetString("udScannerDelay.AccessibleName");
			this.udScannerDelay.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udScannerDelay.Anchor")));
			this.udScannerDelay.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udScannerDelay.Dock")));
			this.udScannerDelay.Enabled = ((bool)(resources.GetObject("udScannerDelay.Enabled")));
			this.udScannerDelay.Font = ((System.Drawing.Font)(resources.GetObject("udScannerDelay.Font")));
			this.udScannerDelay.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udScannerDelay.ImeMode")));
			this.udScannerDelay.Increment = new System.Decimal(new int[] {
																			 100,
																			 0,
																			 0,
																			 0});
			this.udScannerDelay.Location = ((System.Drawing.Point)(resources.GetObject("udScannerDelay.Location")));
			this.udScannerDelay.Maximum = new System.Decimal(new int[] {
																		   60000,
																		   0,
																		   0,
																		   0});
			this.udScannerDelay.Minimum = new System.Decimal(new int[] {
																		   50,
																		   0,
																		   0,
																		   0});
			this.udScannerDelay.Name = "udScannerDelay";
			this.udScannerDelay.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udScannerDelay.RightToLeft")));
			this.udScannerDelay.Size = ((System.Drawing.Size)(resources.GetObject("udScannerDelay.Size")));
			this.udScannerDelay.TabIndex = ((int)(resources.GetObject("udScannerDelay.TabIndex")));
			this.udScannerDelay.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udScannerDelay.TextAlign")));
			this.udScannerDelay.ThousandsSeparator = ((bool)(resources.GetObject("udScannerDelay.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udScannerDelay, resources.GetString("udScannerDelay.ToolTip"));
			this.udScannerDelay.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udScannerDelay.UpDownAlign")));
			this.udScannerDelay.Value = new System.Decimal(new int[] {
																		 3000,
																		 0,
																		 0,
																		 0});
			this.udScannerDelay.Visible = ((bool)(resources.GetObject("udScannerDelay.Visible")));
			this.udScannerDelay.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udScannerDelay.ValueChanged += new System.EventHandler(this.udScannerDelay_ValueChanged);
			// 
			// lblMemoryStep
			// 
			this.lblMemoryStep.AccessibleDescription = resources.GetString("lblMemoryStep.AccessibleDescription");
			this.lblMemoryStep.AccessibleName = resources.GetString("lblMemoryStep.AccessibleName");
			this.lblMemoryStep.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblMemoryStep.Anchor")));
			this.lblMemoryStep.AutoSize = ((bool)(resources.GetObject("lblMemoryStep.AutoSize")));
			this.lblMemoryStep.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblMemoryStep.Dock")));
			this.lblMemoryStep.Enabled = ((bool)(resources.GetObject("lblMemoryStep.Enabled")));
			this.lblMemoryStep.Font = ((System.Drawing.Font)(resources.GetObject("lblMemoryStep.Font")));
			this.lblMemoryStep.Image = ((System.Drawing.Image)(resources.GetObject("lblMemoryStep.Image")));
			this.lblMemoryStep.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblMemoryStep.ImageAlign")));
			this.lblMemoryStep.ImageIndex = ((int)(resources.GetObject("lblMemoryStep.ImageIndex")));
			this.lblMemoryStep.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblMemoryStep.ImeMode")));
			this.lblMemoryStep.Location = ((System.Drawing.Point)(resources.GetObject("lblMemoryStep.Location")));
			this.lblMemoryStep.Name = "lblMemoryStep";
			this.lblMemoryStep.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblMemoryStep.RightToLeft")));
			this.lblMemoryStep.Size = ((System.Drawing.Size)(resources.GetObject("lblMemoryStep.Size")));
			this.lblMemoryStep.TabIndex = ((int)(resources.GetObject("lblMemoryStep.TabIndex")));
			this.lblMemoryStep.Text = resources.GetString("lblMemoryStep.Text");
			this.lblMemoryStep.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblMemoryStep.TextAlign")));
			this.toolTip1.SetToolTip(this.lblMemoryStep, resources.GetString("lblMemoryStep.ToolTip"));
			this.lblMemoryStep.Visible = ((bool)(resources.GetObject("lblMemoryStep.Visible")));
			// 
			// lblMemoryHigh
			// 
			this.lblMemoryHigh.AccessibleDescription = resources.GetString("lblMemoryHigh.AccessibleDescription");
			this.lblMemoryHigh.AccessibleName = resources.GetString("lblMemoryHigh.AccessibleName");
			this.lblMemoryHigh.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblMemoryHigh.Anchor")));
			this.lblMemoryHigh.AutoSize = ((bool)(resources.GetObject("lblMemoryHigh.AutoSize")));
			this.lblMemoryHigh.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblMemoryHigh.Dock")));
			this.lblMemoryHigh.Enabled = ((bool)(resources.GetObject("lblMemoryHigh.Enabled")));
			this.lblMemoryHigh.Font = ((System.Drawing.Font)(resources.GetObject("lblMemoryHigh.Font")));
			this.lblMemoryHigh.Image = ((System.Drawing.Image)(resources.GetObject("lblMemoryHigh.Image")));
			this.lblMemoryHigh.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblMemoryHigh.ImageAlign")));
			this.lblMemoryHigh.ImageIndex = ((int)(resources.GetObject("lblMemoryHigh.ImageIndex")));
			this.lblMemoryHigh.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblMemoryHigh.ImeMode")));
			this.lblMemoryHigh.Location = ((System.Drawing.Point)(resources.GetObject("lblMemoryHigh.Location")));
			this.lblMemoryHigh.Name = "lblMemoryHigh";
			this.lblMemoryHigh.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblMemoryHigh.RightToLeft")));
			this.lblMemoryHigh.Size = ((System.Drawing.Size)(resources.GetObject("lblMemoryHigh.Size")));
			this.lblMemoryHigh.TabIndex = ((int)(resources.GetObject("lblMemoryHigh.TabIndex")));
			this.lblMemoryHigh.Text = resources.GetString("lblMemoryHigh.Text");
			this.lblMemoryHigh.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblMemoryHigh.TextAlign")));
			this.toolTip1.SetToolTip(this.lblMemoryHigh, resources.GetString("lblMemoryHigh.ToolTip"));
			this.lblMemoryHigh.Visible = ((bool)(resources.GetObject("lblMemoryHigh.Visible")));
			// 
			// lblMemoryLow
			// 
			this.lblMemoryLow.AccessibleDescription = resources.GetString("lblMemoryLow.AccessibleDescription");
			this.lblMemoryLow.AccessibleName = resources.GetString("lblMemoryLow.AccessibleName");
			this.lblMemoryLow.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("lblMemoryLow.Anchor")));
			this.lblMemoryLow.AutoSize = ((bool)(resources.GetObject("lblMemoryLow.AutoSize")));
			this.lblMemoryLow.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("lblMemoryLow.Dock")));
			this.lblMemoryLow.Enabled = ((bool)(resources.GetObject("lblMemoryLow.Enabled")));
			this.lblMemoryLow.Font = ((System.Drawing.Font)(resources.GetObject("lblMemoryLow.Font")));
			this.lblMemoryLow.Image = ((System.Drawing.Image)(resources.GetObject("lblMemoryLow.Image")));
			this.lblMemoryLow.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblMemoryLow.ImageAlign")));
			this.lblMemoryLow.ImageIndex = ((int)(resources.GetObject("lblMemoryLow.ImageIndex")));
			this.lblMemoryLow.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("lblMemoryLow.ImeMode")));
			this.lblMemoryLow.Location = ((System.Drawing.Point)(resources.GetObject("lblMemoryLow.Location")));
			this.lblMemoryLow.Name = "lblMemoryLow";
			this.lblMemoryLow.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("lblMemoryLow.RightToLeft")));
			this.lblMemoryLow.Size = ((System.Drawing.Size)(resources.GetObject("lblMemoryLow.Size")));
			this.lblMemoryLow.TabIndex = ((int)(resources.GetObject("lblMemoryLow.TabIndex")));
			this.lblMemoryLow.Text = resources.GetString("lblMemoryLow.Text");
			this.lblMemoryLow.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("lblMemoryLow.TextAlign")));
			this.toolTip1.SetToolTip(this.lblMemoryLow, resources.GetString("lblMemoryLow.ToolTip"));
			this.lblMemoryLow.Visible = ((bool)(resources.GetObject("lblMemoryLow.Visible")));
			// 
			// udScannerStep
			// 
			this.udScannerStep.AccessibleDescription = resources.GetString("udScannerStep.AccessibleDescription");
			this.udScannerStep.AccessibleName = resources.GetString("udScannerStep.AccessibleName");
			this.udScannerStep.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udScannerStep.Anchor")));
			this.udScannerStep.DecimalPlaces = 6;
			this.udScannerStep.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udScannerStep.Dock")));
			this.udScannerStep.Enabled = ((bool)(resources.GetObject("udScannerStep.Enabled")));
			this.udScannerStep.Font = ((System.Drawing.Font)(resources.GetObject("udScannerStep.Font")));
			this.udScannerStep.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udScannerStep.ImeMode")));
			this.udScannerStep.Increment = new System.Decimal(new int[] {
																			100,
																			0,
																			0,
																			393216});
			this.udScannerStep.Location = ((System.Drawing.Point)(resources.GetObject("udScannerStep.Location")));
			this.udScannerStep.Maximum = new System.Decimal(new int[] {
																		  100,
																		  0,
																		  0,
																		  0});
			this.udScannerStep.Minimum = new System.Decimal(new int[] {
																		  0,
																		  0,
																		  0,
																		  0});
			this.udScannerStep.Name = "udScannerStep";
			this.udScannerStep.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udScannerStep.RightToLeft")));
			this.udScannerStep.Size = ((System.Drawing.Size)(resources.GetObject("udScannerStep.Size")));
			this.udScannerStep.TabIndex = ((int)(resources.GetObject("udScannerStep.TabIndex")));
			this.udScannerStep.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udScannerStep.TextAlign")));
			this.udScannerStep.ThousandsSeparator = ((bool)(resources.GetObject("udScannerStep.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udScannerStep, resources.GetString("udScannerStep.ToolTip"));
			this.udScannerStep.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udScannerStep.UpDownAlign")));
			this.udScannerStep.Value = new System.Decimal(new int[] {
																		1000,
																		0,
																		0,
																		393216});
			this.udScannerStep.Visible = ((bool)(resources.GetObject("udScannerStep.Visible")));
			this.udScannerStep.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udScannerStep.ValueChanged += new System.EventHandler(this.udScannerStep_ValueChanged);
			// 
			// udScannerHigh
			// 
			this.udScannerHigh.AccessibleDescription = resources.GetString("udScannerHigh.AccessibleDescription");
			this.udScannerHigh.AccessibleName = resources.GetString("udScannerHigh.AccessibleName");
			this.udScannerHigh.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udScannerHigh.Anchor")));
			this.udScannerHigh.DecimalPlaces = 6;
			this.udScannerHigh.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udScannerHigh.Dock")));
			this.udScannerHigh.Enabled = ((bool)(resources.GetObject("udScannerHigh.Enabled")));
			this.udScannerHigh.Font = ((System.Drawing.Font)(resources.GetObject("udScannerHigh.Font")));
			this.udScannerHigh.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udScannerHigh.ImeMode")));
			this.udScannerHigh.Increment = new System.Decimal(new int[] {
																			100,
																			0,
																			0,
																			327680});
			this.udScannerHigh.Location = ((System.Drawing.Point)(resources.GetObject("udScannerHigh.Location")));
			this.udScannerHigh.Maximum = new System.Decimal(new int[] {
																		  100,
																		  0,
																		  0,
																		  0});
			this.udScannerHigh.Minimum = new System.Decimal(new int[] {
																		  0,
																		  0,
																		  0,
																		  0});
			this.udScannerHigh.Name = "udScannerHigh";
			this.udScannerHigh.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udScannerHigh.RightToLeft")));
			this.udScannerHigh.Size = ((System.Drawing.Size)(resources.GetObject("udScannerHigh.Size")));
			this.udScannerHigh.TabIndex = ((int)(resources.GetObject("udScannerHigh.TabIndex")));
			this.udScannerHigh.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udScannerHigh.TextAlign")));
			this.udScannerHigh.ThousandsSeparator = ((bool)(resources.GetObject("udScannerHigh.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udScannerHigh, resources.GetString("udScannerHigh.ToolTip"));
			this.udScannerHigh.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udScannerHigh.UpDownAlign")));
			this.udScannerHigh.Value = new System.Decimal(new int[] {
																		7220000,
																		0,
																		0,
																		393216});
			this.udScannerHigh.Visible = ((bool)(resources.GetObject("udScannerHigh.Visible")));
			this.udScannerHigh.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udScannerHigh.ValueChanged += new System.EventHandler(this.udScannerHigh_ValueChanged);
			// 
			// udScannerLow
			// 
			this.udScannerLow.AccessibleDescription = resources.GetString("udScannerLow.AccessibleDescription");
			this.udScannerLow.AccessibleName = resources.GetString("udScannerLow.AccessibleName");
			this.udScannerLow.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("udScannerLow.Anchor")));
			this.udScannerLow.DecimalPlaces = 6;
			this.udScannerLow.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("udScannerLow.Dock")));
			this.udScannerLow.Enabled = ((bool)(resources.GetObject("udScannerLow.Enabled")));
			this.udScannerLow.Font = ((System.Drawing.Font)(resources.GetObject("udScannerLow.Font")));
			this.udScannerLow.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("udScannerLow.ImeMode")));
			this.udScannerLow.Increment = new System.Decimal(new int[] {
																		   100,
																		   0,
																		   0,
																		   327680});
			this.udScannerLow.Location = ((System.Drawing.Point)(resources.GetObject("udScannerLow.Location")));
			this.udScannerLow.Maximum = new System.Decimal(new int[] {
																		 100,
																		 0,
																		 0,
																		 0});
			this.udScannerLow.Minimum = new System.Decimal(new int[] {
																		 0,
																		 0,
																		 0,
																		 0});
			this.udScannerLow.Name = "udScannerLow";
			this.udScannerLow.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("udScannerLow.RightToLeft")));
			this.udScannerLow.Size = ((System.Drawing.Size)(resources.GetObject("udScannerLow.Size")));
			this.udScannerLow.TabIndex = ((int)(resources.GetObject("udScannerLow.TabIndex")));
			this.udScannerLow.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("udScannerLow.TextAlign")));
			this.udScannerLow.ThousandsSeparator = ((bool)(resources.GetObject("udScannerLow.ThousandsSeparator")));
			this.toolTip1.SetToolTip(this.udScannerLow, resources.GetString("udScannerLow.ToolTip"));
			this.udScannerLow.UpDownAlign = ((System.Windows.Forms.LeftRightAlignment)(resources.GetObject("udScannerLow.UpDownAlign")));
			this.udScannerLow.Value = new System.Decimal(new int[] {
																	   7200000,
																	   0,
																	   0,
																	   393216});
			this.udScannerLow.Visible = ((bool)(resources.GetObject("udScannerLow.Visible")));
			this.udScannerLow.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.udScannerLow.ValueChanged += new System.EventHandler(this.udScannerLow_ValueChanged);
			// 
			// chkMemoryScanner
			// 
			this.chkMemoryScanner.AccessibleDescription = resources.GetString("chkMemoryScanner.AccessibleDescription");
			this.chkMemoryScanner.AccessibleName = resources.GetString("chkMemoryScanner.AccessibleName");
			this.chkMemoryScanner.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("chkMemoryScanner.Anchor")));
			this.chkMemoryScanner.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("chkMemoryScanner.Appearance")));
			this.chkMemoryScanner.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("chkMemoryScanner.BackgroundImage")));
			this.chkMemoryScanner.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMemoryScanner.CheckAlign")));
			this.chkMemoryScanner.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("chkMemoryScanner.Dock")));
			this.chkMemoryScanner.Enabled = ((bool)(resources.GetObject("chkMemoryScanner.Enabled")));
			this.chkMemoryScanner.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("chkMemoryScanner.FlatStyle")));
			this.chkMemoryScanner.Font = ((System.Drawing.Font)(resources.GetObject("chkMemoryScanner.Font")));
			this.chkMemoryScanner.Image = ((System.Drawing.Image)(resources.GetObject("chkMemoryScanner.Image")));
			this.chkMemoryScanner.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMemoryScanner.ImageAlign")));
			this.chkMemoryScanner.ImageIndex = ((int)(resources.GetObject("chkMemoryScanner.ImageIndex")));
			this.chkMemoryScanner.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("chkMemoryScanner.ImeMode")));
			this.chkMemoryScanner.Location = ((System.Drawing.Point)(resources.GetObject("chkMemoryScanner.Location")));
			this.chkMemoryScanner.Name = "chkMemoryScanner";
			this.chkMemoryScanner.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("chkMemoryScanner.RightToLeft")));
			this.chkMemoryScanner.Size = ((System.Drawing.Size)(resources.GetObject("chkMemoryScanner.Size")));
			this.chkMemoryScanner.TabIndex = ((int)(resources.GetObject("chkMemoryScanner.TabIndex")));
			this.chkMemoryScanner.Text = resources.GetString("chkMemoryScanner.Text");
			this.chkMemoryScanner.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("chkMemoryScanner.TextAlign")));
			this.toolTip1.SetToolTip(this.chkMemoryScanner, resources.GetString("chkMemoryScanner.ToolTip"));
			this.chkMemoryScanner.Visible = ((bool)(resources.GetObject("chkMemoryScanner.Visible")));
			this.chkMemoryScanner.CheckedChanged += new System.EventHandler(this.chkMemoryScanner_CheckedChanged);
			// 
			// radScannerFreq
			// 
			this.radScannerFreq.AccessibleDescription = resources.GetString("radScannerFreq.AccessibleDescription");
			this.radScannerFreq.AccessibleName = resources.GetString("radScannerFreq.AccessibleName");
			this.radScannerFreq.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radScannerFreq.Anchor")));
			this.radScannerFreq.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radScannerFreq.Appearance")));
			this.radScannerFreq.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radScannerFreq.BackgroundImage")));
			this.radScannerFreq.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radScannerFreq.CheckAlign")));
			this.radScannerFreq.Checked = true;
			this.radScannerFreq.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radScannerFreq.Dock")));
			this.radScannerFreq.Enabled = ((bool)(resources.GetObject("radScannerFreq.Enabled")));
			this.radScannerFreq.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radScannerFreq.FlatStyle")));
			this.radScannerFreq.Font = ((System.Drawing.Font)(resources.GetObject("radScannerFreq.Font")));
			this.radScannerFreq.Image = ((System.Drawing.Image)(resources.GetObject("radScannerFreq.Image")));
			this.radScannerFreq.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radScannerFreq.ImageAlign")));
			this.radScannerFreq.ImageIndex = ((int)(resources.GetObject("radScannerFreq.ImageIndex")));
			this.radScannerFreq.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radScannerFreq.ImeMode")));
			this.radScannerFreq.Location = ((System.Drawing.Point)(resources.GetObject("radScannerFreq.Location")));
			this.radScannerFreq.Name = "radScannerFreq";
			this.radScannerFreq.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radScannerFreq.RightToLeft")));
			this.radScannerFreq.Size = ((System.Drawing.Size)(resources.GetObject("radScannerFreq.Size")));
			this.radScannerFreq.TabIndex = ((int)(resources.GetObject("radScannerFreq.TabIndex")));
			this.radScannerFreq.TabStop = true;
			this.radScannerFreq.Text = resources.GetString("radScannerFreq.Text");
			this.radScannerFreq.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radScannerFreq.TextAlign")));
			this.toolTip1.SetToolTip(this.radScannerFreq, resources.GetString("radScannerFreq.ToolTip"));
			this.radScannerFreq.Visible = ((bool)(resources.GetObject("radScannerFreq.Visible")));
			// 
			// radScannerChannel
			// 
			this.radScannerChannel.AccessibleDescription = resources.GetString("radScannerChannel.AccessibleDescription");
			this.radScannerChannel.AccessibleName = resources.GetString("radScannerChannel.AccessibleName");
			this.radScannerChannel.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("radScannerChannel.Anchor")));
			this.radScannerChannel.Appearance = ((System.Windows.Forms.Appearance)(resources.GetObject("radScannerChannel.Appearance")));
			this.radScannerChannel.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("radScannerChannel.BackgroundImage")));
			this.radScannerChannel.CheckAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radScannerChannel.CheckAlign")));
			this.radScannerChannel.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("radScannerChannel.Dock")));
			this.radScannerChannel.Enabled = ((bool)(resources.GetObject("radScannerChannel.Enabled")));
			this.radScannerChannel.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("radScannerChannel.FlatStyle")));
			this.radScannerChannel.Font = ((System.Drawing.Font)(resources.GetObject("radScannerChannel.Font")));
			this.radScannerChannel.Image = ((System.Drawing.Image)(resources.GetObject("radScannerChannel.Image")));
			this.radScannerChannel.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radScannerChannel.ImageAlign")));
			this.radScannerChannel.ImageIndex = ((int)(resources.GetObject("radScannerChannel.ImageIndex")));
			this.radScannerChannel.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("radScannerChannel.ImeMode")));
			this.radScannerChannel.Location = ((System.Drawing.Point)(resources.GetObject("radScannerChannel.Location")));
			this.radScannerChannel.Name = "radScannerChannel";
			this.radScannerChannel.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("radScannerChannel.RightToLeft")));
			this.radScannerChannel.Size = ((System.Drawing.Size)(resources.GetObject("radScannerChannel.Size")));
			this.radScannerChannel.TabIndex = ((int)(resources.GetObject("radScannerChannel.TabIndex")));
			this.radScannerChannel.Text = resources.GetString("radScannerChannel.Text");
			this.radScannerChannel.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("radScannerChannel.TextAlign")));
			this.toolTip1.SetToolTip(this.radScannerChannel, resources.GetString("radScannerChannel.ToolTip"));
			this.radScannerChannel.Visible = ((bool)(resources.GetObject("radScannerChannel.Visible")));
			// 
			// btnSave
			// 
			this.btnSave.AccessibleDescription = resources.GetString("btnSave.AccessibleDescription");
			this.btnSave.AccessibleName = resources.GetString("btnSave.AccessibleName");
			this.btnSave.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnSave.Anchor")));
			this.btnSave.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnSave.BackgroundImage")));
			this.btnSave.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnSave.Dock")));
			this.btnSave.Enabled = ((bool)(resources.GetObject("btnSave.Enabled")));
			this.btnSave.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnSave.FlatStyle")));
			this.btnSave.Font = ((System.Drawing.Font)(resources.GetObject("btnSave.Font")));
			this.btnSave.Image = ((System.Drawing.Image)(resources.GetObject("btnSave.Image")));
			this.btnSave.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnSave.ImageAlign")));
			this.btnSave.ImageIndex = ((int)(resources.GetObject("btnSave.ImageIndex")));
			this.btnSave.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnSave.ImeMode")));
			this.btnSave.Location = ((System.Drawing.Point)(resources.GetObject("btnSave.Location")));
			this.btnSave.Name = "btnSave";
			this.btnSave.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnSave.RightToLeft")));
			this.btnSave.Size = ((System.Drawing.Size)(resources.GetObject("btnSave.Size")));
			this.btnSave.TabIndex = ((int)(resources.GetObject("btnSave.TabIndex")));
			this.btnSave.Text = resources.GetString("btnSave.Text");
			this.btnSave.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnSave.TextAlign")));
			this.toolTip1.SetToolTip(this.btnSave, resources.GetString("btnSave.ToolTip"));
			this.btnSave.Visible = ((bool)(resources.GetObject("btnSave.Visible")));
			this.btnSave.Click += new System.EventHandler(this.btnSave_Click);
			// 
			// btnMemoryRecall
			// 
			this.btnMemoryRecall.AccessibleDescription = resources.GetString("btnMemoryRecall.AccessibleDescription");
			this.btnMemoryRecall.AccessibleName = resources.GetString("btnMemoryRecall.AccessibleName");
			this.btnMemoryRecall.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("btnMemoryRecall.Anchor")));
			this.btnMemoryRecall.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("btnMemoryRecall.BackgroundImage")));
			this.btnMemoryRecall.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("btnMemoryRecall.Dock")));
			this.btnMemoryRecall.Enabled = ((bool)(resources.GetObject("btnMemoryRecall.Enabled")));
			this.btnMemoryRecall.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("btnMemoryRecall.FlatStyle")));
			this.btnMemoryRecall.Font = ((System.Drawing.Font)(resources.GetObject("btnMemoryRecall.Font")));
			this.btnMemoryRecall.Image = ((System.Drawing.Image)(resources.GetObject("btnMemoryRecall.Image")));
			this.btnMemoryRecall.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnMemoryRecall.ImageAlign")));
			this.btnMemoryRecall.ImageIndex = ((int)(resources.GetObject("btnMemoryRecall.ImageIndex")));
			this.btnMemoryRecall.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("btnMemoryRecall.ImeMode")));
			this.btnMemoryRecall.Location = ((System.Drawing.Point)(resources.GetObject("btnMemoryRecall.Location")));
			this.btnMemoryRecall.Name = "btnMemoryRecall";
			this.btnMemoryRecall.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("btnMemoryRecall.RightToLeft")));
			this.btnMemoryRecall.Size = ((System.Drawing.Size)(resources.GetObject("btnMemoryRecall.Size")));
			this.btnMemoryRecall.TabIndex = ((int)(resources.GetObject("btnMemoryRecall.TabIndex")));
			this.btnMemoryRecall.Text = resources.GetString("btnMemoryRecall.Text");
			this.btnMemoryRecall.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("btnMemoryRecall.TextAlign")));
			this.toolTip1.SetToolTip(this.btnMemoryRecall, resources.GetString("btnMemoryRecall.ToolTip"));
			this.btnMemoryRecall.Visible = ((bool)(resources.GetObject("btnMemoryRecall.Visible")));
			this.btnMemoryRecall.Click += new System.EventHandler(this.btnMemoryRecall_Click);
			// 
			// timer_scanner
			// 
			this.timer_scanner.Interval = 3000;
			this.timer_scanner.Tick += new System.EventHandler(this.timer_scanner_Tick);
			// 
			// timer_peak_text
			// 
			this.timer_peak_text.Interval = 500;
			this.timer_peak_text.Tick += new System.EventHandler(this.timer_peak_text_Tick);
			// 
			// grpDateTime
			// 
			this.grpDateTime.AccessibleDescription = resources.GetString("grpDateTime.AccessibleDescription");
			this.grpDateTime.AccessibleName = resources.GetString("grpDateTime.AccessibleName");
			this.grpDateTime.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("grpDateTime.Anchor")));
			this.grpDateTime.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("grpDateTime.BackgroundImage")));
			this.grpDateTime.Controls.Add(this.txtUTCTime);
			this.grpDateTime.Controls.Add(this.txtLocalTime);
			this.grpDateTime.Controls.Add(this.txtDate);
			this.grpDateTime.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("grpDateTime.Dock")));
			this.grpDateTime.Enabled = ((bool)(resources.GetObject("grpDateTime.Enabled")));
			this.grpDateTime.Font = ((System.Drawing.Font)(resources.GetObject("grpDateTime.Font")));
			this.grpDateTime.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("grpDateTime.ImeMode")));
			this.grpDateTime.Location = ((System.Drawing.Point)(resources.GetObject("grpDateTime.Location")));
			this.grpDateTime.Name = "grpDateTime";
			this.grpDateTime.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("grpDateTime.RightToLeft")));
			this.grpDateTime.Size = ((System.Drawing.Size)(resources.GetObject("grpDateTime.Size")));
			this.grpDateTime.TabIndex = ((int)(resources.GetObject("grpDateTime.TabIndex")));
			this.grpDateTime.TabStop = false;
			this.grpDateTime.Text = resources.GetString("grpDateTime.Text");
			this.toolTip1.SetToolTip(this.grpDateTime, resources.GetString("grpDateTime.ToolTip"));
			this.grpDateTime.Visible = ((bool)(resources.GetObject("grpDateTime.Visible")));
			// 
			// txtUTCTime
			// 
			this.txtUTCTime.AccessibleDescription = resources.GetString("txtUTCTime.AccessibleDescription");
			this.txtUTCTime.AccessibleName = resources.GetString("txtUTCTime.AccessibleName");
			this.txtUTCTime.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtUTCTime.Anchor")));
			this.txtUTCTime.AutoSize = ((bool)(resources.GetObject("txtUTCTime.AutoSize")));
			this.txtUTCTime.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtUTCTime.BackgroundImage")));
			this.txtUTCTime.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtUTCTime.Dock")));
			this.txtUTCTime.Enabled = ((bool)(resources.GetObject("txtUTCTime.Enabled")));
			this.txtUTCTime.Font = ((System.Drawing.Font)(resources.GetObject("txtUTCTime.Font")));
			this.txtUTCTime.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtUTCTime.ImeMode")));
			this.txtUTCTime.Location = ((System.Drawing.Point)(resources.GetObject("txtUTCTime.Location")));
			this.txtUTCTime.MaxLength = ((int)(resources.GetObject("txtUTCTime.MaxLength")));
			this.txtUTCTime.Multiline = ((bool)(resources.GetObject("txtUTCTime.Multiline")));
			this.txtUTCTime.Name = "txtUTCTime";
			this.txtUTCTime.PasswordChar = ((char)(resources.GetObject("txtUTCTime.PasswordChar")));
			this.txtUTCTime.ReadOnly = true;
			this.txtUTCTime.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtUTCTime.RightToLeft")));
			this.txtUTCTime.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtUTCTime.ScrollBars")));
			this.txtUTCTime.Size = ((System.Drawing.Size)(resources.GetObject("txtUTCTime.Size")));
			this.txtUTCTime.TabIndex = ((int)(resources.GetObject("txtUTCTime.TabIndex")));
			this.txtUTCTime.Text = resources.GetString("txtUTCTime.Text");
			this.txtUTCTime.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtUTCTime.TextAlign")));
			this.toolTip1.SetToolTip(this.txtUTCTime, resources.GetString("txtUTCTime.ToolTip"));
			this.txtUTCTime.Visible = ((bool)(resources.GetObject("txtUTCTime.Visible")));
			this.txtUTCTime.WordWrap = ((bool)(resources.GetObject("txtUTCTime.WordWrap")));
			// 
			// txtLocalTime
			// 
			this.txtLocalTime.AccessibleDescription = resources.GetString("txtLocalTime.AccessibleDescription");
			this.txtLocalTime.AccessibleName = resources.GetString("txtLocalTime.AccessibleName");
			this.txtLocalTime.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtLocalTime.Anchor")));
			this.txtLocalTime.AutoSize = ((bool)(resources.GetObject("txtLocalTime.AutoSize")));
			this.txtLocalTime.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtLocalTime.BackgroundImage")));
			this.txtLocalTime.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtLocalTime.Dock")));
			this.txtLocalTime.Enabled = ((bool)(resources.GetObject("txtLocalTime.Enabled")));
			this.txtLocalTime.Font = ((System.Drawing.Font)(resources.GetObject("txtLocalTime.Font")));
			this.txtLocalTime.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtLocalTime.ImeMode")));
			this.txtLocalTime.Location = ((System.Drawing.Point)(resources.GetObject("txtLocalTime.Location")));
			this.txtLocalTime.MaxLength = ((int)(resources.GetObject("txtLocalTime.MaxLength")));
			this.txtLocalTime.Multiline = ((bool)(resources.GetObject("txtLocalTime.Multiline")));
			this.txtLocalTime.Name = "txtLocalTime";
			this.txtLocalTime.PasswordChar = ((char)(resources.GetObject("txtLocalTime.PasswordChar")));
			this.txtLocalTime.ReadOnly = true;
			this.txtLocalTime.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtLocalTime.RightToLeft")));
			this.txtLocalTime.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtLocalTime.ScrollBars")));
			this.txtLocalTime.Size = ((System.Drawing.Size)(resources.GetObject("txtLocalTime.Size")));
			this.txtLocalTime.TabIndex = ((int)(resources.GetObject("txtLocalTime.TabIndex")));
			this.txtLocalTime.Text = resources.GetString("txtLocalTime.Text");
			this.txtLocalTime.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtLocalTime.TextAlign")));
			this.toolTip1.SetToolTip(this.txtLocalTime, resources.GetString("txtLocalTime.ToolTip"));
			this.txtLocalTime.Visible = ((bool)(resources.GetObject("txtLocalTime.Visible")));
			this.txtLocalTime.WordWrap = ((bool)(resources.GetObject("txtLocalTime.WordWrap")));
			// 
			// txtDate
			// 
			this.txtDate.AccessibleDescription = resources.GetString("txtDate.AccessibleDescription");
			this.txtDate.AccessibleName = resources.GetString("txtDate.AccessibleName");
			this.txtDate.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("txtDate.Anchor")));
			this.txtDate.AutoSize = ((bool)(resources.GetObject("txtDate.AutoSize")));
			this.txtDate.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("txtDate.BackgroundImage")));
			this.txtDate.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("txtDate.Dock")));
			this.txtDate.Enabled = ((bool)(resources.GetObject("txtDate.Enabled")));
			this.txtDate.Font = ((System.Drawing.Font)(resources.GetObject("txtDate.Font")));
			this.txtDate.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("txtDate.ImeMode")));
			this.txtDate.Location = ((System.Drawing.Point)(resources.GetObject("txtDate.Location")));
			this.txtDate.MaxLength = ((int)(resources.GetObject("txtDate.MaxLength")));
			this.txtDate.Multiline = ((bool)(resources.GetObject("txtDate.Multiline")));
			this.txtDate.Name = "txtDate";
			this.txtDate.PasswordChar = ((char)(resources.GetObject("txtDate.PasswordChar")));
			this.txtDate.ReadOnly = true;
			this.txtDate.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("txtDate.RightToLeft")));
			this.txtDate.ScrollBars = ((System.Windows.Forms.ScrollBars)(resources.GetObject("txtDate.ScrollBars")));
			this.txtDate.Size = ((System.Drawing.Size)(resources.GetObject("txtDate.Size")));
			this.txtDate.TabIndex = ((int)(resources.GetObject("txtDate.TabIndex")));
			this.txtDate.Text = resources.GetString("txtDate.Text");
			this.txtDate.TextAlign = ((System.Windows.Forms.HorizontalAlignment)(resources.GetObject("txtDate.TextAlign")));
			this.toolTip1.SetToolTip(this.txtDate, resources.GetString("txtDate.ToolTip"));
			this.txtDate.Visible = ((bool)(resources.GetObject("txtDate.Visible")));
			this.txtDate.WordWrap = ((bool)(resources.GetObject("txtDate.WordWrap")));
			// 
			// button_IFShiftDownSmall
			// 
			this.button_IFShiftDownSmall.AccessibleDescription = resources.GetString("button_IFShiftDownSmall.AccessibleDescription");
			this.button_IFShiftDownSmall.AccessibleName = resources.GetString("button_IFShiftDownSmall.AccessibleName");
			this.button_IFShiftDownSmall.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("button_IFShiftDownSmall.Anchor")));
			this.button_IFShiftDownSmall.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("button_IFShiftDownSmall.BackgroundImage")));
			this.button_IFShiftDownSmall.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("button_IFShiftDownSmall.Dock")));
			this.button_IFShiftDownSmall.Enabled = ((bool)(resources.GetObject("button_IFShiftDownSmall.Enabled")));
			this.button_IFShiftDownSmall.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("button_IFShiftDownSmall.FlatStyle")));
			this.button_IFShiftDownSmall.Font = ((System.Drawing.Font)(resources.GetObject("button_IFShiftDownSmall.Font")));
			this.button_IFShiftDownSmall.Image = ((System.Drawing.Image)(resources.GetObject("button_IFShiftDownSmall.Image")));
			this.button_IFShiftDownSmall.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_IFShiftDownSmall.ImageAlign")));
			this.button_IFShiftDownSmall.ImageIndex = ((int)(resources.GetObject("button_IFShiftDownSmall.ImageIndex")));
			this.button_IFShiftDownSmall.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("button_IFShiftDownSmall.ImeMode")));
			this.button_IFShiftDownSmall.Location = ((System.Drawing.Point)(resources.GetObject("button_IFShiftDownSmall.Location")));
			this.button_IFShiftDownSmall.Name = "button_IFShiftDownSmall";
			this.button_IFShiftDownSmall.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("button_IFShiftDownSmall.RightToLeft")));
			this.button_IFShiftDownSmall.Size = ((System.Drawing.Size)(resources.GetObject("button_IFShiftDownSmall.Size")));
			this.button_IFShiftDownSmall.TabIndex = ((int)(resources.GetObject("button_IFShiftDownSmall.TabIndex")));
			this.button_IFShiftDownSmall.Text = resources.GetString("button_IFShiftDownSmall.Text");
			this.button_IFShiftDownSmall.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_IFShiftDownSmall.TextAlign")));
			this.toolTip1.SetToolTip(this.button_IFShiftDownSmall, resources.GetString("button_IFShiftDownSmall.ToolTip"));
			this.button_IFShiftDownSmall.Visible = ((bool)(resources.GetObject("button_IFShiftDownSmall.Visible")));
			this.button_IFShiftDownSmall.Click += new System.EventHandler(this.button_IFShiftDownSmall_Click);
			// 
			// button_IFShiftUpSmall
			// 
			this.button_IFShiftUpSmall.AccessibleDescription = resources.GetString("button_IFShiftUpSmall.AccessibleDescription");
			this.button_IFShiftUpSmall.AccessibleName = resources.GetString("button_IFShiftUpSmall.AccessibleName");
			this.button_IFShiftUpSmall.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("button_IFShiftUpSmall.Anchor")));
			this.button_IFShiftUpSmall.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("button_IFShiftUpSmall.BackgroundImage")));
			this.button_IFShiftUpSmall.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("button_IFShiftUpSmall.Dock")));
			this.button_IFShiftUpSmall.Enabled = ((bool)(resources.GetObject("button_IFShiftUpSmall.Enabled")));
			this.button_IFShiftUpSmall.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("button_IFShiftUpSmall.FlatStyle")));
			this.button_IFShiftUpSmall.Font = ((System.Drawing.Font)(resources.GetObject("button_IFShiftUpSmall.Font")));
			this.button_IFShiftUpSmall.Image = ((System.Drawing.Image)(resources.GetObject("button_IFShiftUpSmall.Image")));
			this.button_IFShiftUpSmall.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_IFShiftUpSmall.ImageAlign")));
			this.button_IFShiftUpSmall.ImageIndex = ((int)(resources.GetObject("button_IFShiftUpSmall.ImageIndex")));
			this.button_IFShiftUpSmall.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("button_IFShiftUpSmall.ImeMode")));
			this.button_IFShiftUpSmall.Location = ((System.Drawing.Point)(resources.GetObject("button_IFShiftUpSmall.Location")));
			this.button_IFShiftUpSmall.Name = "button_IFShiftUpSmall";
			this.button_IFShiftUpSmall.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("button_IFShiftUpSmall.RightToLeft")));
			this.button_IFShiftUpSmall.Size = ((System.Drawing.Size)(resources.GetObject("button_IFShiftUpSmall.Size")));
			this.button_IFShiftUpSmall.TabIndex = ((int)(resources.GetObject("button_IFShiftUpSmall.TabIndex")));
			this.button_IFShiftUpSmall.Text = resources.GetString("button_IFShiftUpSmall.Text");
			this.button_IFShiftUpSmall.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_IFShiftUpSmall.TextAlign")));
			this.toolTip1.SetToolTip(this.button_IFShiftUpSmall, resources.GetString("button_IFShiftUpSmall.ToolTip"));
			this.button_IFShiftUpSmall.Visible = ((bool)(resources.GetObject("button_IFShiftUpSmall.Visible")));
			this.button_IFShiftUpSmall.Click += new System.EventHandler(this.button_IFShiftUpSmall_Click);
			// 
			// button_WidthDown
			// 
			this.button_WidthDown.AccessibleDescription = resources.GetString("button_WidthDown.AccessibleDescription");
			this.button_WidthDown.AccessibleName = resources.GetString("button_WidthDown.AccessibleName");
			this.button_WidthDown.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("button_WidthDown.Anchor")));
			this.button_WidthDown.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("button_WidthDown.BackgroundImage")));
			this.button_WidthDown.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("button_WidthDown.Dock")));
			this.button_WidthDown.Enabled = ((bool)(resources.GetObject("button_WidthDown.Enabled")));
			this.button_WidthDown.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("button_WidthDown.FlatStyle")));
			this.button_WidthDown.Font = ((System.Drawing.Font)(resources.GetObject("button_WidthDown.Font")));
			this.button_WidthDown.Image = ((System.Drawing.Image)(resources.GetObject("button_WidthDown.Image")));
			this.button_WidthDown.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_WidthDown.ImageAlign")));
			this.button_WidthDown.ImageIndex = ((int)(resources.GetObject("button_WidthDown.ImageIndex")));
			this.button_WidthDown.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("button_WidthDown.ImeMode")));
			this.button_WidthDown.Location = ((System.Drawing.Point)(resources.GetObject("button_WidthDown.Location")));
			this.button_WidthDown.Name = "button_WidthDown";
			this.button_WidthDown.repeatInterval = 500;
			this.button_WidthDown.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("button_WidthDown.RightToLeft")));
			this.button_WidthDown.Size = ((System.Drawing.Size)(resources.GetObject("button_WidthDown.Size")));
			this.button_WidthDown.TabIndex = ((int)(resources.GetObject("button_WidthDown.TabIndex")));
			this.button_WidthDown.Text = resources.GetString("button_WidthDown.Text");
			this.button_WidthDown.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_WidthDown.TextAlign")));
			this.toolTip1.SetToolTip(this.button_WidthDown, resources.GetString("button_WidthDown.ToolTip"));
			this.button_WidthDown.Visible = ((bool)(resources.GetObject("button_WidthDown.Visible")));
			this.button_WidthDown.Click += new System.EventHandler(this.button_WidthDown_Click);
			// 
			// button_WidthUp
			// 
			this.button_WidthUp.AccessibleDescription = resources.GetString("button_WidthUp.AccessibleDescription");
			this.button_WidthUp.AccessibleName = resources.GetString("button_WidthUp.AccessibleName");
			this.button_WidthUp.Anchor = ((System.Windows.Forms.AnchorStyles)(resources.GetObject("button_WidthUp.Anchor")));
			this.button_WidthUp.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("button_WidthUp.BackgroundImage")));
			this.button_WidthUp.Dock = ((System.Windows.Forms.DockStyle)(resources.GetObject("button_WidthUp.Dock")));
			this.button_WidthUp.Enabled = ((bool)(resources.GetObject("button_WidthUp.Enabled")));
			this.button_WidthUp.FlatStyle = ((System.Windows.Forms.FlatStyle)(resources.GetObject("button_WidthUp.FlatStyle")));
			this.button_WidthUp.Font = ((System.Drawing.Font)(resources.GetObject("button_WidthUp.Font")));
			this.button_WidthUp.Image = ((System.Drawing.Image)(resources.GetObject("button_WidthUp.Image")));
			this.button_WidthUp.ImageAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_WidthUp.ImageAlign")));
			this.button_WidthUp.ImageIndex = ((int)(resources.GetObject("button_WidthUp.ImageIndex")));
			this.button_WidthUp.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("button_WidthUp.ImeMode")));
			this.button_WidthUp.Location = ((System.Drawing.Point)(resources.GetObject("button_WidthUp.Location")));
			this.button_WidthUp.Name = "button_WidthUp";
			this.button_WidthUp.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("button_WidthUp.RightToLeft")));
			this.button_WidthUp.Size = ((System.Drawing.Size)(resources.GetObject("button_WidthUp.Size")));
			this.button_WidthUp.TabIndex = ((int)(resources.GetObject("button_WidthUp.TabIndex")));
			this.button_WidthUp.Text = resources.GetString("button_WidthUp.Text");
			this.button_WidthUp.TextAlign = ((System.Drawing.ContentAlignment)(resources.GetObject("button_WidthUp.TextAlign")));
			this.toolTip1.SetToolTip(this.button_WidthUp, resources.GetString("button_WidthUp.ToolTip"));
			this.button_WidthUp.Visible = ((bool)(resources.GetObject("button_WidthUp.Visible")));
			this.button_WidthUp.Click += new System.EventHandler(this.button_WidthUp_Click);
			// 
			// timer_clock
			// 
			this.timer_clock.Enabled = true;
			this.timer_clock.Interval = 1000;
			this.timer_clock.Tick += new System.EventHandler(this.timer_clock_Tick);
			// 
			// Console
			// 
			this.AccessibleDescription = resources.GetString("$this.AccessibleDescription");
			this.AccessibleName = resources.GetString("$this.AccessibleName");
			this.AutoScaleBaseSize = ((System.Drawing.Size)(resources.GetObject("$this.AutoScaleBaseSize")));
			this.AutoScroll = ((bool)(resources.GetObject("$this.AutoScroll")));
			this.AutoScrollMargin = ((System.Drawing.Size)(resources.GetObject("$this.AutoScrollMargin")));
			this.AutoScrollMinSize = ((System.Drawing.Size)(resources.GetObject("$this.AutoScrollMinSize")));
			this.BackColor = System.Drawing.SystemColors.Control;
			this.BackgroundImage = ((System.Drawing.Image)(resources.GetObject("$this.BackgroundImage")));
			this.ClientSize = ((System.Drawing.Size)(resources.GetObject("$this.ClientSize")));
			this.Controls.Add(this.button_WidthUp);
			this.Controls.Add(this.button_WidthDown);
			this.Controls.Add(this.button_IFShiftUpSmall);
			this.Controls.Add(this.button_IFShiftDownSmall);
			this.Controls.Add(this.chkNB);
			this.Controls.Add(this.grpDateTime);
			this.Controls.Add(this.grpMemory);
			this.Controls.Add(this.grpDisplay2);
			this.Controls.Add(this.grpOptions);
			this.Controls.Add(this.grpSoundControls);
			this.Controls.Add(this.grpVFO);
			this.Controls.Add(this.grpMultimeter);
			this.Controls.Add(this.chkPipe);
			this.Controls.Add(this.grpDSP);
			this.Controls.Add(this.lblCPUMeter);
			this.Controls.Add(this.grpMode);
			this.Controls.Add(this.grpDisplay);
			this.Controls.Add(this.grpVFOA);
			this.Controls.Add(this.grpVFOB);
			this.Controls.Add(this.grpBand);
			this.Controls.Add(this.grpFilter);
			this.Controls.Add(this.chkPower);
			this.Controls.Add(this.tbWidth);
			this.Controls.Add(this.button_IFShiftDown);
			this.Controls.Add(this.button_IFShiftZero);
			this.Controls.Add(this.button_IFShiftUp);
			this.Controls.Add(this.tbFilterShift);
			this.Controls.Add(this.btnFilterShiftReset);
			this.Controls.Add(this.lblFilterShift);
			this.Controls.Add(this.udFilterLow);
			this.Controls.Add(this.lblFilterLow);
			this.Controls.Add(this.lblFilterHigh);
			this.Controls.Add(this.udFilterHigh);
			this.Enabled = ((bool)(resources.GetObject("$this.Enabled")));
			this.Font = ((System.Drawing.Font)(resources.GetObject("$this.Font")));
			this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
			this.ImeMode = ((System.Windows.Forms.ImeMode)(resources.GetObject("$this.ImeMode")));
			this.KeyPreview = true;
			this.Location = ((System.Drawing.Point)(resources.GetObject("$this.Location")));
			this.MaximizeBox = false;
			this.MaximumSize = ((System.Drawing.Size)(resources.GetObject("$this.MaximumSize")));
			this.Menu = this.mainMenu1;
			this.MinimumSize = ((System.Drawing.Size)(resources.GetObject("$this.MinimumSize")));
			this.Name = "Console";
			this.RightToLeft = ((System.Windows.Forms.RightToLeft)(resources.GetObject("$this.RightToLeft")));
			this.StartPosition = ((System.Windows.Forms.FormStartPosition)(resources.GetObject("$this.StartPosition")));
			this.Text = resources.GetString("$this.Text");
			this.toolTip1.SetToolTip(this, resources.GetString("$this.ToolTip"));
			this.KeyDown += new System.Windows.Forms.KeyEventHandler(this.Console_KeyDown);
			this.MouseDown += new System.Windows.Forms.MouseEventHandler(this.WheelTune_MouseDown);
			this.Closing += new System.ComponentModel.CancelEventHandler(this.Console_Closing);
			this.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.Console_KeyPress);
			this.KeyUp += new System.Windows.Forms.KeyEventHandler(this.Console_KeyUp);
			this.MouseWheel += new System.Windows.Forms.MouseEventHandler(this.Console_MouseWheel);
			this.grpVFOA.ResumeLayout(false);
			this.grpVFOB.ResumeLayout(false);
			this.grpDisplay.ResumeLayout(false);
			this.grpMode.ResumeLayout(false);
			this.grpBand.ResumeLayout(false);
			this.grpFilter.ResumeLayout(false);
			((System.ComponentModel.ISupportInitialize)(this.tbFilterShift)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.udFilterHigh)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.udFilterLow)).EndInit();
			this.grpDSP.ResumeLayout(false);
			((System.ComponentModel.ISupportInitialize)(this.tbWidth)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.udXIT)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.udRIT)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.udPWR)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.udAF)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.udMIC)).EndInit();
			this.grpMultimeter.ResumeLayout(false);
			this.grpVFO.ResumeLayout(false);
			this.grpSoundControls.ResumeLayout(false);
			((System.ComponentModel.ISupportInitialize)(this.udSquelch)).EndInit();
			this.grpOptions.ResumeLayout(false);
			this.grpDisplay2.ResumeLayout(false);
			this.grpMemory.ResumeLayout(false);
			((System.ComponentModel.ISupportInitialize)(this.udScannerDelay)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.udScannerStep)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.udScannerHigh)).EndInit();
			((System.ComponentModel.ISupportInitialize)(this.udScannerLow)).EndInit();
			this.grpDateTime.ResumeLayout(false);
			this.ResumeLayout(false);

		}
		#endregion

		#region Main
		// ======================================================
		// Main
		// ======================================================

		[STAThread]
		static void Main(string[] args) 
		{
			try 
			{
				//Application.EnableVisualStyles(); 
				//Application.DoEvents(); 
				Application.Run(new Console(args));
			}
			catch(Exception ex)
			{
				MessageBox.Show(ex.Message+"\n\n"+ex.StackTrace.ToString(), "Fatal Error",
					MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
		}

		#endregion

		#region Misc Routines
		// ======================================================
		// Misc Routines
		// ======================================================

		private void InitConsole()
		{
			Audio.console = this;
#if(NEW_DTTSP)
			chkDSPNB2.Enabled = true;
#endif
			int[] band_stacks = DB.GetBandStackNum();
			band_160m_register = band_stacks[0];
			band_80m_register = band_stacks[1];
			band_60m_register = band_stacks[2];
			band_40m_register = band_stacks[3];
			band_30m_register = band_stacks[4];
			band_20m_register = band_stacks[5];
			band_17m_register = band_stacks[6];
			band_15m_register = band_stacks[7];
			band_12m_register = band_stacks[8];
			band_10m_register = band_stacks[9];
			band_6m_register = band_stacks[10];
			band_2m_register = band_stacks[11];
			band_wwv_register = band_stacks[12];
			band_gen_register = band_stacks[13];

			vfoa_hover_digit = -1;
			vfob_hover_digit = -1;

			atu_tuning = false;
			tune_power = 10;
			calibrating = false;
			run_setup_wizard = true;
			int W = picDisplay.Width;

			// get culture specific decimal separator
			separator = CultureInfo.CurrentCulture.NumberFormat.NumberDecimalSeparator;

			//dsp_display_mutex = new Mutex();
			//dsp_meter_mutex = new Mutex();
			pa_power_mutex = new Mutex();
			high_swr_mutex = new Mutex();

			last_band = "";						// initialize bandstack
			
			average_buffer = new float[display_buffer_size];	// initialize averaging buffer array
			average_buffer[0] = CLEAR_FLAG;		// set the clear flag

			histogram_data = new int[W];		// initialize histogram arrays
			histogram_history = new int[W];
			for(int i=0; i<W; i++)
			{
				histogram_data[i] = Int32.MaxValue;
				histogram_history[i] = 0;
			}

			sc2_cal = new int[101];
			for(int i=0; i<101; i++)
				sc2_cal[i] = 50;

			wheel_tune_list = new double[8];		// initialize wheel tuning list array
			wheel_tune_list[0] = 0.000001;
			wheel_tune_list[1] = 0.00001;
			wheel_tune_list[2] = 0.0001;
			wheel_tune_list[3] = 0.001;
			wheel_tune_list[4] = 0.01;
			wheel_tune_list[5] = 0.1;
			wheel_tune_list[6] = 1.0;
			wheel_tune_list[7] = 10.0;
			wheel_tune_index = 3;

			meter_text_history = new float[multimeter_text_peak_samples];
			display_data = new float[display_buffer_size];

			preamp_offset = new float[4];
			preamp_offset[(int)PreampMode.OFF] = 10.0f;
			preamp_offset[(int)PreampMode.LOW] = 0.0f;
			preamp_offset[(int)PreampMode.MED] = -16.0f;
			preamp_offset[(int)PreampMode.HIGH] = -26.0f;

			display_bitmap = new Bitmap(picDisplay.Width, picDisplay.Height);
			display_graphics = Graphics.FromImage(display_bitmap);
			waterfall_bmp = new Bitmap(W, picDisplay.Height-16, PixelFormat.Format24bppRgb);	// initialize waterfall display

			display_image_mutex = new Mutex(false);			// initialize image synchronization
			background_image_mutex = new Mutex(false);

			this.ActiveControl = chkPower;		// Power has focus initially
			long_crosshair = false;				// initialize long crosshairs

			InitDisplayModes();					// Initialize Display Modes
			InitAGCModes();						// Initialize AGC Modes
			InitMultiMeterModes();				// Initialize MultiMeter Modes

			audio_process_thread = new Thread(	// create audio process thread
				new ThreadStart(DttSP.BeginProcess));
			audio_process_thread.Name = "Audio Process Thread";
			audio_process_thread.Priority = ThreadPriority.Highest;
			audio_process_thread.IsBackground = true;
			audio_process_thread.Start();

			hw = new HW(0x378);					// create hardware object
			hw.Init();							// Power down hardware

			CWForm = new CW(this);				// create CW form
			CWForm.StartPosition = FormStartPosition.Manual;
			//Keyer = new CWKeyer2(this);			// create new Keyer

			SetupForm = new Setup(this);		// create Setup form
			SetupForm.StartPosition = FormStartPosition.Manual;

			CWForm.GetCWOptions();

			MemForm = new Memory(this);			// create Memory form
			MemForm.StartPosition = FormStartPosition.Manual;

			WaveForm = new WaveControl(this);	// create Wave form
			WaveForm.StartPosition = FormStartPosition.Manual;			

			current_filter = Filter.F2600;

			CurrentAGCMode = AGCMode.MED;				// Initialize front panel controls
			comboPreamp.Text = "Med";
			vfob_dsp_mode = DSPMode.LSB;
			vfob_filter = Filter.F2600;
			comboDisplayMode.SelectedIndex = 0;
			comboMeterRXMode.SelectedIndex = 0;
			comboTuneMode.SelectedIndex = 0;
			quick_save_filter = Filter.F2600;
			quick_save_mode = DSPMode.LSB;
			PWR = 50;

			last_filter = new Filter[(int)DSPMode.LAST];
			var1_low = new int[(int)DSPMode.LAST];
			var1_high = new int[(int)DSPMode.LAST];
			var2_low = new int[(int)DSPMode.LAST];
			var2_high = new int[(int)DSPMode.LAST];
			InitFilterVals();					// Initialize filter values

			radModeLSB.Checked = true;
			radFilter2600.Checked = true;
			
			GetState();							// recall saved state

			PAPresent = pa_present;
			txtVFOAFreq_LostFocus(this, new EventArgs());
			txtVFOBFreq_LostFocus(this, new EventArgs());
			udPWR_ValueChanged(this, new EventArgs());
			udAF_ValueChanged(this, new EventArgs());
			udMIC_ValueChanged(this, new EventArgs());
			tbFilterShift_Scroll(this, new EventArgs());
			hw.StandBy();						// initialize hardware device

			wheel_tune_index--;					// Setup wheel tuning
			ChangeWheelTuneLeft();

			SetupForm.initCATandPTTprops();   // wjt added -- get console props setup for cat and ptt 
			if ( CmdLineArgs != null ) 
			{ 
				for ( int i = 0; i < CmdLineArgs.Length; i++ ) 
				{
					if ( CmdLineArgs[i] != null && CmdLineArgs[i] == "--disable-swr-prot-at-my-risk" ) 
					{ 
						this.DisableSWRProtection = true; 
						this.Text = this.Text + "  *** SWR Protection Disabled! ***"; 
					}
				}
			}


			if(comboMeterTXMode.Items.Count > 0 && comboMeterTXMode.SelectedIndex < 0)
				comboMeterTXMode.SelectedIndex = 0;
		}

		public void ExitConsole()
		{
			if(SetupForm != null)		// make sure Setup form is deallocated
				SetupForm.Dispose();
			if(CWForm != null)			// make sure CW form is deallocated
				CWForm.Dispose();
			chkPower.Checked = false;	// make sure power is off		
			
			//			if(draw_display_thread != null)
			//				draw_display_thread.Abort();
#if (!NEW_DTTSP)
			if(audio_process_thread != null)	// kill audio process thread
				audio_process_thread.Abort();
#endif
			
			Parallel.ExitPortTalk();	// close parallel port driver
			PA19.PA_Terminate();		// terminate audio interface
			DB.Exit();					// close and save database
			//Mixer.RestoreState();		// restore initial mixer state
			DttSP.Exit();				// deallocate DSP variables
			Mixer.SetMainVolume(0, main_volume);
			Mixer.SetWaveOutVolume(0, wave_volume);
		}

		public void SaveState()
		{
			// Automatically saves all control settings to the database in the tab
			// pages on this form of the following types: CheckBox, ComboBox,
			// NumericUpDown, RadioButton, TextBox, and TrackBar (slider)

			chkPower.Checked = false;		// turn off the power first

			ArrayList a = new ArrayList();

			foreach(Control c in this.Controls)			// For each control
			{
				if(c.GetType() == typeof(GroupBoxTS))		// if it is a groupbox, check for sub controls
				{
					foreach(Control c2 in ((GroupBoxTS)c).Controls)	// for each sub-control
					{	// check to see if it is a value type we need to save
						if(c2.Enabled)
						{
							if(c2.GetType() == typeof(CheckBoxTS))
								a.Add(c2.Name+"/"+((CheckBoxTS)c2).Checked.ToString());
							else if(c2.GetType() == typeof(ComboBoxTS))
							{
								if(((ComboBoxTS)c2).Items.Count > 0)
									a.Add(c2.Name+"/"+((ComboBoxTS)c2).Text);
							}
							else if(c2.GetType() == typeof(NumericUpDownTS))
								a.Add(c2.Name+"/"+((NumericUpDownTS)c2).Value.ToString());
							else if(c2.GetType() == typeof(RadioButtonTS))
								a.Add(c2.Name+"/"+((RadioButtonTS)c2).Checked.ToString());
							else if(c2.GetType() == typeof(TextBoxTS))
							{
								if(((TextBox)c2).ReadOnly == false)
									a.Add(c2.Name+"/"+((TextBoxTS)c2).Text);
							}
							else if(c2.GetType() == typeof(TrackBarTS))
								a.Add(c2.Name+"/"+((TrackBarTS)c2).Value.ToString());
#if(DEBUG)
							else if(c2.GetType() == typeof(GroupBox) ||
								c2.GetType() == typeof(CheckBox) ||
								c2.GetType() == typeof(ComboBox) ||
								c2.GetType() == typeof(NumericUpDown) ||
								c2.GetType() == typeof(RadioButton) ||
								c2.GetType() == typeof(TextBox) ||
								c2.GetType() == typeof(TrackBar))
								Debug.WriteLine(c2.Name+" needs to be converterd to a Thread Safe control.");
#endif
						}
					}
				}
				else // it is not a group box
				{	// check to see if it is a value type we need to save
					if(c.Enabled)
					{
						if(c.GetType() == typeof(CheckBoxTS))
							a.Add(c.Name+"/"+((CheckBoxTS)c).Checked.ToString());
						else if(c.GetType() == typeof(ComboBoxTS))
						{
							if(((ComboBox)c).SelectedIndex >= 0)
								a.Add(c.Name+"/"+((ComboBoxTS)c).Text);
						}
						else if(c.GetType() == typeof(NumericUpDownTS))
							a.Add(c.Name+"/"+((NumericUpDownTS)c).Value.ToString());
						else if(c.GetType() == typeof(RadioButtonTS))
							a.Add(c.Name+"/"+((RadioButtonTS)c).Checked.ToString());
						else if(c.GetType() == typeof(TextBoxTS))
						{
							if(((TextBoxTS)c).ReadOnly == false)
								a.Add(c.Name+"/"+((TextBoxTS)c).Text);
						}
						else if(c.GetType() == typeof(TrackBarTS))
							a.Add(c.Name+"/"+((TrackBarTS)c).Value.ToString());
#if(DEBUG)
						else if(c.GetType() == typeof(GroupBox) ||
							c.GetType() == typeof(CheckBox) ||
							c.GetType() == typeof(ComboBox) ||
							c.GetType() == typeof(NumericUpDown) ||
							c.GetType() == typeof(RadioButton) ||
							c.GetType() == typeof(TextBox) ||
							c.GetType() == typeof(TrackBar))
							Debug.WriteLine(c.Name+" needs to be converterd to a Thread Safe control.");
#endif
					}
				}
			}

			a.Add("display_cal_offset/"+display_cal_offset.ToString("f3"));
			a.Add("multimeter_cal_offset/"+multimeter_cal_offset);

			a.Add("txtMemoryQuick/"+ txtMemoryQuick.Text);		// save quick memory settings
			a.Add("quick_save_mode/"+(int)quick_save_mode);
			a.Add("quick_save_filter/"+(int)quick_save_filter);

			for(int i=(int)DSPMode.FIRST+1; i<(int)DSPMode.LAST; i++)
			{	// save filter settings per mode
				a.Add("last_filter["+i.ToString()+"]/"+((int)last_filter[i]).ToString());
				a.Add("var1_low["+i.ToString()+"]/"+var1_low[i].ToString());
				a.Add("var1_high["+i.ToString()+"]/"+var1_high[i].ToString());
				a.Add("var2_low["+i.ToString()+"]/"+var2_low[i].ToString());
				a.Add("var2_high["+i.ToString()+"]/"+var2_high[i].ToString());
			}

			for(int i=(int)PreampMode.FIRST+1; i<(int)PreampMode.LAST; i++)
				a.Add("preamp_offset["+i.ToString()+"]/"+preamp_offset[i].ToString());

			a.Add("wheel_tune_index/"+wheel_tune_index.ToString());		// Save wheel tune value

			a.Add("vfob_dsp_mode/"+((int)vfob_dsp_mode).ToString());			// Save VFO B values 
			a.Add("vfob_filter/"+((int)vfob_filter).ToString());
			
			a.Add("console_top/"+this.Top.ToString());		// save form positions
			a.Add("console_left/"+this.Left.ToString());
			a.Add("setup_top/"+SetupForm.Top.ToString());
			a.Add("setup_left/"+SetupForm.Left.ToString());
			a.Add("cw_top/"+CWForm.Top.ToString());
			a.Add("cw_left/"+CWForm.Left.ToString());
			a.Add("mem_top/"+MemForm.Top.ToString());
			a.Add("mem_left/"+MemForm.Left.ToString());

			a.Add("Version/"+this.Text);		// save the current version

			DB.SaveVars("State", ref a);		// save the values to the DB
		}

		public void GetState()
		{
			// Automatically restores all controls from the database in the
			// tab pages on this form of the following types: CheckBox, ComboBox,
			// NumericUpDown, RadioButton, TextBox, and TrackBar (slider)

			ArrayList checkbox_list = new ArrayList();
			ArrayList combobox_list = new ArrayList();
			ArrayList numericupdown_list = new ArrayList();
			ArrayList radiobutton_list = new ArrayList();
			ArrayList textbox_list = new ArrayList();
			ArrayList trackbar_list = new ArrayList();

			//ArrayList controls = new ArrayList();	// list of controls to restore
			foreach(Control c in this.Controls)
			{
				if(c.GetType() == typeof(GroupBoxTS))	// if control is a groupbox, retrieve all subcontrols
				{
					foreach(Control c2 in ((GroupBoxTS)c).Controls)
					{
						if(c2.Enabled)
						{
							if(c2.GetType() == typeof(CheckBoxTS))			// the control is a CheckBox
								checkbox_list.Add(c2);
							else if(c2.GetType() == typeof(ComboBoxTS))		// the control is a ComboBox
								combobox_list.Add(c2);
							else if(c2.GetType() == typeof(NumericUpDownTS))	// the control is a NumericUpDown
								numericupdown_list.Add(c2);
							else if(c2.GetType() == typeof(RadioButtonTS))	// the control is a RadioButton
								radiobutton_list.Add(c2);
							else if(c2.GetType() == typeof(TextBoxTS))		// the control is a TextBox
								textbox_list.Add(c2);
							else if(c2.GetType() == typeof(TrackBarTS))		// the control is a TrackBar (slider)
								trackbar_list.Add(c2);
						}
					}						
				}
				else
				{
					if(c.Enabled)
					{
						if(c.GetType() == typeof(CheckBoxTS))				// the control is a CheckBox
							checkbox_list.Add(c);
						else if(c.GetType() == typeof(ComboBoxTS))		// the control is a ComboBox
							combobox_list.Add(c);
						else if(c.GetType() == typeof(NumericUpDownTS))	// the control is a NumericUpDown
							numericupdown_list.Add(c);
						else if(c.GetType() == typeof(RadioButtonTS))		// the control is a RadioButton
							radiobutton_list.Add(c);
						else if(c.GetType() == typeof(TextBoxTS))			// the control is a TextBox
							textbox_list.Add(c);
						else if(c.GetType() == typeof(TrackBarTS))		// the control is a TrackBar (slider)
							trackbar_list.Add(c);
					}
				}
			}

			ArrayList a = DB.GetVars("State");							// Get the saved list of controls
			a.Sort();
			int num_controls = checkbox_list.Count + combobox_list.Count +
				numericupdown_list.Count + radiobutton_list.Count +
				textbox_list.Count + trackbar_list.Count;

			foreach(string s in a)				// string is in the format "name,value"
			{
				string[] vals = s.Split('/');
				if(vals.Length > 2)
				{
					for(int i=2; i<vals.Length; i++)
						vals[1] += "/"+vals[i];
				}

				string name = vals[0];
				string val = vals[1];
				int num = 0;

				if(name.StartsWith("last_filter["))
				{
					int start = name.IndexOf("[")+1;
					int length = name.IndexOf("]")-start;
					int index = Int32.Parse(name.Substring(start, length));

					last_filter[index] = (Filter)(Int32.Parse(val));
				}
				else if(name.StartsWith("var1_low["))
				{
					int start = name.IndexOf("[")+1;
					int length = name.IndexOf("]")-start;
					int index = Int32.Parse(name.Substring(start, length));

					var1_low[index] = Int32.Parse(val);
				}
				else if(name.StartsWith("var1_high["))
				{
					int start = name.IndexOf("[")+1;
					int length = name.IndexOf("]")-start;
					int index = Int32.Parse(name.Substring(start, length));

					var1_high[index] = Int32.Parse(val);
				}
				else if(name.StartsWith("var2_low["))
				{
					int start = name.IndexOf("[")+1;
					int length = name.IndexOf("]")-start;
					int index = Int32.Parse(name.Substring(start, length));

					var2_low[index] = Int32.Parse(val);
				}
				else if(name.StartsWith("var2_high["))
				{
					int start = name.IndexOf("[")+1;
					int length = name.IndexOf("]")-start;
					int index = Int32.Parse(name.Substring(start, length));

					var2_high[index] = Int32.Parse(val);
				}
				else if(name.StartsWith("preamp_offset["))
				{
					int start = name.IndexOf("[")+1;
					int length = name.IndexOf("]")-start;
					int index = Int32.Parse(name.Substring(start, length));

					preamp_offset[index] = float.Parse(val);
				}

				switch(name)
				{
					case "wheel_tune_index":
						wheel_tune_index = Int32.Parse(val);
						break;
					case "txtMemoryQuick":
						txtMemoryQuick.Text = val;
						break;
					case "display_cal_offset":
						display_cal_offset = float.Parse(val);
						break;
					case "multimeter_cal_offset":
						multimeter_cal_offset = float.Parse(val);
						break;
					case "quick_save_mode":
						quick_save_mode = (DSPMode)(Int32.Parse(val));
						break;
					case "quick_save_filter":
						quick_save_filter = (Filter)(Int32.Parse(val));
						break;
					case "vfob_dsp_mode":
						vfob_dsp_mode = (DSPMode)(Int32.Parse(val));
						break;
					case "vfob_filter":
						vfob_filter = (Filter)(Int32.Parse(val));
						break;
					case "console_top":
						num = Int32.Parse(val);
						if(num < 0 || num > Screen.PrimaryScreen.Bounds.Height)
							num = 0;
						this.Top = num;
						break;
					case "console_left":
						num = Int32.Parse(val);
						if(num < 0 || num > Screen.PrimaryScreen.Bounds.Width)
							num = 0;
						this.Left = num;
						break;
					case "setup_top":
						num = Int32.Parse(val);
						if(num < 0 || num > Screen.PrimaryScreen.Bounds.Height)
							num = 0;
						SetupForm.Top = num;
						break;
					case "setup_left":
						num = Int32.Parse(val);
						if(num < 0 || num > Screen.PrimaryScreen.Bounds.Width)
							num = 0;
						SetupForm.Left = num;
						break;
					case "cw_top":
						num = Int32.Parse(val);
						if(num < 0 || num > Screen.PrimaryScreen.Bounds.Height)
							num = 0;
						CWForm.Top = num;
						break;
					case "cw_left":
						num = Int32.Parse(val);
						if(num < 0 || num > Screen.PrimaryScreen.Bounds.Width)
							num = 0;
						CWForm.Left = num;
						break;
					case "mem_top":
						num = Int32.Parse(val);
						if(num < 0 || num > Screen.PrimaryScreen.Bounds.Height)
							num = 0;
						MemForm.Top = num;
						break;
					case "mem_left":
						num = Int32.Parse(val);
						if(num < 0 || num > Screen.PrimaryScreen.Bounds.Width)
							num = 0;
						MemForm.Left = num;
						break;
					case "SetupWizard":
						if(val == "1")
							run_setup_wizard = false;
						break;
				}
			}

			// restore saved values to the controls
			foreach(string s in a)				// string is in the format "name,value"
			{
				string[] vals = s.Split('/');
				string name = vals[0];
				string val = vals[1];

				if(s.StartsWith("chk"))			// control is a CheckBox
				{
					for(int i=0; i<checkbox_list.Count; i++)
					{	// look through each control to find the matching name
						CheckBoxTS c = (CheckBoxTS)checkbox_list[i];
						if(c.Name.Equals(name))		// name found
						{
							c.Checked = bool.Parse(val);	// restore value
							i = checkbox_list.Count+1;
						}
						if(i == checkbox_list.Count)
							MessageBox.Show("Control not found: "+name, "GetState Error",
								MessageBoxButtons.OK, MessageBoxIcon.Warning);
					}
				}
				else if(s.StartsWith("combo"))	// control is a ComboBox
				{
					for(int i=0; i<combobox_list.Count; i++)
					{	// look through each control to find the matching name
						ComboBoxTS c = (ComboBoxTS)combobox_list[i];
						if(c.Name.Equals(name))		// name found
						{
							c.Text = val;	// restore value
							i = combobox_list.Count+1;
						}
						if(i == combobox_list.Count)
							MessageBox.Show("Control not found: "+name, "GetState Error",
								MessageBoxButtons.OK, MessageBoxIcon.Warning);
					}
				}
				else if(s.StartsWith("ud"))
				{
					for(int i=0; i<numericupdown_list.Count; i++)
					{	// look through each control to find the matching name
						NumericUpDownTS c = (NumericUpDownTS)numericupdown_list[i];
						if(c.Name.Equals(name))		// name found
						{
							decimal num = decimal.Parse(val);

							if(num > c.Maximum) num = c.Maximum;		// check endpoints
							else if(num < c.Minimum) num = c.Minimum;
							c.Value = num;			// restore value
							i = numericupdown_list.Count+1;
						}
						if(i == numericupdown_list.Count)
							MessageBox.Show("Control not found: "+name, "GetState Error",
								MessageBoxButtons.OK, MessageBoxIcon.Warning);	
					}
				}
				else if(s.StartsWith("rad"))
				{	// look through each control to find the matching name
					for(int i=0; i<radiobutton_list.Count; i++)
					{
						RadioButtonTS c = (RadioButtonTS)radiobutton_list[i];
						if(c.Name.Equals(name))		// name found
						{
							if(!val.ToLower().Equals("true") && !val.ToLower().Equals("false"))
								val = "True";
							c.Checked = bool.Parse(val);	// restore value
							i = radiobutton_list.Count+1;
						}
						if(i == radiobutton_list.Count)
							MessageBox.Show("Control not found: "+name, "GetState Error",
								MessageBoxButtons.OK, MessageBoxIcon.Warning);
					}
				}
				else if(s.StartsWith("txt"))
				{	// look through each control to find the matching name
					for(int i=0; i<textbox_list.Count; i++)
					{
						TextBoxTS c = (TextBoxTS)textbox_list[i];
						if(c.Name.Equals(name))		// name found
						{
							c.Text = val;	// restore value
							i = textbox_list.Count+1;
						}
						if(i == textbox_list.Count)
							MessageBox.Show("Control not found: "+name, "GetState Error",
								MessageBoxButtons.OK, MessageBoxIcon.Warning);
					}
				}
				else if(s.StartsWith("tb"))
				{
					// look through each control to find the matching name
					for(int i=0; i<trackbar_list.Count; i++)
					{
						TrackBarTS c = (TrackBarTS)trackbar_list[i];
						if(c.Name.Equals(name))		// name found
						{
							c.Value = Int32.Parse(val);
							i = trackbar_list.Count+1;
						}
						if(i == trackbar_list.Count)
							MessageBox.Show("Control not found: "+name, "GetState Error",
								MessageBoxButtons.OK, MessageBoxIcon.Warning);
					}
				}
			}
		}

		private void InitFilterVals()
		{
			// used to initialize all the filter variables
			for(int i=(int)DSPMode.FIRST+1; i<(int)DSPMode.LAST; i++)
			{
				switch(i)
				{
					case (int)DSPMode.LSB:
					case (int)DSPMode.PSK:
						last_filter[i] = Filter.F2600;
						var1_low[i] = var2_low[i] = -2800;
						var1_high[i] = var2_high[i] = -200;
						break;
					case (int)DSPMode.USB:
						last_filter[i] = Filter.F2600;
						var1_low[i] = var2_low[i] = 200;
						var1_high[i] = var2_high[i] = 2800;
						break;
					case (int)DSPMode.DSB:
						last_filter[i] = Filter.F2600;
						var1_low[i] = var2_low[i] = -2600;
						var1_high[i] = var2_high[i] = 2600;
						break;
					case (int)DSPMode.CWL:
						last_filter[i] = Filter.F1000;
						var1_low[i] = var2_low[i] = -1200;
						var1_high[i] = var2_high[i] = -200;
						break;
					case (int)DSPMode.CWU:
						last_filter[i] = Filter.F1000;
						var1_low[i] = var2_low[i] = 200;
						var1_high[i] = var2_high[i] = 1200;
						break;
					case (int)DSPMode.AM:
					case (int)DSPMode.SAM:
					case (int)DSPMode.FMN:
						last_filter[i] = Filter.F6000;
						var1_low[i] = var2_low[i] = -6000;
						var1_high[i] = var2_high[i] = 6000;
						break;
					case (int)DSPMode.RTTY:
						last_filter[i] = Filter.VAR1;
						var1_low[i] = var2_low[i] = -2465;
						var1_high[i] = var2_high[i] = -1955;
						break;
					default:
						last_filter[i] = Filter.NONE;
						var1_low[i] = var2_low[i] = 0;
						var1_high[i] = var2_high[i] = 0;
						break;
				}
			}
		}

		private void InitDisplayModes()
		{
			// populate the display mode list
			for(DisplayMode dm=DisplayMode.FIRST+1; dm<DisplayMode.LAST; dm++)
			{
				string s = dm.ToString().ToLower();
				s = s.Substring(0, 1).ToUpper() + s.Substring(1, s.Length-1);
				comboDisplayMode.Items.Add(s);
			}
		}

		private void InitAGCModes()
		{
			// populate the AGC mode list
			for(AGCMode agc=AGCMode.FIRST+1; agc<AGCMode.LAST; agc++)
			{
				string s = agc.ToString().ToLower();
				s = s.Substring(0, 1).ToUpper() + s.Substring(1, s.Length-1);
				comboAGC.Items.Add(s);
			}
		}

		private void InitMultiMeterModes()
		{
			comboMeterRXMode.Items.Add("Signal");
			comboMeterRXMode.Items.Add("Sig Avg");
			comboMeterRXMode.Items.Add("ADC L");
			comboMeterRXMode.Items.Add("ADC R");

			comboMeterTXMode.Items.Add("ALC");
			comboMeterTXMode.Items.Add("Fwd Pow");
			comboMeterTXMode.Items.Add("Peak Pow");
			comboMeterTXMode.Items.Add("Rev Pow");
		}

		private void DeselectAllFilters()
		{
			foreach(Control c in grpFilter.Controls)
			{
				if(c.GetType() == typeof(RadioButton))
				{
					((RadioButton)c).Checked = false;
				}
			}
			CurrentFilter = Filter.NONE;
		}

		private void DisableAllFilters()
		{
			foreach(Control c in grpFilter.Controls)
			{
				if(c.GetType() == typeof(RadioButton))
				{
					c.Enabled = false;
				}
			}
		}

		private void EnableAllFilters()
		{
			foreach(Control c in grpFilter.Controls)
			{
				if(c.GetType() == typeof(RadioButton))
				{
					c.Enabled = true;

					if(c.BackColor == vfo_text_dark_color)
					{
						c.BackColor = button_selected_color;
					}
				}
			}
		}

		private void DisableAllBands()
		{
			foreach(Button b in grpBand.Controls)
			{
				b.Enabled = false;

				if(b.BackColor == button_selected_color)
				{
					b.BackColor = vfo_text_dark_color;
				}
			}

		}

		private void EnableAllBands()
		{
			foreach(Button b in grpBand.Controls)
			{
				if(b.Text != "")
				{
					if(b.Text == "2")
						b.Enabled = XVTRPresent;
					else
						b.Enabled = true;
				}
				if(b.BackColor == Color.Olive)
					b.BackColor = button_selected_color;
			}
		}

		private void DisableAllModes()
		{
			foreach(RadioButton r in grpMode.Controls)
			{
				r.Enabled = false;
				if(r.BackColor == button_selected_color)
					r.BackColor = Color.Olive;
			}
		}

		private void EnableAllModes()
		{
			foreach(RadioButton r in grpMode.Controls)
			{
				if(r.Text != "" && r.Text != "RTTY" && r.Text != "PSK")
					r.Enabled = true;
				if(r.BackColor == Color.Olive)
					r.BackColor = button_selected_color;
			}
		}

		private void DisableFilters(int lowcutoff)
		{
			// Disables all filters below the number passed in. 
			// For example, DisableFilters(100) would cause the
			// 100Hz, 50Hz and 25Hz filters to be disabled.

			foreach(Control c in grpFilter.Controls)
			{
				if(c.GetType() == typeof(RadioButton) && c.Name.IndexOf("Var") < 0)
				{
					string name = c.Name;
					int begin, len;
					begin = name.IndexOf("Filter")+6;
					len = name.Length-begin;

					int filter_width = Int32.Parse(name.Substring(begin, len));
					if(filter_width < lowcutoff)
					{
						c.Enabled = false;
						((RadioButton)c).Checked = false;
					}
				}
			}
		}

		private void GetVFOCharWidth()
		{
			// This function calculates the pixel width of the VFO display.
			// This information is used for mouse wheel hover tuning.

			Graphics g = txtVFOAFreq.CreateGraphics();
			SizeF size = g.MeasureString("000000", txtVFOAFreq.Font);
			vfo_pixel_offset = txtVFOAFreq.Width - (int)size.Width + (int)(size.Width/12);
			vfo_char_width = (int)size.Width/6 - 1;
			size = g.MeasureString(CultureInfo.CurrentCulture.NumberFormat.NumberDecimalSeparator, txtVFOAFreq.Font);
			vfo_decimal_width = (int)size.Width/2;
			g.Dispose();
		}

		private void SaveBand()
		{
			// Used in Bandstacking algorithm
			double freq = double.Parse(txtVFOAFreq.Text);
			string filter = "";
			string mode = "";

			foreach(Control c in grpFilter.Controls)
			{
				if(c.GetType() == typeof(RadioButton))
				{
					RadioButton r = (RadioButton)c;
					if(r.Checked)
					{
						filter = r.Name.Substring(9, r.Name.Length-9);
						break;
					}
				}
			}

			foreach(RadioButton r in grpMode.Controls)
			{
				if(r.Checked)
				{
					mode = r.Name.Substring(7, r.Name.Length-7);
					break;
				}
			}

			switch(last_band)
			{
				case "160M":
					if(freq >= 1.8 && freq < 2.0)
						DB.SaveBandStack("160M", band_160m_index, mode, filter, Math.Round(freq, 6));
					break;
				case "80M":
					if(freq >= 3.5 && freq < 4.0)
						DB.SaveBandStack("80M", band_80m_index, mode, filter, Math.Round(freq, 6));
					break;
				case "60M":
					if(freq == 5.3305 || freq == 5.3465 || freq == 5.3665 || freq == 5.3715 || freq == 5.4035)
						DB.SaveBandStack("60M", band_60m_index, "USB", filter, Math.Round(freq, 6));
					break;
				case "40M":
					if(freq >= 7.0 && freq < 7.3)
						DB.SaveBandStack("40M", band_40m_index, mode, filter, Math.Round(freq, 6));
					break;
				case "30M":
					if(freq >= 10.1 && freq < 10.15)
						DB.SaveBandStack("30M", band_30m_index, mode, filter, Math.Round(freq, 6));
					break;
				case "20M":
					if(freq >= 14.0 && freq < 14.350)
						DB.SaveBandStack("20M", band_20m_index, mode, filter, Math.Round(freq, 6));
					break;
				case "17M":
					if(freq >= 18.068 && freq < 18.168)
						DB.SaveBandStack("17M", band_17m_index, mode, filter, Math.Round(freq, 6));
					break;
				case "15M":
					if(freq >= 21.0 && freq < 21.45)
						DB.SaveBandStack("15M", band_15m_index, mode, filter, Math.Round(freq, 6));
					break;
				case "12M":
					if(freq >= 24.890 && freq < 24.990)
						DB.SaveBandStack("12M", band_12m_index, mode, filter, Math.Round(freq, 6));
					break;
				case "10M":
					if(freq >= 28.0 && freq < 29.7)
						DB.SaveBandStack("10M", band_10m_index, mode, filter, Math.Round(freq, 6));
					break;
				case "6M":
					if(freq >= 50.0 && freq < 54.0)
						DB.SaveBandStack("6M", band_6m_index, mode, filter, Math.Round(freq, 6));
					break;
				case "2M":
					if(freq >= 144.0 && freq < 146.0)
						DB.SaveBandStack("2M", band_2m_index, mode, filter, Math.Round(freq, 6));
					break;
				case "WWV":
					if(freq == 2.5 || freq == 5.0 || freq == 10.0 || freq == 15.0 || freq == 20.0)
						DB.SaveBandStack("WWV", band_wwv_index, mode, filter, Math.Round(freq, 6));
					break;
				case "GEN":
					DB.SaveBandStack("GEN", band_gen_index, mode, filter, Math.Round(freq, 6));
					break;
			}
		}

		private void SetBand(string mode, string filter, double freq)
		{
			// Set mode, filter, and frequency according to passed parameters
			mode = "radMode"+mode;
			filter = "radFilter"+filter;
			
			foreach(RadioButton r in grpMode.Controls)
			{
				if(r.Name.Equals(mode))
				{
					r.Checked = true;
					break;
				}
			}

			if(current_dsp_mode != DSPMode.DRM &&
				current_dsp_mode != DSPMode.SPEC)
			{
				foreach(Control c in grpFilter.Controls)
				{
					if(c.GetType() == typeof(RadioButton) &&
						c.Name.Equals(filter))
					{
						RadioButton r = (RadioButton)c;
						r.Checked = true;
						break;
					}
				}
			}

			VFOAFreq = Math.Round(freq, 6);
			txtVFOAFreq_LostFocus(this, new EventArgs());
		}

		public void MemoryRecall(int mode, int filter, double freq, int step, int agc, int squelch)
		{
			// Set mode, filter, and frequency, mouse wheel tune step
			// and AGC according to passed parameters

			CurrentDSPMode = (DSPMode)mode;
			//			((RadioButton)mode_list[mode]).Checked = true;
			if(current_dsp_mode != DSPMode.DRM &&
				current_dsp_mode != DSPMode.SPEC)
				CurrentFilter = (Filter)filter;
			VFOAFreq = Math.Round(freq, 6);
			txtVFOAFreq_LostFocus(this, new EventArgs());
			comboAGC.SelectedIndex = agc;
			udSquelch.Value = squelch;
			wheel_tune_index = step;
			switch(wheel_tune_index)
			{
				case 0: 
					txtWheelTune.Text = "1Hz";
					break;
				case 1:
					txtWheelTune.Text = "10Hz";
					break;
				case 2:
					txtWheelTune.Text = "100Hz";
					break;
				case 3: 
					txtWheelTune.Text = "1KHz";
					break;
				case 4:
					txtWheelTune.Text = "10KHz";
					break;
				case 5:
					txtWheelTune.Text = "100KHz";
					break;
				case 6: 
					txtWheelTune.Text = "1MHz";
					break;
				case 7:
					txtWheelTune.Text = "10MHz";
					break;
			}
		}

		private void ChangeWheelTuneLeft()
		{
			// change mouse wheel tuning step one digit to the left
			wheel_tune_index = (wheel_tune_index+1)%wheel_tune_list.Length;
			switch(wheel_tune_index)
			{
				case 0: 
					txtWheelTune.Text = "1Hz";
					break;
				case 1:
					txtWheelTune.Text = "10Hz";
					break;
				case 2:
					txtWheelTune.Text = "100Hz";
					break;
				case 3: 
					txtWheelTune.Text = "1KHz";
					break;
				case 4:
					txtWheelTune.Text = "10KHz";
					break;
				case 5:
					txtWheelTune.Text = "100KHz";
					break;
				case 6: 
					txtWheelTune.Text = "1MHz";
					break;
				case 7:
					txtWheelTune.Text = "10MHz";
					break;
			}
		}

		private void ChangeWheelTuneRight()
		{
			// change mouse wheel tuning step one digit to the right
			int length = wheel_tune_list.Length;
			wheel_tune_index = (wheel_tune_index-1 + length)%length;
			switch(wheel_tune_index)
			{
				case 0: 
					txtWheelTune.Text = "1Hz";
					break;
				case 1:
					txtWheelTune.Text = "10Hz";
					break;
				case 2:
					txtWheelTune.Text = "100Hz";
					break;
				case 3: 
					txtWheelTune.Text = "1KHz";
					break;
				case 4:
					txtWheelTune.Text = "10KHz";
					break;
				case 5:
					txtWheelTune.Text = "100KHz";
					break;
				case 6: 
					txtWheelTune.Text = "1MHz";
					break;
				case 7:
					txtWheelTune.Text = "10MHz";
					break;
			}
		}

		private void SetBandButtonColor(Band b)
		{
			// Sets band button color based on passed band.

			Button btn = null;
			switch(b)
			{
				case Band.GEN:
					btn = btnBandGEN;
					break;
				case Band.B160M:
					btn = btnBand160;
					break;
				case Band.B80M:
					btn = btnBand80;
					break;
				case Band.B60M:
					btn = btnBand60;
					break;
				case Band.B40M:
					btn = btnBand40;
					break;
				case Band.B30M:
					btn = btnBand30;
					break;
				case Band.B20M:
					btn = btnBand20;
					break;
				case Band.B17M:
					btn = btnBand17;
					break;
				case Band.B15M:
					btn = btnBand15;
					break;
				case Band.B12M:
					btn = btnBand12;
					break;
				case Band.B10M:
					btn = btnBand10;
					break;
				case Band.B6M:
					btn = btnBand6;
					break;
				case Band.B2M:
					btn = btnBand2;
					break;
				case Band.WWV:
					btn = btnBandWWV;
					break;
			}

			foreach(Button b2 in grpBand.Controls)
			{
				Color c = SystemColors.Control;
				if(b2 == btn)
					c = button_selected_color;

				b2.BackColor = c;
			}
		}

		private void SetCurrentBand(float freq)
		{
			Band b = Band.GEN;

			if(!extended)
			{
				if(freq >= 1.8 && freq <= 2.0)
					b = Band.B160M;
				else if(freq >= 3.5 && freq <= 4.0)
					b = Band.B80M;
				else if(freq == 5.3305f || freq == 5.3465f ||
					freq == 5.3665f || freq == 5.3715f ||
					freq == 5.4035f)
					b = Band.B60M;
				else if(freq >= 7.0f && freq <= 7.3f)
					b = Band.B40M;
				else if(freq >= 10.1f && freq <= 10.15f)
					b = Band.B30M;
				else if(freq >= 14.0f && freq <= 14.35f)
					b = Band.B20M;
				else if(freq >= 18.068f && freq <= 18.168f)
					b = Band.B17M;
				else if(freq >= 21.0f && freq <= 21.450f)
					b = Band.B15M;
				else if(freq >= 24.89f && freq <= 24.99f)
					b = Band.B12M;
				else if(freq >= 28.0f && freq <= 29.7f)
					b = Band.B10M;
				else if(freq >= 50.0f && freq <= 54.0f)
					b = Band.B6M;
				else if(freq >= 144.0f && freq <= 148.0f)
					b = Band.B2M;
				else if(freq == 2.5f || freq == 5.0f ||
					freq == 10.0f || freq == 15.0f || 
					freq == 20.0f)
					b = Band.WWV;
				else
					b = Band.GEN;
			}
			else
			{
				if(freq >= 0.0 && freq <= 2.75)
					b = Band.B160M;
				else if(freq > 2.75 && freq <= 5.3305)
					b = Band.B80M;
				else if(freq > 5.3305 && freq < 7.0)
					b = Band.B60M;
				else if(freq >= 7.0 && freq <= 8.7)
					b = Band.B40M;
				else if(freq >= 8.7 && freq <= 12.075)
					b = Band.B30M;
				else if(freq >= 12.075 && freq <= 16.209)
					b = Band.B20M;
				else if(freq >= 16.209 && freq <= 19.584)
					b = Band.B17M;
				else if(freq >= 19.584 && freq <= 23.17)
					b = Band.B15M;
				else if(freq >= 23.17 && freq <= 26.495)
					b = Band.B12M;
				else if(freq >= 26.495 && freq <= 29.7)
					b = Band.B10M;
				else if(freq >= 50.0f && freq <= 54.0f)
					b = Band.B6M;
				else if(freq >= 144.0f && freq <= 148.0f)
					b = Band.B2M;
				else if(freq == 2.5f || freq == 5.0f ||
					freq == 10.0f || freq == 15.0f || 
					freq == 20.0f)
					b = Band.WWV;
				else
					b = Band.GEN;
			}

			CurrentBand = b;
			if(atu_present && 
				comboTuneMode.SelectedIndex > 0 &&
				(ATUTuneMode)comboTuneMode.SelectedIndex != ATUTuneMode.BYPASS &&
				(tuned_band != b) &&
				tuned_band != Band.FIRST)
			{
				hw.PA_ATUTune(ATUTuneMode.BYPASS);
				tuned_band = Band.FIRST;
				chkTUN.BackColor = SystemColors.Control;				
			}

			if(rfe_present)
			{
				if(b == Band.B2M)
				{
					if(comboPreamp.Items.Contains("Off"))
						comboPreamp.Items.Remove("Off");
					if(comboPreamp.Items.Contains("Med"))
						comboPreamp.Items.Remove("Med");
					if(comboPreamp.SelectedIndex < 0)
						comboPreamp.Text = "High";
				}
				else
				{
					if(rfe_present && xvtr_present)
					{
						if(!comboPreamp.Items.Contains("Off"))
							comboPreamp.Items.Insert(0, "Off");
						if(!comboPreamp.Items.Contains("Med"))
							comboPreamp.Items.Insert(2, "Med");
					}
				}
			}
		}

		private float GainByBand(Band b)
		{
			float retval = 0;
			switch(b)
			{
				case Band.B160M:
					retval = SetupForm.PAGain160;
					break;
				case Band.B80M:
					retval = SetupForm.PAGain80;
					break;
				case Band.B60M:
					retval = SetupForm.PAGain60;
					break;
				case Band.B40M:
					retval = SetupForm.PAGain40;
					break;
				case Band.B30M:
					retval = SetupForm.PAGain30;
					break;
				case Band.B20M:
					retval = SetupForm.PAGain20;
					break;
				case Band.B17M:
					retval = SetupForm.PAGain17;
					break;
				case Band.B15M:
					retval = SetupForm.PAGain15;
					break;
				case Band.B12M:
					retval = SetupForm.PAGain12;
					break;
				case Band.B10M:
					retval = SetupForm.PAGain10;
					break;
				default:
					retval = 1000;
					break;
			}

			return retval;
		}

		public void CheckSelectedButtonColor()
		{
			// used when changing the background color of selected buttons
			foreach(Control c in this.Controls)
			{
				if(c.GetType() == typeof(GroupBoxTS))
				{
					foreach(Control c2 in ((GroupBoxTS)c).Controls)
					{
						if(c2.GetType() == typeof(RadioButtonTS))
						{
							RadioButtonTS r = (RadioButtonTS)c2;
							if(r.Checked && r.BackColor != SystemColors.Control)
							{
								c2.BackColor = button_selected_color;
							}
						}
						else if(c2.GetType() == typeof(CheckBoxTS))
						{
							CheckBoxTS chk = (CheckBoxTS)c2;
							if(chk.Checked && chk.BackColor != SystemColors.Control)
							{
								c2.BackColor = button_selected_color;
							}
						}
						else if(c2.GetType() == typeof(NumericUpDownTS))
						{
							NumericUpDownTS ud = (NumericUpDownTS)c2;
							if(ud.BackColor != SystemColors.Window)
							{
								c2.BackColor = button_selected_color;
							}
						}
						else if(c2.GetType() == typeof(ButtonTS))
						{
							ButtonTS b = (ButtonTS)c2;
							if(b.BackColor != SystemColors.Control)
							{
								c2.BackColor = button_selected_color;
							}
						}
					}
				}
				else if(c.GetType() == typeof(RadioButtonTS))
				{
					RadioButtonTS r = (RadioButtonTS)c;
					if(r.Checked && r.BackColor != SystemColors.Control)
						c.BackColor = button_selected_color;
				}
				else if(c.GetType() == typeof(CheckBoxTS))
				{
					CheckBoxTS chk = (CheckBoxTS)c;
					if(chk.Checked && chk.BackColor != SystemColors.Control)
						c.BackColor = button_selected_color;
				}
				else if(c.GetType() == typeof(NumericUpDownTS))
				{
					NumericUpDownTS ud = (NumericUpDownTS)c;
					if(ud.BackColor != SystemColors.Window)
						c.BackColor = button_selected_color;
				}
				else if(c.GetType() == typeof(ButtonTS))
				{
					ButtonTS b = (ButtonTS)c;
					if(b.BackColor != SystemColors.Control)
						c.BackColor = button_selected_color;
				}
			}
		}

		private double PABandOffset(Band b)
		{
			double num = 0;
			switch(b)
			{
				case Band.B160M:
					num = SetupForm.PAADC160;
					break;
				case Band.B80M:
					num = SetupForm.PAADC80;
					break;
				case Band.B60M:
					num = SetupForm.PAADC60;
					break;
				case Band.B40M:
					num = SetupForm.PAADC40;
					break;
				case Band.B30M:
					num = SetupForm.PAADC30;
					break;
				case Band.B20M:
					num = SetupForm.PAADC20;
					break;
				case Band.B17M:
					num = SetupForm.PAADC17;
					break;
				case Band.B15M:
					num = SetupForm.PAADC15;
					break;
				case Band.B12M:
					num = SetupForm.PAADC12;
					break;
				case Band.B10M:
					num = SetupForm.PAADC10;
					break;
			}

			if(num == 0) return 0;
			//return 100000 / Math.Pow(num, 2);
			return (double)108/num;
		}

		private double SWR(double fwd_pow, double rev_pow)
		{
			if(fwd_pow == rev_pow)
				return 1.0;

			double ro = Math.Sqrt(rev_pow/fwd_pow);
			return (1+ro)/(1-ro);
		}

		private double SWR2(int adc_fwd, int adc_rev)
		{
			if(adc_fwd == 0 && adc_rev == 0)
				return 1.0;
			else if(adc_rev > adc_fwd)
				return 50.0;
			
			double Ef = ScaledVoltage(adc_fwd);
			double Er = ScaledVoltage(adc_rev);

			double swr = (Ef + Er)/(Ef - Er);

			return swr;
		}

		private double ScaledVoltage(int adc)
		{
			double v_det = adc * 0.062963;			// scale factor in V/bit including pot ratio
			double v_out = v_det * 10.39853;		// scale factor in V/V for bridge output to detector voltage
			return v_out*PABandOffset(CurrentBand);
			//double v_det = adc * 0.0304;
			//			double v_out = 0;
			//			if(v_det >= 1.6)
			//				v_out = (-0.241259304*v_det+12.07915098)*v_det*PABandOffset(CurrentBand);
			//			else if(v_det > 0.35)
			//				v_out = (1/Math.Pow(v_det, 2)+11.3025111)*v_det*PABandOffset(CurrentBand);
			//return v_out;
		}

		private double ADCtodBm(int adc_data)
		{
			if(adc_data == 0)
				return 0;

			double mult = 100000 / Math.Pow(225/PABandOffset(CurrentBand), 2);
			return 10*Math.Log10(mult*Math.Pow(adc_data, 2));
		}

		private double PAPowerW(int dir)
		{
			int adc = 0;
			if(dir == 0) adc = pa_fwd_power;
			else if(dir == 1) adc = pa_rev_power;
			
			double dBm = ADCtodBm(adc);
			double watts = dBmToWatts(dBm);
			watts = Math.Max(watts, 0.0);
			return watts;
		}

		private double PAPowerW2(int dir)
		{
			int adc = 0;
			if(dir == 0) adc = pa_fwd_power;
			else if(dir == 1) adc = pa_rev_power;

			double v_out = ScaledVoltage(adc);
			double pow = Math.Pow(v_out, 2)/50;
			pow = Math.Max(pow, 0.0);
			return pow;
		}

		private double WattsTodBm(double watts)
		{
			return 10*Math.Log10(watts/0.001);
		}

		private double dBmToWatts(double dBm)
		{
			return Math.Pow(10, dBm/10)*0.001;
		}

		private void CheckForOpenProcesses()
		{
			// find all open PowerSDR processes
			Process[] p = Process.GetProcessesByName("PowerSDR");
			if(p.Length > 1)
			{
				DialogResult dr = MessageBox.Show("Close other instances of PowerSDR?",
					"Close other PowerSDR(s)?",
					MessageBoxButtons.YesNo,
					MessageBoxIcon.Question);
				if(dr == DialogResult.Yes)
				{
					for(int i=0; i<p.Length; i++)	// close all other PowerSDR processes except this one
					{
						if(p[i].Id != Process.GetCurrentProcess().Id)
							p[i].CloseMainWindow();
					}

					int count = 0;
					do
					{
						p = Process.GetProcessesByName("PowerSDR");
						Thread.Sleep(100);
					} while(count++ < 30 && p.Length > 1);

					if(p.Length > 1)
					{
						for(int i=0; i<p.Length; i++)	// close all other PowerSDR processes except this one
						{
							if(p[i].Id != Process.GetCurrentProcess().Id)
								p[i].Kill();
						}
					}
				}
			}
		}

		public int VersionTextToInt(string version)	// takes a version string like "1.0.6" 
		{											// and converts it to an int like 1060.
			int mult = 1000;
			string s = version;
			int retval = 0;

			while(s.IndexOf(".") >= 0)
			{
				int first_index = s.IndexOf(".");
				string temp = s.Substring(0, first_index);
				retval +=  mult * Int32.Parse(temp);
				s = s.Substring(first_index+1);
				mult /= 10;
			}

			retval += mult * Int32.Parse(s);

			return retval;
		}

		public void CheckForUpdates()
		{
			// notes taken from http://free.netartmedia.net/CSharp/CSharp5.html
			try
			{
				// Create a 'WebRequest' object with the specified url. 
				WebRequest myWebRequest = WebRequest.Create("http://www.flex-radio.com/PowerSDR_Update.txt"); 

				// Send the 'WebRequest' and wait for response. 
				WebResponse myWebResponse = myWebRequest.GetResponse(); 

				// Obtain a 'Stream' object associated with the response object. 
				Stream ReceiveStream = myWebResponse.GetResponseStream(); 

				// Pipe the stream to a higher level stream reader. 
				StreamReader readStream = new StreamReader(ReceiveStream);

				// Read the entire stream into a string object.
				string strResponse = readStream.ReadToEnd(); 

				// Release the resources. 
				readStream.Close();
				myWebResponse.Close();

				// Split the lines into a string array.
				string[] lines = strResponse.Split('\n');
				
				string temp = "";
				int begin_index = 0, end_index = 0, line_index = 0;
				int version_value = 0;

				// get current version value in integer form (i.e. 1.1.6 = 116)
				Assembly assembly = Assembly.GetExecutingAssembly();
				FileVersionInfo fvi = FileVersionInfo.GetVersionInfo(assembly.Location);
				VersionTextToInt("1.2.3.4");
				int current_version = VersionTextToInt(fvi.FileVersion);

				if(notify_on_release)
				{
					// find release version value in integer form.
					begin_index = 0;
					end_index = lines[0].IndexOf("Released");
					temp = lines[0].Substring(begin_index, end_index-begin_index);
					temp = temp.Replace("//", "");

					version_value = VersionTextToInt(temp);
					
					// if the new value is higher, display form
					if(version_value > current_version)
					{
						MessageBox.Show("New Version: "+lines[0].Substring(2));
					}
				}

				// find first line of Beta release info
				for(int i=0; i<lines.Length; i++)
				{
					if(lines[i].StartsWith("========"))
					{
						line_index = i+1;
						break;
					}
				}

				// if the line_index = 0, then the response was bad
				// or the file is not formatted correctly.
				if(line_index == 0)
					return;

				if(notify_on_beta)
				{
					// find beta version value in integer form.
					begin_index = lines[line_index].IndexOf(".")-1;
					end_index = lines[line_index].IndexOf("Released");
					temp = lines[line_index].Substring(begin_index, end_index-begin_index);
					temp = temp.Replace("//", "");
					
					version_value = VersionTextToInt(temp);
					
					// if the new value is higher, display form
					if(version_value > current_version)
					{
						MessageBox.Show("New Beta Version: "+lines[line_index].Substring(2));
					}
				}				
				
			}
			catch(Exception e)
			{
				MessageBox.Show(e.Message);
				// do nothing here as internet may just not be present
			}
		}

		public bool IsHamBand(BandPlan b)
		{
			if(extended)
				return true;

			switch(b)
			{
				case BandPlan.IARU1:
					if(dds_freq >= 1.8 && dds_freq <= 2.0)			return true;
					else if(dds_freq >= 3.5 && dds_freq <= 4.0)		return true;
					else if(dds_freq == 5.3305)						return true;
					else if(dds_freq == 5.3465)						return true;
					else if(dds_freq == 5.3665)						return true;
					else if(dds_freq == 5.3715)						return true;
					else if(dds_freq == 5.4035)						return true;
					else if(dds_freq >= 7.0 && dds_freq <= 7.3)		return true;
					else if(dds_freq >= 10.1 && dds_freq <= 10.15)	return true;
					else if(dds_freq >= 14.0 && dds_freq <= 14.35)	return true;
					else if(dds_freq >= 18.068 && dds_freq <= 18.168) return true;
					else if(dds_freq >= 21.0 && dds_freq <= 21.45)	return true;
					else if(dds_freq >= 24.89 && dds_freq <= 24.99) return true;
					else if(dds_freq >= 21.0 && dds_freq <= 21.45)	return true;
					else if(dds_freq >= 28.0 && dds_freq <= 29.7)	return true;
					else if(dds_freq >= 50.0 && dds_freq <= 54.0)	return true;
					else if(dds_freq >= 144.0 && dds_freq <= 146.0)
					{
						if(rfe_present && xvtr_present)
							return true;
						else
							return false;
					}
					else return false;
				default:
					return false;
					// TODO: Implement other bandplans here
			}
		}

		public void SetHWFilters(double freq)
		{
			if(rfe_present)							// RFE is present
			{
				//Use shift registers on RFE to control BPF and LPF banks
				if(freq <= 2.5)					// DC to 2.5MHz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.B160;

					hw.BPFRelay = BPFBand.B160;
					if(freq <= 0.3 && enable_LPF0)
						hw.RFE_LPF = RFELPFBand.AUX;													
					else
						hw.RFE_LPF = RFELPFBand.B160;
				}
				else if(freq <= 4)				// 2.5MHz to 4MHz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.B80;

					hw.BPFRelay = BPFBand.B60;
					hw.RFE_LPF = RFELPFBand.B80;
				}
				else if(freq <= 6)				// 4MHz to 6MHz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.B6040;

					hw.BPFRelay = BPFBand.B60;
					hw.RFE_LPF = RFELPFBand.B60;
				}
				else if(freq <= 7.3)			// 6MHz to 7.3MHz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.B6040;

					hw.BPFRelay = BPFBand.B40;
					hw.RFE_LPF = RFELPFBand.B40;
				}
				else if(freq <= 10.2)			// 7.3MHz to 10.2MHz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.B3020;

					hw.BPFRelay = BPFBand.B40;
					hw.RFE_LPF = RFELPFBand.B30;
				}
				else if(freq <= 12)				// 10.2MHz to 12MHz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.NONE;

					hw.BPFRelay = BPFBand.B40;
					hw.RFE_LPF = RFELPFBand.B30;
				}
				else if(freq <= 14.5)			// 12MHz to 14.5MHz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.B3020;

					hw.BPFRelay = BPFBand.B20;
					hw.RFE_LPF = RFELPFBand.B20;
				}
				else if(freq <= 21.5)			// 14.5MHz to 21.5MHz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.B1715;

					hw.BPFRelay = BPFBand.B20;
					hw.RFE_LPF = RFELPFBand.B1715;
				}
				else if(freq <= 24)				// 21.5MHz to 24MHz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.B1210;

					hw.BPFRelay = BPFBand.B20;
					hw.RFE_LPF = RFELPFBand.B1210;
				}
				else if(freq <= 30)				// 24MHz to 30MHz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.B1210;

					hw.BPFRelay = BPFBand.B10;
					hw.RFE_LPF = RFELPFBand.B1210;
				}
				else if(freq <= 36)				// 30MHz to 36MHz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.NONE;

					hw.BPFRelay = BPFBand.B10;
					hw.RFE_LPF = RFELPFBand.B6;
				}
				else if(freq <= 65)				// 36MHz to 65Mhz
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.NONE;

					hw.BPFRelay = BPFBand.B6;
					hw.RFE_LPF = RFELPFBand.B6;
				}
				else if(xvtr_present && freq >= 144 && freq <= 146)			//28MHz IF for transverter
				{
					if(pa_present)
						hw.PA_LPF = PAFBand.NONE;

					hw.BPFRelay = BPFBand.B10;
					hw.RFE_LPF = RFELPFBand.B1210;
					hw.XVTR_RF = true;
				}
				if(xvtr_present && freq < 144)
					hw.XVTR_RF = false;
			}
			else									// RFE is not present
			{
				//Select the BPF relays using the high frequency cutoff
				if(freq < 2.5)					//DC to 2.5MHz
					hw.BPFRelay = BPFBand.B160;
				else if(freq < 6)				//2.5MHz to 6MHz
					hw.BPFRelay = BPFBand.B60;
				else if(freq < 12)				//6MHz to 12MHz
					hw.BPFRelay = BPFBand.B40;
				else if(freq < 24)				//12MHz to 24MHz
					hw.BPFRelay = BPFBand.B20;
				else if(freq < 36)				//24MHz to 36MHz
					hw.BPFRelay = BPFBand.B10;
				else								//36MHz to 65Mhz
					hw.BPFRelay = BPFBand.B6;
			}
		}

		#endregion

		#region Test and Calibration Routines

		public bool CalibrateFreq(float freq)
		{
			bool retval = false;

			if(!chkPower.Checked)
			{
				MessageBox.Show("Power must be on in order to calibrate.", "Power Is Off",
					MessageBoxButtons.OK, MessageBoxIcon.Stop);
				return false;
			}

			string vfo_freq_text = txtVFOAFreq.Text;		// save current frequency

			DSPMode dsp_mode = current_dsp_mode;			// save current demod mode
			CurrentDSPMode = DSPMode.AM;					// set DSP to AM

			bool rit_on = chkRIT.Checked;							// save current RIT state
			chkRIT.Checked = false;									// set RIT to Off

			int rit_value = RITValue;						// save current RIT value
			RITValue = 0;									// set RIT Value to 0

			Filter filter = CurrentFilter;					// save current filter
			CurrentFilter = Filter.F2600;					// set filter to 2600Hz

			VFOAFreq = freq;								// set frequency to passed value

			Thread.Sleep(300);
			int ret = 0;

			do
			{
				fixed(float* ptr = &display_data[0])
					ret = DttSP.GetSpectrum(ptr);		// get the spectrum values
			} while(ret == 0);

			do
			{
				fixed(float* ptr = &display_data[0])
					ret = DttSP.GetSpectrum(ptr);		// get the spectrum values
			} while(ret == 0);

			float max = float.MinValue;
			float avg = 0;

			int max_index = 0;
			int low = display_buffer_size>>1;
			int high = low;
			low  += (int)((DttSP.RXDisplayLow * display_buffer_size) / DttSP.SampleRate);
			high += (int)((DttSP.RXDisplayHigh * display_buffer_size) / DttSP.SampleRate);

			for(int i=low; i<high; i++)						// find the maximum signal
			{
				avg += display_data[i];
				if(display_data[i] > max)
				{
					max = display_data[i];
					max_index = i;
				}
			}
			avg /= (high - low);

			if(max < avg + 30)
			{
				MessageBox.Show("Peak is less than 30dB from the noise floor.  " +
					"Please use a larger signal for frequency calibration.",
					"Calibration Error - Weak Signal",
					MessageBoxButtons.OK,
					MessageBoxIcon.Error);
				retval = false;
				goto end;
			}

			// Calculate the difference between the known signal and the measured signal
			float diff = (float)(DttSP.SampleRate / display_buffer_size * ((display_buffer_size>>1) - max_index));

			// Calculate the DDS offset
			int offset = (int)(200.0/ freq * diff);

			int current_clock = SetupForm.ClockOffset;
			if((Math.Max(current_clock + offset, current_clock - offset) > 40000) ||
				(Math.Min(current_clock + offset, current_clock - offset) < -40000))
			{
				MessageBox.Show("Peak is outside valid range.",
					"Calibration Error - Range",
					MessageBoxButtons.OK,
					MessageBoxIcon.Error);
				retval = false;
				goto end;
			}

			if((Math.Max(current_clock + offset, current_clock - offset) > 20000) ||
				(Math.Min(current_clock + offset, current_clock - offset) < -20000))
			{
				DialogResult dr = MessageBox.Show("This value is outside the specifications of the oscillator.  " +
					"Do you want to use this value?",
					"Outside Specifications",
					MessageBoxButtons.YesNo);
				if(dr == DialogResult.No)
				{
					retval = false;
					goto end;
				}
			}

			SetupForm.ClockOffset += offset;				// Offset the clock based on the difference
			retval = true;

			end:
				CurrentDSPMode = dsp_mode;						// restore DSP mode
			CurrentFilter = filter;							// restore filter
			chkRIT.Checked = rit_on;									// restore RIT state
			RITValue = rit_value;							// restore RIT value
			VFOAFreq = float.Parse(vfo_freq_text);			// restore frequency

			return retval;
		}

		public bool CalibrateLevel(float level)
		{
			// Calibration routine called by Setup Form.
			bool ret_val = false;
			if(!chkPower.Checked)
			{
				MessageBox.Show("Power must be on in order to calibrate.", "Power Is Off",
					MessageBoxButtons.OK, MessageBoxIcon.Stop);
				return false;
			}

			int screen_delay = display_delay;					// save current FPS
			display_delay = 1000;								// Set FPS to 1

			DisplayMode display_mode = CurrentDisplayMode;		// save current display mode
			CurrentDisplayMode = DisplayMode.SPECTRUM;			// set display mode to spectrum

			DSPMode dsp_mode = current_dsp_mode;				// save current DSP demod mode
			CurrentDSPMode = DSPMode.AM;						// set to AM

			PreampMode preamp = CurrentPreampMode;				// save current preamp mode
			CurrentPreampMode = PreampMode.MED;					// set to medium

			bool avg = DisplayAVG;								// save current average state
			DisplayAVG = false;									// set average state to off

			Thread.Sleep(200);

			int retval = 0;
			
			// get the value of the signal strength meter
			float num = DttSP.CalculateMeter(DttSP.MeterType.SIGNAL_STRENGTH);

			Thread.Sleep(90);
			// read again to clear out changed DSP
			num = DttSP.CalculateMeter(DttSP.MeterType.SIGNAL_STRENGTH);	

			if(rfe_present)
			{
				CurrentPreampMode = PreampMode.HIGH;
				Thread.Sleep(200);
				
				// get the value of the signal strength meter
				float num2 = DttSP.CalculateMeter(DttSP.MeterType.SIGNAL_STRENGTH);

				Thread.Sleep(90);
				// read again to clear out changed DSP
				num2 = DttSP.CalculateMeter(DttSP.MeterType.SIGNAL_STRENGTH);

				float att_offset = num2 - num;

				CurrentPreampMode = PreampMode.OFF;
				Thread.Sleep(200);

				// get the value of the signal strength meter
				num2 = DttSP.CalculateMeter(DttSP.MeterType.SIGNAL_STRENGTH);

				Thread.Sleep(90);
				// read again to clear out changed DSP
				num2 = DttSP.CalculateMeter(DttSP.MeterType.SIGNAL_STRENGTH);

				float gain_offset = num2 - num;

				preamp_offset[(int)PreampMode.OFF] = 10.0f;
				preamp_offset[(int)PreampMode.LOW] = (10.0f - att_offset);
				preamp_offset[(int)PreampMode.MED] = -16.0f;
				preamp_offset[(int)PreampMode.HIGH] = -26.0f + (10.0f - att_offset);
			}
			else
			{
				CurrentPreampMode = PreampMode.HIGH;
				Thread.Sleep(200);

				// get the value of the signal strength meter
				float num2 = DttSP.CalculateMeter(DttSP.MeterType.SIGNAL_STRENGTH);

				Thread.Sleep(90);
				// read again to clear out changed DSP
				num2 = DttSP.CalculateMeter(DttSP.MeterType.SIGNAL_STRENGTH);

				float gain_offset = num2 - num;

				preamp_offset[(int)PreampMode.MED] = 0.0f;
				preamp_offset[(int)PreampMode.HIGH] = -26.0f + (26.0f - gain_offset);
			}

			CurrentPreampMode = PreampMode.MED;
			Thread.Sleep(200);

			do
			{
				fixed(float* ptr = &display_data[0])
					retval = DttSP.GetSpectrum(ptr);		// get the spectrum values
			} while(retval == 0);

			Thread.Sleep(90);

			do
			{
				fixed(float* ptr = &display_data[0])
					retval = DttSP.GetSpectrum(ptr);		// read again to clear out changed DSP
			} while(retval == 0);

			float max = float.MinValue;						// find the max spectrum value
			for(int j=0; j<display_buffer_size; j++)
				if(display_data[j] > max) max = display_data[j];

			// calculate the difference between the current value and the correct multimeter value
			float diff = level - (num + multimeter_cal_offset + preamp_offset[(int)current_preamp_mode]);
			multimeter_cal_offset += diff;

			// calculate the difference between the current value and the correct spectrum value
			diff = level - (max + display_cal_offset + preamp_offset[(int)current_preamp_mode]);
			display_cal_offset += diff;		

			ret_val = true;

			CurrentDisplayMode = display_mode;					// restore display mode
			DisplayAVG = avg;									// restore AVG value

			CurrentPreampMode = preamp;							// restore preamp value
			CurrentDSPMode = dsp_mode;							// restore DSP mode
			display_delay = screen_delay;						// restore FPS

			//			Debug.WriteLine("multimeter_cal_offset: "+multimeter_cal_offset);
			//			Debug.WriteLine("display_cal_offset: "+display_cal_offset);
			//			MessageBox.Show("multimeter_cal_offset: "+multimeter_cal_offset.ToString()+"\n"+
			//				"display_cal_offset: "+display_cal_offset.ToString());
			return ret_val;
		}

		public bool CalibrateImage(float freq, float level, Progress progress)
		{
			//			HiPerfTimer t1 = new HiPerfTimer();
			//			t1.Start();
	
			// Setup Rig for Image Null Cal
			bool ret_val = false;

			if(!chkPower.Checked)
			{
				MessageBox.Show("Power must be on in order to calibrate.", "Power Is Off",
					MessageBoxButtons.OK, MessageBoxIcon.Stop);
				return false;
			}

			int screen_delay = display_delay;				// save current FPS
			display_delay = 333;

			DisplayMode display_mode = CurrentDisplayMode;	// save current display mode
			if(display_mode != DisplayMode.OFF)
				CurrentDisplayMode = DisplayMode.SPECTRUM;	// set display mode to spectrum

			DSPMode dsp_mode = current_dsp_mode;			// save current dsp mode
			CurrentDSPMode = DSPMode.AM;					// set dsp mode to AM

			double vfo_freq = VFOAFreq;						// save current frequency
			VFOAFreq = freq+0.022050f;						// set frequency to passed value + 22KHz

			bool avg = DisplayAVG;								// save current average state
			DisplayAVG = false;									// set average state to off

			SetupForm.ImageGainRX = -500.0;
			SetupForm.ImagePhaseRX = -400.0;

			float[] a = new float[display_buffer_size];
			float[] init_max = new float[4];

			int retval = 0;
			progress.SetPercent(0.0f);
			int counter = 0;

			Thread.Sleep(200);

			do
			{
				fixed(float* ptr = &a[0])
					retval = DttSP.GetSpectrum(ptr);// get the spectrum values
			} while(retval == 0);
			
			Thread.Sleep(200);

			do
			{
					
				fixed(float* ptr = &a[0])
					retval = DttSP.GetSpectrum(ptr);// get the spectrum values
			} while(retval == 0);

			Thread.Sleep(200);

			float max_signal = float.MinValue;				// find the signal value
			int peak_bin = -1;

			// find peak bin
			for(int j=0; j<display_buffer_size; j++)
			{
				if(a[j] > max_signal)
				{
					peak_bin = j;
					max_signal = display_data[j];
				}
			}

			SetupForm.ImageGainRX = 0.0;
			SetupForm.ImagePhaseRX = 0.0;

			// Find starting point for Phase
			for(int i=0; i<4; i++)
			{
				// Take 4 values on either side of likely null
				double val = 0.0;

				switch(i)
				{
					case 0: val = -100.0;
						break;
					case 1: val = -25.0;
						break;
					case 2: val = 25.0;
						break;
					case 3: val = 100.0;
						break;
				}			
				
				SetupForm.ImagePhaseRX = val;
				Thread.Sleep(200);

				do
				{
					fixed(float* ptr = &a[0])
						retval = DttSP.GetSpectrum(ptr);// get the spectrum values
				} while(retval == 0);
			
				Thread.Sleep(200);

				do
				{
					
					fixed(float* ptr = &a[0])
						retval = DttSP.GetSpectrum(ptr);// get the spectrum values
				} while(retval == 0);

				init_max[i] = a[peak_bin];
			}

			// Calculate Phase line coefficients
			double m1 = (init_max[1] - init_max[0])/(-25.0 - -100.0);	// should be negative
			double b1 = init_max[0] - m1*(-100.0);
			double m2 = (init_max[3] - init_max[2])/(100.0 - 25.0);		// should be positive
			double b2 = init_max[3] - m2*(100.0);

			if(m1 > 0 || m2 < 0)
			{
				MessageBox.Show("Error calculating image null.  Please try again.",
					"Image Null Error",
					MessageBoxButtons.OK,
					MessageBoxIcon.Error);
				goto end;
			}

			// Calculate line intersection for phase starting point
			double phase_intercept = (b2-b1)/(m1-m2);
			Debug.WriteLine("phase intercept: "+phase_intercept.ToString("f4"));
			SetupForm.ImagePhaseRX = 0.0;


			// Find starting point for Gain
			for(int i=0; i<4; i++)
			{
				// Take 4 values on either side of likely null
				double val = 0.0;

				switch(i)
				{
					case 0: val = -100.0;
						break;
					case 1: val = -25.0;
						break;
					case 2: val = 25.0;
						break;
					case 3: val = 100.0;
						break;
				}			
				
				SetupForm.ImageGainRX = val;
				Thread.Sleep(200);

				do
				{
					fixed(float* ptr = &a[0])
						retval = DttSP.GetSpectrum(ptr);// get the spectrum values
				} while(retval == 0);
			
				Thread.Sleep(200);

				do
				{
					
					fixed(float* ptr = &a[0])
						retval = DttSP.GetSpectrum(ptr);// get the spectrum values
				} while(retval == 0);

				init_max[i] = a[peak_bin];
			}

			// Calculate Gain line coefficients
			m1 = (init_max[1] - init_max[0])/(-25.0 - -100.0);	// should be negative
			b1 = init_max[0] - m1*(-100.0);
			m2 = (init_max[3] - init_max[2])/(100.0 - 25.0);		// should be positive
			b2 = init_max[3] - m2*(100.0);

			if(m1 > 0 || m2 < 0)
			{
				MessageBox.Show("Error calculating image null.  Please try again.",
					"Image Null Error",
					MessageBoxButtons.OK,
					MessageBoxIcon.Error);
				goto end;
			}

			// Calculate line intersection for Gain starting point
			double gain_intercept = (b2-b1)/(m1-m2);
			Debug.WriteLine("gain_intercept: "+gain_intercept.ToString("f4"));
			SetupForm.ImageGainRX = 0.0;


			bool progressing = true;
			double phase_step = 5;
			double gain_step = 3;
			double phase_range = 80;
			double gain_range = 100;
			double low_phase, high_phase, low_gain, high_gain;
			double phase_index = phase_intercept;
			double gain_index = gain_intercept;
			double global_min_phase = phase_intercept;
			double global_min_gain = gain_intercept;
			double global_min_value = float.MaxValue;

			while(progressing)
			{
				// find minimum of the peak signal over 
				// the range of Phase settings

				low_phase = global_min_phase - phase_range/2;
				high_phase = global_min_phase + phase_range/2;
				float min_signal = float.MaxValue;
				for(double i=(int)Math.Max(low_phase, -400.0); i<high_phase && i<=400.0; i+=phase_step)
				{
					SetupForm.ImagePhaseRX = i;				// phase slider
					Thread.Sleep(100);

					do
					{
						fixed(float* ptr = &a[0])
							retval = DttSP.GetSpectrum(ptr);// get the spectrum values
					} while(retval == 0);

					Thread.Sleep(100);

					do 
					{
						fixed(float* ptr = &a[0])
							retval = DttSP.GetSpectrum(ptr);// get the spectrum values
					} while(retval == 0);

					if(a[peak_bin] < min_signal)			// if image is less than minimum
					{
						min_signal = display_data[peak_bin];			// save new minimum
						phase_index = i;					// save phase index
					}

					if(!progress.Visible)
						goto end;
					else progress.SetPercent((float)((float)counter++/797));
				}	

				if(min_signal < global_min_value)
				{
					global_min_value = min_signal;
					global_min_phase = phase_index;
					global_min_gain = gain_index;
				}

				SetupForm.ImagePhaseRX = global_min_phase;			//set phase slider to min found

				// find minimum of the peak signal over 
				// the range of Gain settings

				low_gain = global_min_gain - gain_range/2;
				high_gain = global_min_gain + gain_range/2;

				min_signal = float.MaxValue;
				for(double i=(int)Math.Max(low_gain, -500.0); i<high_gain && i<=500.0; i+=gain_step)
				{
					SetupForm.ImageGainRX = i;				//set gain slider
					Thread.Sleep(100);

					do
					{
						fixed(float* ptr = &a[0])
							retval = DttSP.GetSpectrum(ptr);// get the spectrum values
					} while(retval == 0);

					Thread.Sleep(100);

					do 
					{
						fixed(float* ptr = &a[0])
							retval = DttSP.GetSpectrum(ptr);// get the spectrum values
					} while(retval == 0);

					if(a[peak_bin] < min_signal)			// if image is less than minimum
					{
						min_signal = display_data[peak_bin];			// save new minimum
						gain_index = i;						// save phase index
					}	

					if(!progress.Visible)
						goto end;
					else progress.SetPercent((float)((float)counter++/797));
				}

				if(min_signal < global_min_value)
				{
					global_min_value = min_signal;
					global_min_phase = phase_index;
					global_min_gain = gain_index;
				}

				SetupForm.ImageGainRX = global_min_gain;			//set gain slider to min found
	
				// narrow search range and use more steps
				phase_step /= 2.0; if(phase_step < 0.01) phase_step = 0.01;
				phase_range /= 2.0; if(phase_range < phase_step*4.0) phase_range = phase_step*4.0;
				gain_step /= 2.0; if(gain_step < 0.01) gain_step = 0.01;
				gain_range /= 2.0; if(gain_range < gain_step*4.0) gain_range = gain_step*4.0;

				// stop when range and step are 1 for gain and phase
				if(phase_range == 0.04 && phase_step == 0.01 &&
					gain_step == 0.01 && gain_range == 0.04)
					progressing = false;
			}

			// Finish the algorithm and reset the values
			ret_val = true;			

			end:
				progress.Hide();
			CurrentDisplayMode = display_mode;					// restore display mode
			CurrentDSPMode = dsp_mode;							// restore dsp mode
			VFOAFreq = vfo_freq;								// restore frequency
			txtVFOAFreq_LostFocus(this, new EventArgs());
			display_delay = screen_delay;						// restore FPS
			DisplayAVG = avg;									// restore average state

			//			t1.Stop();
			//			MessageBox.Show(t1.Duration.ToString());
			return ret_val;
		}

		public bool CalibratePAGain(Progress progress, bool[] run, int target_watts) // calibrate PA Gain values
		{
			//			HiPerfTimer t1 = new HiPerfTimer();
			//			t1.Start();

			bool ret_val = false;

			if(!chkPower.Checked)
			{
				MessageBox.Show("Power must be on in order to calibrate.", "Power Is Off",
					MessageBoxButtons.OK, MessageBoxIcon.Stop);
				return false;
			}

			calibrating = true;

			int screen_delay = display_delay;				// save current FPS
			display_delay = 333;

			DisplayMode display_mode = CurrentDisplayMode;	// save current display mode
			CurrentDisplayMode = DisplayMode.OFF;			// set display mode to off

			DSPMode dsp_mode = current_dsp_mode;			// save current dsp mode
			CurrentDSPMode = DSPMode.USB;					// set dsp mode to CWL

			double vfo_freq = VFOAFreq;						// save current frequency

			int pwr = PWR;									// save current pwr level

			int multimeter_delay = meter_delay;				// save current meter delay
			meter_delay = 1000;								// set meter delay to 1 second

			progress.SetPercent(0.0f);

			float[] band_freqs = { 1.9f, 3.75f, 5.3665f, 7.15f, 10.125f, 14.175f, 18.1f, 21.225f, 24.9f, 28.85f };
				
			if(run[0]) SetupForm.PAGain160 = 48.0f;
			if(run[1]) SetupForm.PAGain80 = 48.0f;
			if(run[2]) SetupForm.PAGain60 = 48.0f;
			if(run[3]) SetupForm.PAGain40 = 48.0f;
			if(run[4]) SetupForm.PAGain30 = 48.0f;
			if(run[5]) SetupForm.PAGain20 = 48.0f;
			if(run[6]) SetupForm.PAGain17 = 48.0f;
			if(run[7]) SetupForm.PAGain15 = 48.0f;
			if(run[8]) SetupForm.PAGain12 = 48.0f;
			if(run[9]) SetupForm.PAGain10 = 48.0f;

			for(int i=0; i<band_freqs.Length; i++)
			{
				if(run[i])
				{
					int error_count = 0;
					VFOAFreq = band_freqs[i];				// set frequency
					if(i == 2)
						udPWR.Value = Math.Min(50, target_watts);
					else udPWR.Value = target_watts;
					int target = (int)udPWR.Value;

					bool good_result = false;
					while(good_result == false)
					{
						if(!second_sound_card_enabled)
							Audio.CurrentAudioState1 = Audio.AudioState.SINL_COSR;
						else
							Audio.CurrentAudioState2 = Audio.AudioState.SINL_COSR;

						chkMOX.Checked = true;
						for(int j=0; j<30; j++)
						{
							Thread.Sleep(100);
							if(!progress.Visible)
								goto end;
						}

						double watts = 0;
						pa_power_mutex.WaitOne();
						if(use_old_swr)
							watts = PAPowerW(0);
						else
							watts = PAPowerW2(0);
						pa_power_mutex.ReleaseMutex();
						chkMOX.Checked = false;
						
						if(!second_sound_card_enabled)
							Audio.CurrentAudioState1 = Audio.AudioState.DTTSP;
						else
							Audio.CurrentAudioState2 = Audio.AudioState.NOL_NOR;

						Debug.WriteLine("watts: "+watts.ToString());

						if(!progress.Visible)
							goto end;

						if(Math.Abs(watts-target) > 4)
						{
							// convert to dBm
							float diff_dBm = (float)Math.Round((WattsTodBm(watts) - WattsTodBm((double)target)), 1);	

							switch(i)										// fix gain value
							{
								case 0:
									if(SetupForm.PAGain160 + diff_dBm < 39.0)
									{	
										error_count++;
										if(error_count == 1)
										{
											SetupForm.PAGain160 = 48.0f;
											break;
										}
										else if(error_count > 1)
											goto error;
									}
									else SetupForm.PAGain160 += diff_dBm;
									break;
								case 1:
									if(SetupForm.PAGain80 + diff_dBm < 39.0)
									{
										error_count++;
										if(error_count == 1)
										{
											SetupForm.PAGain80 = 48.0f;
											break;
										}
										else if(error_count > 1)
											goto error;
									}
									else SetupForm.PAGain80 += diff_dBm;
									break;
								case 2:
									if(SetupForm.PAGain60 + diff_dBm < 39.0)
									{
										error_count++;
										if(error_count == 1)
										{
											SetupForm.PAGain60 = 48.0f;
											break;
										}
										else if(error_count > 1)
											goto error;
									}
									else SetupForm.PAGain60 += diff_dBm;
									break;
								case 3:
									if(SetupForm.PAGain40 + diff_dBm < 39.0)
									{
										error_count++;
										if(error_count == 1)
										{
											SetupForm.PAGain40 = 48.0f;
											break;
										}
										else if(error_count > 1)
											goto error;
									}
									else SetupForm.PAGain40 += diff_dBm;
									break;
								case 4:
									if(SetupForm.PAGain30 + diff_dBm < 39.0)
									{
										error_count++;
										if(error_count == 1)
										{
											SetupForm.PAGain30 = 48.0f;
											break;
										}
										else if(error_count > 1)
											goto error;
									}
									else SetupForm.PAGain30 += diff_dBm;
									break;
								case 5:
									if(SetupForm.PAGain20 + diff_dBm < 39.0)
									{
										error_count++;
										if(error_count == 1)
										{
											SetupForm.PAGain20 = 48.0f;
											break;
										}
										else if(error_count > 1)
											goto error;
									}
									else SetupForm.PAGain20 += diff_dBm;
									break;
								case 6:
									if(SetupForm.PAGain17 + diff_dBm < 39.0)
									{
										error_count++;
										if(error_count == 1)
										{
											SetupForm.PAGain17 = 48.0f;
											break;
										}
										else if(error_count > 1)
											goto error;
									}
									else SetupForm.PAGain17 += diff_dBm;
									break;
								case 7:
									if(SetupForm.PAGain15 + diff_dBm < 39.0)
									{
										error_count++;
										if(error_count == 1)
										{
											SetupForm.PAGain15 = 48.0f;
											break;
										}
										else if(error_count > 1)
											goto error;
									}
									else SetupForm.PAGain15 += diff_dBm;
									break;
								case 8:
									if(SetupForm.PAGain12 + diff_dBm < 39.0)
									{
										error_count++;
										if(error_count == 1)
										{
											SetupForm.PAGain12 = 48.0f;
											break;
										}
										else if(error_count > 1)
											goto error;
									}
									else SetupForm.PAGain12 += diff_dBm;
									break;
								case 9:
									if(SetupForm.PAGain10 + diff_dBm < 39.0)
									{
										error_count++;
										if(error_count == 1)
										{
											SetupForm.PAGain10 = 48.0f;
											break;
										}
										else if(error_count > 1)
											goto error;
									}
									else SetupForm.PAGain10 += diff_dBm;
									break;
							}						
						}
						else good_result = true;
						for(int j=0; j<40; j++)
						{
							Thread.Sleep(100);
							if(!progress.Visible)
								goto end;
						}
					}				
				}
				progress.SetPercent((float)((float)(i+1)/10));
			}

			ret_val = true;
			
			end:
				progress.Hide();
			chkMOX.Checked = false;
			Audio.CurrentAudioState1 = Audio.AudioState.DTTSP;
			CurrentDisplayMode = display_mode;					// restore display mode
			CurrentDSPMode = dsp_mode;							// restore dsp mode
			VFOAFreq = vfo_freq;								// restore frequency
			txtVFOAFreq_LostFocus(this, new EventArgs());
			display_delay = screen_delay;						// restore FPS
			PWR = pwr;											// restore pwr level
			meter_delay = multimeter_delay;						// restore meter delay

			calibrating = false;

			//t1.Stop();
			//MessageBox.Show(t1.Duration.ToString());
			return ret_val;

			error:
				MessageBox.Show("Calculated gain is invalid.  Please double check connections and try again.\n" +
					"If this problem persists, contact eric@flex-radio.com for support.",
					"Invalid Gain Found",
					MessageBoxButtons.OK,
					MessageBoxIcon.Error);
			goto end;
		}

		public bool LowPowerPASweep(Progress progress, int power) // calibrate PA Gain values
		{
			//			HiPerfTimer t1 = new HiPerfTimer();
			//			t1.Start();

			bool ret_val = false;

			if(!chkPower.Checked)
			{
				MessageBox.Show("Power must be on in order to calibrate.", "Power Is Off",
					MessageBoxButtons.OK, MessageBoxIcon.Stop);
				return false;
			}

			calibrating = true;

			int screen_delay = display_delay;				// save current FPS
			display_delay = 333;

			DSPMode dsp_mode = CurrentDSPMode;				// save current DSP Mode
			CurrentDSPMode = DSPMode.USB;					// set DSP Mode to USB

			DisplayMode display_mode = CurrentDisplayMode;	// save current display mode
			CurrentDisplayMode = DisplayMode.OFF;			// set display mode to off

			double vfo_freq = VFOAFreq;						// save current frequency
			
			int pwr = PWR;									// save current pwr level
			PWR = power;										// set pwr level to 100W

			int multimeter_delay = meter_delay;				// save current meter delay
			meter_delay = 1000;								// set meter delay to 1 seconds

			progress.SetPercent(0.0f);

			float[] band_freqs = { 1.9f, 3.75f, 5.3665f, 7.15f, 10.125f, 14.175f, 18.1f, 21.225f, 24.9f, 28.85f };
				
			for(int i=0; i<band_freqs.Length; i++)
			{
				VFOAFreq = band_freqs[i];				// set frequency
				Audio.CurrentAudioState1 = Audio.AudioState.SINL_COSR;
				chkMOX.Checked = true;				
				for(int j=0; j<30; j++)
				{
					Thread.Sleep(100);
					if(!progress.Visible)
						goto end;
				}
				chkMOX.Checked = false;
				Audio.CurrentAudioState1 = Audio.AudioState.DTTSP;

				if(!progress.Visible)
					goto end;

				for(int j=0; j<40; j++)
				{
					Thread.Sleep(100);
					if(!progress.Visible)
						goto end;
				}
				if(!progress.Visible)
					goto end;				

				progress.SetPercent((float)((float)(i+1)/10));
			}

			ret_val = true;
			
			end:
				progress.Hide();
			chkMOX.Checked = false;
			Audio.CurrentAudioState1 = Audio.AudioState.DTTSP;
			CurrentDisplayMode = display_mode;					// restore display mode
			CurrentDSPMode = dsp_mode;							// restore dsp mode
			VFOAFreq = vfo_freq;								// restore frequency
			txtVFOAFreq_LostFocus(this, new EventArgs());
			display_delay = screen_delay;						// restore FPS
			PWR = pwr;											// restore pwr level
			meter_delay = multimeter_delay;						// restore meter delay
			
			calibrating = false;

			//t1.Stop();
			//MessageBox.Show(t1.Duration.ToString());
			return ret_val;
		}

		public bool CalibrateSoundCard(Progress progress, int card)
		{
			if(!chkPower.Checked)
			{
				MessageBox.Show("Power must be on in order to calibrate.", "Power Is Off",
					MessageBoxButtons.OK, MessageBoxIcon.Stop);
				return false;
			}

			progress.SetPercent(0.0f);
			Mixer.SetMainVolume(mixer_id1, 100);
			Mixer.SetWaveOutVolume(mixer_id1, 100);
			double volume = DttSP.TXVolume;			// save current TX volume
			DttSP.TXVolume = 1.0;					// set volume to max
			double radio_vol = Audio.RadioVolume;
			Audio.RadioVolume = 1.0;

			if(card == 1)
				Audio.CurrentAudioState1 = Audio.AudioState.SINL_COSR;	// Start sending tone
			else if(card == 2)
				Audio.CurrentAudioState2 = Audio.AudioState.SINL_COSR;				
			
			progress.Focus();
            
			while(progress.Visible == true)			// keep sending tone until abort is pressed
				Thread.Sleep(100);

			if(card == 1)
				Audio.CurrentAudioState1 = Audio.AudioState.DTTSP;		// stop sending tone
			else if(card == 2)
				Audio.CurrentAudioState2 = Audio.AudioState.NOL_NOR;

			DttSP.TXVolume = volume;				// restore TX volume
			Audio.RadioVolume = radio_vol;			// restore radio volume

			return true;
		}

		public bool CalibrateSecondSoundCard(Progress progress)
		{
			if(!chkPower.Checked)
			{
				MessageBox.Show("Power must be on in order to calibrate.", "Power Is Off",
					MessageBoxButtons.OK, MessageBoxIcon.Stop);
				return false;
			}

			bool retval = false;

			progress.SetPercent(0.0f);
			Mixer.SetMainVolume(mixer_id1, 100);
			Mixer.SetWaveOutVolume(mixer_id1, 100);
			Mixer.SetMainVolume(mixer_id2, 100);
			Mixer.SetWaveOutVolume(mixer_id2, 100);

			double volume = DttSP.RXVolume;			// save current TX volume
			DttSP.RXVolume = 1.0;					// set volume to max

			Audio.AudioState audio_state1 = Audio.CurrentAudioState1;
			Audio.CurrentAudioState1 = Audio.AudioState.SINL_COSR;

			Audio.AudioState audio_state2 = Audio.CurrentAudioState2;
			Audio.CurrentAudioState2 = Audio.AudioState.NOL_NOR;

			int line_in = Mixer.GetLineInRecordVolume(mixer_id2);
			Mixer.SetLineInRecordVolume(mixer_id2, 1);

			bool line_in_mute = Mixer.GetLineInMute(mixer_id2);
			Mixer.SetLineInMute(mixer_id2, false);

			int line_in_gain = 1;

			for(int i = 100; i > 0; i--)
			{
				DttSP.RXVolume = (double)i/100;
				Thread.Sleep(100);
				
				while(Audio.Peak < 0.999 && line_in_gain < 100)
				{
					Mixer.SetLineInRecordVolume(mixer_id2, ++line_in_gain);
					Thread.Sleep(100);
					if(!progress.Visible)
						goto end;			
				}				

				if(line_in_gain > 100) line_in_gain = 100;
				sc2_cal[i] = line_in_gain;
				Debug.WriteLine("sc2_cal["+i+"]: "+line_in_gain);
				progress.SetPercent((100-i)/(float)100);
				if(!progress.Visible)
					goto end;
			}

			retval = true;

			end:
				progress.Hide();
			DttSP.TXVolume = volume;						// restore TXVolume
			Audio.CurrentAudioState1 = audio_state1;		// restore audiostate1
			Audio.CurrentAudioState2 = audio_state2;		// restore audiostate2
			Mixer.SetLineInMute(mixer_id2, line_in_mute);	// restore line in mute
			//Mixer.SetLineInPlayVolume(mixer_id2, line_in);
			udAF_ValueChanged(this, new EventArgs());
			return retval;
		}

		#endregion

		#region Properties
		// ======================================================
		// Properties
		// ======================================================

		private bool rx_only = false;
		public bool RXOnly
		{
			get { return rx_only; }
			set
			{
				rx_only = value;
				if(current_dsp_mode != DSPMode.SPEC &&
					current_dsp_mode != DSPMode.DRM)
					chkMOX.Enabled = !rx_only;
				chkTUN.Enabled = !rx_only;
			}
		}

		private XVTRTRMode current_xvtr_tr_mode = XVTRTRMode.NEGATIVE;
		public XVTRTRMode CurrentXVTRTRMode
		{
			get { return current_xvtr_tr_mode; }
			set	
			{
				current_xvtr_tr_mode = value;
				switch(current_xvtr_tr_mode)
				{
					case XVTRTRMode.NEGATIVE:
						hw.XVTR_TR = true;			// Set to receive
						break;
					case XVTRTRMode.POSITIVE:
						hw.XVTR_TR = false;			// Set to receive
						break;
					case XVTRTRMode.NONE:
						hw.XVTR_TR = false;
						break;
				}
			}
		}

		private double dds_step_size = 0.0;
		private double corrected_dds_clock = 200.0;
		private double dds_clock_correction = 0.0;
		public double DDSClockCorrection
		{
			get	{ return dds_clock_correction; }
			set
			{
				dds_clock_correction = value;
				corrected_dds_clock = 200.0 + dds_clock_correction;
				dds_step_size = corrected_dds_clock / Math.Pow(2, 48);
				DDSFreq = dds_freq;
			}
		}

		private BandPlan current_band_plan = BandPlan.IARU1;
		public BandPlan CurrentBandPlan
		{
			get { return current_band_plan; }
			set { current_band_plan = value; }
		}

		private bool spur_reduction = true;
		public bool SpurReduction
		{
			get { return spur_reduction; }
			set
			{
				spur_reduction = value;
				if(!spur_reduction)
				{
					if(current_dsp_mode == DSPMode.SPEC)
						DttSP.ChangeOsc(0.0);
					else
						DttSP.ChangeOsc(-11025.0);
				}
				DDSFreq = dds_freq;
			}
		}

		private double dds_freq = 7.0;
		public double DDSFreq
		{
			get { return dds_freq; }
			set
			{
				dds_freq = value;
				double vfoFreq = value, f = value;

				//calculate DDS Tuning Word
				if(xvtr_present && f >= 144 && f <= 146)		// If transverter enabled compute 28MHz IF frequency
					f -= 116;									// Subtract 116MHz (144-28) from VFO display frequency

				if(if_shift)
					f -= if_freq;								// adjust for IF shift

				f += vfo_offset;								// adjust for vfo offset

				long tuning_word = (long)(f / corrected_dds_clock * Math.Pow(2, 48));

				if(spur_reduction)
				{
					long sr_tuning_word = (tuning_word &		// start with current tuning word
						~(0x80007fffffff)) | 0x000080000000;	// clear first bit, low 31 bits; set bit 31

					double software_offset = (sr_tuning_word - tuning_word) * dds_step_size;

					double dsp_osc_freq = 0;
					if(if_shift)								//Convert the tuning fraction to rel frq
						dsp_osc_freq = 1000000.0*(software_offset) - 11025.0;
					else
						dsp_osc_freq = 1000000.0* (software_offset);
					DttSP.ChangeOsc(dsp_osc_freq);

					tuning_word = sr_tuning_word;
				}

				hw.DDSTuningWord = tuning_word;		
				SetHWFilters(dds_freq);
			}
		}

		private double min_freq = 0.011025;
		public double MinFreq
		{
			get { return min_freq; }
			set { min_freq = value; }
		}

		private double max_freq = 65.0;
		public double MaxFreq
		{
			get { return max_freq; }
			set { max_freq = value; }
		}

		private double vfo_offset = 0.0;
		public double VFOOffset
		{
			get	{ return vfo_offset; }
			set	{ vfo_offset = value; }
		}

		private double if_freq = 0.011025;
		public double IFFreq
		{
			get { return if_freq; }
			set { if_freq = value; }
		}

		private bool if_shift = true;
		public bool IFShift
		{
			get { return if_shift; }
			set { if_shift = value; }
		}

		private bool extended = false;
		public bool Extended
		{
			get { return extended; }
			set { extended = value; }
		}

		private bool enable_LPF0 = false;
		public bool EnableLPF0
		{
			get { return enable_LPF0; }
			set { enable_LPF0 = value; }
		}

		private int latch_delay = 0;
		public int LatchDelay
		{
			get { return latch_delay;}
			set { latch_delay = value;}
		}

		private bool x2_enabled = false;
		public bool X2Enabled
		{
			get { return x2_enabled; }
			set { x2_enabled = true; }
		}

		private int x2_delay = 500;
		public int X2Delay
		{
			get { return x2_delay; }
			set { x2_delay = value; }
		}

		public bool COMP
		{
			get { return chkDSPComp.Checked; }
			set { chkDSPComp.Checked = value; }
		}

		private HorizontalAlignment spectrum_db_align = HorizontalAlignment.Center;
		public HorizontalAlignment SpectrumdBAlign
		{
			get { return spectrum_db_align; }
			set { spectrum_db_align = value; }
		}

		// props for cat control 
		private SDRSerialPort.Parity cat_parity;
		public SDRSerialPort.Parity CATParity 
		{
			set {cat_parity = value; }
			get { return cat_parity; }
		}


		private SDRSerialPort.StopBits cat_stop_bits; 
		public SDRSerialPort.StopBits CATStopBits 
		{
			set { cat_stop_bits = value; }
			get { return cat_stop_bits; } 
		}
		private SDRSerialPort.DataBits cat_data_bits; 
		public SDRSerialPort.DataBits CATDataBits 
		{
			set { cat_data_bits = value; }
			get { return cat_data_bits; } 
		}
		private int cat_baud_rate; 
		public int CATBaudRate 
		{
			set { cat_baud_rate = value; } 
			get { return cat_baud_rate; } 
		}

		private bool cat_enabled; 
		public bool CATEnabled 
		{
			set 
			{
				cat_enabled = value; 
				if ( siolisten != null )  // if we've got a listener tell them about state change 
				{ 
					if ( cat_enabled ) 
					{ 
						siolisten.enableCAT(); 
					}
					else 
					{
						siolisten.disableCAT(); 
					}
				}
			}
			get { return cat_enabled; } 
		}

		private bool cat_mixw_virt_port; // true if port is a mixw port 
		public bool CATisMixWVirtPort 
		{
			get { return cat_mixw_virt_port; } 
			set { cat_mixw_virt_port = value; } 
		}
		
		private int cat_port; 
		public int CATPort 
		{
			get { return cat_port; }
			set { cat_port = value; } 
		}

		private bool cat_ptt_rts = false; 
		public bool CATPTTRTS 
		{
			get { return cat_ptt_rts; }
			set { cat_ptt_rts = value; }
		}

		private bool cat_ptt_dtr;
		public bool CATPTTDTR
		{
			get { return cat_ptt_dtr; }
			set { cat_ptt_dtr = value; }
		}

		private SerialPortPTT serialPTT = null; 
		private bool ptt_bit_bang_enabled; 
		public bool PTTBitBangEnabled 
		{
			get { return ptt_bit_bang_enabled; }
			set 
			{
				ptt_bit_bang_enabled = value; 
				if ( serialPTT != null )  // kill current serial PTT if we have one 
				{ 
					serialPTT.Destroy(); 
					serialPTT = null; 
				} 
				if ( ptt_bit_bang_enabled )  
				{ 
					// wjt -- don't really like popping a msg box in here ...   nasty when we do a remoted 
					// setup ... will let that wait for the great console refactoring 
					try 
					{ 
						serialPTT = new SerialPortPTT(ptt_bit_bang_port, cat_ptt_rts, cat_ptt_dtr); 
						serialPTT.Init(); 						
					}
					catch ( Exception ex ) 
					{
						ptt_bit_bang_enabled = false; 
						if ( SetupForm != null ) 
						{ 
							SetupForm.copyCATPropsToDialogVars(); // need to make sure the props on the setup page get reset 
						}
						MessageBox.Show("Could not initialize PTT Bit Bang control.  Exception was:\n\n " + ex.Message + 
							"\n\nPTT Bit Bang control has been disabled.", "Error Initializing PTT control", 
							MessageBoxButtons.OK, MessageBoxIcon.Error);
 
					}
				} 
			}
		}


		private int ptt_bit_bang_port; 
		public int PTTBitBangPort 
		{
			get { return ptt_bit_bang_port; }
			set { ptt_bit_bang_port = value; }
		}

		private bool notify_on_beta = true;
		public bool NotifyOnBeta
		{
			get { return notify_on_beta; }
			set { notify_on_beta = value; }
		}

		private bool notify_on_release = true;
		public bool NotifyOnRelease
		{
			get { return notify_on_release; }
			set { notify_on_release = value; }
		}

		private int display_buffer_size = 4096;
		//		private int DisplayBufferSize			// needs synchronization to prevent crashing
		//		{
		//			get { return display_buffer_size; }
		//			set
		//			{
		//				display_buffer_size = value;
		//				if(display_data.Length != display_buffer_size)
		//				{
		//					float[] temp = new float[display_buffer_size];
		//					for(int i=0; i<display_buffer_size; i++)
		//						temp[i] = display_data[i];
		//					display_data = temp;
		//				}
		//			}
		//		}

		private bool use_old_swr = false;
		public bool UseOldSWR
		{
			get { return use_old_swr; }
			set { use_old_swr = value; }
		}

		private int tune_power;								// power setting to use when TUN button is pressed
		public int TunePower
		{
			get { return tune_power; }
			set
			{
				tune_power = value;
				if(SetupForm != null)
					SetupForm.TunePower = tune_power;

				if(chkTUN.Checked)
				{
					if(atu_present)
					{
						if((ATUTuneMode)comboTuneMode.SelectedIndex == ATUTuneMode.BYPASS)
							PWR = tune_power;
					}
					else PWR = tune_power;
				}
			}
		}

		private bool disable_swr_protection = false;
		public bool DisableSWRProtection
		{
			get { return disable_swr_protection; }
			set { disable_swr_protection = value; }
		}

		private int previous_pwr = 50;
		public int PreviousPWR
		{
			get { return previous_pwr; }
			set	{ previous_pwr = value; }
		}

		private bool no_hardware_offset = false;
		public bool NoHardwareOffset
		{
			get { return no_hardware_offset; }
			set
			{
				no_hardware_offset = value;
				comboPreamp_SelectedIndexChanged(this, new EventArgs());
			}
		}

		private Band current_band;
		private Band CurrentBand
		{
			get { return current_band; }
			set
			{
				Band old_band = current_band;
				current_band = value;
				if(current_band != old_band)
					udPWR_ValueChanged(this, new EventArgs());
			}
		}

		private DSPMode current_dsp_mode;
		public DSPMode CurrentDSPMode
		{
			get { return current_dsp_mode; }
			set
			{
				RadioButton r = null;
				switch(value)
				{
					case DSPMode.LSB:
						r = radModeLSB;
						break;
					case DSPMode.USB:
						r = radModeUSB;
						break;
					case DSPMode.DSB:
						r = radModeDSB;
						break;
					case DSPMode.CWL:
						r = radModeCWL;
						break;
					case DSPMode.CWU:
						r = radModeCWU;
						break;
					case DSPMode.FMN:
						r = radModeFMN;
						break;
					case DSPMode.AM:
						r = radModeAM;
						break;
					case DSPMode.SAM:
						r = radModeSAM;
						break;
					case DSPMode.SPEC:
						r = radModeSPEC;
						break;
					case DSPMode.RTTY:
						r = radModeRTTY;
						break;
					case DSPMode.PSK:
						r = radModePSK;
						break;
					case DSPMode.DRM:
						r = radModeDRM;
						break;
				}

				r.Checked = true;
			}
		}

		private Filter current_filter;
		public Filter CurrentFilter
		{
			get { return current_filter; }
			set
			{
					RadioButton r = null;
				switch(value)
				{
					case Filter.F6000:
						r = radFilter6000;
						break;
					case Filter.F4000:
						r = radFilter4000;
						break;
					case Filter.F2600:
						r = radFilter2600;
						break;
					case Filter.F2100:
						r = radFilter2100;
						break;
					case Filter.F1000:
						r = radFilter1000;
						break;
					case Filter.F500:
						r = radFilter500;
						break;
					case Filter.F250:
						r = radFilter250;
						break;
					case Filter.F100:
						r = radFilter100;
						break;
					case Filter.F50:
						r = radFilter50;
						break;
					case Filter.F25:
						r = radFilter25;
						break;
					case Filter.VAR1:
						r = radFilterVar1;
						break;
					case Filter.VAR2:
						r = radFilterVar2;
						break;
					case Filter.NONE:
						foreach(Control c in grpFilter.Controls)
						{
							if(c.GetType() == typeof(RadioButton))
							{
								((RadioButton)c).Checked = false;

								if(c.BackColor != SystemColors.Control)
								{
									((RadioButton)c).BackColor = SystemColors.Control;
								}
							}
						}
						current_filter = Filter.NONE;
						break;
				}

				if(r != null)
				{
					if(r.Checked)
					{
						r.Checked = false;
					}

					r.Checked = true;
				}
			}
		}

		private MeterRXMode current_meter_rx_mode = MeterRXMode.SIGNAL_STRENGTH;
		public MeterRXMode CurrentMeterRXMode
		{
			get { return current_meter_rx_mode; }
			set
			{
				string text = "";
				switch(value)
				{
					case MeterRXMode.SIGNAL_STRENGTH:
						text = "Signal";
						break;
					case MeterRXMode.SIGNAL_AVERAGE:
						text = "Sig Avg";
						break;
					case MeterRXMode.ADC_L:
						text = "ADC L";
						break;
					case MeterRXMode.ADC_R:
						text = "ADC R";
						break;
				}

				if(text == "") return;

				comboMeterRXMode.Text = text;
			}
		}

		private MeterTXMode current_meter_tx_mode = MeterTXMode.FIRST;
		public MeterTXMode CurrentMeterTXMode
		{
			get { return current_meter_tx_mode; }
			set
			{
				string text = "";
				switch(value)
				{
					case MeterTXMode.ALC:
						text = "ALC";
						break;
					case MeterTXMode.FORWARD_POWER:
						text = "Fwd Pow";
						break;
					case MeterTXMode.PEAK_POWER:
						text = "Peak Pow";
						break;
					case MeterTXMode.REVERSE_POWER:
						text = "Rev Pow";
						break;
					case MeterTXMode.SWR:
						text = "SWR";
						break;
				}
				if(text == "") return;

				comboMeterTXMode.Text = text;
			}
		}

		private bool wave_playback = false;
		public bool WavePlayback
		{
			get { return wave_playback; }
			set
			{
				wave_playback = value;
				if(wave_playback)
				{

				}					
			}
		}

		private int cw_pitch = 600;
		public int CWPitch
		{
			get { return cw_pitch; }
			set
			{
				int diff = value - cw_pitch;
				cw_pitch = value;
				CWForm.CWPitch = value;
				Audio.SineFreq1 = value;

				if(current_dsp_mode == DSPMode.CWL ||		// if in CW Mode
					current_dsp_mode == DSPMode.CWU)
				{												// recalculate filter
					txtVFOAFreq_LostFocus(this, new EventArgs());
					CurrentFilter = current_filter;
				}
			}
		}

		private Color waterfall_low_color = Color.Black;
		public Color WaterfallLowColor
		{
			get { return waterfall_low_color; }
			set { waterfall_low_color = value; }
		}

		private Color waterfall_mid_color = Color.Red;
		public Color WaterfallMidColor
		{
			get { return waterfall_mid_color; }
			set { waterfall_mid_color = value; }
		}

		private Color waterfall_high_color = Color.Yellow;
		public Color WaterfallHighColor
		{
			get { return waterfall_high_color; }
			set { waterfall_high_color = value; }
		}

		private float waterfall_high_threshold = -80.0F;
		public float WaterfallHighThreshold
		{
			get { return waterfall_high_threshold; }
			set { waterfall_high_threshold = value; }
		}

		private float waterfall_low_threshold = -130.0F;
		public float WaterfallLowThreshold
		{
			get { return waterfall_low_threshold; }
			set { waterfall_low_threshold = value; }
		}

		private int histogram_hang_time = 100;
		public int HistogramHangTime
		{
			get { return histogram_hang_time; }
			set { histogram_hang_time = value; }
		}

		public double VFOAFreq
		{
			get 
			{
				try
				{
					return double.Parse(txtVFOAFreq.Text);
				}
				catch(Exception)
				{
					return 0;
				}
			}
			set
			{
				txtVFOAFreq.Text = value.ToString("f6");
				txtVFOAFreq_LostFocus(this, new EventArgs());
			}
		}

		public double VFOBFreq
		{
			get 
			{
				try
				{
					return double.Parse(txtVFOBFreq.Text);
				}
				catch(Exception)
				{
					return 0;
				}
			}
			set
			{
				txtVFOBFreq.Text = value.ToString("f6");
				txtVFOBFreq_LostFocus(this, new EventArgs());
			}
		}

		public int PWR
		{
			get { return (int)udPWR.Value; }
			set
			{
				value = Math.Max(0, value);			// lower bound
				value = Math.Min(100, value);		// upper bound

				udPWR.Value = value;
				udPWR_ValueChanged(this, new EventArgs());
			}
		}

		public int AF
		{
			get { return (int)udAF.Value; }
			set
			{
				value = Math.Max(0, value);			// lower bound
				value = Math.Min(100, value);		// upper bound

				udAF.Value = value;
				udAF_ValueChanged(this, new EventArgs());
			}
		}

		public DisplayMode CurrentDisplayMode
		{
			get { return (DisplayMode)comboDisplayMode.SelectedIndex; }
			set	{ comboDisplayMode.SelectedIndex = (int)value; }
		}

		public bool DisplayAVG
		{
			get { return chkDisplayAVG.Checked; }
			set	{ chkDisplayAVG.Checked = value; }
		}

		public bool MOX
		{
			get { return chkMOX.Checked; }
			set	{ chkMOX.Checked = value; }
		}

		public bool MOXEnabled
		{
			get { return chkMOX.Enabled; }
			set	{ chkMOX.Enabled = value; }
		}

		public bool MON
		{
			get { return chkMON.Checked; }
			set	{ chkMON.Checked = value; }
		}

		public bool MUT
		{
			get { return chkMUT.Checked; }
			set	{ chkMUT.Checked = value; }
		}

		public bool TUN
		{
			get { return chkTUN.Checked; }
			set	{ chkTUN.Checked = value; }
		}

		public bool TUNEnabled
		{
			get { return chkTUN.Enabled; }
			set { chkTUN.Enabled = value; }
		}
		
		public int FilterLowValue
		{
			get { return (int)udFilterLow.Value; }
			set { udFilterLow.Value = value; }
		}

		public int FilterHighValue
		{
			get { return (int)udFilterHigh.Value; }
			set	{ udFilterHigh.Value = value; }
		}

		public int FilterShiftValue
		{
			get { return tbFilterShift.Value; }
			set	{ tbFilterShift.Value = value; }
		}

		private PreampMode current_preamp_mode = PreampMode.MED;
		public PreampMode CurrentPreampMode
		{
			get { return current_preamp_mode; }
			set
			{
				current_preamp_mode = value;
				switch(current_preamp_mode)
				{
					case PreampMode.OFF:
						if(rfe_present)
						{
							hw.Attn = true;
							hw.GainRelay = true;	// 0dB
						}
						break;
					case PreampMode.LOW:
						if(rfe_present)
						{
							hw.Attn = false;
							hw.GainRelay = true;	// 0dB
						}
						break;
					case PreampMode.MED:
						if(rfe_present)
						{
							hw.Attn = true;
						}
						hw.GainRelay = false;	// 26dB
						break;
					case PreampMode.HIGH:
						if(rfe_present)
						{
							hw.Attn = false;
							hw.GainRelay = false;
						}
						else
						{
							hw.GainRelay = true;
						}
						break;
				}
			}
		}

		public int Squelch
		{
			get { return (int)udSquelch.Value; }
			set	{ udSquelch.Value = value; }
		}

		public int StepSize
		{
			get { return wheel_tune_index; }
		}

		public AGCMode CurrentAGCMode
		{
			get { return (AGCMode)comboAGC.SelectedIndex; }
			set	{ comboAGC.SelectedIndex = (int)value; }
		}

		public bool VFOSplit
		{
			get { return chkVFOSplit.Checked; }
			set	{ chkVFOSplit.Checked = value; }
		}

		public bool RIT
		{
			get { return chkRIT.Checked; }
			set	{ chkRIT.Checked = value; }
		}

		public bool RITOn
		{
			get { return chkRIT.Checked; }
			set { chkRIT.Checked = value; }
		}

		public int RITValue
		{
			get { return (int)udRIT.Value; }
			set	{ udRIT.Value = value; }
		}

		public bool XITOn
		{
			get { return chkXIT.Checked; }
			set	{ chkXIT.Checked = value; }
		}

		public int XITValue
		{
			get { return (int)udXIT.Value; }
			set	{ udXIT.Value = value; }
		}

		private int tx_filter_high = 3000;
		public int TXFilterHigh
		{
			get{ return tx_filter_high; }
			set
			{
				tx_filter_high = value;
				DttSP.ChangeTXFilters(tx_filter_low, tx_filter_high);
				if(chkMOX.Checked)
				{
					if(current_display_mode == DisplayMode.PANADAPTER ||
						current_display_mode == DisplayMode.SPECTRUM ||
						current_display_mode == DisplayMode.HISTOGRAM ||
						current_display_mode == DisplayMode.WATERFALL)
						DrawBackground();
				}
			}
		}

		private int tx_filter_low = 300;
		public int TXFilterLow
		{
			get{ return tx_filter_low; }
			set
			{
				tx_filter_low = value;
				DttSP.ChangeTXFilters(tx_filter_low, tx_filter_high);
				if(chkMOX.Checked)
				{
					if(current_display_mode == DisplayMode.PANADAPTER ||
						current_display_mode == DisplayMode.SPECTRUM ||
						current_display_mode == DisplayMode.HISTOGRAM ||
						current_display_mode == DisplayMode.WATERFALL)
						DrawBackground();
				}
			}
		}

		private delegate void SetTimerDel(System.Windows.Forms.Timer t, bool enable);
		private void SetTimer(System.Windows.Forms.Timer t, bool enable)
		{
			t.Enabled = enable;
		}

		private bool high_swr = false;
		private bool display_high_swr = false;
		public bool HighSWR
		{
			get { return high_swr; }
			set
			{
				if(high_swr == value)
					return;

				high_swr = value;
				display_high_swr = value;
			}
		}

		private bool disable_ptt = false;
		public bool DisablePTT
		{
			get { return disable_ptt; }
			set { disable_ptt = value; }
		}

		public bool PowerOn
		{
			get { return chkPower.Checked; }
			set { chkPower.Checked = value;	}
		}

		public bool PowerEnabled
		{
			get { return chkPower.Enabled; }
			set { chkPower.Enabled = value; }
		}

		private bool second_sound_card_enabled = false;
		public bool SecondSoundCardEnabled
		{
			get { return second_sound_card_enabled; }
			set { second_sound_card_enabled = value; }
		}

		private int audio_driver_index1 = 0;
		public int AudioDriverIndex1
		{
			get { return audio_driver_index1; }
			set { audio_driver_index1 = value; }
		}

		private int audio_driver_index2 = 0;
		public int AudioDriverIndex2
		{
			get { return audio_driver_index2; }
			set { audio_driver_index2 = value; }
		}

		private int audio_input_index1 = 0;
		public int AudioInputIndex1
		{
			get { return audio_input_index1; }
			set { audio_input_index1 = value; }
		}

		private int audio_input_index2 = 0;
		public int AudioInputIndex2
		{
			get { return audio_input_index2; }
			set { audio_input_index2 = value; }
		}

		private int audio_output_index1 = 0;
		public int AudioOutputIndex1
		{
			get { return audio_output_index1; }
			set { audio_output_index1 = value; }
		}

		private int audio_output_index2 = 0;
		public int AudioOutputIndex2
		{
			get { return audio_output_index2; }
			set { audio_output_index2 = value; }
		}

		private int audio_latency1 = 120;
		public int AudioLatency1
		{
			get { return audio_latency1; }
			set { audio_latency1 = value; }
		}

		private int audio_latency2 = 120;
		public int AudioLatency2
		{
			get { return audio_latency2; }
			set { audio_latency2 = value; }
		}

		private double audio_volts1 = 2.23;
		public double AudioVolts1
		{
			get { return audio_volts1; }
			set { audio_volts1 = value; }
		}

		private double audio_volts2 = 2.23;
		public double AudioVolts2
		{
			get { return audio_volts2; }
			set { audio_volts2 = value; }
		}

		private int mixer_id1 = 0;
		public int MixerID1
		{
			get { return mixer_id1; }
			set { mixer_id1 = value; }
		}

		private int mixer_id2 = 0;
		public int MixerID2
		{
			get { return mixer_id2; }
			set { mixer_id2 = value; }
		}

		private int mixer_rx_mux_id1 = 0;
		public int MixerRXMuxID1
		{
			get { return mixer_rx_mux_id1; }
			set	{ mixer_rx_mux_id1 = value;	}
		}

		private int mixer_tx_mux_id1 = 0;
		public int MixerTXMuxID1
		{
			get { return mixer_tx_mux_id1; }
			set { mixer_tx_mux_id1 = value; }
		}

		private int mixer_rx_mux_id2 = 0;
		public int MixerRXMuxID2
		{
			get { return mixer_rx_mux_id2; }
			set { mixer_rx_mux_id2 = value; }
		}

		private int mixer_tx_mux_id2 = 0;
		public int MixerTXMuxID2
		{
			get { return mixer_tx_mux_id2; }
			set { mixer_tx_mux_id2 = value; }
		}

		private int sample_rate1 = 48000;
		public int SampleRate1
		{
			get { return sample_rate1; }
			set { sample_rate1 = value; }
		}

		private int sample_rate2 = 48000;
		public int SampleRate2
		{
			get { return sample_rate2; }
			set { sample_rate2 = value; }
		}

		private int block_size1;
		public int BlockSize1
		{
			get { return block_size1; }
			set
			{
				block_size1 = value;
				CWForm.CWBlockSize = block_size1;
			}
		}

		private int block_size2;
		public int BlockSize2
		{
			get { return block_size2; }
			set { block_size2 = value; }
		}

		private bool cw_key_mode = false;
		public bool CWKeyMode
		{
			get { return cw_key_mode; }
			set { cw_key_mode = value; }
		}

		private float average_mult_avg = 1 - (float)1/8;
		private float average_mult_samp = (float)1/8;
		private int average_samples = 8;
		public int AverageSamples
		{
			get { return average_samples; }
			set
			{
				average_samples = value;
				average_mult_avg = 1 - (float)1/average_samples;
				average_mult_samp = (float)1/average_samples;
			}
		}

		private int spectrum_grid_max = 0;
		public int SpectrumGridMax
		{
			get{ return spectrum_grid_max;}
			set
			{	
				spectrum_grid_max = value;
				DrawBackground();
			}
		}

		private int spectrum_grid_min = -150;
		public int SpectrumGridMin
		{
			get{ return spectrum_grid_min;}
			set
			{
				spectrum_grid_min = value;
				DrawBackground();
			}
		}

		private int spectrum_grid_step = 10;
		public int SpectrumGridStep
		{
			get{ return spectrum_grid_step;}
			set
			{
				spectrum_grid_step = value;
				DrawBackground();
			}
		}

		private int peak_text_delay = 500;
		public int PeakTextDelay
		{
			get { return peak_text_delay; }
			set
			{
				peak_text_delay = value;
				timer_peak_text.Interval = value;
			}
		}

		private int meter_delay = 100;
		public int MeterDelay
		{
			get { return meter_delay; }
			set
			{
				meter_delay = value;
				MultimeterPeakHoldTime = MultimeterPeakHoldTime;
			}
		}

		private int cpu_meter_delay = 1000;
		public int CPUMeterDelay
		{
			get { return cpu_meter_delay; }
			set
			{
				cpu_meter_delay = value;
				timer_cpu_meter.Interval = value;
			}
		}

		private int display_fps = 15;
		private int display_delay = 1000 / 15;
		public int DisplayFPS
		{
			get{ return display_fps;}
			set
			{
				display_fps = value;
				display_delay = 1000 / display_fps;
			}
		}

		private int multimeter_peak_hold_time = 1000;
		private int multimeter_peak_hold_samples = 10;
		public int MultimeterPeakHoldTime
		{
			get { return multimeter_peak_hold_time; }
			set
			{
				multimeter_peak_hold_time = value;
				multimeter_peak_hold_samples = value / meter_delay;
			}
		}

		private int multimeter_text_peak_time = 500;
		private int multimeter_text_peak_samples = 5;
		public int MultimeterTextPeakTime
		{
			get { return multimeter_text_peak_time; }
			set
			{
				multimeter_text_peak_time = value;
				multimeter_text_peak_samples = value / meter_delay;
				if(multimeter_text_peak_samples > meter_text_history.Length)
				{
					float[] temp = new float[multimeter_text_peak_samples];
					for(int i=0; i<meter_text_history.Length; i++)
						temp[i] = meter_text_history[i];
					meter_text_history = temp;
				}
			}
		}

		private int phase_num_pts = 100;
		public int PhaseNumPts
		{
			get{ return phase_num_pts;}
			set{ phase_num_pts = value;}
		}

		private Color vfo_text_light_color = Color.Yellow;
		public Color VFOTextLightColor
		{
			get { return vfo_text_light_color; }
			set
			{
				if(txtVFOAFreq.ForeColor == vfo_text_light_color)
					txtVFOAFreq.ForeColor = value;
				if(txtVFOBFreq.ForeColor == vfo_text_light_color)
					txtVFOBFreq.ForeColor = value;
				vfo_text_light_color = value;
			}
		}

		private Color vfo_text_dark_color = Color.Olive;
		public Color VFOTextDarkColor
		{
			get { return vfo_text_dark_color; }
			set
			{
				if(txtVFOAFreq.ForeColor == vfo_text_dark_color)
					txtVFOAFreq.ForeColor = value;
				if(txtVFOBFreq.ForeColor == vfo_text_dark_color)
					txtVFOBFreq.ForeColor = value;
				vfo_text_dark_color = value;
			}
		}

		private Color band_text_light_color = Color.Lime;
		public Color BandTextLightColor
		{
			get { return band_text_light_color; }
			set
			{
				if(txtVFOABand.ForeColor == band_text_light_color)
					txtVFOABand.ForeColor = value;
				if(txtVFOBBand.ForeColor == band_text_light_color)
					txtVFOBBand.ForeColor = value;
				band_text_light_color = value;
			}
		}

		private Color band_text_dark_color = Color.Green;
		public Color BandTextDarkColor
		{
			get { return band_text_dark_color; }
			set
			{
				if(txtVFOABand.ForeColor == band_text_dark_color)
					txtVFOABand.ForeColor = value;
				if(txtVFOBBand.ForeColor == band_text_dark_color)
					txtVFOBBand.ForeColor = value;
				band_text_dark_color = value;
			}
		}

		private Color peak_text_color = Color.DodgerBlue;
		public Color PeakTextColor
		{
			get { return peak_text_color; }
			set
			{
				peak_text_color = value;
				txtDisplayCurPos.ForeColor = value;
				txtDisplayPeak.ForeColor = value;
			}
		}

		private Color button_selected_color = Color.Yellow;
		public Color ButtonSelectedColor
		{
			get { return button_selected_color; }
			set
			{
				button_selected_color = value;
				CheckSelectedButtonColor();
			}
		}

		private Color meter_left_color = Color.Green;
		public Color MeterLeftColor
		{
			get { return meter_left_color; }
			set
			{
				meter_left_color = value;
				picMultiMeterDigital.Invalidate();
			}
		}

		private Color meter_right_color = Color.Lime;
		public Color MeterRightColor
		{
			get { return meter_right_color; }
			set
			{
				meter_right_color = value;
				picMultiMeterDigital.Invalidate();
			}
		}

		private Color grid_text_color = Color.Yellow;
		public Color GridTextColor
		{
			get{ return grid_text_color;}
			set
			{
				grid_text_color = value;
				DrawBackground();
			}
		}

		private Color grid_zero_color = Color.Red;
		public Color GridZeroColor
		{
			get{ return grid_zero_color;}
			set
			{
				grid_zero_color = value;
				DrawBackground();
			}
		}

		private Color grid_color = Color.Purple;
		public Color GridColor
		{
			get{ return grid_color;}
			set
			{
				grid_color = value;
				DrawBackground();
			}
		}

		private Color data_line_color = Color.LightGreen;
		public Color DataLineColor
		{
			get{ return data_line_color;}
			set
			{
				data_line_color = value;
				DrawBackground();
			}
		}

		private Color display_filter_color = Color.Green;
		public Color DisplayFilterColor
		{
			get { return display_filter_color; }
			set
			{
				display_filter_color = value;
				DrawBackground();
			}
		}

		private Color display_background_color = Color.Black;
		public Color DisplayBackgroundColor
		{
			get{ return display_background_color;}
			set
			{
				display_background_color = value;
				DrawBackground();
			}
		}		

		private float display_line_width = 1.0F;
		public float DisplayLineWidth
		{
			get{ return display_line_width;}
			set{ display_line_width = value;}
		}

		private Keys key_tune_up_1 = Keys.Q;
		public Keys KeyTuneUp1
		{
			get{ return key_tune_up_1;}
			set{ key_tune_up_1 = value;}
		}

		private Keys key_tune_down_1 = Keys.A;
		public Keys KeyTuneDown1
		{
			get{ return key_tune_down_1;}
			set{ key_tune_down_1 = value;}
		}

		private Keys key_tune_up_2 = Keys.W;
		public Keys KeyTuneUp2
		{
			get{ return key_tune_up_2;}
			set{ key_tune_up_2 = value;}
		}

		private Keys key_tune_down_2 = Keys.S;
		public Keys KeyTuneDown2
		{
			get{ return key_tune_down_2;}
			set{ key_tune_down_2 = value;}
		}

		private Keys key_tune_up_3 = Keys.E;
		public Keys KeyTuneUp3
		{
			get{ return key_tune_up_3;}
			set{ key_tune_up_3 = value;}
		}

		private Keys key_tune_down_3 = Keys.D;
		public Keys KeyTuneDown3
		{
			get{ return key_tune_down_3;}
			set{ key_tune_down_3 = value;}
		}

		private Keys key_tune_up_4 = Keys.R;
		public Keys KeyTuneUp4
		{
			get{ return key_tune_up_4;}
			set{ key_tune_up_4 = value;}
		}

		private Keys key_tune_down_4 = Keys.F;
		public Keys KeyTuneDown4
		{
			get{ return key_tune_down_4;}
			set{ key_tune_down_4 = value;}
		}

		private Keys key_tune_up_5 = Keys.T;
		public Keys KeyTuneUp5
		{
			get{ return key_tune_up_5;}
			set{ key_tune_up_5 = value;}
		}

		private Keys key_tune_down_5 = Keys.G;
		public Keys KeyTuneDown5
		{
			get{ return key_tune_down_5;}
			set{ key_tune_down_5 = value;}
		}

		private Keys key_tune_up_6 = Keys.Y;
		public Keys KeyTuneUp6
		{
			get{ return key_tune_up_6;}
			set{ key_tune_up_6 = value;}
		}

		private Keys key_tune_down_6 = Keys.H;
		public Keys KeyTuneDown6
		{
			get{ return key_tune_down_6;}
			set{ key_tune_down_6 = value;}
		}

		private Keys key_tune_up_7 = Keys.U;
		public Keys KeyTuneUp7
		{
			get{ return key_tune_up_7;}
			set{ key_tune_up_7 = value;}
		}

		private Keys key_tune_down_7 = Keys.J;
		public Keys KeyTuneDown7
		{
			get{ return key_tune_down_7;}
			set{ key_tune_down_7 = value;}
		}

		private Keys key_band_up = Keys.N;
		public Keys KeyBandUp
		{
			get{ return key_band_up;}
			set{ key_band_up = value;}
		}

		private Keys key_band_down = Keys.M;
		public Keys KeyBandDown
		{
			get{ return key_band_down;}
			set{ key_band_down = value;}
		}

		private Keys key_filter_up = Keys.B;
		public Keys KeyFilterUp
		{
			get{ return key_filter_up;}
			set{ key_filter_up = value;}
		}

		private Keys key_filter_down = Keys.V;
		public Keys KeyFilterDown
		{
			get{ return key_filter_down;}
			set{ key_filter_down = value;}
		}

		private Keys key_mode_up = Keys.X;
		public Keys KeyModeUp
		{
			get{ return key_mode_up;}
			set{ key_mode_up = value;}
		}

		private Keys key_mode_down = Keys.Z;
		public Keys KeyModeDown
		{
			get{ return key_mode_down;}
			set{ key_mode_down = value;}
		}

		private Keys key_cw_dot = Keys.OemPeriod;
		public Keys KeyCWDot
		{
			get { return key_cw_dot; }
			set { key_cw_dot = value; }
		}

		private Keys key_cw_dash = Keys.OemQuestion;
		public Keys KeyCWDash
		{
			get { return key_cw_dash; }
			set { key_cw_dash = value; }
		}

		public bool MemoryScanner
		{
			get { return chkMemoryScanner.Checked; }
			set
			{
				chkMemoryScanner.Checked = value;
			}
		}

		private bool rfe_present = true;
		public bool RFEPresent
		{
			get { return rfe_present;}
			set
			{
				rfe_present = value;
				hw.RFEPresent = value;
				hw.UpdateHardware = hw.UpdateHardware;
				if(value)
				{
					if(!comboPreamp.Items.Contains("Off"))
						comboPreamp.Items.Insert(0, "Off");
					if(!comboPreamp.Items.Contains("Low"))
						comboPreamp.Items.Insert(1, "Low");
				}
				else
				{
					if(comboPreamp.Items.Contains("Off"))
						comboPreamp.Items.Remove("Off");
					if(comboPreamp.Items.Contains("Low"))
						comboPreamp.Items.Remove("Low");
					if(comboPreamp.SelectedIndex < 0)
						comboPreamp.SelectedIndex = 0;
				}
			}
		}

		private bool xvtr_present = false;
		public bool XVTRPresent
		{
			get { return xvtr_present; }
			set 
			{
				xvtr_present = value;
				btnBand2.Enabled = value;
				hw.XVTRPresent = value;
				if(value)
					MaxFreq = 146.0;
				else
					MaxFreq = 65.0;
			}
		}

		private bool pa_present = false;
		public bool PAPresent
		{
			get { return pa_present; }
			set
			{
				pa_present = value;
				hw.PAPresent = value;
				if(pa_present && !comboMeterTXMode.Items.Contains("Rev Pow"))
				{
					comboMeterTXMode.Items.Add("Rev Pow");
					if(comboMeterTXMode.SelectedIndex < 0)
						comboMeterTXMode.SelectedIndex = 0;
				}
				else if(!pa_present && comboMeterTXMode.Items.Contains("Rev Pow"))
				{
					comboMeterTXMode.Items.Remove("Rev Pow");
					current_meter_tx_mode = MeterTXMode.ALC;
					if(comboMeterTXMode.SelectedIndex < 0 &&
						comboMeterTXMode.Items.Count > 0)
						comboMeterTXMode.SelectedIndex = 0;
				}
				udPWR_ValueChanged(this, new EventArgs());
			}
		}

		private bool atu_present = false;
		public bool ATUPresent
		{
			get { return atu_present; }
			set
			{
				atu_present = value;
				comboTuneMode.Visible = atu_present;
				comboTuneMode.SelectedIndex = 0;
			}
		}

		private bool usb_present = false;
		public bool USBPresent
		{
			get { return usb_present; }
			set
			{
				usb_present = value;
				hw.USBPresent = value;
			}
		}

		private static PerformanceCounter cpu_usage;
		public static float CpuUsage
		{
			get
			{
				if (cpu_usage == null)
				{
					cpu_usage = new PerformanceCounter(
						"Processor", "% Processor Time", "_Total", true);
				}
				return cpu_usage.NextValue(); 
			}
		}

		#endregion

		#region Drawing Routines
		// ======================================================
		// Drawing Routines
		// ======================================================

		private void DrawBackground()
		{
			// draws the background image for the display based
			// on the current selected display mode.

			int W = picDisplay.Width;
			int H = picDisplay.Height;

			switch(current_display_mode)
			{
				case DisplayMode.SPECTRUM:
					DrawSpectrumGrid(ref background_bmp, W, H);
					break;
				case DisplayMode.PANADAPTER:
					DrawPanadapterGrid(ref background_bmp, W, H);
					break;
				case DisplayMode.SCOPE:
					DrawScopeGrid(ref background_bmp, W, H);
					break;
				case DisplayMode.PHASE:
					DrawPhaseGrid(ref background_bmp, W, H);
					break;	
				case DisplayMode.PHASE2:
					DrawPhaseGrid(ref background_bmp, W, H);
					break;
				case DisplayMode.WATERFALL:
					DrawWaterfallGrid(ref background_bmp, W, H);
					break;
				case DisplayMode.HISTOGRAM:
					DrawSpectrumGrid(ref background_bmp, W, H);
					break;
				case DisplayMode.OFF:
					DrawOffBackground(ref background_bmp, W, H);
					break;
				default:
					break;
			}

			// if radio is Powered on, replace the background image
			if(!chkPower.Checked)
			{
				display_image_mutex.WaitOne();
				if(picDisplay.Image != null)
					picDisplay.Image.Dispose();
				picDisplay.Image = (Image)background_bmp.Clone();
				display_image_mutex.ReleaseMutex();
			}
		}

		private void DrawPhaseGrid(ref Bitmap b, int W, int H)
		{
			Bitmap bmp = new Bitmap(W, H);				// create bitmap
			Graphics g = Graphics.FromImage(bmp);		// create graphics object for drawing

			// draw background
			g.FillRectangle(new SolidBrush(display_background_color), 0, 0, W, H);	

			for(double i=0.50; i < 3; i+=.50)	// draw 3 concentric circles
			{
				g.DrawEllipse(new Pen(grid_color), (int)(W/2-H*i/2), (int)(H/2-H*i/2), (int)(H*i), (int)(H*i));
			}

			// save bitmap
			background_image_mutex.WaitOne();
			if(b != null) b.Dispose();
			b = (Bitmap)bmp.Clone();
			background_image_mutex.ReleaseMutex();
			bmp.Dispose();
			g.Dispose();
		}

		private void DrawScopeGrid(ref Bitmap b, int W, int H)
		{
			Bitmap bmp = new Bitmap(W, H);				// create bitmap
			Graphics g = Graphics.FromImage(bmp);		// create graphics object for drawing

			// draw background
			g.FillRectangle(new SolidBrush(display_background_color), 0, 0, W, H);

			g.DrawLine(new Pen(grid_color), 0, H/2, W, H/2);	// draw horizontal line
			g.DrawLine(new Pen(grid_color), W/2, 0, W/2, H);	// draw vertical line

			// save bitmap
			background_image_mutex.WaitOne();
			if(b != null) b.Dispose();
			b = (Bitmap)bmp.Clone();
			background_image_mutex.ReleaseMutex();
			bmp.Dispose();
			g.Dispose();
		}

		private void DrawSpectrumGrid(ref Bitmap b, int W, int H)
		{
			Bitmap bmp = new Bitmap(W, H);				// create new bmp
			Graphics g = Graphics.FromImage(bmp);		// create graphics object for drawing

			// draw background
			g.FillRectangle(new SolidBrush(display_background_color), 0, 0, W, H);

			int low = 0;								// init limit variables
			int high = 0;

			if(!chkMOX.Checked)
			{
				low = DttSP.RXDisplayLow;				// get RX display limits
				high = DttSP.RXDisplayHigh;
			}
			else
			{
				low = DttSP.TXDisplayLow;				// get TX display limits
				high = DttSP.TXDisplayHigh;
			}

			int mid_w = W/2;
			int[] step_list = {10, 20, 25, 50};
			int step_power = 1;
			int step_index = 0;
			int freq_step_size = 50;

			Font font = new Font("Arial", 9);
			SolidBrush grid_text_brush = new SolidBrush(grid_text_color);
			Pen grid_pen = new Pen(grid_color);
			int y_range = spectrum_grid_max - spectrum_grid_min;

			if(high == 0)
			{
				int f = -low;
				// Calculate horizontal step size
				while(f/freq_step_size > 7)
				{
					freq_step_size = step_list[step_index]*(int)Math.Pow(10.0, step_power);
					step_index = (step_index+1)%4;
					if(step_index == 0) step_power++;
				}
				float pixel_step_size = (float)(W*freq_step_size/f);

				int num_steps = f/freq_step_size;

				// Draw vertical lines
				for(int i=1; i<=num_steps; i++)
				{
					int x = W-(int)Math.Floor(i*pixel_step_size);	// for negative numbers
					
					g.DrawLine(grid_pen, x, 0, x, H);				// draw right line
				
					// Draw vertical line labels
					int num = i*freq_step_size;
					string label = num.ToString();
					int offset = (int)((label.Length+1)*4.1);
					if(x-offset >= 0)
						g.DrawString("-"+label, font, grid_text_brush, x-offset, (float)Math.Floor(H*.01));
				}

				// Draw horizontal lines
				int V = (int)(spectrum_grid_max - spectrum_grid_min);
				num_steps = V/spectrum_grid_step;
				pixel_step_size = H/num_steps;

				for(int i=1; i<num_steps; i++)
				{
					int xOffset = 0;
					int num = spectrum_grid_max - i*spectrum_grid_step;
					int y = (int)Math.Floor((spectrum_grid_max - num)*H/y_range);

					g.DrawLine(grid_pen, 0, y, W, y);

					// Draw horizontal line labels
					string label = num.ToString();
					int offset = (int)(label.Length*4.1);
					if(label.Length == 3)
						xOffset = (int)g.MeasureString("-", font).Width - 2;
					SizeF size = g.MeasureString(label, font);

					y -= 8;
//					int x = 0;
//					switch(spectrum_db_align)
//					{
//						case HorizontalAlignment.Left:
//							x = xOffset + 3;
//							break;
//						case HorizontalAlignment.Right:
//							x = (int)(W-size.Width);
//							break;
//						case HorizontalAlignment.Center:
//							x = W/2-xOffset;
//							break;
//					}

					if(y+9 < H)
						g.DrawString(label, font, grid_text_brush, W-size.Width, y);
				}

				// Draw middle vertical line
				g.DrawLine(new Pen(grid_zero_color), W-1, 0, W-1, H);
				g.DrawLine(new Pen(grid_zero_color), W-2, 0, W-2, H);
			}
			else if(low == 0)
			{
				int f = high;
				// Calculate horizontal step size
				while(f/freq_step_size > 7)
				{
					freq_step_size = step_list[step_index]*(int)Math.Pow(10.0, step_power);
					step_index = (step_index+1)%4;
					if(step_index == 0) step_power++;
				}
				float pixel_step_size = (float)(W*freq_step_size/f);
				int num_steps = f/freq_step_size;

				// Draw vertical lines
				for(int i=1; i<=num_steps; i++)
				{
					int x = (int)Math.Floor(i*pixel_step_size);// for positive numbers
					
					g.DrawLine(grid_pen, x, 0, x, H);			// draw right line
				
					// Draw vertical line labels
					int num = i*freq_step_size;
					string label = num.ToString();
					int offset = (int)(label.Length*4.1);
					if(x-offset+label.Length*7 < W)
						g.DrawString(label, font, grid_text_brush, x-offset, (float)Math.Floor(H*.01));
				}

				// Draw horizontal lines
				int V = (int)(spectrum_grid_max - spectrum_grid_min);
				int numSteps = V/spectrum_grid_step;
				pixel_step_size = H/numSteps;
				for(int i=1; i<numSteps; i++)
				{
					int xOffset = 0;
					int num = spectrum_grid_max - i*spectrum_grid_step;
					int y = (int)Math.Floor((spectrum_grid_max - num)*H/y_range);

					g.DrawLine(grid_pen, 0, y, W, y);

					// Draw horizontal line labels
					
					string label = num.ToString();
					if(label.Length == 3)
						xOffset = (int)g.MeasureString("-", font).Width - 2;
					int offset = (int)(label.Length*4.1);

					y -= 8;
					if(y+9 < H)
						g.DrawString(label, font, grid_text_brush, 3 + xOffset, y);
				}

				// Draw middle vertical line
				g.DrawLine(new Pen(grid_zero_color), 0, 0, 0, H);
				g.DrawLine(new Pen(grid_zero_color), 1, 0, 1, H);
			}
			if(low < 0 && high > 0)
			{
				int f = high;

				// Calculate horizontal step size
				while(f/freq_step_size > 4)
				{
					freq_step_size = step_list[step_index]*(int)Math.Pow(10.0, step_power);
					step_index = (step_index+1)%4;
					if(step_index == 0) step_power++;
				}
				int pixel_step_size = W/2*freq_step_size/f;
				int num_steps = f/freq_step_size;

				// Draw vertical lines
				for(int i=1; i<=num_steps; i++)
				{
					int xLeft = mid_w-(i*pixel_step_size);			// for negative numbers
					int xRight = mid_w+(i*pixel_step_size);		// for positive numbers
					g.DrawLine(grid_pen, xLeft, 0, xLeft, H);		// draw left line
					g.DrawLine(grid_pen, xRight, 0, xRight, H);		// draw right line
				
					// Draw vertical line labels
					int num = i*freq_step_size;
					string label = num.ToString();
					int offsetL = (int)((label.Length+1)*4.1);
					int offsetR = (int)(label.Length*4.1);
					if(xLeft-offsetL >= 0)
					{
						g.DrawString("-"+label, font, grid_text_brush, xLeft-offsetL, (float)Math.Floor(H*.01));
						g.DrawString(label, font, grid_text_brush, xRight-offsetR, (float)Math.Floor(H*.01));
					}
				}

				// Draw horizontal lines
				int V = (int)(spectrum_grid_max - spectrum_grid_min);
				int numSteps = V/spectrum_grid_step;
				pixel_step_size = H/numSteps;
				for(int i=1; i<numSteps; i++)
				{
					int xOffset = 0;
					int num = spectrum_grid_max - i*spectrum_grid_step;
					int y = (int)Math.Floor((spectrum_grid_max - num)*H/y_range);
					g.DrawLine(grid_pen, 0, y, W, y);

					// Draw horizontal line labels
					string label = num.ToString();
					if(label.Length == 3) xOffset = 7;
					int offset = (int)(label.Length*4.1);

					y -= 8;
					if(y+9 < H)
						g.DrawString(label, font, grid_text_brush, mid_w+2 + xOffset, y);
				}

				// Draw middle vertical line
				g.DrawLine(new Pen(grid_zero_color), mid_w, 0, mid_w, H);
				g.DrawLine(new Pen(grid_zero_color), mid_w-1, 0, mid_w-1, H);
			}

			// save bitmap
			background_image_mutex.WaitOne();
			if(b != null) b.Dispose();
			b = (Bitmap)bmp.Clone();
			background_image_mutex.ReleaseMutex();
			bmp.Dispose();
			g.Dispose();
		}

		private void DrawPanadapterGrid(ref Bitmap b, int W, int H)
		{
			Bitmap bmp = new Bitmap(W, H);				// create bitmap
			Graphics g = Graphics.FromImage(bmp);		// create graphics object for drawing

			// draw background
			g.FillRectangle(new SolidBrush(display_background_color), 0, 0, W, H);

			int low = -10000;					// initialize variables
			int high = 10000;
			int mid_w = W/2;
			int[] step_list = {10, 20, 25, 50};
			int step_power = 1;
			int step_index = 0;
			int freq_step_size = 50;

			Font font = new Font("Arial", 9);
			SolidBrush grid_text_brush = new SolidBrush(grid_text_color);
			Pen grid_pen = new Pen(grid_color);
			int y_range = spectrum_grid_max - spectrum_grid_min;
			int filter_low, filter_high;

			if(chkMOX.Checked)								// initialize right and left drawing limits
			{
				filter_low = DttSP.TXFilterLowCut;
				filter_high = DttSP.TXFilterHighCut;
			}
			else
			{
				filter_low = DttSP.RXFilterLowCut;
				filter_high = DttSP.RXFilterHighCut;
			}

			if(current_dsp_mode == DSPMode.DRM)
			{
				filter_low = -6000;
				filter_high = 6000;
			}

			// get filter limits
			int filter_left_x = (int)((float)(filter_low-low)/(high-low)*W);
			int filter_right_x = (int)((float)(filter_high-low)/(high-low)*W);

			g.FillRectangle(new SolidBrush(display_filter_color),	// draw filter overlay
				filter_left_x, 0, filter_right_x-filter_left_x, H);

			if(high == 0)
			{
				int f = -low;
				// Calculate horizontal step size
				while(f/freq_step_size > 7)
				{
					freq_step_size = step_list[step_index]*(int)Math.Pow(10.0, step_power);
					step_index = (step_index+1)%4;
					if(step_index == 0) step_power++;
				}
				float pixelStepSize = (float)(W*freq_step_size/f);
				int num_steps = f/freq_step_size;

				// Draw vertical lines
				for(int i=1; i<=num_steps; i++)
				{
					int x = W-(int)Math.Floor(i*pixelStepSize);			// for negative numbers
					
					g.DrawLine(grid_pen, x, 0, x, H);		// draw right line
				
					// Draw vertical line labels
					int num = i*freq_step_size;
					string label = num.ToString();
					int offset = (int)((label.Length+1)*4.1);
					if(x-offset >= 0)
						g.DrawString("-"+label, font, grid_text_brush, x-offset, (float)Math.Floor(H*.01));
				}

				// Draw horizontal lines
				int V = (int)(spectrum_grid_max - spectrum_grid_min);
				int numSteps = V/spectrum_grid_step;
				pixelStepSize = H/numSteps;
				for(int i=1; i<numSteps; i++)
				{
					int num = spectrum_grid_max - i*spectrum_grid_step;
					int y = (int)Math.Floor((spectrum_grid_max - num)*H/y_range);

					g.DrawLine(grid_pen, 0, y, W, y);

					// Draw horizontal line labels
					num = spectrum_grid_max - i*spectrum_grid_step;
					string label = num.ToString();
					int offset = (int)(label.Length*4.1);
					SizeF size = g.MeasureString(label, font);

					y -= 8;
					if(y+9 < H)
						g.DrawString(label, font, grid_text_brush, W-size.Width, y);
				}

				// Draw middle vertical line
				g.DrawLine(new Pen(grid_zero_color), W-1, 0, W-1, H);
				g.DrawLine(new Pen(grid_zero_color), W-2, 0, W-2, H);
			}
			else if(low == 0)
			{
				int f = high;
				// Calculate horizontal step size
				while(f/freq_step_size > 7)
				{
					freq_step_size = step_list[step_index]*(int)Math.Pow(10.0, step_power);
					step_index = (step_index+1)%4;
					if(step_index == 0) step_power++;
				}
				float pixelStepSize = (float)(W*freq_step_size/f);
				int num_steps = f/freq_step_size;

				// Draw vertical lines
				for(int i=1; i<=num_steps; i++)
				{
					int x = (int)Math.Floor(i*pixelStepSize);// for positive numbers
					
					g.DrawLine(grid_pen, x, 0, x, H);			// draw right line
				
					// Draw vertical line labels
					int num = i*freq_step_size;
					string label = num.ToString();
					int offset = (int)(label.Length*4.1);
					if(x-offset+label.Length*7 < W)
						g.DrawString(label, font, grid_text_brush, x-offset, (float)Math.Floor(H*.01));
				}

				// Draw horizontal lines
				int V = (int)(spectrum_grid_max - spectrum_grid_min);
				int numSteps = V/spectrum_grid_step;
				pixelStepSize = H/numSteps;
				for(int i=1; i<numSteps; i++)
				{
					int xOffset = 0;
					int num = spectrum_grid_max - i*spectrum_grid_step;
					int y = (int)Math.Floor((spectrum_grid_max - num)*H/y_range);
					g.DrawLine(grid_pen, 0, y, W, y);

					// Draw horizontal line labels
					num = spectrum_grid_max - i*spectrum_grid_step;
					string label = num.ToString();
					if(label.Length == 3)
						xOffset = (int)g.MeasureString("-", font).Width - 2;
					int offset = (int)(label.Length*4.1);

					y -= 8;
					if(y+9 < H)
						g.DrawString(label, font, grid_text_brush, 3 + xOffset, y);
				}

				// Draw middle vertical line
				g.DrawLine(new Pen(grid_zero_color), 0, 0, 0, H);
				g.DrawLine(new Pen(grid_zero_color), 1, 0, 1, H);
			}
			if(low < 0 && high > 0)
			{
				int f = high;

				// Calculate horizontal step size
				while(f/freq_step_size > 4)
				{
					freq_step_size = step_list[step_index]*(int)Math.Pow(10.0, step_power);
					step_index = (step_index+1)%4;
					if(step_index == 0) step_power++;
				}
				int pixelStepSize = W/2*freq_step_size/f;
				int num_steps = f/freq_step_size;

				// Draw vertical lines
				for(int i=1; i<=num_steps; i++)
				{
					int xLeft = mid_w-(i*pixelStepSize);			// for negative numbers
					int xRight = mid_w+(i*pixelStepSize);		// for positive numbers
					g.DrawLine(grid_pen, xLeft, 0, xLeft, H);		// draw left line
					g.DrawLine(grid_pen, xRight, 0, xRight, H);		// draw right line
				
					// Draw vertical line labels
					int num = i*freq_step_size;
					string label = num.ToString();
					int offsetL = (int)((label.Length+1)*4.1);
					int offsetR = (int)(label.Length*4.1);
					if(xLeft-offsetL >= 0)
					{
						g.DrawString("-"+label, font, grid_text_brush, xLeft-offsetL, (float)Math.Floor(H*.01));
						g.DrawString(label, font, grid_text_brush, xRight-offsetR, (float)Math.Floor(H*.01));
					}
				}

				// Draw horizontal lines
				int V = (int)(spectrum_grid_max - spectrum_grid_min);
				int numSteps = V/spectrum_grid_step;
				pixelStepSize = H/numSteps;
				for(int i=1; i<numSteps; i++)
				{
					int xOffset = 0;
					int num = spectrum_grid_max - i*spectrum_grid_step;
					int y = (int)Math.Floor((spectrum_grid_max - num)*H/y_range);
					g.DrawLine(grid_pen, 0, y, W, y);

					// Draw horizontal line labels
					num = spectrum_grid_max - i*spectrum_grid_step;
					string label = num.ToString();
					if(label.Length == 3) xOffset = 7;
					//int offset = (int)(label.Length*4.1);
					if(current_dsp_mode == DSPMode.USB ||
						current_dsp_mode == DSPMode.CWU)
						xOffset -= 32;

					y -= 8;
					if(y+9 < H)
						g.DrawString(label, font, grid_text_brush, mid_w+2 + xOffset, y);
				}

				// Draw middle vertical line
				g.DrawLine(new Pen(grid_zero_color), mid_w, 0, mid_w, H);
				g.DrawLine(new Pen(grid_zero_color), mid_w-1, 0, mid_w-1, H);
			}

			// save bitmap
			background_image_mutex.WaitOne();
			if(b != null) b.Dispose();
			b = (Bitmap)bmp.Clone();
			background_image_mutex.ReleaseMutex();
			bmp.Dispose();
			g.Dispose();
		}

		private void DrawWaterfallGrid(ref Bitmap b, int W, int H)
		{
			Bitmap bmp = new Bitmap(W, H);				// create bitmap
			Graphics g = Graphics.FromImage(bmp);		// create graphics object for drawing

			// draw background
			g.FillRectangle(new SolidBrush(display_background_color), 0, 0, W, H);

			int low = 0;
			int high = 0;

			if(!chkMOX.Checked)					// get left and right drawing limits
			{
				low = DttSP.RXDisplayLow;
				high = DttSP.RXDisplayHigh;
			}
			else
			{
				low = DttSP.TXDisplayLow;
				high = DttSP.TXDisplayHigh;
			}

			int mid_w = W/2;					// initialize variables
			int[] step_list = {10, 20, 25, 50};
			int step_power = 1;
			int step_index = 0;
			int freq_step_size = 50;

			Font font = new Font("Arial", 9);
			SolidBrush grid_text_brush = new SolidBrush(grid_text_color);
			Pen grid_pen = new Pen(grid_color);
			int y_range = spectrum_grid_max - spectrum_grid_min;

			if(high == 0)
			{
				int f = -low;
				// Calculate horizontal step size
				while(f/freq_step_size > 7)
				{
					freq_step_size = step_list[step_index]*(int)Math.Pow(10.0, step_power);
					step_index = (step_index+1)%4;
					if(step_index == 0) step_power++;
				}
				float pixelStepSize = (float)(W*freq_step_size/f);
				int num_steps = f/freq_step_size;

				// Draw vertical lines
				for(int i=1; i<=num_steps; i++)
				{
					int x = W-(int)Math.Floor(i*pixelStepSize);			// for negative numbers
					
					g.DrawLine(grid_pen, x, 0, x, H);		// draw right line
				
					// Draw vertical line labels
					int num = i*freq_step_size;
					string label = num.ToString();
					int offset = (int)((label.Length+1)*4.1);
					if(x-offset >= 0)
						g.DrawString("-"+label, font, grid_text_brush, x-offset, (float)Math.Floor(H*.01));
				}

				// Draw horizontal lines
				int V = (int)(spectrum_grid_max - spectrum_grid_min);
				int numSteps = V/spectrum_grid_step;
				pixelStepSize = H/numSteps;
				for(int i=1; i<numSteps; i++)
				{
					int num = spectrum_grid_max - i*spectrum_grid_step;
					int y = (int)Math.Floor((spectrum_grid_max - num)*H/y_range);
					g.DrawLine(grid_pen, 0, y, W, y);
				}

				// Draw zero line
				g.DrawLine(new Pen(grid_zero_color), W-1, 0, W-1, H);
				g.DrawLine(new Pen(grid_zero_color), W-2, 0, W-2, H);
			}
			else if(low == 0)
			{
				int f = high;
				// Calculate horizontal step size
				while(f/freq_step_size > 7)
				{
					freq_step_size = step_list[step_index]*(int)Math.Pow(10.0, step_power);
					step_index = (step_index+1)%4;
					if(step_index == 0) step_power++;
				}
				float pixelStepSize = (float)(W*freq_step_size/f);
				int num_steps = f/freq_step_size;

				// Draw vertical lines
				for(int i=1; i<=num_steps; i++)
				{
					int x = (int)Math.Floor(i*pixelStepSize);// for positive numbers
					
					g.DrawLine(grid_pen, x, 0, x, H);			// draw right line
				
					// Draw vertical line labels
					int num = i*freq_step_size;
					string label = num.ToString();
					int offset = (int)(label.Length*4.1);
					if(x-offset+label.Length*7 < W)
						g.DrawString(label, font, grid_text_brush, x-offset, (float)Math.Floor(H*.01));
				}

				// Draw horizontal lines
				int V = (int)(spectrum_grid_max - spectrum_grid_min);
				int numSteps = V/spectrum_grid_step;
				pixelStepSize = H/numSteps;
				for(int i=1; i<numSteps; i++)
				{
					int num = spectrum_grid_max - i*spectrum_grid_step;
					int y = (int)Math.Floor((spectrum_grid_max - num)*H/y_range);
					g.DrawLine(grid_pen, 0, y, W, y);
				}

				// Draw zero line
				g.DrawLine(new Pen(grid_zero_color), 0, 0, 0, H);
				g.DrawLine(new Pen(grid_zero_color), 1, 0, 1, H);
			}
			if(low < 0 && high > 0)
			{
				int f = high;

				// Calculate horizontal step size
				while(f/freq_step_size > 4)
				{
					freq_step_size = step_list[step_index]*(int)Math.Pow(10.0, step_power);
					step_index = (step_index+1)%4;
					if(step_index == 0) step_power++;
				}
				int pixelStepSize = W/2*freq_step_size/f;
				int num_steps = f/freq_step_size;

				// Draw vertical lines
				for(int i=1; i<=num_steps; i++)
				{
					int xLeft = mid_w-(i*pixelStepSize);			// for negative numbers
					int xRight = mid_w+(i*pixelStepSize);		// for positive numbers
					g.DrawLine(grid_pen, xLeft, 0, xLeft, H);		// draw left line
					g.DrawLine(grid_pen, xRight, 0, xRight, H);		// draw right line
				
					// Draw vertical line labels
					int num = i*freq_step_size;
					string label = num.ToString();
					int offsetL = (int)((label.Length+1)*4.1);
					int offsetR = (int)(label.Length*4.1);
					if(xLeft-offsetL >= 0)
					{
						g.DrawString("-"+label, font, grid_text_brush, xLeft-offsetL, (float)Math.Floor(H*.01));
						g.DrawString(label, font, grid_text_brush, xRight-offsetR, (float)Math.Floor(H*.01));
					}
				}

				// Draw horizontal lines
				int V = (int)(spectrum_grid_max - spectrum_grid_min);
				int numSteps = V/spectrum_grid_step;
				pixelStepSize = H/numSteps;
				for(int i=1; i<numSteps; i++)
				{
					int num = spectrum_grid_max - i*spectrum_grid_step;
					int y = (int)Math.Floor((spectrum_grid_max - num)*H/y_range);
					g.DrawLine(grid_pen, 0, y, W, y);
				}

				// Draw zero line
				g.DrawLine(new Pen(grid_zero_color), mid_w, 0, mid_w, H);
				g.DrawLine(new Pen(grid_zero_color), mid_w-1, 0, mid_w-1, H);
			}

			// save bitmap
			background_image_mutex.WaitOne();
			if(b != null) b.Dispose();
			b = (Bitmap)bmp.Clone();
			background_image_mutex.ReleaseMutex();
			bmp.Dispose();
			g.Dispose();
		}

		private void DrawOffBackground(ref Bitmap b, int W, int H)
		{
			Bitmap bmp = new Bitmap(W, H);				// create bitmap
			Graphics g = Graphics.FromImage(bmp);		// create graphics object for drawing

			// draw background
			g.FillRectangle(new SolidBrush(display_background_color), 0, 0, W, H);

			// save bitmap
			background_image_mutex.WaitOne();
			if(b != null) b.Dispose();
			b = (Bitmap)bmp.Clone();
			background_image_mutex.ReleaseMutex();
			bmp.Dispose();
			g.Dispose();
		}

		private bool DrawScope(ref Graphics g, int W, int H)
		{
			int retval = -1;

			// get data from DSP
			fixed(float *ptr = &display_data[0])
				retval = DttSP.GetScope(ptr, W);

			Point[] points = new Point[W];				// create Point array
			for(int i=0; i<W; i++)						// fill point array
			{	
				int pixels = (int)(H/2 * display_data[i]);
				int y = H/2 - pixels;
				if(y < display_max_y)
				{
					display_max_y = y;
					display_max_x = i;
				}
				points[i].X = i;
				points[i].Y = y;
			}

			// draw the connected points
			g.DrawLines(new Pen(data_line_color, display_line_width), points);

			// draw long cursor
			if(long_crosshair)
			{
				g.DrawLine(new Pen(grid_text_color), display_cursor_x, 0, display_cursor_x, H);
				g.DrawLine(new Pen(grid_text_color), 0, display_cursor_y, W, display_cursor_y);
			}

			if(display_high_swr)
				g.DrawString("High SWR", new Font("Arial", 14, FontStyle.Bold), new SolidBrush(Color.Red), 245, 20);

			return true;
		}

		private bool DrawPhase(ref Graphics g, int W, int H)
		{
			int num_points = phase_num_pts;

			//float[] a = new float[display_buffer_size];		// declare array for data -- 2x because x,y 
			int retval = -1;

			fixed(float *ptr = &display_data[0])					// get data from DSP
				retval = DttSP.GetPhase(ptr, num_points);

			Point[] points = new Point[num_points];		// declare Point array
			for(int i=0,j=0; i<num_points; i++,j+=8)	// fill point array
			{
				int x = (int)(display_data[i*2]*H/2);
				int y = (int)(display_data[i*2+1]*H/2);
				points[i].X = W/2+x;
				points[i].Y = H/2+y;
			}
			
			// draw each point
			for(int i=0; i<num_points; i++)
				g.DrawRectangle(new Pen(data_line_color, display_line_width), points[i].X, points[i].Y, 1, 1);

			// draw long cursor
			if(long_crosshair)
			{
				g.DrawLine(new Pen(grid_text_color), display_cursor_x, 0, display_cursor_x, H);
				g.DrawLine(new Pen(grid_text_color), 0, display_cursor_y, W, display_cursor_y);
			}

			if(display_high_swr)
				g.DrawString("High SWR", new Font("Arial", 14, FontStyle.Bold), new SolidBrush(Color.Red), 245, 20);

			return true;
		}

		private void DrawPhase2(ref Graphics g, int W, int H)
		{
			int num_points = phase_num_pts;

			float[] a = new float[num_points*2];

			//			dsp_meter_mutex.WaitOne();
			//			float gain = DttSP.CalculateMeter(DttSP.MeterType.AGC_GAIN);
			//			dsp_meter_mutex.ReleaseMutex();
			//
			//			gain = (float)Math.Pow(10, (gain-20)/20);

			Audio.phase_mutex.WaitOne();
			for(int i=0; i<num_points; i++)
			{
				display_data[i*2]   = Audio.phase_buf_l[i];
				display_data[i*2+1] = Audio.phase_buf_r[i];
			}
			Audio.phase_mutex.ReleaseMutex();

			Point[] points = new Point[num_points];		// declare Point array
			for(int i=0; i<num_points; i++)	// fill point array
			{
				int x = (int)(display_data[i*2]*H*0.5*500);
				int y = (int)(display_data[i*2+1]*H*0.5*500);
				points[i].X = (int)(W*0.5+x);
				points[i].Y = (int)(H*0.5+y);
			}
			
			// draw each point
			for(int i=0; i<num_points; i++)
				g.DrawRectangle(new Pen(data_line_color, display_line_width), points[i].X, points[i].Y, 1, 1);

			// draw long cursor
			if(long_crosshair)
			{
				g.DrawLine(new Pen(grid_text_color), display_cursor_x, 0, display_cursor_x, H);
				g.DrawLine(new Pen(grid_text_color), 0, display_cursor_y, W, display_cursor_y);
			}

			if(display_high_swr)
				g.DrawString("High SWR", new Font("Arial", 14, FontStyle.Bold), new SolidBrush(Color.Red), 245, 20);
		}

		unsafe private bool DrawSpectrum(ref Graphics g, int W, int H)
		{
			Point[] points = new Point[W];			// array of points to display
			float slope = 0.0f;						// samples to process per pixel
			int num_samples = 0;					// number of samples to process
			int start_sample_index = 0;				// index to begin looking at samples
			int low = 0;
			int high = 0;

			display_max_y = Int32.MinValue;

			if(!chkMOX.Checked)
			{
				low = DttSP.RXDisplayLow;
				high = DttSP.RXDisplayHigh;
			}
			else
			{
				low = DttSP.TXDisplayLow;
				high = DttSP.TXDisplayHigh;
			}

			if(current_dsp_mode == DSPMode.DRM)
			{
				low = 2000;
				high = 22000;
			}

			int yRange = spectrum_grid_max - spectrum_grid_min;

			int retval = -1;
			fixed(float* ptr = &display_data[0])
				retval = DttSP.GetSpectrum(ptr);

			if(average_display)
			{
				if(average_buffer[0] == CLEAR_FLAG)
				{
					for(int i=0; i<display_buffer_size; i++)
						average_buffer[i] = display_data[i];
				}
				else
				{
					for(int i=0; i<display_buffer_size; i++)
						average_buffer[i] = display_data[i] = (float)(display_data[i]*average_mult_samp + average_buffer[i]*average_mult_avg);
				}
			}

			start_sample_index = (display_buffer_size>>1) + (int)((low * display_buffer_size) / DttSP.SampleRate);
			num_samples = (int)((high - low) * display_buffer_size / DttSP.SampleRate);

			if (start_sample_index < 0) start_sample_index = 0;
			if ((num_samples - start_sample_index) > (display_buffer_size+1))
				num_samples = display_buffer_size - start_sample_index;

			slope = (float)num_samples/(float)W;
			for(int i=0; i<W; i++)
			{
				float max = float.MinValue;
				float dval = i*slope + start_sample_index;
				int lindex = (int)Math.Floor(dval);
				if (slope <= 1) 
					max =  display_data[lindex]*((float)lindex-dval+1) + display_data[lindex+1]*(dval-(float)lindex);
				else 
				{
					int rindex = (int)Math.Floor(dval + slope);
					if (rindex > display_buffer_size) rindex = display_buffer_size;
					for(int j=lindex;j<rindex;j++)
						if (display_data[j] > max) max=display_data[j];

				}

				max = max + display_cal_offset + preamp_offset[(int)current_preamp_mode];
				if(max > display_max_y)
				{
					display_max_y = max;
					display_max_x = i;
				}

				points[i].X = i;
				points[i].Y = (int)(Math.Floor((spectrum_grid_max - max)*H/yRange));
			}

			g.DrawLines(new Pen(data_line_color, display_line_width), points);

			// draw long cursor
			if(long_crosshair)
			{
				g.DrawLine(new Pen(grid_text_color), display_cursor_x, 0, display_cursor_x, H);
				g.DrawLine(new Pen(grid_text_color), 0, display_cursor_y, W, display_cursor_y);
			}

			if(display_high_swr)
				g.DrawString("High SWR", new Font("Arial", 14, FontStyle.Bold), new SolidBrush(Color.Red), 245, 20);

			return true;
		}

		unsafe private bool DrawPanadapter(ref Graphics g, int W, int H)
		{
			Point[] points = new Point[W];			// array of points to display
			float slope = 0.0F;						// samples to process per pixel
			int num_samples = 0;					// number of samples to process
			int start_sample_index = 0;				// index to begin looking at samples
			int Low = -10000;
			int High = 10000;
			int yRange = spectrum_grid_max - spectrum_grid_min;

			if(current_dsp_mode == DSPMode.DRM)
			{
				Low = 2000;
				High = 22000;
			}

			display_max_y = Int32.MinValue;

			int retval = -1;
			fixed(float* ptr = &display_data[0])
				retval = DttSP.GetSpectrum(ptr);

			if(average_display)
			{
				if(average_buffer[0] == CLEAR_FLAG)
				{
					for(int i=0; i<display_buffer_size; i++)
						average_buffer[i] = display_data[i];
				}
				else
				{
					for(int i=0; i<display_buffer_size; i++)
						average_buffer[i] = display_data[i] = (float)(display_data[i]*average_mult_samp + average_buffer[i]*average_mult_avg);
				}
			}

			num_samples = (High - Low);

			start_sample_index = (display_buffer_size>>1) +(int)((Low * display_buffer_size) / DttSP.SampleRate);
			num_samples = (int)((High - Low) * display_buffer_size / DttSP.SampleRate);
			if (start_sample_index < 0) start_sample_index = 0;
			if ((num_samples - start_sample_index) > (display_buffer_size+1))
				num_samples = display_buffer_size-start_sample_index;

			slope = (float)num_samples/(float)W;
			for(int i=0; i<W; i++)
			{
				float max = float.MinValue;
				float dval = i*slope + start_sample_index;
				int lindex = (int)Math.Floor(dval);
				int rindex = (int)Math.Floor(dval + slope);
				if (rindex > display_buffer_size) rindex = display_buffer_size;
				
				for(int j=lindex;j<rindex;j++)
					if (display_data[j] > max) max=display_data[j];

				max = max + display_cal_offset + preamp_offset[(int)current_preamp_mode];
				if(max > display_max_y)
				{
					display_max_y = max;
					display_max_x = i;
				}

				points[i].X = i;
				points[i].Y = (int)(Math.Floor((spectrum_grid_max - max)*H/yRange));
			} 
			
			g.DrawLines(new Pen(data_line_color, display_line_width), points);
			
			points = null;

			// draw long cursor
			if(long_crosshair)
			{
				g.DrawLine(new Pen(grid_text_color), display_cursor_x, 0, display_cursor_x, H);
				g.DrawLine(new Pen(grid_text_color), 0, display_cursor_y, W, display_cursor_y);
			}

			if(display_high_swr)
				g.DrawString("High SWR", new Font("Arial", 14, FontStyle.Bold), new SolidBrush(Color.Red), 245, 20);

			return true;
		}

		unsafe private bool DrawWaterfall(ref Graphics g, int W, int H)
		{
			float[] data = new float[W];			// array of points to display
			float slope = 0.0F;						// samples to process per pixel
			int num_samples = 0;					// number of samples to process
			int start_sample_index = 0;				// index to begin looking at samples
			int low = 0;
			int high = 0;
			display_max_y = Int32.MinValue;

			if(!chkMOX.Checked)
			{
				low = DttSP.RXDisplayLow;
				high = DttSP.RXDisplayHigh;
			}
			else
			{
				low = DttSP.TXDisplayLow;
				high = DttSP.TXDisplayHigh;
			}

			if(current_dsp_mode == DSPMode.DRM)
			{
				low = 2000;
				high = 22000;
			}

			int yRange = spectrum_grid_max - spectrum_grid_min;

			//float[] a = new float[display_buffer_size];
			int retval = -1;

			fixed(float* ptr = &display_data[0])
				retval = DttSP.GetSpectrum(ptr);

			if(average_display)
			{
				if(average_buffer[0] == CLEAR_FLAG)
				{
					for(int i=0; i<display_buffer_size; i++)
						average_buffer[i] = display_data[i];
				}
				else
				{
					for(int i=0; i<display_buffer_size; i++)
						average_buffer[i] = display_data[i] = (float)(display_data[i]*average_mult_samp + average_buffer[i]*average_mult_avg);
				}
			}

			num_samples = (high - low);

			start_sample_index = (display_buffer_size>>1) +(int)((low * display_buffer_size) / DttSP.SampleRate);
			num_samples = (int)((high - low) * display_buffer_size / DttSP.SampleRate);
			if (start_sample_index < 0) start_sample_index = 0;
			if ((num_samples - start_sample_index) > (display_buffer_size+1)) 
				num_samples = display_buffer_size-start_sample_index;

			slope = (float)num_samples/(float)W;
			for(int i=0; i<W; i++)
			{
				float max = float.MinValue;
				float dval = i*slope + start_sample_index;
				int lindex = (int)Math.Floor(dval);
				if (slope <= 1) 
					max =  display_data[lindex]*((float)lindex-dval+1) + display_data[lindex+1]*(dval-(float)lindex);
				else 
				{
					int rindex = (int)Math.Floor(dval + slope);
					if (rindex > display_buffer_size) rindex = display_buffer_size;
					for(int j=lindex;j<rindex;j++)
					{
						if (display_data[j] > max) 
							max=display_data[j];
					}
				}

				max = max + display_cal_offset + preamp_offset[(int)current_preamp_mode];
				switch(current_dsp_mode)
				{
					case DSPMode.SPEC:
						max += 6.0F;
						break;
				}
				if(max > display_max_y)
				{
					display_max_y = max;
					display_max_x = i;
				}

				data[i] = max;
			}

			BitmapData bitmapData = waterfall_bmp.LockBits(
				new Rectangle(0, 0, waterfall_bmp.Width, waterfall_bmp.Height),
				ImageLockMode.ReadWrite,
				waterfall_bmp.PixelFormat);

			int pixel_size = 3;
			byte* row = null;

			// first scroll image
			int total_size = bitmapData.Stride * bitmapData.Height;		// find buffer size
			MSCVRT.memcpy(new IntPtr((int)bitmapData.Scan0+bitmapData.Stride).ToPointer(),
				bitmapData.Scan0.ToPointer(),
				total_size-bitmapData.Stride);
			//			byte[] buffer = new byte[total_size];						// create buffer
			//			Marshal.Copy(bitmapData.Scan0, buffer, 0, total_size);		// copy bitmap to buffer
			//			Marshal.Copy(buffer, 0,
			//				new IntPtr((int)bitmapData.Scan0+bitmapData.Stride),	// slide memory and copy back
			//				total_size - bitmapData.Stride);
			//			buffer = null;

			row = (byte *)bitmapData.Scan0;
				
			// draw new data
			for(int i=0; i<W; i++)	// for each pixel in the new line
			{
				int R, G, B;		// variables to save Red, Green and Blue component values

				if(data[i] <= waterfall_low_threshold)		// if less than low threshold, just use low color
				{
					R = waterfall_low_color.R;
					G = waterfall_low_color.G;
					B = waterfall_low_color.B;
				}
				else if(data[i] >= waterfall_high_threshold)// if more than high threshold, just use high color
				{
					R = waterfall_high_color.R;
					G = waterfall_high_color.G;
					B = waterfall_high_color.B;
				}
				else // use a color between high and low
				{
					float percent = (data[i] - waterfall_low_threshold)/(waterfall_high_threshold - waterfall_low_threshold);
					if(percent <= 0.5)	// use a gradient between low and mid colors
					{
						percent *= 2;

						R = (int)((1-percent)*waterfall_low_color.R + percent*waterfall_mid_color.R);
						G = (int)((1-percent)*waterfall_low_color.G + percent*waterfall_mid_color.G);
						B = (int)((1-percent)*waterfall_low_color.B + percent*waterfall_mid_color.B);
					}
					else				// use a gradient between mid and high colors
					{
						percent = (float)(percent-0.5)*2;

						R = (int)((1-percent)*waterfall_mid_color.R + percent*waterfall_high_color.R);
						G = (int)((1-percent)*waterfall_mid_color.G + percent*waterfall_high_color.G);
						B = (int)((1-percent)*waterfall_mid_color.B + percent*waterfall_high_color.B);
					}
				}

				// set pixel color
				row[i*pixel_size + 0] = (byte)B;	// set color in memory
				row[i*pixel_size + 1] = (byte)G;
				row[i*pixel_size + 2] = (byte)R;
			}
			waterfall_bmp.UnlockBits(bitmapData);

			g.DrawImageUnscaled(waterfall_bmp, 0, 16);	// draw the image on the background	

			waterfall_counter++;

			// draw long cursor
			if(long_crosshair)
			{
				g.DrawLine(new Pen(grid_text_color), display_cursor_x, 0, display_cursor_x, H);
				g.DrawLine(new Pen(grid_text_color), 0, display_cursor_y, W, display_cursor_y);
			}

			if(display_high_swr)
				g.DrawString("High SWR", new Font("Arial", 14, FontStyle.Bold), new SolidBrush(Color.Red), 245, 20);

			return true;
		}

		unsafe private bool DrawHistogram(ref Graphics g, int W, int H)
		{
			Point[] points = new Point[W];			// array of points to display
			float slope = 0.0F;						// samples to process per pixel
			int num_samples = 0;					// number of samples to process
			int start_sample_index = 0;				// index to begin looking at samples
			int low = 0;
			int high = 0;
			display_max_y = Int32.MinValue;

			if(!chkMOX.Checked)								// Receive Mode
			{
				low = DttSP.RXDisplayLow;
				high = DttSP.RXDisplayHigh;
			}
			else									// Transmit Mode
			{
				low = DttSP.TXDisplayLow;
				high = DttSP.TXDisplayHigh;
			}

			if(current_dsp_mode == DSPMode.DRM)
			{
				low = 2000;
				high = 22000;
			}

			int yRange = spectrum_grid_max - spectrum_grid_min;

			//			float[] a = new float[display_buffer_size];
			int retval = -1;

			fixed(float* ptr = &display_data[0])
				retval = DttSP.GetSpectrum(ptr);

			if(average_display)
			{
				if(average_buffer[0] == CLEAR_FLAG)
				{
					for(int i=0; i<display_buffer_size; i++)
						average_buffer[i] = display_data[i];
				}
				else
				{
					for(int i=0; i<display_buffer_size; i++)
						average_buffer[i] = display_data[i] = (float)(display_data[i]*average_mult_samp + average_buffer[i]*average_mult_avg);
				}
			}

			num_samples = (high - low);

			start_sample_index = (display_buffer_size>>1) +(int)((low * display_buffer_size) / DttSP.SampleRate);
			num_samples = (int)((high - low) * display_buffer_size / DttSP.SampleRate);
			if (start_sample_index < 0) start_sample_index = 0;
			if ((num_samples - start_sample_index) > (display_buffer_size+1))
				num_samples = display_buffer_size-start_sample_index;

			slope = (float)num_samples/(float)W;
			for(int i=0; i<W; i++)
			{
				float max = float.MinValue;
				float dval = i*slope + start_sample_index;
				int lindex = (int)Math.Floor(dval);
				if (slope <= 1) 
					max =  display_data[lindex]*((float)lindex-dval+1) + display_data[lindex+1]*(dval-(float)lindex);
				else 
				{
					int rindex = (int)Math.Floor(dval + slope);
					if (rindex > display_buffer_size) rindex = display_buffer_size;
					for(int j=lindex;j<rindex;j++)
						if (display_data[j] > max) max=display_data[j];

				}

				
				max = max + display_cal_offset + preamp_offset[(int)current_preamp_mode];
				switch(current_dsp_mode)
				{
					case DSPMode.SPEC:
						max += 6.0F;
						break;
				}
				if(max > display_max_y)
				{
					display_max_y = max;
					display_max_x = i;
				}

				points[i].X = i;
				points[i].Y = (int)(Math.Floor((spectrum_grid_max - max)*H/yRange));
			} 

			// get the average
			float avg = 0.0F;
			int sum = 0;
			foreach(Point p in points)
				sum += p.Y;

			avg = (float)((float)sum/points.Length / 1.12);

			for(int i=0; i<W; i++)
			{
				if(points[i].Y < histogram_data[i])
				{
					histogram_history[i] = 0;
					histogram_data[i] = points[i].Y;
				}
				else
				{
					histogram_history[i]++;
					if(histogram_history[i] > 51)
					{
						histogram_history[i] = 0;
						histogram_data[i] = points[i].Y;
					}

					int alpha = (int)Math.Max(255-histogram_history[i]*5, 0);
					Color c = Color.FromArgb(alpha, 0, 255, 0);
					int height = points[i].Y-histogram_data[i];
					g.DrawRectangle(new Pen(c), i, histogram_data[i], 1, height);
				}

				if(points[i].Y >= avg)		// value is below the average
				{
					Color c = Color.FromArgb(150, 0, 0, 255);
					g.DrawRectangle(new Pen(c), points[i].X, points[i].Y, 1, H-points[i].Y);
				}
				else 
				{
					g.DrawRectangle(new Pen(Color.FromArgb(150, 0, 0, 255)), points[i].X, (int)Math.Floor(avg), 1, H-(int)Math.Floor(avg));
					g.DrawRectangle(new Pen(Color.FromArgb(150, 255, 0, 0)), points[i].X, points[i].Y, 1, (int)Math.Floor(avg)-points[i].Y);
				}
			}

			// draw long cursor
			if(long_crosshair)
			{
				g.DrawLine(new Pen(grid_text_color), display_cursor_x, 0, display_cursor_x, H);
				g.DrawLine(new Pen(grid_text_color), 0, display_cursor_y, W, display_cursor_y);
			}

			if(display_high_swr)
				g.DrawString("High SWR", new Font("Arial", 14, FontStyle.Bold), new SolidBrush(Color.Red), 245, 20);

			return true;
		}

		private void UpdatePeakText()
		{
			if(txtVFOAFreq.Text == "" ||
				txtVFOAFreq.Text == "." ||
				txtVFOAFreq.Text == ",")
				return;

			// update peak value
			float mx, my;
			PixelToSpectrum(display_max_x, display_max_y, out mx, out my);
			my = display_max_y;

			double freq = double.Parse(txtVFOAFreq.Text) + (double)mx*0.0000010;

			if(current_dsp_mode == DSPMode.CWL)
				freq += (double)cw_pitch*0.0000010;
			else if(current_dsp_mode == DSPMode.CWU)
				freq -= (double)cw_pitch*0.0000010;
			
			display_max_y = picDisplay.Height;
			txtDisplayPeak.Text = mx.ToString("f1") + "Hz, " + my.ToString("f1") + "dBm, " + freq.ToString("f6") + "MHz";
		}

		private void PixelToSpectrum(float inx, float iny, out float outx, out float outy)
		{
			int low, high;
			if(current_display_mode == DisplayMode.PANADAPTER)
			{
				low = -10000;
				high = 10000;
			}
			else
			{
				if(!chkMOX.Checked)
				{
					low = DttSP.RXDisplayLow;
					high = DttSP.RXDisplayHigh;
				}
				else
				{
					low = DttSP.TXDisplayLow;
					high = DttSP.TXDisplayHigh;
				}
			}
			outx = (float)(low + ((double)inx*(high - low)/picDisplay.Width));
			outy = (float)(spectrum_grid_max - iny * (double)(spectrum_grid_max - spectrum_grid_min) / picDisplay.Height);
		}

		private void ResetDisplayAverage()
		{
			average_buffer[0] = CLEAR_FLAG;	// set reset flag
		}

		#endregion

		#region Paint Event Handlers
		// ======================================================
		// Paint Event Handlers
		// ======================================================

		private void picDisplay_Paint(object sender, PaintEventArgs e)
		{
			if(chkPower.Checked &&					// if Power is on and
				(Audio.CurrentAudioState1 == Audio.AudioState.DTTSP ||
				Audio.CurrentAudioState2 == Audio.AudioState.DTTSP)) // DttSP is running
			{
				int W = picDisplay.Width;
				int H = picDisplay.Height;

				BitmapData display_bmpData = display_bitmap.LockBits(
					new Rectangle(0, 0, display_bitmap.Width, display_bitmap.Height),
					ImageLockMode.WriteOnly,
					display_bitmap.PixelFormat);

				background_image_mutex.WaitOne();			// get background image

				BitmapData background_bmpData = background_bmp.LockBits(
					new Rectangle(0, 0, background_bmp.Width, background_bmp.Height),
					ImageLockMode.ReadOnly,
					background_bmp.PixelFormat);
				
				int total_size = background_bmpData.Stride * background_bmpData.Height;		// find buffer size

				byte *srcptr = (byte *)background_bmpData.Scan0.ToPointer();
				byte *destptr = (byte *)display_bmpData.Scan0.ToPointer();

				MSCVRT.memcpy(destptr, srcptr, total_size);

				background_bmp.UnlockBits(background_bmpData);
				background_image_mutex.ReleaseMutex();

				display_bitmap.UnlockBits(display_bmpData);

				//Graphics g = Graphics.FromImage(display_bitmap);
				//g.SmoothingMode = SmoothingMode.AntiAlias;
				bool update = true;
				
				switch (current_display_mode) 
				{
					case DisplayMode.SPECTRUM:
						update = DrawSpectrum(ref display_graphics, W, H);
						break;
					case DisplayMode.PANADAPTER:
						update = DrawPanadapter(ref display_graphics, W, H);
						break;
					case DisplayMode.SCOPE:
						update = DrawScope(ref display_graphics, W, H);
						break;
					case DisplayMode.PHASE:
						update = DrawPhase(ref display_graphics, W, H);
						break;
					case DisplayMode.PHASE2:
						DrawPhase2(ref display_graphics, W, H);
						break;
					case DisplayMode.WATERFALL:
						update = DrawWaterfall(ref display_graphics, W, H);
						break;
					case DisplayMode.HISTOGRAM:
						update = DrawHistogram(ref display_graphics, W, H);
						break;
					case DisplayMode.OFF:
						//Thread.Sleep(1000);
						break;
					default:
						break;
				}

				if(update)
				{
					e.Graphics.DrawImage(display_bitmap, 0, 0);
				}
				else
				{
					Debug.WriteLine("display update = false");
				}
			}
		}

		private void picMultiMeterDigital_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
		{
			int H = picMultiMeterDigital.Height;
			int W = picMultiMeterDigital.Width;
			int pixel_x = 0;

			if(!chkMOX.Checked)
			{
				if(txtMultiText.Text == "")
					return;

				MeterRXMode mode = CurrentMeterRXMode;
				double num = 0;
				switch(mode)
				{
					case MeterRXMode.SIGNAL_STRENGTH:
					case MeterRXMode.SIGNAL_AVERAGE:
					case MeterRXMode.ADC_L:
					case MeterRXMode.ADC_R:
						string text = txtMultiText.Text;
						int length = text.IndexOf(separator) + 1;
						if(length == 0) return;
						string numtext = text.Substring(0, length);
						num = double.Parse(numtext);
						break;
				}

				switch(mode)
				{
					case MeterRXMode.SIGNAL_STRENGTH:
					case MeterRXMode.SIGNAL_AVERAGE:
						double s = (num + 127)/6;
						if(s <= 9.0F)
							pixel_x = (int)((s * 7.5) + 2);
						else
						{
							double over_s9 = num + 73;
							pixel_x = 69 + (int)(over_s9 * 1.05);
						}	
						break;
					case MeterRXMode.ADC_L:
					case MeterRXMode.ADC_R:
						pixel_x = (int)(((num + 100)*1.2) + 12);
						break;
				}
			}
			else
			{
				if(txtMultiText.Text == "")
					return;

				MeterTXMode mode = CurrentMeterTXMode;
				double num = 0;
				switch(mode)
				{
					case MeterTXMode.ALC:
						string text = txtMultiText.Text;
						int length = text.IndexOf(separator) + 1;
						if(length == 0) return;
						string numtext = text.Substring(0, length);
						num = -double.Parse(numtext);
						break;
					case MeterTXMode.FORWARD_POWER:
					case MeterTXMode.PEAK_POWER:
						if(pa_present)
						{
							pa_power_mutex.WaitOne();
							if(use_old_swr)
								num = PAPowerW(0);
							else
								num = PAPowerW2(0);
							pa_power_mutex.ReleaseMutex();
						}
						else
						{
							text = txtMultiText.Text;
							length = text.IndexOf(separator) + 2;
							if(length == 0) return;
							numtext = text.Substring(0, length);
							num = double.Parse(numtext);
						}
						break;
					case MeterTXMode.REVERSE_POWER:
						pa_power_mutex.WaitOne();
						if(use_old_swr)
							num = PAPowerW(1);
						else
							num = PAPowerW2(1);
						pa_power_mutex.ReleaseMutex();
						break;
					case MeterTXMode.SWR:
						pa_power_mutex.WaitOne();
						if(use_old_swr)
							num = SWR(PAPowerW(0), PAPowerW(1));
						else
							num = SWR2(pa_fwd_power, pa_rev_power);
						pa_power_mutex.ReleaseMutex();
						break;
				}

				switch(mode)
				{
					case MeterTXMode.ALC:
						if(num <= -20.0f)
							pixel_x = (int)(0+(num+21.0)*11);
						else if(num <= -10.0f)
							pixel_x = (int)(11+(num+20.0)/10.0*22);
						else if(num <= 0.0f)
							pixel_x = (int)(33+(num+10.0)/10.0*21);
						else if(num <= 5.0f)
							pixel_x = (int)(54+(num-0.0)/5.0*24);
						else if(num <= 10.0f)
							pixel_x = (int)(78+(num-5.0)/5.0*26);
						else if(num <= 20.0f)
							pixel_x = (int)(104+(num-10.0)/10.0*25);
						else
							pixel_x = (int)(129 + (num-20.0)/5.0*8);
						break;
					case MeterTXMode.FORWARD_POWER:
					case MeterTXMode.PEAK_POWER:
						if(pa_present)
						{
							if(num <= 1.0f)
								pixel_x = (int)(0 + num*2);
							else if(num <= 5.0f)
								pixel_x = (int)(2 + (num-1)/4*24);
							else if(num <= 10.0f)
								pixel_x = (int)(26 + (num-5)/5*24);
							else if(num <= 50.0f)
								pixel_x = (int)(50 + (num-10)/40*24);
							else if(num <= 100.0f)
								pixel_x = (int)(74 + (num-50)/50*24);
							else if(num <= 120.0f)
								pixel_x = (int)(98 + (num-100)/20*24);
							else
								pixel_x = (int)(122 + (num-120)/20*16);
						}
						else
						{
							if(num == 0)
								pixel_x = 2;
							if(num <= 0.1f)
								pixel_x = (int)(2 + num/0.1*29);
							else if(num <= 0.2f)
								pixel_x = (int)(31 + (num-0.1)/0.1*30);
							else if(num <= 0.5f)
								pixel_x = (int)(61 + (num-0.2)/0.3*30);
							else if(num <= 1.0f)
								pixel_x = (int)(91 + (num-0.5)/0.5*39);
							else
								pixel_x = (int)(130 + (num-1.0)/0.2*9);							
						}
						break;
					case MeterTXMode.REVERSE_POWER:
						if(num <= 1.0f)
							pixel_x = (int)(0 + num*2);
						else if(num <= 5.0f)
							pixel_x = (int)(2 + (num-1)/4*24);
						else if(num <= 10.0f)
							pixel_x = (int)(26 + (num-5)/5*24);
						else if(num <= 50.0f)
							pixel_x = (int)(50 + (num-10)/40*24);
						else if(num <= 100.0f)
							pixel_x = (int)(74 + (num-50)/50*24);
						else if(num <= 120.0f)
							pixel_x = (int)(98 + (num-100)/20*24);
						else
							pixel_x = (int)(122 + (num-120)/20*16);
						break;
					case MeterTXMode.SWR:
						if(double.IsInfinity(num))
							pixel_x = 139;
						else if(num <= 1.0f)
							pixel_x = (int)(0 + num * 3);
						else if(num <= 1.5f)
							pixel_x = (int)(3 + (num-1.0)/0.5*27);
						else if(num <= 2.0f)
							pixel_x = (int)(30 + (num-1.5)/0.5*20);
						else if(num <= 3.0f)
							pixel_x = (int)(50 + (num-2.0)/1.0*21);
						else if(num <= 5.0f)
							pixel_x = (int)(71 + (num-3.0)/2.0*21);
						else if(num <= 10.0f)
							pixel_x = (int)(92 + (num-5.0)/5.0*21);
						else
							pixel_x = (int)(113 + (num-10.0)/15.0*26);
						break;
				}
			}

			if(pixel_x > 139) pixel_x = 139;
			if(pixel_x == 0) pixel_x = 1;

			LinearGradientBrush brush = new LinearGradientBrush(new Rectangle(0, 0, pixel_x, H),
				meter_left_color, meter_right_color, LinearGradientMode.Horizontal);
			Graphics g = e.Graphics;
			g.FillRectangle(brush, 0, 0, pixel_x, H);
							
			for(int i=0; i<17; i++)
				g.DrawLine(new Pen(Color.Black), 6+i*8, 0, 6+i*8, H);

			g.DrawLine(new Pen(Color.Red), pixel_x, 0, pixel_x, H);
			g.FillRectangle(new SolidBrush(Color.Black), pixel_x+1, 0, W-pixel_x, H);

			if(pixel_x >= meter_peak_value)
			{
				meter_peak_count = 0;
				meter_peak_value = pixel_x;
			}
			else
			{
				if(meter_peak_count++ >= multimeter_peak_hold_samples)
				{
					meter_peak_count = 0;
					meter_peak_value = pixel_x;
				}
				else
				{
					g.DrawLine(new Pen(Color.Red), meter_peak_value, 0, meter_peak_value, H);
					g.DrawLine(new Pen(Color.Red), meter_peak_value-1, 0, meter_peak_value-1, H);
				}
			}
		}

		private void panelVFOAHover_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
		{
			if(vfoa_hover_digit < 0)
				return;

			int x = 0;

			x += vfo_char_width*vfoa_hover_digit;
			if(vfoa_hover_digit > 1)
				x += vfo_decimal_width;

			e.Graphics.DrawLine(new Pen(txtVFOAFreq.ForeColor, 2.0f), x, 1, x+vfo_char_width-2, 1);
		}

		private void panelVFOBHover_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
		{
			if(vfob_hover_digit < 0)
				return;

			int x = 0;

			x += vfo_char_width*vfob_hover_digit;
			if(vfob_hover_digit > 1)
				x += vfo_decimal_width;

			e.Graphics.DrawLine(new Pen(txtVFOBFreq.ForeColor, 2.0f), x, 1, x+vfo_char_width-2, 1);
		}

		#endregion

		#region Thread and Timer Routines
		// ======================================================
		// Thread Routines
		// ======================================================

		private void RunDisplay()
		{
			while(chkPower.Checked && current_display_mode != DisplayMode.OFF)
			{
				picDisplay.Invalidate();
				
				if(chkPower.Checked)
					Thread.Sleep(display_delay);
			}	
		}

		private void UpdateMultimeter()
		{
			while(chkPower.Checked)
			{
				string output = "";
				if(!chkMOX.Checked)
				{
					if(Audio.CurrentAudioState1 != Audio.AudioState.DTTSP)
						goto end;

					MeterRXMode mode = CurrentMeterRXMode;
					float num = 0f;
				
					switch(mode)
					{
						case MeterRXMode.SIGNAL_STRENGTH:
							num = DttSP.CalculateMeter(DttSP.MeterType.SIGNAL_STRENGTH);
							num = num + multimeter_cal_offset + preamp_offset[(int)current_preamp_mode];
							output = num.ToString("f1")+" dBm";
							break;
						case MeterRXMode.SIGNAL_AVERAGE:
							num = DttSP.CalculateMeter(DttSP.MeterType.AVG_SIGNAL_STRENGTH);
							num = num + multimeter_cal_offset + preamp_offset[(int)current_preamp_mode];
							output = num.ToString("f1")+" dBm";
							break;
						case MeterRXMode.ADC_L:
							num = DttSP.CalculateMeter(DttSP.MeterType.ADC_REAL);
							output = num.ToString("f1")+" dBFS";
							break;
						case MeterRXMode.ADC_R:
							num = DttSP.CalculateMeter(DttSP.MeterType.ADC_IMAG);
							output = num.ToString("f1")+" dBFS";
							break;
					}
				}
				else
				{
					MeterTXMode mode = CurrentMeterTXMode;
					float num = 0f;

					switch(mode)
					{
						case MeterTXMode.ALC:
							if(Audio.CurrentAudioState1 == Audio.AudioState.DTTSP ||
								Audio.CurrentAudioState2 == Audio.AudioState.DTTSP)
							{
								num = DttSP.CalculateMeter(DttSP.MeterType.ALC);
								Debug.WriteLine((int)DttSP.MeterType.ALC);
								output = num.ToString("f1")+" dB";
							}
							else output = "0.0 dB";
							break;
						case MeterTXMode.FORWARD_POWER:
							if(pa_present && VFOAFreq < 30.0)
							{
								double power = 0.0;
								pa_power_mutex.WaitOne();
								if(use_old_swr)
									power = PAPowerW(0);
								else
									power = PAPowerW2(0);
								pa_power_mutex.ReleaseMutex();
							
								output = power.ToString("f0")+" W";
							}
							else
							{
								if(Audio.CurrentAudioState1 == Audio.AudioState.DTTSP ||
									Audio.CurrentAudioState2 == Audio.AudioState.DTTSP)
								{
									num = DttSP.CalculateMeter(DttSP.MeterType.PWR);
									num *= (float)(PWR*0.01);
									output = num.ToString("f2")+" W";
								}
								else output = "0.00 W";
							}
							break;
						case MeterTXMode.PEAK_POWER:
							if(pa_present && VFOAFreq < 30.0)
							{
								num = DttSP.CalculateMeter(DttSP.MeterType.PKPWR);
								num *= PWR;
							
								meter_text_history[meter_text_history_index] = num;
								meter_text_history_index = (meter_text_history_index+1)%multimeter_text_peak_samples;
								float max = float.MinValue;
								for(int i=0; i<multimeter_text_peak_samples; i++)
								{
									if(meter_text_history[i] > max)
										max = meter_text_history[i];
								}
								num = max;

								output = num.ToString("f0")+" W";
							}
							else
							{
								if(Audio.CurrentAudioState1 == Audio.AudioState.DTTSP ||
									Audio.CurrentAudioState2 == Audio.AudioState.DTTSP)
								{
									num = DttSP.CalculateMeter(DttSP.MeterType.PKPWR);
									num *= (float)(PWR*0.01);
									output = num.ToString("f2")+" W";
								}
								else output = "0.00 W";
							}
							break;
						case MeterTXMode.REVERSE_POWER:
							pa_power_mutex.WaitOne();
							if(use_old_swr)
								output = PAPowerW(1).ToString("f0")+" W";
							else
								output = PAPowerW2(1).ToString("f0")+" W";
							pa_power_mutex.ReleaseMutex();
							break;
						case MeterTXMode.SWR:
							pa_power_mutex.WaitOne();
							if(use_old_swr)
								output = SWR(PAPowerW(0), PAPowerW(1)).ToString("f1")+" : 1";
							else
								output = SWR2(pa_fwd_power, pa_rev_power).ToString("f1")+" : 1";
							pa_power_mutex.ReleaseMutex();
							break;
					}
				}
						
				txtMultiText.Text = output;

				picMultiMeterDigital.Invalidate();

			end:
				if(chkPower.Checked)
					Thread.Sleep(meter_delay);

			}
		}

		private void PollPTT()
		{
			int mox_mode = 0;

			while(chkPower.Checked)
			{
				if(!manual_mox && !disable_ptt)
				{
					byte b = hw.StatusPort();
					bool mic_ptt = (b & (byte)Rig.StatusPin.Dot) != 0;
					
					// wjt added 
					if ( !mic_ptt && PTTBitBangEnabled && serialPTT != null) 
					{
						if ( serialPTT.isPTT() ) mic_ptt = true; 
					}
					// wjt added ends 

					bool cw_ptt = false;
					if(usb_present)
						cw_ptt = (b & (byte)Rig.StatusPin.PIN_11) == 0;
					else
						cw_ptt = (b & (byte)Rig.StatusPin.PIN_11) != 0;

					if(cw_ptt && !chkMOX.Checked)
					{
						if(cw_key_mode)
							CWForm.Send = true;
						else
						{
							chkMOX.Checked = true;
							if(!chkMOX.Checked)
								chkPower.Checked = false;
						}
						mox_mode = 2;
					}
					else if(mic_ptt && !chkMOX.Checked && !cw_key_mode)
					{
						chkMOX.Checked = true;
						if(!chkMOX.Checked)
							chkPower.Checked = false;
						mox_mode = 1;
					}
					else if(!mic_ptt && chkMOX.Checked && mox_mode == 1 && !cw_key_mode)
					{
						chkMOX.Checked = false;
						mox_mode = 0;
					}
					else if(!cw_ptt && mox_mode == 2)
					{
						if(cw_key_mode)
							CWForm.Send = false;
						else
							chkMOX.Checked = false;
						mox_mode = 0;
					}
				}
				Thread.Sleep(50);
			}
		}

		private void PollPAPWR()
		{
			int first_time_count = 0;
			int high_swr_count = 0;

			while(chkPower.Checked && pa_present)
			{
				if(chkMOX.Checked)
				{
					if(!atu_tuning)
					{
						if(first_time_count++ >= 5)
						{
							pa_power_mutex.WaitOne();

							int temp = (int)hw.PA_GetADC(0);
							if(temp >= 0) pa_fwd_power = temp;
							temp = (int)hw.PA_GetADC(1);
							if(temp >= 0) pa_rev_power = temp;

							//pa_fwd_power = (int)numericUpDown1.Value;
							//pa_rev_power = (int)numericUpDown2.Value;

							//Debug.WriteLine("fwd: "+pa_fwd_power.ToString()+"  rev: "+pa_rev_power.ToString());

							double rev_power = 0;
							if(use_old_swr)
								rev_power = PAPowerW(1);
							else 
								rev_power = PAPowerW2(1);

							if(rev_power >= 11.0)
							{
								if(high_swr_count++ >= 3)
									HighSWR = true;
							}
							else
								high_swr_count = 0;

							pa_power_mutex.ReleaseMutex();

							if(high_swr && !disable_swr_protection)
							{
								if(DttSP.PAHighSWRScale != 0.1)
									DttSP.PAHighSWRScale = 0.1;
							}
						}
					}
				}
				else 
				{
					pa_power_mutex.WaitOne();
					pa_fwd_power = 0;
					pa_rev_power = 0;
					first_time_count = 0;
					HighSWR = false;
					pa_power_mutex.ReleaseMutex();					
				}

				Thread.Sleep(100);
			}
		}


		private void timer_scanner_Tick(object sender, System.EventArgs e)
		{
			if(radScannerFreq.Checked)
			{
				if(VFOAFreq >= (double)udScannerHigh.Value)
				{
					timer_scanner.Enabled = false;
					chkMemoryScanner.Checked = false;
				}
				else
					VFOAFreq += (double)udScannerStep.Value;
			}
			else
			{
				if(!MemForm.ScanMemory(false))
				{
					timer_scanner.Enabled = false;
					chkMemoryScanner.Checked = false;
				}
			}
		}

		private void timer_cpu_meter_Tick(object sender, System.EventArgs e)
		{
			lblCPUMeter.Text = "CPU %: " + CpuUsage.ToString("f1");
		}

		private void timer_peak_text_Tick(object sender, System.EventArgs e)
		{
			switch(current_display_mode)
			{
				case DisplayMode.HISTOGRAM:
				case DisplayMode.PANADAPTER:
				case DisplayMode.SPECTRUM:
				case DisplayMode.WATERFALL:
					UpdatePeakText();
					break;
				default:
					txtDisplayPeak.Text = "";
					break;
			}
		}

		private void timer_clock_Tick(object sender, System.EventArgs e)
		{
			txtDate.Text = DateTime.Today.ToShortDateString();
			txtLocalTime.Text = "LOC "+DateTime.Now.ToString("HH:mm:ss");
			txtUTCTime.Text = "UTC "+DateTime.UtcNow.ToString("HH:mm:ss");
		}

		#endregion

		#region Event Handlers
		// ======================================================
		// Event Handlers
		// ======================================================

		// Console Events

		private void Console_KeyPress(object sender, System.Windows.Forms.KeyPressEventArgs e)
		{
			if(e.KeyChar == (char)Keys.Enter)
				btnHidden.Focus();
		}

		private void Console_KeyUp(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			if(e.Shift == false)
				shift_down = false;
			
			if(e.KeyCode == key_cw_dot)
			{
				CWForm.KBDot = false;
			}
			else if(e.KeyCode == key_cw_dash)
			{
				CWForm.KBDash = false;
			}
		}

		private void Console_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			if(e.Shift == true)
				shift_down = true;

			if(e.Control == true && e.Shift == true && e.KeyCode == Keys.T && pa_present)
			{
				if(PAQualForm == null || PAQualForm.IsDisposed)
					PAQualForm = new PAQualify(this);
				PAQualForm.Show();
				PAQualForm.Focus();
			}
			else if(e.Control == true)		// control key is pressed
			{
				switch(e.KeyCode)
				{
					case Keys.Left:
						ChangeWheelTuneLeft();
						e.Handled = true;
						break;
					case Keys.Right:
						ChangeWheelTuneRight();
						e.Handled = true;
						break;
					case Keys.Up:
						Console_MouseWheel(this, new MouseEventArgs(MouseButtons.None, 0, 0, 0, 120));
						e.Handled = true;
						break;
					case Keys.Down:
						Console_MouseWheel(this, new MouseEventArgs(MouseButtons.None, 0, 0, 0, -120));
						e.Handled = true;
						break;
				}
			}
			else
			{
				if(this.ActiveControl is TextBox) return;
				if(this.ActiveControl is UpDownBase) return;

				if(e.KeyCode == Keys.Multiply)
					chkMUT.Checked = !chkMUT.Checked;
				else if(e.KeyCode == Keys.Add)
				{
					if(udAF.Value != udAF.Maximum)
						udAF.Value++;
				}
				else if(e.KeyCode == Keys.Subtract)
				{
					if(udAF.Value != udAF.Minimum)
						udAF.Value--;
				}
				else if(e.KeyCode == key_tune_up_1)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq += 1.0;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_down_1)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq -= 1.0;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_up_2)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq += 0.1;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_down_2)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq -= 0.1;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_up_3)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq += 0.01;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_down_3)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq -= 0.01;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_up_4)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq += 0.001;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_down_4)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq -= 0.001;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_up_5)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq += 0.0001;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_down_5)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq -= 0.0001;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_up_6)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq += 0.00001;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_down_6)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq -= 0.00001;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_up_7)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq += 0.000001;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_tune_down_7)
				{
					double freq = Double.Parse(txtVFOAFreq.Text);
					freq -= 0.000001;
					VFOAFreq = freq;
				}
				else if(e.KeyCode == key_filter_up)
				{
					if(current_filter == Filter.NONE)
						return;
					if(current_filter == Filter.VAR2)
						CurrentFilter = Filter.F6000;
					else
						CurrentFilter++;
				}
				else if(e.KeyCode == key_filter_down)
				{
					if(current_filter == Filter.NONE)
						return;
					if(current_filter == Filter.F6000)
						CurrentFilter = Filter.VAR2;
					else
						CurrentFilter--;
				}
				else if(e.KeyCode == key_mode_up)
				{
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
							CurrentDSPMode = DSPMode.USB;
							break;
						case DSPMode.USB:
							CurrentDSPMode = DSPMode.DSB;
							break;
						case DSPMode.DSB:
							CurrentDSPMode = DSPMode.CWL;
							break;
						case DSPMode.CWL:
							CurrentDSPMode = DSPMode.CWU;
							break;
						case DSPMode.CWU:
							CurrentDSPMode = DSPMode.FMN;
							break;
						case DSPMode.FMN:
							CurrentDSPMode = DSPMode.AM;
							break;
						case DSPMode.AM:
							CurrentDSPMode = DSPMode.SAM;
							break;
						case DSPMode.SAM:
							CurrentDSPMode = DSPMode.SPEC;
							break;
						case DSPMode.SPEC:
							CurrentDSPMode = DSPMode.DRM;
							break;
						case DSPMode.DRM:
							CurrentDSPMode = DSPMode.LSB;
							break;
					}
				}
				else if(e.KeyCode == key_mode_down)
				{
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
							CurrentDSPMode = DSPMode.DRM;
							break;
						case DSPMode.USB:
							CurrentDSPMode = DSPMode.LSB;
							break;
						case DSPMode.DSB:
							CurrentDSPMode = DSPMode.USB;
							break;
						case DSPMode.CWL:
							CurrentDSPMode = DSPMode.DSB;
							break;
						case DSPMode.CWU:
							CurrentDSPMode = DSPMode.CWL;
							break;
						case DSPMode.FMN:
							CurrentDSPMode = DSPMode.CWU;
							break;
						case DSPMode.AM:
							CurrentDSPMode = DSPMode.FMN;
							break;
						case DSPMode.SAM:
							CurrentDSPMode = DSPMode.AM;
							break;
						case DSPMode.SPEC:
							CurrentDSPMode = DSPMode.SAM;
							break;
						case DSPMode.DRM:
							CurrentDSPMode = DSPMode.SPEC;
							break;
					}
				}
				else if(e.KeyCode == key_band_up)
				{
					switch(current_band)
					{
						case Band.B160M:
							if(band_160m_index == 2)
							{
								band_80m_index = 0;
								btnBand80_Click(this, new EventArgs());
							}
							else
							{
								btnBand160_Click(this, new EventArgs());
							}
							break;
						case Band.B80M:
							if(band_80m_index == 2)
							{
								band_60m_index = 0;
								btnBand60_Click(this, new EventArgs());
							}
							else
							{
								btnBand80_Click(this, new EventArgs());
							}
							break;
						case Band.B60M:
							if(band_60m_index == 4)
							{
								band_40m_index = 0;
								btnBand40_Click(this, new EventArgs());
							}
							else
							{
								btnBand60_Click(this, new EventArgs());
							}
							break;
						case Band.B40M:
							if(band_40m_index == 2)
							{
								band_30m_index = 0;
								btnBand30_Click(this, new EventArgs());
							}
							else
							{
								btnBand40_Click(this, new EventArgs());
							}
							break;
						case Band.B30M:
							if(band_30m_index == 2)
							{
								band_20m_index = 0;
								btnBand20_Click(this, new EventArgs());
							}
							else
							{
								btnBand30_Click(this, new EventArgs());
							}
							break;
						case Band.B20M:
							if(band_20m_index == 2)
							{
								band_17m_index = 0;
								btnBand17_Click(this, new EventArgs());
							}
							else
							{
								btnBand20_Click(this, new EventArgs());
							}
							break;
						case Band.B17M:
							if(band_17m_index == 2)
							{
								band_15m_index = 0;
								btnBand15_Click(this, new EventArgs());
							}
							else
							{
								btnBand17_Click(this, new EventArgs());
							}
							break;
						case Band.B15M:
							if(band_15m_index == 2)
							{
								band_12m_index = 0;
								btnBand12_Click(this, new EventArgs());
							}
							else
							{
								btnBand15_Click(this, new EventArgs());
							}
							break;
						case Band.B12M:
							if(band_12m_index == 2)
							{
								band_10m_index = 0;
								btnBand10_Click(this, new EventArgs());
							}
							else
							{
								btnBand12_Click(this, new EventArgs());
							}
							break;
						case Band.B10M:
							if(band_10m_index == 2)
							{
								band_6m_index = 0;
								btnBand6_Click(this, new EventArgs());
							}
							else
							{
								btnBand10_Click(this, new EventArgs());
							}
							break;
						case Band.B6M:
							if(band_6m_index == 2)
							{
								if(XVTRPresent)
								{
									band_2m_index = 0;
									btnBand2_Click(this, new EventArgs());
								}
								else
								{
									band_wwv_index = 0;
									btnBandWWV_Click(this, new EventArgs());
								}
							}
							else
							{
								btnBand6_Click(this, new EventArgs());
							}
							break;
						case Band.B2M:
							if(band_2m_index == 2)
							{
								band_wwv_index = 0;
								btnBandWWV_Click(this, new EventArgs());
							}
							else
							{
								btnBand2_Click(this, new EventArgs());
							}
							break;
						case Band.WWV:
							if(band_wwv_index == 4)
							{
								band_gen_index = 0;
								btnBandGEN_Click(this, new EventArgs());
							}
							else
							{
								btnBandWWV_Click(this, new EventArgs());
							}
							break;
						case Band.GEN:
							if(band_gen_index == 4)
							{
								band_160m_index = 0;
								btnBand160_Click(this, new EventArgs());
							}
							else
							{
								btnBandGEN_Click(this, new EventArgs());
							}
							break;
					}
				}
				else if(e.KeyCode == key_band_down)
				{
					switch(current_band)
					{
						case Band.B160M:
							if(band_160m_index == 0)
							{
								band_gen_index = 4;
								btnBandGEN_Click(this, new EventArgs());
							}
							else
							{
								last_band = "160M";
								band_160m_index = (band_160m_index+1)%3;
								btnBand160_Click(this, new EventArgs());
							}
							break;
						case Band.B80M:
							if(band_80m_index == 0)
							{
								band_160m_index = 2;
								btnBand160_Click(this, new EventArgs());
							}
							else
							{
								last_band = "80M";
								band_80m_index = (band_80m_index+1)%3;
								btnBand80_Click(this, new EventArgs());
							}
							break;
						case Band.B60M:
							if(band_60m_index == 0)
							{
								band_80m_index = 2;
								btnBand80_Click(this, new EventArgs());
							}
							else
							{
								last_band = "60M";
								band_60m_index = (band_60m_index+3)%5;
								btnBand60_Click(this, new EventArgs());
							}
							break;
						case Band.B40M:
							if(band_40m_index == 0)
							{
								band_60m_index = 4;
								btnBand60_Click(this, new EventArgs());
							}
							else
							{
								last_band = "40M";
								band_40m_index = (band_40m_index+1)%3;
								btnBand40_Click(this, new EventArgs());
							}
							break;
						case Band.B30M:
							if(band_30m_index == 0)
							{
								band_40m_index = 2;
								btnBand40_Click(this, new EventArgs());
							}
							else
							{
								last_band = "30M";
								band_30m_index = (band_30m_index+1)%3;
								btnBand30_Click(this, new EventArgs());
							}
							break;
						case Band.B20M:
							if(band_20m_index == 0)
							{
								band_30m_index = 2;
								btnBand30_Click(this, new EventArgs());
							}
							else
							{
								last_band = "20M";
								band_20m_index = (band_20m_index+1)%3;
								btnBand20_Click(this, new EventArgs());
							}
							break;
						case Band.B17M:
							if(band_17m_index == 0)
							{
								band_20m_index = 2;
								btnBand20_Click(this, new EventArgs());
							}
							else
							{
								last_band = "17M";
								band_17m_index = (band_17m_index+1)%3;
								btnBand17_Click(this, new EventArgs());
							}
							break;
						case Band.B15M:
							if(band_15m_index == 0)
							{
								band_17m_index = 2;
								btnBand17_Click(this, new EventArgs());
							}
							else
							{
								last_band = "15M";
								band_15m_index = (band_15m_index+1)%3;
								btnBand15_Click(this, new EventArgs());
							}
							break;
						case Band.B12M:
							if(band_12m_index == 0)
							{
								band_15m_index = 2;
								btnBand15_Click(this, new EventArgs());
							}
							else
							{
								last_band = "12M";
								band_12m_index = (band_12m_index+1)%3;
								btnBand12_Click(this, new EventArgs());
							}
							break;
						case Band.B10M:
							if(band_10m_index == 0)
							{
								band_12m_index = 2;
								btnBand12_Click(this, new EventArgs());
							}
							else
							{
								last_band = "10M";
								band_10m_index = (band_10m_index+1)%3;
								btnBand10_Click(this, new EventArgs());
							}
							break;
						case Band.B6M:
							if(band_6m_index == 0)
							{
								band_10m_index = 2;
								btnBand10_Click(this, new EventArgs());
							}
							else
							{
								last_band = "6M";
								band_6m_index = (band_6m_index+1)%3;
								btnBand6_Click(this, new EventArgs());
							}
							break;
						case Band.B2M:
							if(band_2m_index == 0)
							{
								band_6m_index = 2;
								btnBand6_Click(this, new EventArgs());
							}
							else
							{
								last_band = "2M";
								band_2m_index = (band_2m_index+1)%3;
								btnBand6_Click(this, new EventArgs());
							}
							break;
						case Band.WWV:
							if(band_wwv_index == 0)
							{
								if(xvtr_present)
								{
									band_2m_index = 2;
									btnBand2_Click(this, new EventArgs());
								}
								else
								{
									band_6m_index = 2;
									btnBand6_Click(this, new EventArgs());
								}
							}
							else
							{
								last_band = "WWV";
								band_wwv_index = (band_wwv_index+3)%5;
								btnBandWWV_Click(this, new EventArgs());
							}
							break;
						case Band.GEN:
							if(band_gen_index == 0)
							{
								band_wwv_index = 4;
								btnBandWWV_Click(this, new EventArgs());
							}
							else
							{
								last_band = "GEN";
								band_gen_index = (band_gen_index+3)%5;
								btnBandGEN_Click(this, new EventArgs());
							}
							break;
					}

				}
				else if(e.KeyCode == key_cw_dot)
				{
					CWForm.KBDot = true;
				}
				else if(e.KeyCode == key_cw_dash)
				{
					CWForm.KBDash = true;
				}
				else if((int)e.KeyCode >= 48 && (int)e.KeyCode <= 57)
				{
					txtVFOAFreq.Focus();
					txtVFOAFreq.Text = ((int)(e.KeyCode - 48)).ToString();
					txtVFOAFreq.Select(1,0);
				}
				else if((int)e.KeyCode >= 96 && (int)e.KeyCode <= 105)
				{
					txtVFOAFreq.Focus();
					txtVFOAFreq.Text = ((int)e.KeyCode-96).ToString();
					txtVFOAFreq.Select(1,0);
				}
				else if(e.KeyCode.Equals(Keys.Decimal) ||
					(separator == "." && e.KeyCode.Equals(Keys.OemPeriod)) ||
					(separator == "," && e.KeyCode.Equals(Keys.Oemcomma)))
				{
					txtVFOAFreq.Focus();
					txtVFOAFreq.Text = separator;
					txtVFOAFreq.Select(1,0);
				}
			}
		}

		private void Console_MouseWheel(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if(this.ActiveControl is TextBox && this.ActiveControl != txtVFOAFreq
				&& this.ActiveControl != txtVFOBFreq) return;
			if(this.ActiveControl is UpDownBase) return;

			int numberToMove = e.Delta / 120;	// 1 per click
			
			if(vfo_pixel_offset == 0)
				GetVFOCharWidth();
			
			if (numberToMove != 0) 
			{
				int left, right, top, bottom;
				left = grpVFOA.Left+txtVFOAFreq.Left;
				right = left + txtVFOAFreq.Width;
				top = grpVFOA.Top+txtVFOAFreq.Top;
				bottom = top + txtVFOAFreq.Height;

				if(e.X > left && e.X < right &&			// Update VFOA
					e.Y > top && e.Y < bottom)
				{	
					double freq = double.Parse(txtVFOAFreq.Text);
					double mult = 10.0;
					int x = vfo_pixel_offset+left - vfo_char_width - vfo_decimal_width;
					while(x < e.X && mult > 0.0000011)
					{
						mult /= 10;
						x += vfo_char_width;
						if(mult == 1.0)
							x += vfo_decimal_width;
					}
					freq += mult*numberToMove;
					VFOAFreq = freq;

				}
				else
				{
					left = grpVFOB.Left+txtVFOBFreq.Left;
					right = left + txtVFOBFreq.Width;
					top = grpVFOB.Top+txtVFOBFreq.Top;
					bottom = top + txtVFOBFreq.Height;
					if(e.X > left && e.X < right &&		// Update VFOB
						e.Y > top && e.Y < bottom)
					{
						double freq = double.Parse(txtVFOBFreq.Text);
						double mult = 1.0;
						int x = vfo_pixel_offset+left;
						while(x < e.X && mult > 0.0000011)
						{
							mult /= 10;
							x += vfo_char_width;
							if(mult == 1.0)
								x += vfo_decimal_width;
						}
						freq += mult*numberToMove;
						VFOBFreq = freq;
					}
					else
					{
						double freq = Double.Parse(txtVFOAFreq.Text);
						double mult = wheel_tune_list[wheel_tune_index];
						if(shift_down && mult >= 0.000009) mult /= 10;

						//						if(mult == 0.001 && (freq * 1000000)%1000 != 0)
						//						{
						//							if(numberToMove < 0)
						//							{
						//								double temp = freq - mult*0.5;
						//								temp = (double)Math.Round(temp, 4);
						//								freq = (double)Math.Round(temp, 3);
						//							}
						//							else
						//							{
						//								double temp = freq + mult*0.5;
						//								temp = (double)Math.Round(temp, 4);
						//								freq = (double)Math.Round(temp, 3);
						//							}						
						//						}
						//						else
						//							freq += mult*numberToMove;

						if(numberToMove < 0)
						{
							double temp = freq;
							numberToMove += 1;
							freq = Math.Floor(freq/mult);
							if ((double)Math.Round(freq*mult, 6) == temp) freq -= 1.0;
							freq = (freq+numberToMove)*mult;	
						}
						else
						{
							double temp = freq;
							if (numberToMove > 0) numberToMove -= 1;
							freq = Math.Floor((freq/mult) + 1.0);
							if ((double)Math.Round(freq*mult,6) == temp) freq += 1.0;
							freq = (freq+numberToMove)*mult;
						}

						VFOAFreq = freq;
					}
				}				
			}
		}

		// chkPower
		private void chkPower_CheckedChanged(object sender, System.EventArgs e)
		{
			DttSP.AudioReset();
			if(chkPower.Checked)
			{
				chkPower.Text = "On";
				chkPower.BackColor = button_selected_color;
				txtVFOAFreq.ForeColor = vfo_text_light_color;
				txtVFOABand.ForeColor = band_text_light_color;
				hw.PowerOn();
				txtVFOAFreq_LostFocus(this, new EventArgs());

				// wjt added 
				if ( PTTBitBangEnabled && serialPTT == null ) // we are enabled but don't have port object 
				{
					//Debug.WriteLine("Forcing property set on PTTBitBangEnabled"); 
					PTTBitBangEnabled = true; // force creation of serial ptt 
				}
				// wjt added ends 
				SetupForm.AudioReceiveMux1 = SetupForm.AudioReceiveMux1;		// set receive mux

				Audio.CurrentAudioState1 = Audio.AudioState.DTTSP;
				if(SetupForm.SoundCardIndex == (int)SoundCard.DELTA_44)
				{
					Audio.StartAudio(ref callback4port, (uint)block_size1, sample_rate1, audio_driver_index1,
						audio_input_index1, audio_output_index1, 4, 0, audio_latency1);
				}
				else
				{				
					Audio.StartAudio(ref callback1, (uint)block_size1, sample_rate1, audio_driver_index1,
						audio_input_index1, audio_output_index1, 2, 0, audio_latency1);

					Audio.CurrentAudioState2 = Audio.AudioState.NOL_NOR;
					if(second_sound_card_enabled)
						Audio.StartAudio(ref callback2, (uint)block_size2, sample_rate2, audio_driver_index2,
							audio_input_index2, audio_output_index2, 2, 1, audio_latency2);				
				}

				draw_display_thread = new Thread(new ThreadStart(RunDisplay));
				draw_display_thread.Name = "Draw Display Thread";
				draw_display_thread.Priority = ThreadPriority.BelowNormal;
				draw_display_thread.IsBackground = true;
				draw_display_thread.Start();

				multimeter_thread = new Thread(new ThreadStart(UpdateMultimeter));
				multimeter_thread.Name = "Multimeter Thread";
				multimeter_thread.Priority = ThreadPriority.Lowest;
				multimeter_thread.IsBackground = true;
				multimeter_thread.Start();

				poll_ptt_thread = new Thread(new ThreadStart(PollPTT));
				poll_ptt_thread.Name = "Poll PTT Thread";
				poll_ptt_thread.Priority = ThreadPriority.BelowNormal;
				poll_ptt_thread.IsBackground = true;
				poll_ptt_thread.Start();

				if(pa_present)
				{
					poll_pa_pwr_thread = new Thread(new ThreadStart(PollPAPWR));
					poll_pa_pwr_thread.Name = "Poll PA PWR Thread";
					poll_pa_pwr_thread.Priority = ThreadPriority.BelowNormal;
					poll_pa_pwr_thread.IsBackground = true;
					poll_pa_pwr_thread.Start();
				}

				if(!rx_only)
				{
					chkMOX.Enabled = true;
					chkTUN.Enabled = true;
				}
				chkMemoryScanner.Enabled = true;

				timer_peak_text.Enabled = true;

				if(atu_present)
				{
					int counter = 0;
					while(((hw.StatusPort() & (byte)StatusPin.PA_DATA)) == 0)
					{
						Thread.Sleep(50);
						if(counter++ > 100)		// 5 second time out
						{
							MessageBox.Show("ATU Initialization Timeout.\n"+
								"Please check power to radio",
								"ATU Init Error",
								MessageBoxButtons.OK,
								MessageBoxIcon.Hand);
							chkPower.Checked = false;
							return;
						}
					}
					hw.PA_ATUTune(ATUTuneMode.BYPASS);
				}
			}
			else
			{
                chkPower.Text = "Standby";

				if(CWForm.Send)
				{
					Audio.CurrentAudioState1 = Audio.AudioState.DTTSP;
					CWForm.Send = false;					
				}

				// wjt added 
				if ( serialPTT != null )  // let go of serial port
				{ 
					serialPTT.Destroy(); 
					serialPTT = null; 
				}
				// wjt added ends 

				chkMOX.Checked = false;
				chkMOX.Enabled = false;
				chkTUN.Checked = false;
				chkTUN.Enabled = false;
				if(MemoryScanner)
					MemoryScanner = false;
				chkMemoryScanner.Enabled = false;

				Audio.StopAudio1();
				if(second_sound_card_enabled)
					Audio.StopAudio2();
#if (NEW_DTTSP)
				DttSP.Suspend();
#endif
				
				hw.StandBy();
				chkPower.BackColor = SystemColors.Control;
				txtVFOAFreq.ForeColor = vfo_text_dark_color;
				txtVFOABand.ForeColor = band_text_dark_color;

				timer_peak_text.Enabled = false;

				ResetDisplayAverage();
			}

			panelVFOAHover.Invalidate();
			panelVFOBHover.Invalidate();
		}

		//txtVFOAFreq Events
		private void txtVFOAFreq_LostFocus(object sender, System.EventArgs e)
		{
			if(txtVFOAFreq.Text == ".") return;
			if(txtVFOAFreq.Text == "") txtVFOAFreq.Text = "0";
			
			double freq = double.Parse(txtVFOAFreq.Text);
			if(freq < min_freq) freq = min_freq;
			else if(freq > max_freq) freq = max_freq;
			txtVFOAFreq.Text = freq.ToString("f6");

			// update Band Info
			string bandInfo;
			bool transmit_allowed = DB.BandText(freq, out bandInfo);
			if(!transmit_allowed)
			{
				txtVFOABand.BackColor = Color.Red;
				if(!chkVFOSplit.Checked && chkMOX.Checked)
					chkMOX.Checked = false;
			}
			else txtVFOABand.BackColor = Color.Black;
			txtVFOABand.Text = bandInfo;
			SetCurrentBand((float)freq);
			SetBandButtonColor(current_band);

			if(CurrentBand == Band.B60M)
			{
				chkXIT.Enabled = false;
				chkXIT.Checked = false;
			}
			else
				chkXIT.Enabled = true;

			if(chkMOX.Checked &&
				(CurrentDSPMode == DSPMode.AM ||
				CurrentDSPMode == DSPMode.SAM ||
				CurrentDSPMode == DSPMode.FMN))
				freq -= 0.011025;

			if(current_dsp_mode == DSPMode.CWL)
				freq += (double)CWForm.CWPitch * 0.0000010;
			else if(current_dsp_mode == DSPMode.CWU)
				freq -= (double)CWForm.CWPitch * 0.0000010;

			if(freq < min_freq) freq = min_freq;
			else if(freq > max_freq) freq = max_freq;

			if(chkRIT.Checked && !chkMOX.Checked && !chkVFOSplit.Checked)
				freq += (int)udRIT.Value * 0.000001;
			else if(chkXIT.Checked && chkMOX.Checked && !chkVFOSplit.Checked)
				freq += (int)udXIT.Value * 0.000001;

			if(freq < min_freq) freq = min_freq;
			else if(freq > max_freq) freq = max_freq;
			
			if(chkPower.Checked)
			{
				DDSFreq = freq;
			}

			// BT 04/2005
			// raises the serial io event           }
			// EventArgs sioe = new EventArgs();
			// OnNotifySIO(sioe);
		}

		private void txtVFOAFreq_KeyPress(object sender, System.Windows.Forms.KeyPressEventArgs e)
		{
			string separator = System.Globalization.CultureInfo.CurrentCulture.NumberFormat.NumberDecimalSeparator;
			int KeyCode = (int)e.KeyChar;  
			if((KeyCode < 48 || KeyCode > 57) && KeyCode != 8 && !e.KeyChar.ToString().Equals(separator))
			{
				e.Handled=true;
			}
			else
			{
				if(e.KeyChar.ToString().Equals(separator))
				{
					e.Handled = (((TextBox)sender).Text.IndexOf(separator)>-1);
				}
			}
			if(e.KeyChar == (char)Keys.Enter)
			{
				txtVFOAFreq_LostFocus(txtVFOAFreq, new System.EventArgs());
				btnHidden.Focus();
			}
		}

		private void txtVFOAFreq_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if(this.ContainsFocus)
			{
				int old_digit = vfoa_hover_digit;
				int digit_index = 0;
				if(vfo_pixel_offset == 0)
					GetVFOCharWidth();

				int x = vfo_pixel_offset - vfo_char_width - vfo_decimal_width - 2;
				while(x < e.X)
				{
					digit_index++;
					x += vfo_char_width;
					if(digit_index == 1)
						x += vfo_decimal_width;
				}

				vfoa_hover_digit = digit_index;
				if(vfoa_hover_digit != old_digit)
					panelVFOAHover.Invalidate();
			}
		}

		private void txtVFOAFreq_MouseLeave(object sender, System.EventArgs e)
		{
			vfoa_hover_digit = -1;
			panelVFOAHover.Invalidate();
		}

		// txtVFOBFreq
		private void txtVFOBFreq_LostFocus(object sender, System.EventArgs e)
		{
			if(txtVFOBFreq.Text == "") txtVFOBFreq.Text = "0";
			if(txtVFOBFreq.Text == ".") return;

			double freq = double.Parse(txtVFOBFreq.Text);
			if(freq < min_freq) freq = min_freq; 
			else if(freq > max_freq) freq = max_freq; 
			txtVFOBFreq.Text = freq.ToString("f6"); 
			
			if(chkPower.Checked && chkMOX.Checked && chkVFOSplit.Checked)
			{
				if(current_dsp_mode == DSPMode.CWL)
					freq += (double)CWForm.CWPitch * 0.0000010;
				else if(current_dsp_mode == DSPMode.CWU)
					freq -= (double)CWForm.CWPitch * 0.0000010;
                			
				if(chkXIT.Checked)
					freq += (int)udXIT.Value * 0.0001;
					
				if(freq < min_freq) freq = min_freq;
				else if(freq > max_freq) freq = max_freq;
			
				DDSFreq = freq;
			}			

			// update Band Info
			string bandInfo;
			bool transmit = DB.BandText(freq, out bandInfo);
			if(transmit == false)
			{
				txtVFOBBand.BackColor = Color.Red;
				if(chkVFOSplit.Checked && chkMOX.Checked)
					chkMOX.Checked = false;
			}

			else txtVFOBBand.BackColor = Color.Black;
			txtVFOBBand.Text = bandInfo;

			//BT 04/2005
			// raises the serial io event
			// EventArgs sioe = new EventArgs();
			// OnNotifySIO(sioe);
		}

		private void txtVFOBFreq_KeyPress(object sender, System.Windows.Forms.KeyPressEventArgs e)
		{
			string separator = System.Globalization.CultureInfo.CurrentCulture.NumberFormat.NumberDecimalSeparator;
			int KeyCode = (int)e.KeyChar;  
			if((KeyCode < 48 || KeyCode > 57) && KeyCode != 8 && !e.KeyChar.ToString().Equals(separator))
			{
				e.Handled = true;
			}
			else
			{
				if(e.KeyChar.ToString().Equals(separator))
				{
					e.Handled = (((TextBox)sender).Text.IndexOf(separator)>-1);
				}
			}
			if(e.KeyChar == (char)Keys.Enter)
			{
				txtVFOAFreq_LostFocus(txtVFOAFreq, new System.EventArgs());
				btnHidden.Focus();
			}
		}

		private void txtVFOBFreq_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if(this.ContainsFocus)
			{
				int old_digit = vfob_hover_digit;
				int digit_index = 0;
				if(vfo_pixel_offset == 0)
					GetVFOCharWidth();

				int x = vfo_pixel_offset - vfo_char_width - vfo_decimal_width - 2;
				while(x < e.X)
				{
					digit_index++;
					x += vfo_char_width;
					if(digit_index == 1)
						x += vfo_decimal_width;
				}

				vfob_hover_digit = digit_index;
				if(vfob_hover_digit != old_digit)
					panelVFOBHover.Invalidate();
			}
		}

		private void txtVFOBFreq_MouseLeave(object sender, System.EventArgs e)
		{
			vfob_hover_digit = -1;
			panelVFOBHover.Invalidate();
		}

		private void panelVFOAHover_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			txtVFOAFreq_MouseMove(sender, new MouseEventArgs(MouseButtons.None, 0,
				e.X+panelVFOAHover.Left-10, e.Y+panelVFOAHover.Top, 0));
		}

		private void panelVFOBHover_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			txtVFOBFreq_MouseMove(sender, new MouseEventArgs(MouseButtons.None, 0,
				e.X+panelVFOBHover.Left-10, e.Y+panelVFOBHover.Top, 0));
		}

		private void picDisplay_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			try
			{
				switch(current_display_mode)
				{
					case DisplayMode.HISTOGRAM:
					case DisplayMode.PANADAPTER:
					case DisplayMode.SPECTRUM:
					case DisplayMode.WATERFALL:
						float x, y;
						display_cursor_x = e.X;
						display_cursor_y = e.Y;
						PixelToSpectrum(e.X, e.Y, out x, out y);
						double freq = double.Parse(txtVFOAFreq.Text) + (double)x*0.0000010;

						if(current_dsp_mode == DSPMode.CWL)
							freq += (double)cw_pitch*0.0000010;
						else if(current_dsp_mode == DSPMode.CWU)
							freq -= (double)cw_pitch*0.0000010;
						txtDisplayCurPos.Text = x.ToString("f1") + "Hz, " + y.ToString("f1") + "dBm, " + freq.ToString("f6") + "MHz";
						break;
					default:
						txtDisplayCurPos.Text = "";
						break;
				}
			}
			catch(Exception)
			{

			}
		}

		private void picDisplay_MouseLeave(object sender, System.EventArgs e)
		{
			txtDisplayCurPos.Text = "";
			display_cursor_x = -1;
			display_cursor_y = -1;
		}

		private void picDisplay_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if(e.Button == MouseButtons.Left)
			{
				if(long_crosshair)
				{
					switch(current_display_mode)
					{
						case DisplayMode.SPECTRUM:
						case DisplayMode.WATERFALL:
						case DisplayMode.HISTOGRAM:
						case DisplayMode.PANADAPTER:
							float x,y;
							PixelToSpectrum(e.X, e.Y, out x, out y);
							double freq = double.Parse(txtVFOAFreq.Text) + (double)x*0.0000010;
							if(current_dsp_mode == DSPMode.CWL)
								freq += (float)CWForm.CWPitch*0.0000010;
							else if(current_dsp_mode == DSPMode.CWU)
								freq -= (float)CWForm.CWPitch*0.0000010;
							VFOAFreq = Math.Round(freq, 6);;
							break;
						default:
							break;
					}
				}
			}
			else if(e.Button == MouseButtons.Right)
			{
				if(long_crosshair)
					long_crosshair = false;
				else long_crosshair = true;
			}
			else if(e.Button == MouseButtons.Middle)
				ChangeWheelTuneLeft();
		}

		private void comboDisplayMode_SelectedIndexChanged(object sender, System.EventArgs e)
		{			
			if(current_display_mode == DisplayMode.OFF)
			{
				draw_display_thread = new Thread(new ThreadStart(RunDisplay));
				draw_display_thread.Name = "Draw Display Thread";
				draw_display_thread.Priority = ThreadPriority.BelowNormal;
				draw_display_thread.Start();
			}
			
			switch(comboDisplayMode.Text)
			{
				case "Spectrum":
					current_display_mode = DisplayMode.SPECTRUM;
#if (NEW_DTTSP)
					DttSP.ChangePWSSubmode(0);
#endif
					break;
				case "Panadapter":
					current_display_mode = DisplayMode.PANADAPTER;
#if (NEW_DTTSP)
					DttSP.ChangePWSSubmode(1);
#endif
					break;
				case "Scope":
					current_display_mode = DisplayMode.SCOPE;
#if (NEW_DTTSP)
					DttSP.ChangePWSSubmode(2);
#endif
					break;
				case "Phase":
					current_display_mode = DisplayMode.PHASE;
#if (NEW_DTTSP)
					DttSP.ChangePWSSubmode(3);
#endif
					break;
				case "Phase2":
					current_display_mode = DisplayMode.PHASE2;
#if (NEW_DTTSP)
					DttSP.ChangePWSSubmode(4);
#endif
					break;
				case "Waterfall":
					current_display_mode = DisplayMode.WATERFALL;
#if (NEW_DTTSP)
					DttSP.ChangePWSSubmode(5);
#endif
					break;
				case "Histogram":
					current_display_mode = DisplayMode.HISTOGRAM;
#if (NEW_DTTSP)
					DttSP.ChangePWSSubmode(6);
#endif
					break;
				case "Off":
					current_display_mode = DisplayMode.OFF;
#if (NEW_DTTSP)
					DttSP.ChangePWSSubmode(7);
#endif
					break;
			}

			if(current_display_mode == DisplayMode.PHASE ||
				current_display_mode == DisplayMode.SCOPE ||
				current_display_mode == DisplayMode.OFF)
			{
				chkDisplayAVG.Enabled = false;
				chkDisplayAVG.Checked = false;
			}
			else
				chkDisplayAVG.Enabled = true;

			if(current_display_mode == DisplayMode.PANADAPTER)
				DttSP.ChangePWSmode(1);
			else DttSP.ChangePWSmode(0);

			if(current_display_mode == DisplayMode.PHASE2)
				Audio.phase = true;
			else Audio.phase = false;

			if(chkDisplayAVG.Checked)
				ResetDisplayAverage();

			DrawBackground();

			if(comboDisplayMode.Focused)
				btnHidden.Focus();
		}

		private void btnMemoryRecall_Click(object sender, System.EventArgs e)
		{
			if(MemForm.IsDisposed)
				MemForm = new Memory(this);
			MemForm.Show();
			MemForm.Focus();
		}

		private void chkBIN_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkBIN.Checked)
			{
				chkBIN.BackColor = button_selected_color;
#if(!NEW_DTTSP)
				chkNR.Enabled = false;
				chkNR.Checked = false;
				chkANF.Enabled = false;
				chkANF.Checked = false;
#endif
				DttSP.ChangeBIN(1);
			}
			else
			{
				chkBIN.BackColor = SystemColors.Control;
#if(!NEW_DTTSP)
				chkNR.Enabled = true;
				chkANF.Enabled = true;
#endif
				DttSP.ChangeBIN(0);
			}
		}

		private void chkNR_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkNR.Checked)
			{
				chkNR.BackColor = button_selected_color;
				DttSP.ChangeNR(1);
			}
			else
			{
				chkNR.BackColor = SystemColors.Control;
				DttSP.ChangeNR(0);
			}
		}

		private void chkANF_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkANF.Checked)
			{
				chkANF.BackColor = button_selected_color;
				DttSP.ChangeANF(1);
			}
			else
			{
				chkANF.BackColor = SystemColors.Control;
				DttSP.ChangeANF(0);
			}				
		}

		private void chkNB_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkNB.Checked)
			{
				chkNB.BackColor = button_selected_color;
				DttSP.ChangeNB(1);
			}
			else
			{
				chkNB.BackColor = SystemColors.Control;
				DttSP.ChangeNB(0);
			}
		}

		private void chkDSPComp_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkDSPComp.Checked)
			{
				chkDSPComp.BackColor = button_selected_color;
				DttSP.ChangeTXAGCFF(1);
			}
			else
			{
				chkDSPComp.BackColor = SystemColors.Control;
				DttSP.ChangeTXAGCFF(0);
			}
		}	

		private void comboAGC_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			DttSP.ChangeAGC((AGCMode)comboAGC.SelectedIndex);
			if(comboAGC.Focused)
				btnHidden.Focus();
		}

		private void Console_Closing(object sender, System.ComponentModel.CancelEventArgs e)
		{
			this.Hide();
			chkPower.Checked = false;
			btnFilterShiftReset_Click(this, new EventArgs());
			SaveState();
			CWForm.SaveCWOptions();
			SetupForm.SaveOptions();
		}

		private void comboPreamp_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			if(comboPreamp.Text == "Off")
				CurrentPreampMode = PreampMode.OFF;
			else if(comboPreamp.Text == "Low")
				CurrentPreampMode = PreampMode.LOW;
			else if(comboPreamp.Text == "Med")
				CurrentPreampMode = PreampMode.MED;
			else if(comboPreamp.Text == "High")
				CurrentPreampMode = PreampMode.HIGH;

			if(comboPreamp.Focused)
				btnHidden.Focus();
		}

		private void chkMUT_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkMUT.Checked)
				chkMUT.BackColor = button_selected_color;
			else
				chkMUT.BackColor = SystemColors.Control;

			if(SetupForm.SoundCardIndex != (int)SoundCard.DELTA_44)
			{
				hw.MuteRelay = chkMUT.Checked;
			}
			else	// Delta 44
			{
				if(chkMUT.Checked)
					Audio.MonitorVolume = 0.0;
				else
					udAF_ValueChanged(this, new EventArgs());
			}
		}

		private void udPWR_ValueChanged(object sender, System.EventArgs e)
		{
			if(SetupForm == null)
				return;

			if(pa_present && VFOAFreq < 29.7f)
			{
				if(udPWR.Value == 0)
				{
					DttSP.TXVolume = 0;
					return;
				}

				double target_dbm = 10*(double)Math.Log10((double)udPWR.Value/0.001);
				target_dbm -= GainByBand(CurrentBand);

				double target_volts = Math.Sqrt(Math.Pow(10, target_dbm/10)*0.05);		// E = Sqrt(P * R) 

				if(SetupForm.SoundCardIndex == (int)SoundCard.DELTA_44)
				{
					Audio.RadioVolume = target_volts/audio_volts1;
				}
				else
				{
					if(!second_sound_card_enabled)
						DttSP.TXVolume = target_volts/audio_volts1;
					else
						Audio.RadioVolume = target_volts/audio_volts2;
				}
			}
			else
			{
				if(SetupForm.SoundCardIndex == (int)SoundCard.DELTA_44)
					Audio.RadioVolume = (double)Math.Sqrt((double)udPWR.Value/100.0);
				else
				{
					if(!second_sound_card_enabled)
						DttSP.TXVolume = (double)Math.Sqrt((double)udPWR.Value/100.0);
					else
						Audio.RadioVolume = (double)Math.Sqrt((double)udPWR.Value/100.0);
				}
			}
			if(udPWR.Focused)
				btnHidden.Focus();
		}

		private void udAF_ValueChanged(object sender, System.EventArgs e)
		{
			if(SetupForm == null)
				return;

			if(SetupForm.SoundCardIndex != (int)SoundCard.DELTA_44)
			{
				DttSP.RXVolume = (double)((double)udAF.Value/100.0);
				if(second_sound_card_enabled)
				{
					Mixer.SetLineInRecordVolume(mixer_id2, sc2_cal[(int)(DttSP.RXVolume*100)]);
					if(chkMOX.Checked)
						DttSP.TXVolume = DttSP.RXVolume;
				}
			}
			else
			{
				if(chkMUT.Checked)
					Audio.MonitorVolume = 0.0;
                else if(!(chkMOX.Checked && (!chkMON.Checked)) && !chkMUT.Checked)
					Audio.MonitorVolume = (double)udAF.Value/100.0;
			}
		
			if(udAF.Focused)
				btnHidden.Focus();
		}

		private void udMIC_ValueChanged(object sender, System.EventArgs e)
		{
			if(second_sound_card_enabled)
				Mixer.SetMicRecordVolume(mixer_id2, (int)udMIC.Value);
			else
				Mixer.SetMicRecordVolume(mixer_id1, (int)udMIC.Value);

			if(udMIC.Focused)
				btnHidden.Focus();
		}

		private void udSquelch_ValueChanged(object sender, System.EventArgs e)
		{
			DttSP.ChangeSquelchVal(-(float)udSquelch.Value - preamp_offset[(int)current_preamp_mode] - multimeter_cal_offset);
			if(udSquelch.Focused)
				btnHidden.Focus();
		}

		private void WheelTune_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if(e.Button == MouseButtons.Middle)
				ChangeWheelTuneLeft();
		}

		private void chkPipe_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkPipe.Checked)
				Audio.CurrentAudioState1 = Audio.AudioState.PIPE;
			else
				Audio.CurrentAudioState1 = Audio.AudioState.DTTSP;
		}

		private void chkMON_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkMON.Checked)
				chkMON.BackColor = button_selected_color;
			else
				chkMON.BackColor = SystemColors.Control;

			if(SetupForm.SoundCardIndex == (int)SoundCard.DELTA_44)
			{
				if(!(chkMON.Checked == false && chkMOX.Checked))
					udAF_ValueChanged(this, new EventArgs());
				else
					Audio.MonitorVolume = 0.0;
			}
			else
			{
				if(chkPower.Checked && chkMOX.Checked)
					hw.MuteRelay = !chkMON.Checked;
			}
		}

		private void chkMOX_CheckedChanged(object sender, System.EventArgs e)
		{
			const int NUM_SWITCH_BUFFERS = 2;
//			bool cw = (current_dsp_mode == DSPMode.CWL) ||
//					  (current_dsp_mode == DSPMode.CWU);

			if(rx_only)
			{
				chkMOX.Checked = false;
				return;
			}

			double tx_vol = DttSP.TXVolume;
			double rx_vol = DttSP.RXVolume;

			meter_peak_count = multimeter_peak_hold_samples;		// reset multimeter peak

			if(chkMOX.Checked)
			{
				double freq = double.Parse(txtVFOAFreq.Text);
				if(chkVFOSplit.Checked)
				{
					freq = double.Parse(txtVFOBFreq.Text);
					if(current_dsp_mode == DSPMode.CWL)
						freq += (double)CWForm.CWPitch * 0.0000010;
					else if(current_dsp_mode == DSPMode.CWU)
						freq -= (double)CWForm.CWPitch * 0.0000010;
				}
				else if(chkXIT.Checked)
				{
					freq += (int)udXIT.Value * 0.000001;
				}

				if(!calibrating)
				{
					if(!IsHamBand(current_band_plan))	// out of band
					{
						MessageBox.Show("The frequency "+dds_freq.ToString("f6")+"MHz is not within the "+
							"IARU Band specifications.",
							"Transmit Error: Out Of Band",
							MessageBoxButtons.OK,
							MessageBoxIcon.Error);
						chkMOX.Checked = false;
						return;
					}

					if(btnBand60.BackColor == button_selected_color &&
						current_dsp_mode != DSPMode.USB)
					{
						MessageBox.Show(DttSP.CurrentMode.ToString()+" mode is not allowed on 60M band.",
							"Transmit Error: Mode/Band",
							MessageBoxButtons.OK,
							MessageBoxIcon.Error);
						chkMOX.Checked = false;
						return;
					}
				}

				hw.UpdateHardware = false;

				if(!cw_key_mode)
				{
					if(!second_sound_card_enabled)
					{
						if(Audio.CurrentAudioState1 == Audio.AudioState.DTTSP ||
							Audio.CurrentAudioState1 == Audio.AudioState.SWITCH) 
						{
							Audio.SwitchCount = 2048/BlockSize1*NUM_SWITCH_BUFFERS;
							Audio.NextAudioState1 = Audio.AudioState.DTTSP;
							Audio.CurrentAudioState1 = Audio.AudioState.SWITCH;
							Thread.Sleep(43);
							DttSP.SetTRX(DttSP.TransmitState.ON);
						}
					
						if(current_display_mode == DisplayMode.PANADAPTER ||
							current_display_mode == DisplayMode.SPECTRUM ||
							current_display_mode == DisplayMode.HISTOGRAM ||
							current_display_mode == DisplayMode.WATERFALL)
							DrawBackground();

						Mixer.SetMux(mixer_id1, mixer_tx_mux_id1);
					}
					else
					{
						DttSP.AudioReset();
						Mixer.SetMux(mixer_id2, mixer_tx_mux_id2);
						if(Audio.CurrentAudioState1 == Audio.AudioState.DTTSP ||
							Audio.CurrentAudioState1 == Audio.AudioState.SWITCH) 
						{
							Audio.CurrentAudioState1 = Audio.AudioState.NOL_NOR;
							Audio.SwitchCount = 2048/BlockSize2*NUM_SWITCH_BUFFERS;
							Audio.NextAudioState2 = Audio.AudioState.DTTSP;
							Audio.CurrentAudioState2 = Audio.AudioState.SWITCH;
							Thread.Sleep(43);
							DttSP.SetTRX(DttSP.TransmitState.ON);
						}

						if(current_display_mode == DisplayMode.PANADAPTER ||
							current_display_mode == DisplayMode.SPECTRUM ||
							current_display_mode == DisplayMode.HISTOGRAM ||
							current_display_mode == DisplayMode.WATERFALL)
							DrawBackground();
					}
				}				

				comboMeterRXMode.ForeColor = Color.Gray;
				comboMeterTXMode.ForeColor = Color.Black;
				comboMeterTXMode_SelectedIndexChanged(this, new EventArgs());

				if(SetupForm.SoundCardIndex != (int)SoundCard.DELTA_44)
				{
					DttSP.TXVolume = 0;
					DttSP.RXVolume = 0;
					Mixer.SetMainMute(mixer_id1, true);
					if(second_sound_card_enabled)
						Mixer.SetMainMute(mixer_id2, true);
				}
				else
				{
					Audio.RadioVolume = 0;
					Audio.MonitorVolume = 0;
				}
				
				if(SetupForm.SoundCardIndex == (int)SoundCard.AUDIGY_2 ||
					SetupForm.SoundCardIndex == (int)SoundCard.AUDIGY_2_ZS)
					Mixer.SetLineInMute(mixer_id1, true);

				if(current_dsp_mode == DSPMode.AM ||
					current_dsp_mode == DSPMode.SAM ||
					current_dsp_mode == DSPMode.FMN)
				{
					freq -= 0.011025;
				}

				chkMOX.BackColor = button_selected_color;

				SetupForm.SpurRedEnabled = false;
				spur_reduction = false;
				if_shift = false;

				if(x2_enabled)
				{
					hw.X2 |= 1 << 6;
					Thread.Sleep(x2_delay);
				}

				DDSFreq = freq;
				hw.MuteRelay = !chkMON.Checked;
				if(rfe_present)
				{					
					hw.GainRelay = true;		// 0dB
					hw.Attn = false;
					
					if(xvtr_present && freq >= 144.0)
					{
						hw.XVTR_RF = true;
						if(current_xvtr_tr_mode == XVTRTRMode.POSITIVE)
							hw.XVTR_TR = true;
						else if(current_xvtr_tr_mode == XVTRTRMode.NEGATIVE)
							hw.XVTR_TR = false;
					}
					else
					{
						hw.RFE_TR = true;
						if(pa_present)
						{
							hw.PA_TR_Relay = true;
							hw.PABias = true;
						}
					}
					
				}
				else
					hw.GainRelay = false;		// 26dB
									
				hw.TransmitRelay = true;	
				hw.UpdateHardware = true;
						
				DisableAllBands();
				DisableAllModes();
				chkVFOSplit.Enabled = false;
				btnVFOAtoB.Enabled = false;
				btnVFOBtoA.Enabled = false;
				btnVFOSwap.Enabled = false;
				chkMemoryScanner.Enabled = false;
				chkPower.BackColor = Color.Red;
				
				if(cw_key_mode)
				{
					if(!second_sound_card_enabled)
					{
						if(!chkTUN.Checked)
						{
							Audio.SwitchCount  = 2048/BlockSize1*NUM_SWITCH_BUFFERS;
							Audio.NextAudioState1 = Audio.AudioState.SWITCH;
							Audio.CurrentAudioState1 = Audio.AudioState.CW;
						}
					}
					else
					{
						DttSP.AudioReset();
						if(!chkTUN.Checked)
						{
							Mixer.SetMux(mixer_id2, mixer_rx_mux_id2);
							Audio.SwitchCount = 2048/BlockSize1*2;
							Audio.NextAudioState1 = Audio.AudioState.SWITCH;
							Audio.CurrentAudioState1 = Audio.AudioState.CW;
							Audio.CurrentAudioState2 = Audio.AudioState.CW;
						}
					}
				}

				if(SetupForm.SoundCardIndex != (int)SoundCard.DELTA_44)
				{
					Mixer.SetMainMute(mixer_id1, false);
					if(second_sound_card_enabled)
						Mixer.SetMainMute(mixer_id2, false);
					DttSP.RXVolume = rx_vol;
					DttSP.TXVolume = tx_vol;
				}
				else
				{
					udPWR_ValueChanged(this, new EventArgs());
					udAF_ValueChanged(this, new EventArgs());
				}
			}
			else
			{    // Going from TX to RX
				hw.UpdateHardware = false;
				if(SetupForm.SoundCardIndex != (int)SoundCard.DELTA_44)
				{
					DttSP.TXVolume = 0;
					DttSP.RXVolume = 0;
					Mixer.SetMainMute(mixer_id1, true);
					if(second_sound_card_enabled)
						Mixer.SetMainMute(mixer_id2, true);
				}
				else
				{
					Audio.RadioVolume = 0;
					Audio.MonitorVolume = 0;
				}
				
				if(CWForm.Send)
					CWForm.Send = false;

				if(rfe_present)
				{
					if(xvtr_present && hw.XVTR_RF)
					{
						hw.XVTR_RF = false;
						if(current_xvtr_tr_mode == XVTRTRMode.POSITIVE)
							hw.XVTR_TR = false;
						else if(current_xvtr_tr_mode == XVTRTRMode.NEGATIVE)
							hw.XVTR_TR = true;
					}
					else
					{				
						hw.RFE_TR = false;
						if(pa_present)
						{
							hw.PABias = false;
							hw.PA_TR_Relay = false;
						}
					}
				}
				hw.TransmitRelay = false;

				if(x2_enabled)
				{
					Thread.Sleep(x2_delay);
					hw.X2 &= ~(1 << 6);
				}

				if(!second_sound_card_enabled)
				{
					if(cw_key_mode)
					{
						Audio.SwitchCount = 2048/BlockSize1*NUM_SWITCH_BUFFERS;
						Audio.NextAudioState1 = Audio.AudioState.DTTSP;
						Audio.CurrentAudioState1 = Audio.AudioState.SWITCH;
						Thread.Sleep(43);
						DttSP.SetTRX(DttSP.TransmitState.OFF);
					}
					else
					{
						if(Audio.CurrentAudioState1 == Audio.AudioState.DTTSP ||
							Audio.CurrentAudioState1 == Audio.AudioState.SWITCH)
						{
							Audio.SwitchCount = 2048/BlockSize1*NUM_SWITCH_BUFFERS;
							Audio.NextAudioState1 = Audio.AudioState.DTTSP;
							Audio.CurrentAudioState1 = Audio.AudioState.SWITCH;
							Thread.Sleep(43);
							DttSP.SetTRX(DttSP.TransmitState.OFF);
							DttSP.AudioReset();
						}
						if(current_display_mode == DisplayMode.PANADAPTER ||
							current_display_mode == DisplayMode.SPECTRUM ||
							current_display_mode == DisplayMode.HISTOGRAM ||
							current_display_mode == DisplayMode.WATERFALL)
							DrawBackground();
					}
				}
				else
				{
					if(cw_key_mode)
					{
						Audio.CurrentAudioState2 = Audio.AudioState.NOL_NOR;
						Audio.SwitchCount = 2048/BlockSize1*NUM_SWITCH_BUFFERS;
						Audio.NextAudioState1 = Audio.AudioState.DTTSP;
						Audio.CurrentAudioState1 = Audio.AudioState.SWITCH;
						Thread.Sleep(43);
						DttSP.SetTRX(DttSP.TransmitState.OFF);
					}
					else
					{
						DttSP.SetTRX(DttSP.TransmitState.OFF);
						Audio.CurrentAudioState2 = Audio.AudioState.NOL_NOR;
						Audio.SwitchCount = 2048/BlockSize1*NUM_SWITCH_BUFFERS;
						Audio.CurrentAudioState1 = Audio.AudioState.SWITCH;
						Audio.NextAudioState1 = Audio.AudioState.DTTSP;
					}
					DttSP.AudioReset();
					if(current_display_mode == DisplayMode.PANADAPTER ||
						current_display_mode == DisplayMode.SPECTRUM ||
						current_display_mode == DisplayMode.HISTOGRAM ||
						current_display_mode == DisplayMode.WATERFALL)
						DrawBackground();
				}

				comboMeterTXMode.ForeColor = Color.Gray;
				comboMeterRXMode.ForeColor = Color.Black;
				comboMeterRXMode_SelectedIndexChanged(this, new EventArgs());

				chkMOX.BackColor = SystemColors.Control;
				spur_reduction = SetupForm.chkGeneralSpurRed.Checked;
				if(!spur_reduction &&
					(current_dsp_mode == DSPMode.AM ||
					current_dsp_mode == DSPMode.SAM ||
					current_dsp_mode == DSPMode.FMN))
					DttSP.ChangeOsc(-11025.0);
				
				SetupForm.SpurRedEnabled = true;
				if(current_dsp_mode != DSPMode.DRM &&
					current_dsp_mode != DSPMode.SPEC)
					if_shift = true;

				txtVFOAFreq_LostFocus(this, new EventArgs());
				
				Mixer.SetMux(mixer_id1, mixer_rx_mux_id1);
				
				EnableAllBands();
				EnableAllModes();

				chkVFOSplit.Enabled = true;
				btnVFOAtoB.Enabled = true;
				btnVFOBtoA.Enabled = true;
				btnVFOSwap.Enabled = true;
				chkMemoryScanner.Enabled = true;
				chkPower.BackColor = button_selected_color;
				if(!chkMUT.Checked)
					hw.MuteRelay = false;

				hw.UpdateHardware = true;

				if(SetupForm.SoundCardIndex == (int)SoundCard.AUDIGY_2 ||
					SetupForm.SoundCardIndex == (int)SoundCard.AUDIGY_2_ZS)
					Mixer.SetLineInMute(mixer_id1, false);

				CurrentPreampMode = current_preamp_mode;


				if(SetupForm.SoundCardIndex != (int)SoundCard.DELTA_44)
				{
					Mixer.SetMainMute(mixer_id1, false);
					if(second_sound_card_enabled)
						Mixer.SetMainMute(mixer_id2, false);
					DttSP.RXVolume = rx_vol;
					DttSP.TXVolume = tx_vol;
				}
				else
				{
					udPWR_ValueChanged(this, new EventArgs());
					udAF_ValueChanged(this, new EventArgs());
				}

				pa_fwd_power = 0;
				pa_rev_power = 0;

				DttSP.PAHighSWRScale = 1.0;
				HighSWR = false;

				for(int i=0; i<meter_text_history.Length; i++)
					meter_text_history[i] = 0.0f;
			}	

			SetupForm.MOX = chkMOX.Checked;
		}

		private void chkMOX_Click(object sender, System.EventArgs e)
		{
			if(chkMOX.Checked)			// because the CheckedChanged event fires first
				manual_mox = true;
			else
			{
				manual_mox = false;
				if(chkTUN.Checked)
					chkTUN.Checked = false;
			}
		}

		private void picMultimeter_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
		{
			int x1, x2, y1, y2;
			Pen pen = new Pen(Color.Red);

			x1 = picMultimeterAnalog.Width/2;
			y1 = (int)(picMultimeterAnalog.Height*1.2);

			float val = (float)(display_max_y - 80) / 100;
			int angle = (int)(-45 + 90*val);

			x2 = (int)(x1 - picMultimeterAnalog.Height*1.1*Math.Cos(Math.PI/2+angle*Math.PI/180));
			y2 = (int)(y1 - picMultimeterAnalog.Height*1.1*Math.Sin(Math.PI/2+angle*Math.PI/180));

			e.Graphics.DrawLine(pen, x1, y1, x2, y2);
		}

		private void comboMeterRXMode_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			MeterRXMode mode = (MeterRXMode)comboMeterRXMode.SelectedIndex;
			current_meter_rx_mode = mode;
			if(!chkMOX.Checked)
			{
				switch(mode)
				{
					case MeterRXMode.SIGNAL_STRENGTH:
					case MeterRXMode.SIGNAL_AVERAGE:
						lblMultiSMeter.Text = "  1   3   5   7   9  +20 +40 +60";
						break;
					case MeterRXMode.ADC_L:
					case MeterRXMode.ADC_R:
						lblMultiSMeter.Text = "-100  -80   -60   -40   -20    0";
						break;
				}
			}

			if(comboMeterRXMode.Focused)
				btnHidden.Focus();
		}

		private void comboMeterTXMode_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			if(comboMeterTXMode.Items.Count == 0)
				current_meter_tx_mode = MeterTXMode.FIRST;
			else
			{
				MeterTXMode mode = MeterTXMode.FIRST;
				switch(comboMeterTXMode.Text)
				{
					case "ALC":
						mode = MeterTXMode.ALC;
						break;
					case "Fwd Pow":
						mode = MeterTXMode.FORWARD_POWER;
						break;
					case "Peak Pow":
						mode = MeterTXMode.PEAK_POWER;
						break;
					case "Rev Pow":
						mode = MeterTXMode.REVERSE_POWER;
						break;
					case "SWR":
						mode = MeterTXMode.SWR;
						break;
				}
				current_meter_tx_mode = mode;
			}

			if(chkMOX.Checked)
			{
				switch(current_meter_tx_mode)
				{
					case MeterTXMode.FIRST:
						lblMultiSMeter.Text = "";
						break;
					case MeterTXMode.ALC:
						lblMultiSMeter.Text = "-20   -10    0      5      10    20";
						break;
					case MeterTXMode.FORWARD_POWER:
					case MeterTXMode.PEAK_POWER:
						if(pa_present)
							lblMultiSMeter.Text = "1      5     10    50   100  120+";
						else
							lblMultiSMeter.Text = "0      0.1     0.2     0.5        1.0";
						break;
					case MeterTXMode.REVERSE_POWER:
						lblMultiSMeter.Text = "1      5     10    50   100  120+";
						break;					
					case MeterTXMode.SWR:
						lblMultiSMeter.Text = "1      1.5   2     3     5    10";
						break;
				}
			}

			if(comboMeterTXMode.Focused)
				btnHidden.Focus();
		}

		private void chkDisplayAVG_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkDisplayAVG.Checked)
			{
				chkDisplayAVG.BackColor = button_selected_color;
				ResetDisplayAverage();
			}
			else
			{
				chkDisplayAVG.BackColor = SystemColors.Control;
			}

			average_display = chkDisplayAVG.Checked;
		}

		private void chkSquelch_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkSquelch.Checked)
			{
				chkSquelch.BackColor = button_selected_color;
				DttSP.ChangeSquelchState(1);
			}
			else
			{
				chkSquelch.BackColor = SystemColors.Control;
				DttSP.ChangeSquelchState(0);
			}
		}

		private void chkTUN_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkTUN.Checked)
			{
				if(!chkPower.Checked)
				{
					MessageBox.Show("Power must be on to turn on the Tune function.",
						"Power is off",
						MessageBoxButtons.OK,
						MessageBoxIcon.Hand);
					chkTUN.Checked = false;
					return;
				}

				if(!second_sound_card_enabled)
					Audio.CurrentAudioState1 = Audio.AudioState.SINL_COSR;
				else 
				{
					Audio.CurrentAudioState1 = Audio.AudioState.NOL_NOR;
					Audio.CurrentAudioState2 = Audio.AudioState.SINL_COSR;
				}
				CWForm.SendEnabled = false;
				PreviousPWR = PWR;
				PWR = tune_power;
				chkMOX.Checked = true;				
				if(!chkMOX.Checked)
				{
					chkTUN.Checked = false;
					return;
				}		

				if(atu_present)
				{
					switch((ATUTuneMode)comboTuneMode.SelectedIndex)
					{
						case ATUTuneMode.MEMORY:
							PWR = 16;
							atu_tuning = true;
							if(!hw.PA_ATUTune(ATUTuneMode.MEMORY))
								goto atu_error;
							chkTUN.Checked = false;
							tuned_band = current_band;
							break;
						case ATUTuneMode.FULL:
							PWR = 16;
							atu_tuning = true;
							if(!hw.PA_ATUTune(ATUTuneMode.FULL))
								goto atu_error;
							chkTUN.Checked = false;
							tuned_band = current_band;
							break;
					}
					return;
					
				atu_error:
					chkTUN.Checked = false;
					MessageBox.Show("Error communicating with the ATU",
						"ATU Error",
						MessageBoxButtons.OK,
						MessageBoxIcon.Error);
					return;
				}

				if(pa_present && !comboMeterTXMode.Items.Contains("SWR"))
				{
					int index = comboMeterTXMode.SelectedIndex;
					comboMeterTXMode.Items.Add("SWR");
					comboMeterTXMode.SelectedIndex = index;
				}
				chkTUN.BackColor = button_selected_color;
			}
			else
			{
				chkMOX.Checked = false;
				if(!second_sound_card_enabled)
				{
					Audio.NextAudioState1 = Audio.AudioState.DTTSP;
					Audio.SwitchCount = 2048/BlockSize1*2;
					Audio.CurrentAudioState1 = Audio.AudioState.SWITCH;
				}
				else
				{
					Audio.NextAudioState1 = Audio.AudioState.DTTSP;
					Audio.SwitchCount = 2048/BlockSize1*2;
					Audio.CurrentAudioState1 = Audio.AudioState.SWITCH;
					Audio.CurrentAudioState2 = Audio.AudioState.NOL_NOR;
				}

				CWForm.SendEnabled = true;

				if(atu_present)
				{
					atu_tuning = false;
					switch((ATUTuneMode)comboTuneMode.SelectedIndex)
					{
						case ATUTuneMode.BYPASS:
							tune_power = PWR;
							PWR = PreviousPWR;
							break;
						case ATUTuneMode.MEMORY:
							atu_tuning = false;
							PWR = PreviousPWR;
							break;
						case ATUTuneMode.FULL:
							atu_tuning = false;
							PWR = PreviousPWR;
							break;
					}
				}
				else
				{
					chkTUN.BackColor = SystemColors.Control;
					if(pa_present && comboMeterTXMode.Items.Contains("SWR"))
					{
						comboMeterTXMode.Items.Remove("SWR");
						if(comboMeterTXMode.SelectedIndex < 0)
							comboMeterTXMode.SelectedIndex = 0;
					}
					TunePower = PWR;
					PWR = PreviousPWR;
				}
			}
		}

		private void comboTuneMode_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			if(atu_present)
			{
				if(comboTuneMode.SelectedIndex == (int)ATUTuneMode.BYPASS)
				{
					hw.PA_ATUTune(ATUTuneMode.BYPASS);	
					tuned_band = Band.FIRST;
				}
			}

			if(comboTuneMode.Focused)
				btnHidden.Focus();
		}

		private void chkDSPNB2_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkDSPNB2.Checked)
			{
				chkDSPNB2.BackColor = button_selected_color;
				DttSP.ChangeSDROM(1);
			}
			else
			{
				chkDSPNB2.BackColor = SystemColors.Control;
				DttSP.ChangeSDROM(0);
			}
		}

		private void HideFocus(object sender, EventArgs e)
		{
			btnHidden.Focus();
		}

		// BT 04/2005
		// method called to raise the event that notifies the serial port to read the status
		protected virtual void OnNotifySIO(EventArgs e)
		{
			SIOEventHandler seh = NotifySIO;
			if(seh != null)
			{
				seh(this,e);
			}
		}

		#endregion

		#region Band Button Events
// ======================================================
// Band Button Events
// ======================================================

		private void btnBand160_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("160M"))
				band_160m_index = (band_160m_index+1)%band_160m_register;
			last_band = "160M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_160m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}
			
			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBand80_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("80M"))
				band_80m_index = (band_80m_index+1)%band_80m_register;
			last_band = "80M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_80m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBand60_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("60M"))
				band_60m_index = (band_60m_index+1)%band_60m_register;
			last_band = "60M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_60m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBand40_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("40M"))
				band_40m_index = (band_40m_index+1)%band_40m_register;
			last_band = "40M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_40m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBand30_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("30M"))
				band_30m_index = (band_30m_index+1)%band_30m_register;
			last_band = "30M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_30m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBand20_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("20M"))
				band_20m_index = (band_20m_index+1)%band_20m_register;
			last_band = "20M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_20m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBand17_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("17M"))
				band_17m_index = (band_17m_index+1)%band_17m_register;
			last_band = "17M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_17m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBand15_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("15M"))
				band_15m_index = (band_15m_index+1)%band_15m_register;
			last_band = "15M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_15m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBand12_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("12M"))
				band_12m_index = (band_12m_index+1)%band_12m_register;
			last_band = "12M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_12m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBand10_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("10M"))
				band_10m_index = (band_10m_index+1)%band_10m_register;
			last_band = "10M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_10m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBand6_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("6M"))
				band_6m_index = (band_6m_index+1)%band_6m_register;
			last_band = "6M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_6m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBand2_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("2M"))
				band_2m_index = (band_2m_index+1)%band_2m_register;
			last_band = "2M";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_2m_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBandWWV_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band.Equals("WWV"))
				band_wwv_index = (band_wwv_index+1)%band_wwv_register;
			last_band = "WWV";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_wwv_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		private void btnBandGEN_Click(object sender, System.EventArgs e)
		{
			hw.UpdateHardware = false;
			SaveBand();
			if(last_band == "GEN")
				band_gen_index = (band_gen_index+1)%band_gen_register;
			last_band = "GEN";

			string filter, mode;
			double freq;
			if(DB.GetBandStack(last_band, band_gen_index, out mode, out filter, out freq))
			{
				SetBand(mode, filter, freq);
			}

			if(chkPower.Checked)
				hw.UpdateHardware = true;
		}

		#endregion

		#region Mode Button Events
// ======================================================
// Mode Button Events
// ======================================================

		private void ChangeMode(DSPMode new_mode)
		{
			if(new_mode == DSPMode.RTTY || new_mode == DSPMode.PSK)
				DttSP.CurrentMode = DttSP.Mode.LSB;
			else
				DttSP.CurrentMode = (DttSP.Mode)new_mode;				// set new DSP mode

			float freq = float.Parse(txtVFOAFreq.Text);

			// close CW Form if old mode was CW and new mode is not
			if((current_dsp_mode == DSPMode.CWL || current_dsp_mode == DSPMode.CWU) &&
				(new_mode != DSPMode.CWL && new_mode != DSPMode.CWU))
				CWForm.Close();

			tbFilterShift.Value = 0;
			btnFilterShiftReset.BackColor = SystemColors.Control;

			switch(current_dsp_mode)
			{
				case DSPMode.LSB:
					radModeLSB.BackColor = SystemColors.Control;
					break;
				case DSPMode.USB:
					radModeUSB.BackColor = SystemColors.Control;
					break;
				case DSPMode.DSB:
					radModeDSB.BackColor = SystemColors.Control;
					break;
				case DSPMode.CWL:
					radModeCWL.BackColor = SystemColors.Control;
					//Keyer.PollPort.Suspend();

				switch(new_mode)
				{
					case DSPMode.USB:
						freq -= (float)(cw_pitch*0.0000010);
						break;
					case DSPMode.CWU:
						break;
					default:
						freq += (float)(cw_pitch*0.0000010);
						break;
				}
					txtVFOAFreq.Text = freq.ToString("f6");
					break;
				case DSPMode.CWU:
					radModeCWU.BackColor = SystemColors.Control;
					//Keyer.PollPort.Suspend();

				switch(new_mode)
				{
					case DSPMode.LSB:
						freq += (float)(cw_pitch*0.0000010);
						break;
					case DSPMode.CWL:
						break;
					default:
						freq -= (float)(cw_pitch*0.0000010);
						break;
				}
					txtVFOAFreq.Text = freq.ToString("f6");
					break;
				case DSPMode.FMN:
					radModeFMN.BackColor = SystemColors.Control;
					chkMON.Enabled = true;
					break;
				case DSPMode.AM:
					radModeAM.BackColor = SystemColors.Control;
					chkMON.Enabled = true;
					break;
				case DSPMode.SAM:
					radModeSAM.BackColor = SystemColors.Control;
					chkMON.Enabled = true;
					break;
				case DSPMode.SPEC:
					radModeSPEC.BackColor = SystemColors.Control;
					comboDisplayMode.Items.Insert(1, "Panadapter");
					tbFilterShift.Enabled = true;
					btnFilterShiftReset.Enabled = true;
					if(new_mode != DSPMode.DRM)
						EnableAllFilters();
					if_shift = true;
					if(!spur_reduction)
						DttSP.ChangeOsc(-11025.0);
					break;
				case DSPMode.RTTY:
					radModeRTTY.BackColor = SystemColors.Control;
					break;
				case DSPMode.PSK:
					radModePSK.BackColor = SystemColors.Control;
					break;
				case DSPMode.DRM:
					radModeDRM.BackColor = SystemColors.Control;
					tbFilterShift.Enabled = true;
					btnFilterShiftReset.Enabled = true;
					if(new_mode != DSPMode.SPEC)
						EnableAllFilters();
					vfo_offset = 0.0;
					if_shift = true;
					break;
			}

			switch(new_mode)
			{
				case DSPMode.LSB:
					radModeLSB.BackColor = button_selected_color;
					grpMode.Text = "Mode - LSB";
					DttSP.ChangeTXOsc(0.0);
					if(!rx_only)
						chkMOX.Enabled = true;
					DttSP.ChangeTXFilters(tx_filter_low, tx_filter_high);
					SetupForm.UpdateEQ();
					break;
				case DSPMode.USB:
					radModeUSB.BackColor = button_selected_color;
					grpMode.Text = "Mode - USB";
					DttSP.ChangeTXOsc(0.0);
					if(!rx_only)
						chkMOX.Enabled = true;
					DttSP.ChangeTXFilters(tx_filter_low, tx_filter_high);
					SetupForm.UpdateEQ();
					break;
				case DSPMode.DSB:
					radModeDSB.BackColor = button_selected_color;
					grpMode.Text = "Mode - DSB";
					DttSP.ChangeTXOsc(0.0);
					if(!rx_only)
						chkMOX.Enabled = true;
					DttSP.ChangeTXFilters(tx_filter_low, tx_filter_high);
					SetupForm.UpdateEQ();
					break;
				case DSPMode.CWL:
					radModeCWL.BackColor = button_selected_color;
					grpMode.Text = "Mode - CWL";
					DttSP.ChangeTXOsc(0.0);
					if(!rx_only) 
					{
						chkMOX.Enabled = true;
						//Keyer.PollPort.Resume();
					}
					DttSP.ChangeTXFilters(tx_filter_low, tx_filter_high);
					SetupForm.UpdateEQ();
					

					switch(current_dsp_mode)
					{
						case DSPMode.USB:
							freq += (float)(cw_pitch*0.0000010);
							break;
						case DSPMode.CWU:
							break;
						default:
							freq -= (float)(cw_pitch*0.0000010);
							break;
					}
					txtVFOAFreq.Text = freq.ToString("f6");
					break;
				case DSPMode.CWU:
					radModeCWU.BackColor = button_selected_color;
					grpMode.Text = "Mode - CWU";
					DttSP.ChangeTXOsc(0.0);
					if(!rx_only) 
					{
						chkMOX.Enabled = true;
						//Keyer.PollPort.Resume();
					}
					DttSP.ChangeTXFilters(tx_filter_low, tx_filter_high);
					SetupForm.UpdateEQ();

					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
							freq -= (float)(cw_pitch*0.0000010);
							break;
						case DSPMode.CWL:
							break;
						default:
							freq += (float)(cw_pitch*0.0000010);
							break;
					}
					txtVFOAFreq.Text = freq.ToString("f6");
					break;
				case DSPMode.FMN:
					radModeFMN.BackColor = button_selected_color;
					grpMode.Text = "Mode - FMN";
					if(!rx_only)
						chkMOX.Enabled = true;
					chkMON.Checked = false;
					chkMON.Enabled = false;
					DttSP.ChangeTXFilters(tx_filter_low, tx_filter_high);
					SetupForm.UpdateEQ();
					DttSP.ChangeTXOsc(11025.0);
					break;
				case DSPMode.AM:
					radModeAM.BackColor = button_selected_color;
					grpMode.Text = "Mode - AM";
					if(!rx_only)
						chkMOX.Enabled = true;
					chkMON.Checked = false;
					chkMON.Enabled = false;
					DttSP.ChangeTXFilters(tx_filter_low, tx_filter_high);
					SetupForm.UpdateEQ();
					DttSP.ChangeTXOsc(11025.0);
					break;
				case DSPMode.SAM:
					radModeSAM.BackColor = button_selected_color;
					grpMode.Text = "Mode - SAM";
					if(!rx_only)
						chkMOX.Enabled = true;
					chkMON.Checked = false;
					chkMON.Enabled = false;
					DttSP.ChangeTXFilters(tx_filter_low, tx_filter_high);
					SetupForm.UpdateEQ();
					DttSP.ChangeTXOsc(11025.0);
					break;
				case DSPMode.SPEC:
					radModeSPEC.BackColor = button_selected_color;
					grpMode.Text = "Mode - SPEC";
					if_shift = false;
					DttSP.ChangeTXOsc(0.0);			
					DisableAllFilters();
					grpFilter.Text = "Filter - "+(DttSP.SampleRate/1000).ToString("f0")+"KHz";
					DttSP.RXDisplayLow = -(int)DttSP.SampleRate/2;
					DttSP.RXDisplayHigh = (int)DttSP.SampleRate/2;
					tbFilterShift.Enabled = false;
					btnFilterShiftReset.Enabled = false;
					if(CurrentDisplayMode == DisplayMode.PANADAPTER)
						CurrentDisplayMode = DisplayMode.SPECTRUM;
					comboDisplayMode.Items.RemoveAt(1);
					break;
				case DSPMode.RTTY:
					radModeRTTY.BackColor = button_selected_color;
					grpMode.Text = "Mode - RTTY";					
					break;
				case DSPMode.PSK:
					radModePSK.BackColor = button_selected_color;
					grpMode.Text = "Mode - PSK";
					break;
				case DSPMode.DRM:
					if_shift = false;
					vfo_offset = -0.012;
					radModeDRM.BackColor = button_selected_color;
					grpMode.Text = "Mode - DRM";
					chkMOX.Enabled = false;
					DisableAllFilters();
					tbFilterShift.Enabled = false;
					btnFilterShiftReset.Enabled = false;
					grpFilter.Text = "Filter - 20KHz";
					DttSP.ChangeRXFilters(6000, 18000);
					DttSP.RXDisplayLow = -10000;
					DttSP.RXDisplayHigh = 10000;
					break;
			}

			current_dsp_mode = new_mode;
			if(current_dsp_mode == DSPMode.SPEC ||
				current_dsp_mode == DSPMode.DRM)
				CurrentFilter = Filter.NONE;
			else CurrentFilter = last_filter[(int)new_mode];

			tbWidthScroll_newMode(); // wjt 

			DrawBackground();
			txtVFOAFreq_LostFocus(this, new EventArgs());            
		}

		private void radModeLSB_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModeLSB.Checked)
			{
				ChangeMode(DSPMode.LSB);
			}
		}

		private void radModeUSB_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModeUSB.Checked)
			{
				ChangeMode(DSPMode.USB);
			}
		}

		private void radModeDSB_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModeDSB.Checked)
			{
				ChangeMode(DSPMode.DSB);
			}
		}

		private void radModeCWL_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModeCWL.Checked)
			{
				ChangeMode(DSPMode.CWL);
			}

		}

		private void radModeCWU_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModeCWU.Checked)
			{
				ChangeMode(DSPMode.CWU);
			}
		}

		private void radModeFMN_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModeFMN.Checked)
			{
				ChangeMode(DSPMode.FMN);
			}
		}

		private void radModeAM_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModeAM.Checked)
			{
				ChangeMode(DSPMode.AM);
			}
		}

		private void radModeSAM_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModeSAM.Checked)
			{
				ChangeMode(DSPMode.SAM);
			}
		}

		private void radModePSK_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModePSK.Checked)
			{
				ChangeMode(DSPMode.PSK);
			}
		}

		private void radModeSPEC_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModeSPEC.Checked)
			{
				ChangeMode(DSPMode.SPEC);
			}
		}

		private void radModeRTTY_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModeRTTY.Checked)
			{
				ChangeMode(DSPMode.RTTY);
			}
		}

		private void radModeDRM_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radModeDRM.Checked)
			{
				ChangeMode(DSPMode.DRM);
			}
		}

		#endregion

		#region Filter Button Events
// ======================================================
// Filter Button Events
// ======================================================

		private void ChangeFilter(Filter new_filter)
		{
			int low = 0, high = 0;
			last_filter[(int)current_dsp_mode] = new_filter;

			switch(current_filter)
			{
				case Filter.F6000:
					radFilter6000.BackColor = SystemColors.Control;
					break;
				case Filter.F4000:
					radFilter4000.BackColor = SystemColors.Control;
					break;
				case Filter.F2600:
					radFilter2600.BackColor = SystemColors.Control;
					break;
				case Filter.F2100:
					radFilter2100.BackColor = SystemColors.Control;
					break;
				case Filter.F1000:
					radFilter1000.BackColor = SystemColors.Control;
					break;
				case Filter.F500:
					radFilter500.BackColor = SystemColors.Control;
					break;
				case Filter.F250:
					radFilter250.BackColor = SystemColors.Control;
					break;
				case Filter.F100:
					radFilter100.BackColor = SystemColors.Control;
					break;
				case Filter.F50:
					radFilter50.BackColor = SystemColors.Control;
					break;
				case Filter.F25:
					radFilter25.BackColor = SystemColors.Control;
					break;
				case Filter.VAR1:
					radFilterVar1.BackColor = SystemColors.Control;
					udFilterLow.BackColor = SystemColors.Window;
					udFilterHigh.BackColor = SystemColors.Window;
					udFilterLow.Enabled = false;
					udFilterHigh.Enabled = false;
					break;
				case Filter.VAR2:
					radFilterVar2.BackColor = SystemColors.Control;
					udFilterLow.BackColor = SystemColors.Window;
					udFilterHigh.BackColor = SystemColors.Window;
					udFilterLow.Enabled = false;
					udFilterHigh.Enabled = false;
					break;
			}

			int saved_filter_shift = tbFilterShift.Value;
			tbFilterShift.Value = 0;
			last_filter_shift = 0;
			last_var1_shift = 0;
			last_var2_shift = 0;
//			btnFilterShiftReset.BackColor = SystemColors.Control;

			current_filter = new_filter;

			switch(new_filter)
			{
				case Filter.F6000:
					radFilter6000.BackColor = button_selected_color;
					grpFilter.Text = "Filter - 6.0KHz";
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
						case DSPMode.PSK:
						case DSPMode.RTTY:
							low = -6000; high = -100;
							break;
						case DSPMode.USB:
 							low = 100; high = 6000;
							break;
						case DSPMode.CWL:
							high = Math.Min(-cw_pitch + 3000, -100);
							low = high - 6000;
							break;
						case DSPMode.CWU:
							low = Math.Max(cw_pitch - 3000, 100);
							high = low + 6000;
							break;
						case DSPMode.DSB:
						case DSPMode.AM:
						case DSPMode.SAM:
						case DSPMode.FMN:
							low = -6000; high = 6000;
							break;
						default:
							low = -6000; high = 6000;
							break;
					}
					break;
				case Filter.F4000:
					radFilter4000.BackColor = button_selected_color;
					grpFilter.Text = "Filter - 4.0KHz";
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
						case DSPMode.PSK:
						case DSPMode.RTTY:
							low = -4000; high = -100;
							break;
						case DSPMode.USB:
							low = 100; high = 4000;
							break;
						case DSPMode.CWL:
							high = Math.Min(-cw_pitch + 2000, -100);
							low = high - 4000;
							break;
						case DSPMode.CWU:
							low = Math.Max(cw_pitch - 2000, 100);
							high = low + 4000;
							break;
						case DSPMode.DSB:
						case DSPMode.AM:
						case DSPMode.SAM:
						case DSPMode.FMN:
							low = -4000; high = 4000;
							break;
						default:
							low = -4000; high = 4000;
							break;
					}
					break;
				case Filter.F2600:
					radFilter2600.BackColor = button_selected_color;
					grpFilter.Text = "Filter - 2.6KHz";
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
						case DSPMode.PSK:
						case DSPMode.RTTY:
							low = -2800; high = -200;
							break;
						case DSPMode.USB:
							low = 200; high = 2800;
							break;
						case DSPMode.CWL:
							high = Math.Min(-cw_pitch + 1300, -100);
							low = high - 2600;
							break;
						case DSPMode.CWU:
							low = Math.Max(cw_pitch - 1300, 100);
							high = low + 2600;
							break;
						case DSPMode.DSB:
						case DSPMode.AM:
						case DSPMode.SAM:
						case DSPMode.FMN:
							low = -2600; high = 2600;
							break;
						default:
							low = -2600; high = 2600;
							break;
					}
					break;
				case Filter.F2100:
					radFilter2100.BackColor = button_selected_color;
					grpFilter.Text = "Filter - 2.1KHz";
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
						case DSPMode.PSK:
						case DSPMode.RTTY:
							low = -2300; high = -200;
							break;
						case DSPMode.USB:
							low = 200; high = 2300;
							break;
						case DSPMode.CWL:
							high = Math.Min(-cw_pitch + 1050, -100);
							low = high - 2100;
							break;
						case DSPMode.CWU:
							low = Math.Max(cw_pitch - 1050, 100);
							high = low + 2100;
							break;
						case DSPMode.DSB:
						case DSPMode.AM:
						case DSPMode.SAM:
						case DSPMode.FMN:
							low = -2100; high = 2100;
							break;
						default:
							low = -2100; high = 2100;
							break;
					}
					break;
				case Filter.F1000:
					radFilter1000.BackColor = button_selected_color;
					grpFilter.Text = "Filter - 1.0KHz";
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
						case DSPMode.PSK:
						case DSPMode.RTTY:
							low = -1200;
							high = -200;
							break;
						case DSPMode.USB:
							low = 200;
							high = 1200;
							break;
						case DSPMode.CWL:
							high = Math.Min(-cw_pitch + 500, -100);
							low = high - 1000;
							break;
						case DSPMode.CWU:
							low = Math.Max(cw_pitch - 500, 100);
							high = low + 1000;
							break;
						case DSPMode.DSB:
						case DSPMode.AM:
						case DSPMode.SAM:
						case DSPMode.FMN:
							low = -1000; high = 1000;
							break;
						default:
							low = -1000; high = 1000;
							break;
					}
					break;
				case Filter.F500:
					radFilter500.BackColor = button_selected_color;
					grpFilter.Text = "Filter - 500Hz";
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
						case DSPMode.PSK:
						case DSPMode.RTTY:
							low = -850; high = -350;
							break;
						case DSPMode.USB:
							low = 350; high = 850;
							break;
						case DSPMode.CWL:
							high = Math.Min(-cw_pitch + 250, -100);
							low = high - 500;
							break;
						case DSPMode.CWU:
							low = Math.Max(cw_pitch - 250, 100);
							high = low + 500;
							break;
						case DSPMode.DSB:
						case DSPMode.AM:
						case DSPMode.SAM:
						case DSPMode.FMN:
							low = -500; high = 500;
							break;
						default:
							low = -500; high = 500;
							break;
					}
					break;
				case Filter.F250:
					radFilter250.BackColor = button_selected_color;
					grpFilter.Text = "Filter - 250Hz";
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
						case DSPMode.PSK:
						case DSPMode.RTTY:
							low = -725; high = -475;
							break;
						case DSPMode.USB:
							low = 475; high = 725;
							break;
						case DSPMode.CWL:
							high = Math.Min(-cw_pitch + 125, -100);
							low = high - 250;
							break;
						case DSPMode.CWU:
							low = Math.Max(cw_pitch - 125, 100);
							high = low + 250;
							break;
						case DSPMode.DSB:
						case DSPMode.AM:
						case DSPMode.SAM:
						case DSPMode.FMN:
							low = -250; high = 250;
							break;
						default:
							low = -250; high = 250;
							break;
					}
					break;
				case Filter.F100:
					radFilter100.BackColor = button_selected_color;
					grpFilter.Text = "Filter - 100Hz";
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
						case DSPMode.PSK:
						case DSPMode.RTTY:
							low = -650; high = -550;
							break;
						case DSPMode.USB:
							low = 550; high = 650;
							break;
						case DSPMode.CWL:
							high = Math.Min(-cw_pitch + 50, -100);
							low = high - 100;
							break;
						case DSPMode.CWU:
							low = Math.Max(cw_pitch - 50, 100);
							high = low + 100;
							break;
						case DSPMode.DSB:
						case DSPMode.AM:
						case DSPMode.SAM:
						case DSPMode.FMN:
							low = -100; high = 100;
							break;
						default:
							low = -100; high = 100;
							break;
					}
					break;
				case Filter.F50:
					radFilter50.BackColor = button_selected_color;
					grpFilter.Text = "Filter - 50Hz";
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
						case DSPMode.PSK:
						case DSPMode.RTTY:
							low = -625; high = -575;
							break;
						case DSPMode.USB:
							low = 575; high = 625;
							break;
						case DSPMode.CWL:
							high = Math.Min(-cw_pitch + 25, -100);
							low = high - 50;
							break;
						case DSPMode.CWU:
							low = Math.Max(cw_pitch - 25, 100);
							high = low + 50;
							break;
						case DSPMode.DSB:
						case DSPMode.AM:
						case DSPMode.SAM:
						case DSPMode.FMN:
							low = -50; high = 50;
							break;
						default:
							low = -50; high = 50;
							break;
					}
					break;
				case Filter.F25:
					radFilter25.BackColor = button_selected_color;
					grpFilter.Text = "Filter - 25Hz";
					switch(current_dsp_mode)
					{
						case DSPMode.LSB:
						case DSPMode.PSK:
						case DSPMode.RTTY:
							low = -613; high = -587;
							break;
						case DSPMode.USB:
							low = 587; high = 613;
							break;
						case DSPMode.CWL:
							high = Math.Min(-cw_pitch + 13, -100);
							low = high - 25;
							break;
						case DSPMode.CWU:
							low = Math.Max(cw_pitch - 13, 100);
							high = low + 25;
							break;
						case DSPMode.DSB:
						case DSPMode.AM:
						case DSPMode.SAM:
						case DSPMode.FMN:
							low = -25; high = 25;
							break;
						default:
							low = -25; high = 25;
							break;
					}
					break;
				case Filter.VAR1:
					radFilterVar1.BackColor = button_selected_color;
					udFilterLow.BackColor = button_selected_color;
					udFilterHigh.BackColor = button_selected_color;
					grpFilter.Text = "Filter - Var1";
					udFilterLow.Enabled = udFilterHigh.Enabled = false;
					udFilterLow.Value = low = var1_low[(int)current_dsp_mode];
					udFilterHigh.Value = high = var1_high[(int)current_dsp_mode];		
					udFilterLow.Enabled = udFilterHigh.Enabled = true;
					DttSP.ChangeRXFilters(low, high);
					break;
				case Filter.VAR2:
					radFilterVar2.BackColor = button_selected_color;
					udFilterLow.BackColor = button_selected_color;
					udFilterHigh.BackColor = button_selected_color;
					grpFilter.Text = "Filter - Var2";
					udFilterLow.Enabled = udFilterHigh.Enabled = false;
					udFilterLow.Value = low = var2_low[(int)current_dsp_mode];
					udFilterHigh.Value = high = var2_high[(int)current_dsp_mode];	
					udFilterLow.Enabled = udFilterHigh.Enabled = true;
					DttSP.ChangeRXFilters(low, high);
					break;
			}

			if(new_filter != Filter.VAR1 &&
				new_filter != Filter.VAR2 &&
				new_filter != Filter.NONE)
			{
				DttSP.ChangeRXFilters(low, high);
				udFilterLow.Value = low;
				udFilterHigh.Value = high;
			}

			tbWidth_FilterWidthChanged(low, high); // wjt added -- tell the width control to adjust 

			pre_shift_filter_low = low;
			pre_shift_filter_high = high;

			FilterShiftValue = saved_filter_shift;
			tbFilterShift_Scroll(this, new EventArgs());

			DrawBackground();

			if(chkDisplayAVG.Checked)
				ResetDisplayAverage();

			if((high - low) > 250)
			{
				udXIT.Increment = 50;
				udRIT.Increment = 50;
			}
			else
			{
				udXIT.Increment = 5;
				udRIT.Increment = 5;
			}
		}

		private void radFilter6000_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilter6000.Checked)
			{
				ChangeFilter(Filter.F6000);
			}
		}

		private void radFilter4000_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilter4000.Checked)
				ChangeFilter(Filter.F4000);
		}

		private void radFilter2600_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilter2600.Checked)
				ChangeFilter(Filter.F2600);
		}

		private void radFilter2100_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilter2100.Checked)
				ChangeFilter(Filter.F2100);
		}

		private void radFilter1000_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilter1000.Checked)
				ChangeFilter(Filter.F1000);
		}

		private void radFilter500_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilter500.Checked)
				ChangeFilter(Filter.F500);
		}

		private void radFilter250_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilter250.Checked)
				ChangeFilter(Filter.F250);
		}

		private void radFilter100_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilter100.Checked)
				ChangeFilter(Filter.F100);
		}

		private void radFilter50_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilter50.Checked)
				ChangeFilter(Filter.F50);
		}

		private void radFilter25_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilter25.Checked)
				ChangeFilter(Filter.F25);
		}

		private void radFilterVar1_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilterVar1.Checked)
				ChangeFilter(Filter.VAR1);
		}

		private void radFilterVar2_CheckedChanged(object sender, System.EventArgs e)
		{
			if(radFilterVar2.Checked)
				ChangeFilter(Filter.VAR2);
		}

		private void udFilterLow_ValueChanged(object sender, System.EventArgs e)
		{
			if(udFilterLow.Enabled)
			{
				if(udFilterLow.Value >= udFilterHigh.Value-10)
					udFilterLow.Value = udFilterHigh.Value-10;
				else
				{
					int i = (int)current_dsp_mode;
					if(current_filter == Filter.VAR1)	// save variable filter values
					{
						if(tbFilterShift.Value != 0)
						{
							var1_low[i] = pre_shift_filter_low;
							var1_high[i] = pre_shift_filter_high;
						}
						else
						{
							var1_low[i] = (int)udFilterLow.Value;
							var1_high[i] = (int)udFilterHigh.Value;
						}
					}
					else if(current_filter == Filter.VAR2)
					{
						if(tbFilterShift.Value != 0)
						{
							var2_low[i] = pre_shift_filter_low;
							var2_high[i] = pre_shift_filter_high;
						}
						else
						{
							var2_low[i] = (int)udFilterLow.Value;
							var2_high[i] = (int)udFilterHigh.Value;
						}
					}
					DttSP.ChangeRXFilters((int)udFilterLow.Value, (int)udFilterHigh.Value);
					DrawBackground();
					if(chkDisplayAVG.Checked)
						ResetDisplayAverage();
				}						
				tbWidth_FilterWidthChanged((int)udFilterLow.Value, (int)udFilterHigh.Value); 
			}
			if(udFilterLow.Focused)
				btnHidden.Focus();		
		}

		private void udFilterHigh_ValueChanged(object sender, System.EventArgs e)
		{
			if(udFilterHigh.Enabled)
			{
				if(udFilterHigh.Value <= udFilterLow.Value+10)
					udFilterHigh.Value = udFilterLow.Value+10;
				else
				{
					int i = (int)current_dsp_mode;
					if(current_filter == Filter.VAR1)	// save variable filter values
					{
						if(tbFilterShift.Value != 0)
						{
							var1_low[i] = pre_shift_filter_low;
							var1_high[i] = pre_shift_filter_high;
						}
						else
						{
							var1_low[i] = (int)udFilterLow.Value;
							var1_high[i] = (int)udFilterHigh.Value;
						}
					}
					else if(current_filter == Filter.VAR2)
					{
						if(tbFilterShift.Value != 0)
						{
							var2_low[i] = pre_shift_filter_low;
							var2_high[i] = pre_shift_filter_high;
						}
						else
						{
							var2_low[i] = (int)udFilterLow.Value;
							var2_high[i] = (int)udFilterHigh.Value;
						}
					}
					DttSP.ChangeRXFilters((int)udFilterLow.Value, (int)udFilterHigh.Value);
					DrawBackground();
					if(chkDisplayAVG.Checked)
						ResetDisplayAverage();
				}
				tbWidth_FilterWidthChanged((int)udFilterLow.Value, (int)udFilterHigh.Value); 
			}
			if(udFilterHigh.Focused)
				btnHidden.Focus();
			

		}

		private void DoFilterShift(int shift, bool redraw)
		{
			// VK6APH: Does the Filter Shift function, alters the filter low and high frequency values 
			// as the Filter Shift slider is moved. We need to keep the last Filter Shift values
			// that the variable filters use since, unlike the other filters, there are 
			// no pre-set bandwidths that they can default to when the Filter Shift is 
			// turned off. These values are stored in the public variables last_var1_shift and
			// last_var2_shift. 
			int IFShift;
			int low;
			int high;
			int bandwidth;
			int max_shift = 9999;		// needed when using variable filters so we can't exceed +/- 10kHz DSP limits

			if(current_dsp_mode == DSPMode.SPEC ||
				current_dsp_mode == DSPMode.DRM)
				return;

			bandwidth = (int)Math.Abs(udFilterHigh.Value - udFilterLow.Value); // calculate current filter bandwidth 

			// set the maximum IF Shift depending on filter bandwidth in use 
			if (bandwidth > 800)
			{
				tbFilterShift.Maximum = 1000;  // max IF Shift +/- 1kHz for filters > 800Hz wide
				tbFilterShift.Minimum = -1000;
			}
			else 
			{
				tbFilterShift.Maximum = 500;	// max IF Shift +/- 500Hz for filters < 800Hz wide
				tbFilterShift.Minimum = -500;	
			}
			// calculate how far the IF Shift slider has moved
			// if we are using variable bandwidth filters need to use their last shift value
			if (current_filter == Filter.VAR1)
				IFShift = shift - last_var1_shift;
			else if (current_filter == Filter.VAR2)
				IFShift = shift - last_var2_shift;
			else
				IFShift = shift - last_filter_shift; 

			high = (int)Math.Min(udFilterHigh.Value + IFShift, max_shift);	// limit high shift to maximum value
			low = (int)Math.Max(udFilterLow.Value + IFShift, -max_shift);	// limit low shift to maximum value
			
			DttSP.ChangeRXFilters(low, high);					// select new filters
			udFilterLow.Value = low;							// display new low value 
			udFilterHigh.Value = high;							// display new high value
			if(redraw) DrawBackground();						// draw new background for updated filter values
	
			// store the last IF Shift applied for use next time
			if (current_filter == Filter.VAR1)
				last_var1_shift = last_var1_shift + IFShift;
			else if (current_filter == Filter.VAR2)
				last_var2_shift = last_var2_shift + IFShift;
			else 
				last_filter_shift = last_filter_shift + IFShift;				
			// show the IF Shift is active by setting the zero button colour
			if (shift != 0)
				btnFilterShiftReset.BackColor = button_selected_color;		
		}

		private void tbFilterShift_Scroll(object sender, System.EventArgs e)
		{
			DoFilterShift(tbFilterShift.Value, true);

			if(tbFilterShift.Focused)
				btnHidden.Focus();
		}

		private void btnFilterShiftReset_Click(object sender, System.EventArgs e)
		{
			tbFilterShift.Value = 0;
			btnFilterShiftReset.BackColor = SystemColors.Control;	// make button grey
			tbFilterShift_Scroll(this, new EventArgs()); 
		}

		#endregion

		#region VFO Button Events
// ======================================================
// VFO Button Events
// ======================================================

		private void btnVFOAtoB_Click(object sender, System.EventArgs e)
		{
			txtVFOBFreq.Text = txtVFOAFreq.Text;
			txtVFOBFreq_LostFocus(this, new EventArgs());
			vfob_dsp_mode = current_dsp_mode;
			vfob_filter = current_filter;
		}

		private void btnVFOBtoA_Click(object sender, System.EventArgs e)
		{
			CurrentDSPMode = vfob_dsp_mode;
			CurrentFilter = vfob_filter;
			txtVFOAFreq.Text = txtVFOBFreq.Text;
			txtVFOAFreq_LostFocus(this, new EventArgs());
		}

		private void btnVFOSwap_Click(object sender, System.EventArgs e)
		{
			string temp = txtVFOAFreq.Text;

			DSPMode mode = current_dsp_mode;
			Filter filter = current_filter;

			CurrentDSPMode = vfob_dsp_mode;
			CurrentFilter = vfob_filter;
			vfob_dsp_mode = mode;
			vfob_filter = filter;
			
			txtVFOAFreq.Text = txtVFOBFreq.Text;
			txtVFOBFreq.Text = temp;
			txtVFOAFreq_LostFocus(this, new EventArgs());
			txtVFOBFreq_LostFocus(this, new EventArgs());
		}

		private void chkVFOSplit_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkVFOSplit.Checked)
			{
				XITOn = false;
				chkVFOSplit.BackColor = button_selected_color;
				grpVFOB.Font = new Font("Microsoft Sans Serif", 8.25F, FontStyle.Bold);
				grpVFOB.ForeColor = Color.Red;
				txtVFOBFreq.ForeColor = Color.Red;
				txtVFOBBand.ForeColor = band_text_light_color;
			}
			else
			{
				chkVFOSplit.BackColor = SystemColors.Control;
				grpVFOB.Font = new Font("Microsoft Sans Serif", 8.25F, FontStyle.Regular);
				grpVFOB.ForeColor = Color.Black;
				txtVFOBFreq.ForeColor = vfo_text_dark_color;
				txtVFOBBand.ForeColor = band_text_dark_color;
			}
			// BT 04/2005
			// raises the serial io event
			// EventArgs sioe = new EventArgs();
			// OnNotifySIO(sioe);
		}

		private void chkXIT_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkXIT.Checked)
			{
				chkXIT.BackColor = button_selected_color;
				chkVFOSplit.Checked = false;
				chkRIT.Checked = false;
			}
			else
			{
				chkXIT.BackColor = SystemColors.Control;
			}

			if(chkMOX.Checked)
			{
				if(chkVFOSplit.Checked)
					txtVFOBFreq_LostFocus(this, new EventArgs());
				else
					txtVFOAFreq_LostFocus(this, new EventArgs());
			}
		}

		private void chkRIT_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkRIT.Checked)
			{
				chkRIT.BackColor = button_selected_color;
				XITOn = false;
			}
			else
				chkRIT.BackColor = SystemColors.Control;

			txtVFOAFreq_LostFocus(this, new EventArgs());
		}

		private void udRIT_ValueChanged(object sender, System.EventArgs e)
		{
			if(chkRIT.Checked && !chkMOX.Checked)
				txtVFOAFreq_LostFocus(this, new EventArgs());
			if(udRIT.Focused)
				btnHidden.Focus();
		}

		private void udXIT_ValueChanged(object sender, System.EventArgs e)
		{
			if(chkXIT.Checked && chkMOX.Checked)
			{
				if(chkVFOSplit.Checked)
					txtVFOBFreq_LostFocus(this, new EventArgs());
				else
					txtVFOAFreq_LostFocus(this, new EventArgs());
			}

			if(udXIT.Focused)
				btnHidden.Focus();
		}

		private void btnXITReset_Click(object sender, System.EventArgs e)
		{
			XITValue = 0;
		}

		private void btnRITReset_Click(object sender, System.EventArgs e)
		{
			RITValue = 0;
		}

		#endregion

		#region Memory Events
// ======================================================
// Memory Events
// ======================================================

		private void btnSave_Click(object sender, System.EventArgs e)
		{
			SaveMem save = new SaveMem(this);
			save.Show();
			save.Focus();
		}

		private void chkMemoryScanner_CheckedChanged(object sender, System.EventArgs e)
		{
			if(chkMemoryScanner.Checked)
			{
				chkMemoryScanner.BackColor = button_selected_color;
				radScannerChannel.Enabled = false;
				radScannerFreq.Enabled = false;

				udScannerDelay.BackColor = button_selected_color;

				if(radScannerFreq.Checked)
				{
					udScannerLow.BackColor = button_selected_color;
					udScannerHigh.BackColor = button_selected_color;
					udScannerStep.BackColor = button_selected_color;

					udScannerLow.Enabled = false;
					udScannerHigh.Enabled = false;
					udScannerStep.Enabled = false;

					VFOAFreq = (float)udScannerLow.Value;
				}
				else 
				{
					MemForm.ScanMemory(true);
				}

				chkMOX.Enabled = false;
								
				timer_scanner.Enabled = true;				
			}
			else
			{
				timer_scanner.Enabled = false;

				udScannerDelay.BackColor = SystemColors.Window;

				radScannerChannel.Enabled = true;
				radScannerFreq.Enabled = true;

				if(radScannerFreq.Checked)
				{
					udScannerLow.BackColor = Color.White;
					udScannerHigh.BackColor = Color.White;
					udScannerStep.BackColor = Color.White;

					udScannerLow.Enabled = true;
					udScannerHigh.Enabled = true;
					udScannerStep.Enabled = true;
				}

				chkMOX.Enabled = true;
				radScannerChannel.Enabled = true;
				radScannerFreq.Enabled = true;
				chkMemoryScanner.BackColor = SystemColors.Control;
			}
		}

		private void udScannerDelay_ValueChanged(object sender, System.EventArgs e)
		{
			timer_scanner.Interval = (int)udScannerDelay.Value;
			if(udScannerDelay.Focused)
				btnHidden.Focus();
		}

		private void btnMemoryQuickSave_Click(object sender, System.EventArgs e)
		{
			txtMemoryQuick.Text = txtVFOAFreq.Text;
			quick_save_mode = CurrentDSPMode;
			quick_save_filter = CurrentFilter;
		}

		private void btnMemoryQuickRestore_Click(object sender, System.EventArgs e)
		{
			CurrentDSPMode = quick_save_mode;
			txtVFOAFreq.Text = txtMemoryQuick.Text;
			txtVFOAFreq_LostFocus(this, new EventArgs());
			CurrentFilter = quick_save_filter;
		}

		private void udScannerLow_ValueChanged(object sender, System.EventArgs e)
		{
			if(udScannerLow.Focused)
				btnHidden.Focus();
		}

		private void udScannerHigh_ValueChanged(object sender, System.EventArgs e)
		{
			if(udScannerHigh.Focused)
				btnHidden.Focus();
		}

		private void udScannerStep_ValueChanged(object sender, System.EventArgs e)
		{
			if(udScannerStep.Focused)
				btnHidden.Focus();
		}

		#endregion

		#region Menu Events

		private void menuItem1_Click(object sender, System.EventArgs e)
		{
			if(SetupForm.IsDisposed)
				SetupForm = new Setup(this);
			SetupForm.Show();
			SetupForm.Focus();
		}

		private void menuItem2_Click(object sender, System.EventArgs e)
		{
			if(WaveForm.IsDisposed)
				WaveForm = new WaveControl(this);
			WaveForm.Show();
			WaveForm.Focus();
		}

		private void menuItem3_Click(object sender, System.EventArgs e)
		{
			if(current_dsp_mode == DSPMode.LSB)
				CurrentDSPMode = DSPMode.CWL;
			else if(current_dsp_mode == DSPMode.USB)
				CurrentDSPMode = DSPMode.CWU;

			if(current_dsp_mode != DSPMode.CWL &&
				current_dsp_mode != DSPMode.CWU)
			{
				MessageBox.Show("The radio must be in CWL or CWU mode in order to open the "+
					"CW Control Form.",
					"CW Error: Wrong Mode",
					MessageBoxButtons.OK,
					MessageBoxIcon.Error);
				return;
			}

			cw_key_mode = true;
			if(CWForm.IsDisposed)
				CWForm = new CW(this);
			CWForm.Show();
			CWForm.Focus();
		}

		#endregion


		
		// kd5tfd -- called from change filter to make the width control display the 
		// current filter width 
		// 
		private void tbWidth_FilterWidthChanged(int low, int high) 
		{ 
						
			// System.Console.WriteLine("fhi: " + udFilterHigh.Value + " flo: " + udFilterLow.Value);  	
			System.Console.WriteLine("fwc hi: " + high + " lo: " + low);  	
			// int width = (int)(udFilterHigh.Value = udFilterLow.Value); 

			int width; 
			if ( current_dsp_mode == DSPMode.AM || current_dsp_mode == DSPMode.SAM || current_dsp_mode  == DSPMode.FMN ) 
			// symetric width centered mode 
			{
				width = high; 
			}
			else 
			{ 
				width = high - low; 
				if ( width < 0 ) width = - width; 
			}

			// convert to log scale the control uses 
			double width_d = (double)width; 

			double ws_d; // double of value to feed scroller 		
			ws_d = Math.Exp((width_d - 10001.0)*(-9.2103/10000));
			int ws = (int)ws_d; 	
			ws = 10001 - ws;  // reverse sense 
			// ws = ws / 10; // scale from original 10k control 
			
			
			// make sure we don't hose the control 
			if ( ws < tbWidth.Minimum ) 
			{ 
				ws = tbWidth.Minimum; 
				System.Console.WriteLine("filter width min limited"); 
			}
			else if	( ws > tbWidth.Maximum ) 
			{ 
				ws = tbWidth.Maximum; 
				System.Console.WriteLine("filter width max limited"); 
			} 
			tbWidth.Value = ws; 
			System.Console.WriteLine("fwc width: " + width + "  ws: " + ws); 
			centerSave = 0; // reset our save center point 

		}

		

		private void tbWidthScroll_newMode() 
		{
			switch ( current_dsp_mode ) 
			{
				case DSPMode.SPEC: 
				case DSPMode.DRM: 
					tbWidth.Enabled = false; 
					break; 

				default: 
					tbWidth.Enabled = true; 
					break; 
			}
		}

		int centerSave = 0;  // remember the nominal center point 

		private void tbWidth_Scroll(object sender, System.EventArgs e)
		{

			if ( current_dsp_mode == DSPMode.DRM || current_dsp_mode == DSPMode.SPEC ) 
			{
				return;  // no good in this mode 
			}

			SelectVarFilter(); // make filter variable 

			// System.Console.WriteLine("tbWidth_Scroll!");   
			// System.Console.WriteLine("fhi: " + udFilterHigh.Value + " flo: " + udFilterLow.Value);  	
			
			// select filter width with a logarathmic scale 
			// 
			int width_scroll = tbWidth.Value;			
			// width_scroll = width_scroll * 10;     // orig control ranged to 10k, so scale up 
			width_scroll = 10001 - width_scroll;  // invert sense 			
			// double log_width_scroll = Math.Log((double)width_scroll, 10.0); 
			double log_width_scroll = Math.Log((double)width_scroll); 
			// scroller ranges 1-10000, so log ranges from 0 to 4 
			// full width is 10khz ... so width is log(scroller)/4 * 10 khz 
			// double width_d = 10001 - ((log_width_scroll/4.0)*(10000)); 
			double width_d = 10001 - ((log_width_scroll/9.2103)*(10000)); 
			int width = (int)width_d; 
			if ( width < 10) width = 10; // minimum width 
			else if ( width > 9999 ) width = 9999;  // maximum  width 

			int center = 0; 
			bool side_band_style_mode = false; 
			switch ( current_dsp_mode ) 
			{ 
				case DSPMode.CWL:  // sideband style modes 
				case DSPMode.LSB:
				case DSPMode.USB:
				case DSPMode.CWU: 
					side_band_style_mode = true; 
					int h = (int)udFilterHigh.Value; 
					int l = (int)udFilterLow.Value; 
					if ( current_dsp_mode  == DSPMode.LSB || current_dsp_mode == DSPMode.CWL ) 
					{ 
						int save_int; 
						save_int = l; 
						l = -h; 
						h = -save_int; 						
					} 
					center = ((h - l)/2) + l; 
					break; 


				case DSPMode.AM: 
				case DSPMode.SAM: 
				case DSPMode.DSB: 
					center = 0; 
					break; 


			} 
		
			int low; 
			int high; 

			low = center - width/2; 
			high = center + width/2; 


			if ( side_band_style_mode ) 
			{ 
				if ( low < 0 )  // low can't be less than zero -- adjust 				
				{ 			
					if ( (udFilterHigh.Value > 0 && udFilterLow.Value < 0) || 
						(udFilterHigh.Value < 0 && udFilterLow.Value > 0) )
					{
						// filter is already spanning dc (prob from of shfit), don't cap the bottom @ dc 
						centerSave = 0; 
					}
					else 
					{ 
						if ( centerSave == 0 ) 
						{	
							System.Console.WriteLine("setting centerSave: " + center); 
							centerSave = center; // remember the prior center 
						}
						int too_low = - low; 
						high = high + too_low; 
						low = 0; 
					} 
				}
				else if ( centerSave != 0 )  // we'd previously shifted the center ... see if we can shift back to it 
				{
					int new_low; 
					int new_hi; 
					new_low = centerSave - width/2; 
					new_hi = centerSave + width/2; 
					if ( new_low >= 0 )  // ok, we can go with this one as it fits 
					{ 
						System.Console.WriteLine("resotring prev center of: " + centerSave); 
						centerSave = 0; 
						low = new_low; 
						high = new_hi; 
					} 
					else  // adjust it keeping 
					{ 
						System.Console.WriteLine("not back to centerSave yet!"); 
						int too_low = - new_low; 
						low = 0; 
						high = new_hi + too_low; 					
					} 
		
				}

				if ( high - low <= 11 ) 
				{ 
					high = high + ( 11 - (high - low)) ; 
				} 
				

				if ( current_dsp_mode == DSPMode.LSB ||  // mirror image things for lower sideband
					current_dsp_mode == DSPMode.CWL ) 
				{
					int save_int = low; 
					low	= -high; 
					high = -save_int; 					
				}				
			}
			else  // symetric style mode 
			{ 
				low = - width; 
				high = width; 
			}


			
			System.Console.WriteLine("low: " + low + " high: " + high + " center: " + center + " width: " + width + " lws: " + log_width_scroll); 

			if ( high > 9999 ) 
			{ 
				high = 9999; 
			} 
			if ( low < -9999 ) 
			{ 
				low = -9999; 
			} 

			DttSP.ChangeRXFilters(low, high);					// select new filters
			udFilterLow.Value = low;							// display new low value 
			udFilterHigh.Value = high;							// display new high value
			DrawBackground();									// draw new background for updated filter values
					
		}



		// shift scale is percentage of current bandwidth to shift filter, with direction indicated by sign. 
		//              A value of 0 says to reset the shift to 0 
		private void DoFilterShift2(int shift_scale, bool redraw)
		{

			// kd5tfd edition 

			// VK6APH: Does the Filter Shift function, alters the filter low and high frequency values 
			// as the Filter Shift slider is moved. We need to keep the last Filter Shift values
			// that the variable filters use since, unlike the other filters, there are 
			// no pre-set bandwidths that they can default to when the Filter Shift is 
			// turned off. These values are stored in the public variables last_var1_shift and
			// last_var2_shift. 
			int shift_delta; 
			// int shift = 0; 
			int IFShift;
			int low;
			int high;
			int bandwidth;
			int max_shift = 9999;		// needed when using variable filters so we can't exceed +/- 10kHz DSP limits

			if(current_dsp_mode == DSPMode.SPEC ||
				current_dsp_mode == DSPMode.DRM)
				return;

			// limit shift scale to +/- 1-100 
			if ( shift_scale != 0 ) 
			{ 
				centerSave = 0; // throw away any saved center point as it does not make sense after a shift ... 
				                // perhaps we should shift it as well?    

				if ( shift_scale < -100 ) 
				{
					shift_scale = -100; 
				} 
				else if ( shift_scale > 100 ) 
				{
					shift_scale = 100; 
				}
			

				bandwidth = (int)Math.Abs(udFilterHigh.Value - udFilterLow.Value); // calculate current filter bandwidth 
				shift_delta = (int)(((double)bandwidth)/100);   // make base delta 1% of current bandwidth 
				shift_delta *= shift_scale;  // scale up base on input parm 
			
				IFShift = shift_delta; 

				System.Console.WriteLine("IFShift: " + IFShift); 
				// make sure IF shift does not exceed max/min 
				if ( udFilterHigh.Value + IFShift > max_shift ) 
				{
					IFShift = max_shift - (int)udFilterHigh.Value; 
					System.Console.WriteLine("ifshift max limited"); 
				}
				if ( udFilterLow.Value + IFShift < -max_shift ) 
				{
					IFShift = -max_shift - (int)udFilterLow.Value;
					System.Console.WriteLine("ifshift -max limited"); 
				}
				high = (int)udFilterHigh.Value + IFShift; 
				low = (int)udFilterLow.Value + IFShift; 

				// this does not work well as the stored value for shift value is not correct in the case of the shift 
				// being limited -- replace by code above 
				//	high = (int)Math.Min(udFilterHigh.Value + IFShift, max_shift);	// limit high shift to maximum value
				//	low = (int)Math.Max(udFilterLow.Value + IFShift, -max_shift);	// limit low shift to maximum value
			}
			else  // shift value is 0 -- reset 
			{ 
				int last_shift; 
				if ( current_filter == Filter.VAR1 ) 
				{
					last_shift = last_var1_shift;
				}
				else if ( current_filter == Filter.VAR2 ) 
				{ 
					last_shift = last_var2_shift; 
				} 
				else 
				{ 
					last_shift = last_filter_shift; 
				}
				high = (int)udFilterHigh.Value - last_shift; 
				low = (int)udFilterLow.Value - last_shift; 
				IFShift = 0; 						
			}
			
			DttSP.ChangeRXFilters(low, high);					// select new filters
			udFilterLow.Value = low;							// display new low value 
			udFilterHigh.Value = high;							// display new high value
			if(redraw) DrawBackground();						// draw new background for updated filter values
	
			if ( IFShift != 0 ) 
			{ 
				System.Console.WriteLine("IFShift store"); 
				// store the last IF Shift applied for use next time
				if (current_filter == Filter.VAR1) 
				{ 
					last_var1_shift = last_var1_shift + IFShift;
					//shift = last_var1_shift; 
				}
				else if (current_filter == Filter.VAR2) 
				{ 
					last_var2_shift = last_var2_shift + IFShift;
					//shift = last_var2_shift; 
				}
				else 
				{ 
					last_filter_shift = last_filter_shift + IFShift;				
					//shift = last_filter_shift; 
				}
			}
			else  // ifshift is 0 -- reset 
			{ 
				System.Console.WriteLine("IFShfit reset"); 
				if (current_filter == Filter.VAR1 ) 
				{
					last_var1_shift = 0; 
				}
				else if ( current_filter == Filter.VAR2 ) 
				{ 
					last_var2_shift = 0; 
				} 
				else 
				{ 
					last_filter_shift = 0; 
				} 
			}
			// show the IF Shift is active by setting the zero button colour
			//if (shift != 0)
			//	btnFilterShiftReset.BackColor = button_selected_color;		
		}

		private void SelectVarFilter() 
		{ 
			if ( current_filter == Filter.VAR1 ) return; 
			if ( current_filter == Filter.VAR2 ) return; 

			// save current filter bounds, reset to var, set filter bounds 
			decimal hi = udFilterHigh.Value;
			decimal lo = udFilterLow.Value; 
			radFilterVar1.Checked = true;
			ChangeFilter(Filter.VAR1); 
			udFilterHigh.Value = hi; 
			udFilterLow.Value = lo; 
		} 

		private void button_IFShiftDown_Click(object sender, System.EventArgs e)
		{
			SelectVarFilter(); 
			DoFilterShift2(-33, true); 		
		}

		private void button_IFShiftUp_Click(object sender, System.EventArgs e)
		{
			SelectVarFilter(); 
			DoFilterShift2(33, true); 	
		}

		private void button_IFShiftZero_Click(object sender, System.EventArgs e)
		{
			SelectVarFilter(); 
			DoFilterShift2(0, true); 					
		}

		private void button_IFShiftDownSmall_Click(object sender, System.EventArgs e)
		{
			SelectVarFilter(); 
			DoFilterShift2(-5, true); 
		}

		private void button_IFShiftUpSmall_Click(object sender, System.EventArgs e)
		{
			SelectVarFilter(); 
			DoFilterShift2(5, true); 
		}



		private void button_WidthDown_Click(object sender, System.EventArgs e)
		{
			SelectVarFilter(); 
			int new_val = tbWidth.Value - 50; 
			if ( new_val < tbWidth.Minimum ) 
			{
				new_val = tbWidth.Minimum;  
			}
			tbWidth.Value = new_val; 
			tbWidth_Scroll(sender, e); 		
		}

		private void button_WidthUp_Click(object sender, System.EventArgs e)
		{			
			SelectVarFilter(); 
			int new_val = tbWidth.Value + 50; 
			if ( new_val > tbWidth.Maximum ) 
			{
				new_val = tbWidth.Maximum; 
			}
			tbWidth.Value = new_val;			
			tbWidth_Scroll(sender, e); 
		}

		private void grpBand_Enter(object sender, System.EventArgs e)
		{
		
		}				
	}
}
