//=================================================================
// SIOListener.cs
//=================================================================
// Copyright (C) 2005  Bob Tracy
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// You may contact the author via email at: btracy@wimberley-tx.com
//=================================================================

#define DBG_PRINT

using System;
using System.Text;
using System.Collections;
using System.Threading;
using System.Windows.Forms; // needed for MessageBox (wjt) 

namespace PowerSDR
{
	/// <summary>
	/// Summary description for SIOListner.
	/// </summary>
	/// 

	// BT 04/2005 added delegate for serial IO event
	public delegate void SIOEventHandler(object sender, EventArgs e);

	public class SIOListener
	{
		#region Constructor

		public SIOListener(Console c)
		{
			console = c;
			console.Activated += new EventHandler(console_Activated);
			console.NotifySIO += new SIOEventHandler(sdr_NotifySIO);
			console.Closing += new System.ComponentModel.CancelEventHandler(console_Closing);

			if ( console.CATEnabled )  // if CAT is on fire it up 
			{ 
				try 
				{ 
					enableCAT();  
				}
				catch ( Exception ex ) 
				{					
					// fixme??? how cool is to to pop a msg box from an exception handler in a constructor ?? 
					//  seems ugly to me (wjt) 
					console.CATEnabled = false; 
					if ( console.SetupForm != null ) 
					{ 
						console.SetupForm.copyCATPropsToDialogVars(); // need to make sure the props on the setup page get reset 
					}
					MessageBox.Show("Could not initialize CAT control.  Exception was:\n\n " + ex.Message + 
						"\n\nCAT control has been disabled.", "Error Initializing CAT control", 
						MessageBoxButtons.OK, MessageBoxIcon.Error);
				}
			}
		}

		public void enableCAT() 
		{

			lock ( this ) 
			{
				if ( cat_enabled ) return; // nothing to do already enabled 
				cat_enabled = true; 
			}
			int port_num = console.CATPort; 
			SIO = new SDRSerialSupport.SDRSerialPort(port_num);
			if ( console.CATisMixWVirtPort ) 
			{
				SIO.setVirtual(true); 
			}
			else 
			{ 
				SIO.setCommParms(console.CATBaudRate, console.CATParity, console.CATDataBits, console.CATStopBits); 
			}

			++rxThreadNum; // bump counter so we know we have an RXthread running 
			rxThread = new Thread(new ThreadStart(RXMonitor));
			rxThread.Name = "SIOListener.rxMonitor #" + rxThreadNum; 
			rxThread.IsBackground = true; 
			rxThread.Start();

			Initialize();			
		}

		// typically called when the end user has disabled CAT control through a UI element ... this 
		// kills the recvThread, closes the serial port and neutralized the listeners we have in place
		public void disableCAT() 
		{
			lock ( this ) 
			{ 
				if ( !cat_enabled )  return; /* nothing to do already disabled */ 
				cat_enabled = false; 
			}

			++rxThreadNum; // this will stop the RxThread; 
			rxThread = null;
			if ( SIO != null ) 
			{
				SIO.Destroy(); 
				SIO = null; 
			}
			Fpass = true; // reset init flag 
			return; 									
		}

		#endregion Constructor

		#region variables

		SDRSerialSupport.SDRSerialPort SIO; 
		Console console;
		ASCIIEncoding AE = new ASCIIEncoding();
//		private bool IDSent = false;
		private int mode;
		private bool Fpass = true;
		private bool cat_enabled = false;  // is cat currently enabled by user? 
		Thread rxThread;
		private int rxThreadPollTime = 200;
		// private bool rxThreadRun = true;
		private int rxThreadNum = 0;  // id number of the currently running RXthread ... when the thread is stopped we increment 
		                              // the number.  The running copy of rxthread stores this as a local when it starts and keeps 
		                              // running as long as the local stored copy mathces the instance copy.  This allows us to kill 
		                              // and immediately start a new RxThread without having to wait for the prior running copy to die
		CAT cat = new CAT();

		#endregion variables

		#region Methods

		// Converts a vfo frequency to a proper CAT frequency string
		private string StrVFOFreq(string vfo)
		{
			// Added 05/01/05 BT for globalization
			string separator = System.Globalization.CultureInfo.CurrentCulture.NumberFormat.NumberDecimalSeparator;

			double freq;
			string cmd_string = "";

			if(vfo == "A")
				freq = console.VFOAFreq;
			else
				freq = console.VFOBFreq;
			
			if((int) freq < 1)
			{
				cmd_string += "00000"+freq.ToString();
			}
			else if((int) freq < 10)
			{
				cmd_string += "0000"+freq.ToString();
			}
			else if((int) freq < 100)
			{
				cmd_string += "000"+freq.ToString();
			}
			else if((int) freq < 1000)
			{
				cmd_string += "00"+freq.ToString();
			}
			else
			{
				cmd_string += "0"+freq.ToString();
			}
			// Get rid of the decimal separator and pad the right side with 0's 
			// Modified 05/01/05 BT for globalization
			if(cmd_string.IndexOf(separator) > 0)
				cmd_string = cmd_string.Remove(cmd_string.IndexOf(separator),1);
			cmd_string = cmd_string.PadRight(11,'0');
			return cmd_string;
		}

		private static void dbgWriteLine(string s) 
		{ 
#if(!DBG_PRINT) 
			Console.dbgWriteLine("SIOListener: " + s); 
#endif
		}

		private void CmdLookup(string cmd)
		{
			// Added 05/01/05 BT for globalization
			string separator = System.Globalization.CultureInfo.CurrentCulture.NumberFormat.NumberDecimalSeparator;

			// Split out the command string prefix and suffix
			string prefix = cmd.Substring(0,2);
			string suffix = cmd.Substring(2);
			// Get rid of the semicolon on the end
			suffix = suffix.Remove(suffix.Length-1,1);
			// dbgWriteLine("CmdLookup >"  + cmd + "<"); 

			switch(prefix)
			{
				case "FA":
					// If it is a frequency command, insert the decimal place
					// and send the frequency to the vfo.
					if(suffix.Length == 11)
					{
						// Modified 05/01/05 BT for globalization
						suffix = suffix.Insert(5,separator);
						console.VFOAFreq = double.Parse(suffix);
					}
					else
					{
						byte[] cmd_id1 = AE.GetBytes("FA" + StrVFOFreq("A") + ";");
						uint cnt1 = 14;
						SIO.put(cmd_id1,cnt1);					
					}
					break;
				case "FB":
					if(cmd.Length == 11)
					{
						// Modified 05/01/05 BT for globalization
						suffix = suffix.Insert(5,separator);
						console.VFOBFreq = double.Parse(suffix);
					}
					else
					{
						byte[] cmd_id2 = AE.GetBytes("FB" + StrVFOFreq("B") + ";");
						uint cnt2 = 14;
						SIO.put(cmd_id2,cnt2);		
					}
					break;
				case "FR":
					break;
				case "FT":
//					console.VFOSplitEnabled = true;
					break;
				case "IF":
					sdr_NotifySIO(this, new EventArgs());
					break;
				case "MD":
					if(suffix == "")		// received a mode status request
						sdr_NotifySIO(this, new EventArgs());
					else if(suffix == "1")
						console.CurrentDSPMode = DSPMode.LSB;
					else if(suffix == "2")
						console.CurrentDSPMode = DSPMode.USB;
					else if(suffix == "3")
						console.CurrentDSPMode = DSPMode.CWU;
					else if(suffix == "4")
						console.CurrentDSPMode = DSPMode.FMN;
					else if(suffix == "5")
						console.CurrentDSPMode = DSPMode.AM;
					break;
				case "RX":
					console.MOX = false;
					// dbgWriteLine("Mox set to false"); 
					break;
				case "TX":
					console.MOX = true;
					// dbgWriteLine("Mox set to true"); 
					break;

				case "ID":
					string id = "ID019;";
					byte[] cmd_id = AE.GetBytes(id);
					uint cnt = 6;
					SIO.put(cmd_id,cnt);
//					sdr_NotifySIO(this, new EventArgs());
					break;
				case "SH":
					console.FilterHighValue = GetFilterValue("SH", suffix);
					// console.ChangeFilter(Filter.VAR1);
					console.CurrentFilter = Filter.VAR1;
					break;
				case "SL":
					console.FilterLowValue = GetFilterValue("SL", suffix);
					// console.ChangeFilter(Filter.VAR1);
					console.CurrentFilter = Filter.VAR1;
					break;

		        case "ZZ":   // kd5tfd extension to Kenwod CAT vocab 
					processKD5TFDextendedCAT(suffix); 
					break; 					

				case "SM":
					byte[] cmd_id5 = AE.GetBytes("SM" + "00000" + ";");
					uint cnt5 = 8;
					SIO.put(cmd_id5,cnt5);	
					break;

				default:
					dbgWriteLine("CAT Ignore >" + cmd + "<"); 
					break;
			}
		}

		// 
		// kd5tfd's extensions to cat 
		// 
		// ZZSFccccwwww  Set Filter, cccc=center freq www=width both in hz 
		// 
		//
		// note -- this thing expects the ZZ to be parsed off first 
		private void processKD5TFDextendedCAT(string zz_suffix) 
		{ 
			string suffix = zz_suffix.Substring(0,2); 
			string args = zz_suffix.Substring(2); 

			switch ( suffix ) 
			{ 
				case "SF": 
					int center = Convert.ToInt32(args.Substring(0,4), 10); 
					int width = Convert.ToInt32(args.Substring(4), 10); 
					SetFilterCenterAndWidth(center, width); 
					break; 

					
					// filter width 
					// ZZLW+NNN -- relative increment ... NNN is somewhat arbitrary, is num steps of the log width ctl
					// ZZLW-NNN -- relative increment  
					// ZZLWNNNN -- absolute setting (not implemented) 


				case "LW":
					if ( args[0] == '+' || args[0] == '-' )  //
					{
						int incr = Convert.ToInt32(args, 10); 
						// make scroller update -- wonder of the goodnes of wacking 
						// a UI control from the CAT thread? 
						System.Console.WriteLine("lw incr: " + incr); 
						console.doFilterWidthIncrement(incr, null, null); 						
					}
					else 
					{ 
						// wjt?? todo implement absolute form 
					} 
					break; 

			    // ZZIF+NNN
				// ZZIF-NNN
				// ZZIF0000 - set zero if shift 
			    // change IF +/- NNN % 
				case "IF":					
					if ( args[0] == '+' || args[0] == '-' || args[0] == '0' )  //
					{
						int incr = Convert.ToInt32(args, 10); 
						System.Console.WriteLine("if incr: " + incr); 
						// make scroller update -- wonder of the goodnes of wacking 
						// a UI control from the CAT thread? 
						console.doIFShiftIncrement(incr); 						
					}
					else 
					{	
						// wjt?? todo implement absolute form 
					} 
					break;

					
			    default: 
					break; 
			} 
		} 

		// set variable filter 1 to indicate center and width 
		//  
		// assumes filter center is given w/o  sigm, so swizzles, sign appropriately 
		// for lsb 
		// 
		// if either center or width is zero, current value of center or width is 
		// contained 
		// fixme ... what should this thing do for am, fm, dsb ... ignore width? 
		private void SetFilterCenterAndWidth(int center, int width) 
		{ 
			int new_lo; 
			int new_hi; 
			
			
			switch ( console.CurrentDSPMode ) 
			{
				// check for modes that don't make sense or are not implemented 
				case DSPMode.SPEC: 				
				case DSPMode.DRM:
				case DSPMode.PSK:
				case DSPMode.RTTY:
					return; 

				// for lower sideband adjust center to be correct center
				case DSPMode.LSB: 
				case DSPMode.CWL: 
					center = -center; 
					break; 
					
			}
			if  ( center == 0 && width == 0 )  // nothing to do 
			{ 
				return; // nothing to do 
			} 
			if ( width < 0 )  // makes no sense, bail out to do no damage
			{ 
				return; 
			} 
			if ( center == 0 )  // use current center 
			{ 
				center = (console.FilterHighValue - console.FilterLowValue)/2 + console.FilterLowValue; 
			}
			if ( width == 0 )  // use current width 
			{
				width = console.FilterHighValue - console.FilterLowValue; 		
			}

						
			System.Console.WriteLine("center: " + center  + " width: " + width); 
			new_lo = center - (width/2); 
			new_hi = center + (width/2); 							

			// see if this is a sideband mode, if so make sure we don't move filter below dc 
			// unles the current filter is already below dc 
						
			switch ( console.CurrentDSPMode ) 
			{
				case DSPMode.USB: 
				case DSPMode.CWU: 
					if ( console.FilterLowValue < 0 ) 
					{
						break; // filter already below dc, no need to limit 
					}
					// make sure we don't put new filter below dc 
					if ( new_lo < 0  ) 
					{
						new_lo = 0; 
					}
					break; 
					

					
				case DSPMode.LSB:
				case DSPMode.CWL:
					if ( console.FilterHighValue > 0 ) 
					{
						break;  // filtr is aleady spanning dc, don't limit 
					}
					if ( new_hi > 0 ) 
					{
						new_hi = 0; 
					}
					break; 
					
			}
			
//			// new_lo and new_hi calculated assuming a USB mode .. do the right thing 
//			// for lsb and other modes 
//			// fixme -- needs more thinking 
//			switch ( console.CurrentDSPMode ) 
//			{ 
//				case DSPMode.LSB: 
//				case DSPMode.CWL:
//					int scratch = new_hi; 
//					new_hi = -new_lo; 
//					new_lo = -scratch; 
//					break; 
//
//				case DSPMode.AM: 
//				case DSPMode.SAM: 
//				case DSPMode.DSB:
//				case DSPMode.FMN:
//					new_lo = -new_hi; 
//					break; 
//			} 			
			console.CurrentFilter = Filter.VAR1; 
			console.FilterLowValue = new_lo; 
			console.FilterHighValue = new_hi; 
			return; 
		} 

		private int GetFilterValue(string filter, string index)
		{
			int freq = 0;
			switch(index)
			{
				case "00":
					if(filter == "SL")
						freq = 10;
					else
						freq = 1400;
					break;
				case "01":
					if(filter == "SL")
						freq = 50;
					else
						freq = 1600;
					break;
				case "02":
					if(filter == "SL")
						freq = 100;
					else
						freq = 1800;
					break;
				case "03":
					if(filter == "SL")
						freq = 200;
					else
						freq = 2000;
					break;
				case "04":
					if(filter == "SL")
						freq = 300;
					else
						freq = 2200;
					break;
				case "05":
					if(filter == "SL")
						freq = 400;
					else
						freq = 2400;
					break;
				case "06":
					if(filter == "SL")
						freq = 500;
					else
						freq = 2600;
					break;
				case "07":
					if(filter == "SL")
						freq = 600;
					else
						freq = 2800;
					break;
				case "08":
					if(filter == "SL")
						freq = 700;
					else
						freq = 3000;
					break;
				case "09":
					if(filter == "SL")
						freq = 800;
					else
						freq = 3400;
					break;
				case "10":
					if(filter == "SL")
						freq = 900;
					else
						freq = 4000;
					break;
				case "11":
					if(filter == "SL")
						freq = 1000;
					else
						freq = 5000;
					break;
				default:
					break;
			}
			return freq;
		}

		// Called when the console is activated for the first time.  Keeps the
		// ports from polling while the console is initializing.
		private void Initialize()
		{		 

			if(Fpass)
			{
				SIO.Create();
				sdr_NotifySIO(this, new EventArgs());
				Fpass = false;
			}
		}

		private void RXMonitor()
		{
			int my_thread_num = rxThreadNum; 

			byte[] rxData = new byte[100];
			System.Console.WriteLine("SIOListener.RXMonitor alive and well"); 
			while( rxThreadNum == my_thread_num )
			{								
				uint numread; 
				try 
				{ 
					if ( (numread = SIO.get(rxData, (uint)100)) != 0 )  // wjt fixme ... small race condition here when cat is disabled 
						// on the other thread SIO could end up being null .. window is small 
						// as we check that thread is still running .. prob ok to just catch 
						// the exception and bail out of the thread if we take a null reference
						// exception in here 
					{ 
						ParseString(rxData, numread);
					}
					Thread.Sleep(rxThreadPollTime);  
				}
				catch ( NullReferenceException ) 
				{
					System.Console.WriteLine("SIOLisener -- NullReferenceException ignored"); 
					// bail out 
					rxThread = null; 
					++rxThreadNum; 
					my_thread_num = 0; 

				}				
			}
			System.Console.WriteLine("SIOListener.RXMonitor dies"); 
		}

		private char[] ParseLeftover = null; 

		// segment incoming string into CAT commands ... handle leftovers from when we read a parial 
		// 
		private void ParseString(byte[] rxdata, uint count) 
		{ 
	        if ( count == 0 ) return;  // nothing to do 
			int cmd_char_count = 0; 
			int left_over_char_count = ( ParseLeftover == null ? 0 : ParseLeftover.Length ); 
			char[] cmd_chars = new char[count + left_over_char_count]; 			
			if ( ParseLeftover != null )  // seed with leftovers from last read 
			{ 
				for ( int j = 0; j < left_over_char_count; j++ )  // wjt fixme ... use C# equiv of System.arraycopy 
				{
					cmd_chars[cmd_char_count] = ParseLeftover[j]; 
					++cmd_char_count; 
				}
				ParseLeftover = null; 
			}
			for ( int j = 0; j < count; j++ )   // while we have chars to play with 
			{ 
				cmd_chars[cmd_char_count] = (char)rxdata[j]; 
				++cmd_char_count; 
				if ( rxdata[j] == ';' )  // end of cmd -- parse it and execute it 
				{ 
					string cmdword = new String(cmd_chars, 0, cmd_char_count); 
					dbgWriteLine("cmdword: >" + cmdword + "<");  
					if ( cat.IsValidCAT(cmdword) ) 
					{
						CmdLookup(cmdword); // has side effect of executing 
					}
					else 
					{ 
						dbgWriteLine("invalid cmdword >" + cmdword + "<"); 
					} 
					cmd_char_count = 0; // reset word counter 
				}
			} 
			// when we get here have processed all of the incoming buffer, if there's anyting 
			// in cmd_chars we need to save it as we've not pulled a full command so we stuff 
			// it in leftover for the next time we come through 
			if ( cmd_char_count != 0 ) 
			{ 
				ParseLeftover = new char[cmd_char_count]; 
				for ( int j = 0; j < cmd_char_count; j++ )  // wjt fixme ... C# equiv of Sytsem.arraycopy 
				{
					ParseLeftover[j] = cmd_chars[j]; 
				}
			} 
#if DBG_PRINT
			if ( ParseLeftover != null) 
			{
				dbgWriteLine("Leftover >" + new String(ParseLeftover) + "<"); 
			}
#endif
			return; 
		}

//		private void ParseString(byte[] rxdata, uint count)
//		{
//			string data = AE.GetString(rxdata, 0, (int)count);
//			if(data != null & data.IndexOf(";") > 0)
//			{
//				dbgWriteLine("data is >" + data + "<"); 
//				string word = data.Substring(0,data.IndexOf(";")+1);
//				dbgWriteLine("word is >" + word + "<"); 
//				if(cat.IsValidCAT(word))
//				{
//					CmdLookup(word);
//				}
//			}
//
//
//		}



		#endregion Methods

		#region Events

		private void sdr_NotifySIO(object sender, EventArgs e)
		{
			if ( !cat_enabled ) return;  // do nothing if cat control not enabled. 

			// Initalize the command word, Pn refers to the Kenwood TS2000 CAT table
			string cmd_string = "IF";
			// Get VFOA's frequency with the correct padding (P1 - 11 bytes)
			cmd_string += StrVFOFreq("A");

			// Get the step size index (P2 - 4 bytes)
			cmd_string += console.StepSize.ToString().PadLeft(4,'0');

			// Determine which incremental tuning control is active
			int ITValue = 0;
			if(console.RITOn)
				ITValue = console.RITValue;
			else
				ITValue = console.XITValue;

			// Add the polarity and value to the command string (P3 - 6 bytes)
			if(ITValue >= 0)
				cmd_string += "+";
			else
				cmd_string += "-";
			cmd_string += ITValue.ToString().PadLeft(5,'0');

			// Add the flag for RIT/XIT on/off (P4 & P5, 1 byte each)
			if(!console.RITOn & !console.XITOn)
				cmd_string += "00";
			else if(console.RITOn)
				cmd_string += "01";
			else
				cmd_string += "10";

			// Skip the memory channel stuff, the SDR1K doesn't use banks and channels per se
			// (P6 - 1 byte, P7 - 2 bytes)
			cmd_string += "000";

			// Set the current MOX state (P8 - 1 byte)(what the heck is this for?)
			if(console.MOX)
				cmd_string += "1";
			else
				cmd_string += "0";

			// Get the SDR mode.  There are more than 9 SDR modes, some have to be
			// defaulted to AM (or whatever we want the default to be) (P9 - 1 byte)
			mode = 0;
			switch(console.CurrentDSPMode.ToString())
			{
				case "LSB":
					mode = 1;
					break;
				case "USB":
					mode = 2;
					break;
				case "CWU":
					mode = 3;
					break;
				case "FM":
					mode = 4;
					break;
				case "AM":
					mode = 5;
					break;
				case "RTTY":
					mode = 6;
					break;
				case "CWL":
					mode = 7;
					break;
				case "DSB":
					mode = 8;
					break;
				case "SAM":
					mode = 9;
					break;
				default:
					mode = 5;
					break;
			}
			cmd_string += mode.ToString();

			// Set the FR/FT commands which determines the transmit and receive
			// VFO's. VFO A = 0, VFO B = 1. (P10 - 1 byte)
			//			if(console.VFOSplit)
			//				cmd_string += "1";
			//			else
			cmd_string += "0";

			// Set the Scan code to 0 
			// The Scan code might be implemented but the frequency range would
			// have to be manually entered. (P11 - 1 byte)
			cmd_string += "0";

			// Set the Split operation code (P12 - 1 byte)
			// Only apparent effect is to set Commander split check box.
			//			if(console.VFOSplit)
			//				cmd_string += "1";
			//			else
			cmd_string += "0";

			// Set the remaining CTCSS tone and shift bits to 0 and terminate the string (P13 - P15, 4 bytes)
			cmd_string += "0000;";
			byte[] out_string = AE.GetBytes(cmd_string);
			uint result = SIO.put(out_string, (uint) out_string.Length);

			/*
			// Send VFO B frequency
			cmd_string = "FB"+StrVFOFreq("B")+";";
			out_string = AE.GetBytes(cmd_string);
			result = SIO.put(out_string, (uint) out_string.Length);
			*/

		}

		private void console_Closing(object sender, System.ComponentModel.CancelEventArgs e)
		{

			// rxThreadRun = false;
			++rxThreadNum;  // causes running thread to die 
			if ( SIO != null ) 
			{ 
				SIO.Destroy(); 
			}
		}

		private void console_Activated(object sender, EventArgs e)
		{
			if ( console.CATEnabled ) 
			{ 
				// Initialize();   // wjt enable CAT calls Initialize 
				enableCAT(); 
			}
		}
		#endregion Events
	}
}
