//=================================================================
// dsp.cs
//=================================================================
// PowerSDR is a C# implementation of a Software Defined Radio.
// Copyright (C) 2004, 2005  FlexRadio Systems
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
// You may contact us via email at: sales@flex-radio.com.
// Paper mail may be sent to: 
//    FlexRadio Systems, 8900 Marybank Dr., Austin, TX  78750, USA.
//=================================================================


//#define NEW_KEYER
using System;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Diagnostics;

namespace PowerSDR
{
	unsafe class DttSP
	{
		#region Dll Variable and Method Definition
		// ======================================================
		// DLL Variable and Method Definition
		// ======================================================

		public enum Mode
		{
			FIRST = -1,
			LSB,
			USB,
			DSB,
			CWL,
			CWU,
			FMN,
			AM,
			PSK,
			SPEC,
			RTTY,
			SAM,
			DRM,
			LAST,
		}

		public enum MeterType
		{
			SIGNAL_STRENGTH = 0, 
			AVG_SIGNAL_STRENGTH,  
			ADC_REAL, 
			ADC_IMAG, 
			AGC_GAIN,
			ALC,
			PWR,
			PKPWR,
		}

		public enum TransmitState
		{
			OFF = 0,
			ON = 1,
		}

		public enum Window
		{
			FIRST = -1,
			RECTANGULAR,
			HANNING,
			WELCH,
			PARZEN,
			BARTLETT,
			HAMMING,
			BLACKMAN2,
			BLACKMAN3,
			BLACKMAN4,
			EXPONENTIAL,
			RIEMANN,
			BLKHARRIS,
			LAST,
		}


		[DllImport("DttSP.dll", EntryPoint="setup_sdr")]
		private static extern void SetupSDR();

		[DllImport("DttSP.dll", EntryPoint="resize_sdr")]
		public static extern void ResizeSDR(int dspsize);

		[DllImport("DttSP.dll", EntryPoint="destroy_sdr")]
		public static extern void Exit();


		[DllImport("DttSP.dll", EntryPoint="process_samples_thread")]
		public static extern void BeginProcess();

		[DllImport("DttSP.dll", EntryPoint="audioreset")]
		public static extern void AudioReset();

		[DllImport("DttSP.dll", EntryPoint="calculate_autocor")]
		public static extern void CalculateAutoCorr(void *input_l, void *input_r,int size);

#if (NEW_KEYER)
		[DllImport("DttSP.dll", EntryPoint="cwtimerfired")]
		public static extern void CWTimerFired();

		[DllImport("DttSP.dll", EntryPoint="StartKeyer")]
		public static extern void StartKeyer();

		[DllImport("DttSP.dll", EntryPoint="StopKeyer")]
		public static extern void StopKeyer();

		[DllImport("DttSP.dll", EntryPoint="key_thread_process")]
		public static extern void KeyValue(double del, int dash, int dot);

		[DllImport("DttSP.dll", EntryPoint="NewKeyer")]
		public static extern void NewKeyer(double freq, int iambic, double gain, double ramp, double wpm, double SampleRate);

		[DllImport("DttSP.dll", EntryPoint="KeyerClockFireRelease")]
		public static extern void KeyerClockFireRelease();

		[DllImport("DttSP.dll", EntryPoint="KeyerPlaying")]
		public static extern int KeyerPlaying();

		[DllImport("DttSP.dll", EntryPoint="KeyerClockFireWait")]
		public static extern void KeyerClockFireWait();

		[DllImport("DttSP.dll", EntryPoint="KeyerStartedRelease")]
		public static extern void KeyerStartedRelease();

		[DllImport("DttSP.dll", EntryPoint="KeyerStartedWait")]
		public static extern void KeyerStartedWait();

		[DllImport("DttSP.dll", EntryPoint="PollTimerRelease")]
		public static extern void PollTimerRelease();

		[DllImport("DttSP.dll", EntryPoint="PollTimerWait")]
		public static extern void PollTimerWait();

		[DllImport("DttSP.dll", EntryPoint="delKeyer")]
		public static extern void DeleteKeyer();

		[DllImport("DttSP.dll", EntryPoint="sound_thread_keyd")]
		public static extern void KeyerSoundThread();

		[DllImport("DttSP.dll", EntryPoint="CWtoneExchange")]
		public static extern void CWtoneExchange(float *bufl,float*bufr,int nframes);

#endif

		[DllImport("DttSP.dll", EntryPoint="audio_callback")]
		public static extern void ExchangeSamples(void *input_l, void *input_r, void *output_l, void *output_r, int numsamples);

		[DllImport("DttSP.dll", EntryPoint="changeMode")]
		private static extern int ChangeMode(Mode m);

		[DllImport("DttSP.dll", EntryPoint="changeFilter")]
		private static extern int ChangeFilter(double low, double high, int taps, TransmitState trx_on);

		[DllImport("DttSP.dll", EntryPoint="changeOsc")]
		public static extern int ChangeOsc(double freq);

		[DllImport("DttSP.dll", EntryPoint="changeTXOsc")]
		public static extern int ChangeTXOsc(double freq);

		[DllImport("DttSP.dll", EntryPoint="changeSampleRate")]
		private static extern int ChangeSampleRate(double sampleRate);

		[DllImport("DttSP.dll", EntryPoint="changeNR")]
		public static extern void ChangeNR(int setit);

		[DllImport("DttSP.dll", EntryPoint="changeNRvals")]
		public static extern void ChangeNRvals(int taps, int delay, double gain, double leak);

		[DllImport("DttSP.dll", EntryPoint="changeANF")]
		public static extern void ChangeANF(int setit);

		[DllImport("DttSP.dll", EntryPoint="changeANFvals")]
		public static extern void ChangeANFvals(int taps, int delay, double gain, double leak);

		[DllImport("DttSP.dll", EntryPoint="changeAGC")]
		public static extern void ChangeAGC(AGCMode setit);

		[DllImport("DttSP.dll", EntryPoint="changeTXAGC")]
		public static extern void ChangeTXAGC(int setit);

		[DllImport("DttSP.dll", EntryPoint="changeTXAGCLimit")]
		public static extern void ChangeTXAGCLimit(int setit);

		[DllImport("DttSP.dll", EntryPoint="changeTXAGCFF")]
		public static extern void ChangeTXAGCFF(int setit);

		[DllImport("DttSP.dll", EntryPoint="changeTXAGCCompression")]
		public static extern void ChangeTXAGCCompression(double txcompression);

		[DllImport("DttSP.dll", EntryPoint="changeTXAGCFFCompression")]
		public static extern void ChangeTXAGCFFCompression(double txcompression);

		[DllImport("DttSP.dll", EntryPoint="changeTXAGCHang")]
		public static extern void ChangeTXAGCHang(int txhang);

		[DllImport("DttSP.dll", EntryPoint="setTXScale")]
		public static extern void SetTXPostScale(double scale);

		[DllImport("DttSP.dll", EntryPoint="setTXPreScale")]
		public static extern void SetTXPreScale(double scale);

		[DllImport("DttSP.dll", EntryPoint="setTXPreGain")]
		public static extern void SetTXPreGain(double db_scale);

		[DllImport("DttSP.dll", EntryPoint="setRXScale")]
		public static extern void SetRXScale(double scale);

		[DllImport("DttSP.dll", EntryPoint="changeDCBlock")]
		public static extern void ChangeDCBlock(int setit);


		[DllImport("DttSP.dll", EntryPoint="oldsetTXEQ")]
		public static extern void SetTXEQ(int[] txeq);


		[DllImport("DttSP.dll", EntryPoint="changeNB")]
		public static extern void ChangeNB(int setit);

		[DllImport("DttSP.dll", EntryPoint="changeNBvals")]
		public static extern void ChangeNBvals(double threshold);

		[DllImport("DttSP.dll", EntryPoint="changecorrectIQgain")]
		public static extern void ChangeIQgain(double setit);

		[DllImport("DttSP.dll", EntryPoint="changecorrectTXIQgain")]
		public static extern void ChangeTXIQgain(double setit);

		[DllImport("DttSP.dll", EntryPoint="changecorrectTXIQphase")]
		public static extern void ChangeTXIQphase(double setit);

		[DllImport("DttSP.dll", EntryPoint="changeSDROM")]
		public static extern void ChangeSDROM(int setit);

		[DllImport("DttSP.dll", EntryPoint="changeSDROMvals")]
		public static extern void ChangeSDROMvals(double threshold);

		[DllImport("DttSP.dll", EntryPoint="changefixedAGC")]
		public static extern void ChangeFixedAGC(double fixed_agc);

		[DllImport("DttSP.dll", EntryPoint="changePosition")]
		public static extern void ChangePosition(double pos);

		[DllImport("DttSP.dll", EntryPoint="changemaxAGC")]
		public static extern void ChangeAGCMaxGain(double max_agc);

		[DllImport("DttSP.dll", EntryPoint="changeRXAGCLimit")]
		public static extern void ChangeRXAGCLimit(int setit);

		[DllImport("DttSP.dll", EntryPoint="changeWindow")]
		public static extern void ChangeWindow(Window windowset);

		[DllImport("DttSP.dll", EntryPoint="changeBIN")]
		public static extern void ChangeBIN(int setit);

		[DllImport("DttSP.dll", EntryPoint="changeSquelchVal")]
		public static extern void ChangeSquelchVal(float setit);

		[DllImport("DttSP.dll", EntryPoint="changeSquelchState")]
		public static extern void ChangeSquelchState(int state);

		[DllImport("DttSP.dll", EntryPoint="changePWSmode")]
		public static extern void ChangePWSmode(int setit);

		[DllImport("DttSP.dll", EntryPoint="changecorrectIQphase")]
		public static extern void ChangeIQphase(double setit);

		[DllImport("DttSP.dll", EntryPoint="process_spectrum")]
		unsafe public static extern void GetSpectrum(float* results);

		[DllImport("DttSP.dll", EntryPoint="process_phase")]
		unsafe public static extern void GetPhase(float* results, int numpoints);

		[DllImport("DttSP.dll", EntryPoint="process_scope")]
		unsafe public static extern void GetScope(float* results, int numpoints);


		[DllImport("DttSP.dll", EntryPoint="oldsetTRX")]
		unsafe public static extern void SetTRX(TransmitState trx_on);

		[DllImport("DttSP.dll", EntryPoint="calculate_meters")]
		public static extern float CalculateMeter(MeterType MT);


		[DllImport("DttSP.dll", EntryPoint="release_update")]
		unsafe public static extern void ReleaseUpdate();
 
		[DllImport("DttSP.dll", EntryPoint="suspend_sdr")]
		unsafe public static extern void Suspend();


		#endregion

		#region Properties
// ======================================================
// Properties
// ======================================================

		private static double rx_volume;
		public static double RXVolume
		{
			get { return rx_volume; }
			set
			{
				rx_volume = value;
				SetRXScale(rx_volume);
			}
		}

		private static double tx_volume;
		public static double TXVolume
		{
			get
			{
				return tx_volume; 
			}
			set
			{
				//Debug.WriteLine("TX Volume: "+value.ToString("f3"));
				value = Math.Max(0, value);
				value = Math.Min(1.0, value);
				tx_volume = value;
				SetTXPostScale(tx_volume * pa_high_swr_scale);
			}
		}

		private static double pa_high_swr_scale = 1.0;
		public static double PAHighSWRScale
		{
			get { return pa_high_swr_scale; }
			set
			{
				//Debug.WriteLine("SWR Scale: "+value.ToString("f3"));
				value = Math.Max(0, value);
				value = Math.Min(1.0, value);
				pa_high_swr_scale = value;
				SetTXPostScale(tx_volume * pa_high_swr_scale);
			}
		}

		private static int block_size;
		public static int BlockSize
		{
			get { return block_size; }
		}

		private static Mode current_mode;
		public static Mode CurrentMode
		{
			get	{ return current_mode;}
			set 
			{
				current_mode = value;
                int i = ChangeMode(current_mode);
				if(i != 0)
					MessageBox.Show("Error in DttSP.ChangeMode: "+i);
			}
		}

		private static Window current_window = Window.HANNING;
		public static Window CurrentWindow
		{
			get { return current_window; }
			set
			{
				current_window = value;
				ChangeWindow(current_window);
			}
		}

		private static int rx_display_low = -4000;
		public static int RXDisplayLow
		{
			get { return rx_display_low; }
			set { rx_display_low = value; }
		}

		private static int rx_display_high = 4000;
		public static int RXDisplayHigh
		{
			get { return rx_display_high; }
			set { rx_display_high = value; }
		}

		private static int tx_display_low = -4000;
		public static int TXDisplayLow
		{
			get { return tx_display_low; }
			set { tx_display_low = value; }
		}

		private static int tx_display_high = 4000;
		public static int TXDisplayHigh
		{
			get { return tx_display_high; }
			set { tx_display_high = value; }
		}

		private static int rx_filter_low_cut = 200;
		public static int RXFilterLowCut
		{
			get { return rx_filter_low_cut;}
			set 
			{
				rx_filter_low_cut = value;
				UpdateRXDisplayVars();

				int i = ChangeFilter(rx_filter_low_cut, rx_filter_high_cut, filter_taps, 0);
				if(i != 0)
					MessageBox.Show("Error in DttSP.ChangeFilter(RXFilterLowCut): "+i);
			}
		}

		private static int rx_filter_high_cut = 2800;
		public static int RXFilterHighCut
		{
			get { return rx_filter_high_cut;}
			set 
			{
				rx_filter_high_cut = value;
				UpdateRXDisplayVars();

				int i = ChangeFilter(rx_filter_low_cut, rx_filter_high_cut, filter_taps, 0);
				if(i != 0)
					MessageBox.Show("Error in DttSP.ChangeFilter(RXFilterHighCut): "+i);
			}
		}

		private static int tx_filter_low_cut = 300;
		public static int TXFilterLowCut
		{
			get { return tx_filter_low_cut;}
			set 
			{
				tx_filter_low_cut = value;
				UpdateTXDisplayVars();

				int i = ChangeFilter(tx_filter_low_cut, tx_filter_high_cut, filter_taps, TransmitState.ON);
				if(i != 0)
					MessageBox.Show("Error in DttSP.ChangeFilter(TXFilterLowCut): "+i);
			}
		}

		private static int tx_filter_high_cut = 3000;
		public static int TXFilterHighCut
		{
			get { return tx_filter_high_cut;}
			set 
			{
				tx_filter_high_cut = value;
				UpdateTXDisplayVars();

				int i = ChangeFilter(tx_filter_low_cut, tx_filter_high_cut, filter_taps, TransmitState.ON);
				if(i != 0)
					MessageBox.Show("Error in DttSP.ChangeFilter(TXFilterHighCut): "+i);
			}
		}

		private static double sample_rate;
		public static double SampleRate
		{
			get { return sample_rate;}
			set
			{
				sample_rate = value;
				int i = ChangeSampleRate(sample_rate);
				if(i != 0)
					MessageBox.Show("Error in DttSP.ChangeSampleRate: "+i);
			}
		}

		private static int filter_taps;
		public static int FilterTaps
		{
			get { return filter_taps;}
			set
			{
				filter_taps = value;
				int i = ChangeFilter(rx_filter_low_cut, rx_filter_high_cut, filter_taps, 0);
				if(i != 0)
					MessageBox.Show("Error in DttSP.ChangeFilter(FilterTaps): "+i);
			}
		}

		private static double second_if;
		public static double SecondIF
		{
			get { return second_if; }
			set
			{
				second_if = value;
				int i = ChangeOsc(second_if);
				if(i != 0)
					MessageBox.Show("Error in ChangeOsc(SecondIF): "+i);
			}
		}

		#endregion

		#region Misc Routines
// ======================================================
// Misc Routines
// ======================================================

		public static void Init()
		{
			block_size = 2048;
			SetupSDR();
			ReleaseUpdate();
			rx_filter_low_cut = 8175;
			rx_filter_high_cut = 10775;
			filter_taps = BlockSize;
			SampleRate = 48000.0;
			SecondIF = -11025.0;
		}

		public static void ChangeRXFilters(int low, int high)
		{
			rx_filter_low_cut = low;
			rx_filter_high_cut = high;
			UpdateRXDisplayVars();

			int i = ChangeFilter(rx_filter_low_cut, rx_filter_high_cut, filter_taps, TransmitState.OFF);
			if(i != 0)
				MessageBox.Show("Error in DttSP.ChangeRXFilters (ChangeFilter): "+i);
		}

		public static void ChangeTXFilters(int low, int high)
		{
			switch(CurrentMode)
			{
				case Mode.LSB:
				case Mode.CWL:
					tx_filter_low_cut = -high;
					tx_filter_high_cut = -low;
					break;
				case DttSP.Mode.USB:
				case DttSP.Mode.CWU:
					tx_filter_low_cut = low;
					tx_filter_high_cut = high;
					break;
				case DttSP.Mode.DSB:
					tx_filter_low_cut = -high;
					tx_filter_high_cut = high;
					break;
				case DttSP.Mode.AM:
				case DttSP.Mode.SAM:
					tx_filter_low_cut = -high;
					tx_filter_high_cut = high;
					break;
				case DttSP.Mode.FMN:
					tx_filter_low_cut = -6000;
					tx_filter_high_cut = 6000;
					break;
			}
			UpdateTXDisplayVars();

			int i = ChangeFilter(tx_filter_low_cut, tx_filter_high_cut, filter_taps, TransmitState.ON);
			if(i != 0)
				MessageBox.Show("Error in DttSP.ChangeTXFilters (ChangeFilter): "+i);
		}

		private static void UpdateRXDisplayVars()
		{
			if(rx_filter_low_cut < 0 && rx_filter_high_cut <= 0)
			{
				rx_display_high = 0;
				if(rx_filter_low_cut >= -910)
					rx_display_low = -1000;
				else
					rx_display_low = (int)(rx_filter_low_cut*1.1);
			}
			else if(rx_filter_low_cut >= 0 && rx_filter_high_cut > 0)
			{
				rx_display_low = 0;
				if(rx_filter_high_cut <= 910)
					rx_display_high = 1000;
				else
					rx_display_high = (int)(rx_filter_high_cut*1.1);
			}
			else if(rx_filter_low_cut < 0 && rx_filter_high_cut > 0)
			{
				int max_edge = Math.Max(-rx_filter_low_cut, rx_filter_high_cut);
				rx_display_low = (int)(max_edge*-1.1);
				rx_display_high = (int)(max_edge*1.1);
			}	
		}

		private static void UpdateTXDisplayVars()
		{
			if(tx_filter_low_cut < 0 && tx_filter_high_cut <= 0)
			{
				tx_display_high = 0;
				if(tx_filter_low_cut >= -910)
					tx_display_low = -1000;
				else
					tx_display_low = (int)(tx_filter_low_cut*1.1);
			}
			else if(tx_filter_low_cut >= 0 && tx_filter_high_cut > 0)
			{
				tx_display_low = 0;
				if(tx_filter_high_cut <= 910)
					tx_display_high = 1000;
				else
					tx_display_high = (int)(tx_filter_high_cut*1.1);
			}
			else if(tx_filter_low_cut < 0 && tx_filter_high_cut > 0)
			{
				int max_edge = Math.Max(-tx_filter_low_cut, tx_filter_high_cut);
				tx_display_low = (int)(max_edge*-1.1);
				tx_display_high = (int)(max_edge*1.1);
			}			
		}

		#endregion
	}
}